﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Globalization;
using System.Linq;
using System.Reflection;
using Produmex.Foundation.Data.DbClient;
using Produmex.Foundation.Data.Sbo.BusinessObjects.Definitions.Tables;
using Produmex.Foundation.Diagnostics;
using Produmex.Foundation.SlimScreen;
using Produmex.Foundation.Workflows;
using Produmex.Foundation.Workflows.Parameters;
using Produmex.Foundation.Wwf.Sbo.LocalServices;
using Produmex.Sbo.Logex.Data.BusinessObjects;
using Produmex.Sbo.Logex.Data.BusinessObjects.Definitions.Tables;
using Produmex.Sbo.Logex.Data.DataObjects;
using Produmex.Sbo.Logex.ProLaser.BusinessObjects.Definitions.Tables;

// PmxWorkflowExecutionTypes.HOOK_FLOW
namespace Produmex.Sbo.Logex.WorkflowScripts
{
	/// <summary>
	/// Hook flow for after a Logistic unit is received.
	/// This flow is called after the print, but BEFORE it is actually added to the database!
	/// </summary>
	/// <seealso cref="Produmex.Foundation.Workflows.WorkflowInstanceScriptBase" />
	public class WorkflowScript_AfterLogisticUnitReceivedHookScript : WorkflowInstanceScriptBase
	{
		private static readonly ILog s_log = LogProvider.GetLogger(MethodInfo.GetCurrentMethod().DeclaringType);

		// Input parameters * do not change *
		public ReadOnlyBinder<CultureInfo> DefaultCultureInfo;
		public ReadOnlyBinder<PmxOseCompany> PmxOseCompany;
		public ReadWriteBinder<List<LogisticUnitGoodsReceipt>> LogisticUnits;

		public WorkflowScript_AfterLogisticUnitReceivedHookScript(WorkflowInstanceBase parent, WorkflowInstanceFactory factory)
			: base(parent, factory)
		{
		}

		#region WorkflowInstanceScriptBase Members
		protected override void Execute()
		{
			// Parameters in scope
			Session session = GetScopeParameter("Session") as Session;
			ISboProviderService sboProviderService = GetScopeParameter("<WwfService>ISboService") as ISboProviderService;

			foreach (LogisticUnitGoodsReceipt logisticUnitGoodsReceipt in LogisticUnits.Get())
			{
				if (logisticUnitGoodsReceipt.LogisticUnitId.HasValue)
				{
					string cardCode = GetCardCodeFromASNByLuid(logisticUnitGoodsReceipt.LogisticUnitId.Value, sboProviderService);

					if (!string.IsNullOrEmpty(cardCode))
					{
						bool printItemLabel = GetPrintItemLabelFlagByCardCode(cardCode, sboProviderService);

						if (printItemLabel)
						{
							foreach (LogisticUnitItemGoodReceipt itemOnLogisticUnit in logisticUnitGoodsReceipt.ItemsOnLogisticUnit)
							{
								double numberOfItems = itemOnLogisticUnit.Quantity * itemOnLogisticUnit.QuantityPerUom;
								int numberOfBoxes = 0;

								if (itemOnLogisticUnit.ListOfPackagingTypes != null && itemOnLogisticUnit.ListOfPackagingTypes.Count > 0)
								{
									double minPackagingTypeInfoQuantity = itemOnLogisticUnit.ListOfPackagingTypes.Min(pti => pti.Quantity);
									numberOfBoxes = Convert.ToInt32(Math.Ceiling(numberOfItems / minPackagingTypeInfoQuantity));
								}

								int numberOfItemLabels = Convert.ToInt32(Math.Ceiling(numberOfItems)) + numberOfBoxes;
								PrintItemLabels(itemOnLogisticUnit, numberOfItemLabels);
							}
						}
					}
				}
			}
		}

		private class PrintItemLabelFlowData
		{
			public bool BackRequested;
			public string LocationCode;
			public string NeededItemCode;
			public string NeededBatchNumber;
			public string NeededBatchNumber2;
			public DateTime? NeededBestBeforeDate;
			public int? NeededNrOfLabels;
			public bool AskBatchInfo;
			public double? EnteredQuantity;
			public bool ShowPrintMessage;
		}

		private void PrintItemLabels(LogisticUnitItemGoodReceipt itemOnLogisticUnit, int numberOfItemLabels)
		{
			bool next;

			do
			{
				PrintItemLabelFlowData printData = new PrintItemLabelFlowData()
				{
					BackRequested = false,
					ShowPrintMessage = false,
					LocationCode = itemOnLogisticUnit.OverrideLocationCode,
					NeededBatchNumber = itemOnLogisticUnit.BatchNumber1,
					NeededBatchNumber2 = itemOnLogisticUnit.BatchNumber2,
					NeededBestBeforeDate = itemOnLogisticUnit.BestBeforeDate,
					NeededItemCode = itemOnLogisticUnit.ItemCode,
					NeededNrOfLabels = numberOfItemLabels,
					AskBatchInfo = false,
					EnteredQuantity = itemOnLogisticUnit.Quantity
				};

				WorkflowInstanceBase printItemLabelFlow = Create("PrintItemLabelScript");

				printItemLabelFlow.BindParameters(
					DefaultCultureInfo,
					PmxOseCompany,
					new ReadOnlyBinder<string>(nameof(printData.LocationCode), printData, nameof(printData.LocationCode)),
					new ReadOnlyBinder<string>(nameof(printData.NeededItemCode), printData, nameof(printData.NeededItemCode)),
					new ReadOnlyBinder<string>(nameof(printData.NeededBatchNumber), printData, nameof(printData.NeededBatchNumber)),
					new ReadOnlyBinder<string>(nameof(printData.NeededBatchNumber2), printData, nameof(printData.NeededBatchNumber2)),
					new ReadOnlyBinder<DateTime?>(nameof(printData.NeededBestBeforeDate), printData, nameof(printData.NeededBestBeforeDate)),
					new ReadOnlyBinder<int?>(nameof(printData.NeededNrOfLabels), printData, nameof(printData.NeededNrOfLabels)),
					new ReadOnlyBinder<bool>(nameof(printData.AskBatchInfo), printData, nameof(printData.AskBatchInfo)),
					new ReadWriteBinder<bool>(nameof(printData.BackRequested), printData, nameof(printData.BackRequested)),
					new ReadWriteBinder<double?>(nameof(printData.EnteredQuantity), printData, nameof(printData.EnteredQuantity)),
					new ReadWriteBinder<bool>(nameof(printData.ShowPrintMessage), printData, nameof(printData.ShowPrintMessage))
				);

				StartSubFlowAndWait(printItemLabelFlow, true);

				next = !printData.BackRequested;

			} while (!next);
		}

		private static bool GetPrintItemLabelFlagByCardCode(string cardCode, ISboProviderService sboProviderService)
		{
			bool result = false;

			string query = BuildQuery.GetPrintItemLabelFlagByCardCode(cardCode, sboProviderService);
			DataSet ds = sboProviderService.RunView(true, null, null, query);

			if (GetNumberOfRows(ds) > 0)
			{
				result = Convert.ToString(ds.Tables[0].Rows[0][PmxBusinessPartnerTablePL.Columns.PrintItemLabels.NAME]) == SboBooleanColumnValues.Y.VALUE;
			}

			return result;
		}

		private static string GetCardCodeFromASNByLuid(int logisticUnitId, ISboProviderService sboProviderService)
		{
			string result = null;

			string query = BuildQuery.GetCardCodeFromASNByLuid(logisticUnitId, sboProviderService);
			DataSet ds = sboProviderService.RunView(true, null, null, query);

			if (GetNumberOfRows(ds) > 0)
			{
				result = Convert.ToString(ds.Tables[0].Rows[0][PmxAdvanceShippingNoticeTable.Columns.CardCode.NAME]);
			}

			return result;
		}

		private static int GetNumberOfRows(DataSet ds)
		{
			if (ds != null && ds.Tables.Count > 0)
			{
				return ds.Tables[0].Rows.Count;
			}

			return 0;
		}
		#endregion

		#region Query
		private static class BuildQuery
		{
			public static string GetCardCodeFromASNByLuid(int logisticUnitId, ISboProviderService sboProviderService)
			{
				DbQueryBuilder qb = sboProviderService.GetDbTool().GetQueryBuilder();

				qb = qb.Select
					.EscColumnName(PmxAdvanceShippingNoticeTable.Columns.CardCode.NAME)
					.From
					.EscTableName(PmxAdvanceShippingNoticeTable.NAME)
					.WithNoLock
					.Where
					.EscColumnName(PmxAdvanceShippingNoticeTable.Columns.LogUnitIdentKey.NAME)
					.Eq
					.Append(logisticUnitId);

				return qb.ToString();
			}

			public static string GetPrintItemLabelFlagByCardCode(string cardCode, ISboProviderService sboProviderService)
			{
				DbQueryBuilder qb = sboProviderService.GetDbTool().GetQueryBuilder();

				qb = qb.Select
					.EscColumnName(PmxBusinessPartnerTablePL.Columns.PrintItemLabels.NAME)
					.From
					.EscTableName(PmxBusinessPartnerTablePL.NAME)
					.WithNoLock
					.Where
					.EscColumnName(PmxBusinessPartnerTablePL.Columns.CardCode.NAME)
					.Eq
					.String_(cardCode);

				return qb.ToString();
			}
		}
		#endregion
	}
}
