﻿using System.Globalization;
using System.Reflection;
using Produmex.Foundation.Data.Sbo.DataObjects;
using Produmex.Foundation.Diagnostics;
using Produmex.Foundation.Workflows;
using Produmex.Foundation.Workflows.Parameters;
using Produmex.Sbo.Logex.Data.BusinessObjects;
using System.Collections.Generic;
using Produmex.Sbo.Logex.Data.ViewObjects.Definitions.Views;
using System.Text;
using Produmex.Foundation.Data.SqlClient;
using Produmex.Sbo.Logex.Data.BusinessObjects.Definitions.Tables;
using Produmex.Foundation.Data.Sbo.BusinessObjects.Definitions.Tables;
using System;
using Produmex.Foundation.Data.Sbo;
using Produmex.Foundation.Wwf.Sbo.LocalServices;
using System.Collections.ObjectModel;
using Produmex.Foundation.Data.DbClient;
using System.Data;
using Produmex.Foundation.Data;
using Produmex.Sbo.Logex.ProLaser.BusinessObjects.Definitions.Tables;
using Produmex.Foundation.SlimScreen;
using Produmex.Foundation.Collections;

// PmxWorkflowExecutionTypes.HOOK_FLOW
namespace WorkflowScript_BeforeFinishLUIDPackingHookScript
{
	/// <summary>
	/// This is a hook script, meant to be customized on a "per project" basis
	/// </summary>
	public class WorkflowScript_BeforeFinishLUIDPackingHookScript : WorkflowInstanceScriptBase
	{
		private static double DEFAULT_TOLERANCE = 5;

		private static readonly ILog s_log = LogProvider.GetLogger(MethodInfo.GetCurrentMethod().DeclaringType);

		// Input parameters * do not change *
		public ReadOnlyBinder<CultureInfo> DefaultCultureInfo;
		public ReadOnlyBinder<PmxOseCompany> PmxOseCompany;
		public ReadOnlyBinder<List<int>> LUIDs;
		public ReadOnlyBinder<Collection<int>> ListOfPickListEntries;

		// Output parameters * do not change *
		public ReadWriteBinder<bool> BackRequested;

		// Sub-flows
		// <none>

		private static PmxDictionary<string, double> UomsInMiligrams = new PmxDictionary<string, double>();
		private static PmxDictionary<string, double> LogisticCarriersTolernce = new PmxDictionary<string, double>();

		/// <summary>
		/// Initializes a new instance of the <see cref="WorkflowScript_BeforeFinishLUIDPackingHookScript"/> class.
		/// </summary>
		/// <param name="parent">The parent.</param>
		/// <param name="factory">The factory.</param>
		public WorkflowScript_BeforeFinishLUIDPackingHookScript(WorkflowInstanceBase parent, WorkflowInstanceFactory factory)
			: base(parent, factory)
		{
		}

		#region WorkflowInstanceScriptBase Members

		protected override void Execute()
		{
			// Parameters in scope
			Session session = GetScopeParameter("Session") as Session;
			ISboProviderService sboProviderService = GetScopeParameter("<WwfService>ISboService") as ISboProviderService;

			List<int> luids = LUIDs.Get();
			string errorMessage = null;

			foreach (int luid in luids)
			{			
				string theoreticalWeightQuery = BuildQuery.GetQuerySumTheoreticalWeigtOfItemsOnLuid(luid, sboProviderService.GetDbTool());

				double? theoreticalWeight = GetDataFromQuery(sboProviderService, theoreticalWeightQuery);

				string luidInfoQuery = BuildQuery.GetQueryLuidDetails(luid, sboProviderService.GetDbTool());
				DataSet dsLuidInfo = sboProviderService.RunView(false, null, null, luidInfoQuery);

				double uomInMilligrams = 1;
				double? weight = null;
				string uom = null;
				string logisticCarrierCode;
				double logisticCarrierTolerance = 0;

				if (this.GetNumberOfRows(dsLuidInfo) > 0)
				{
					weight = Convertor.ConvertToDoubleNullable(dsLuidInfo.Tables[0].Rows[0][PmxLogisticUnitIDTable.Columns.Weight.NAME]);
					uom = dsLuidInfo.Tables[0].Rows[0][PmxLogisticUnitIDTable.Columns.WeightUom.NAME] as string;
					logisticCarrierCode = dsLuidInfo.Tables[0].Rows[0][PmxLogisticUnitIDTable.Columns.LogisticCarrierCode.NAME] as string;

					if (string.IsNullOrWhiteSpace(uom))
					{
						errorMessage = $"UoM is not set for luid {luid}";
					}

					if (string.IsNullOrWhiteSpace(logisticCarrierCode))
					{
						errorMessage = $"Logistic carrier is not set for luid {luid}";
					}

					if (errorMessage == null)
					{
						if (!UomsInMiligrams.TryGetValue(uom, out uomInMilligrams))
						{
							string weightInMgQuery = BuildQuery.GetConversionToMiligrams(uom, sboProviderService.GetDbTool());

							DataSet dsWeightInMiligram = sboProviderService.RunView(false, null, null, weightInMgQuery);

							if (this.GetNumberOfRows(dsWeightInMiligram) > 0)
							{
								uomInMilligrams = Convertor.ConvertToDouble(dsWeightInMiligram.Tables[0].Rows[0][0]);

								UomsInMiligrams.Add(uom, uomInMilligrams);
							}
							else
							{
								errorMessage = $"UoM \"{uom}\" not found in OWGT";
							}
						}

						if (!LogisticCarriersTolernce.TryGetValue(logisticCarrierCode, out logisticCarrierTolerance))
						{
							string toleranceQuery = BuildQuery.GetToleranceOfLogisticCarrer(logisticCarrierCode, sboProviderService.GetDbTool());

							DataSet dsLogisticCarrierTolerance = sboProviderService.RunView(false, null, null, toleranceQuery);

							if (this.GetNumberOfRows(dsLogisticCarrierTolerance) > 0)
							{
								double? tolerance = Convertor.ConvertToDoubleNullable(dsLogisticCarrierTolerance.Tables[0].Rows[0][0]);

								logisticCarrierTolerance = tolerance.HasValue ? tolerance.Value : DEFAULT_TOLERANCE;

								LogisticCarriersTolernce.Add(logisticCarrierCode, logisticCarrierTolerance);
							}
							else
							{
								errorMessage = $"Logistic carrier \"{logisticCarrierCode}\" not found";
							}
						}
					}
				}
				else
				{
					errorMessage = $"LUID \"{luid}\" not found";
				}

				if (errorMessage != null)
				{
					BackRequested.Set(true);

					// Inform the user with the error
					session.ShowScreen(typeof(Produmex.Foundation.SlimScreen.Interfaces.IShowMessageScreen),
						DefaultCultureInfo.Get(), BuildParamCollection(
							"MessageKey", errorMessage,
							"NoTranslationOfMessageKey", false,
							"IsError", true,
							"ShowButton", true));
					WaitForMessage();

					return;
				}

				weight = weight.HasValue ? weight : 0;
				theoreticalWeight = (theoreticalWeight.HasValue ? theoreticalWeight : 0) / uomInMilligrams;

				decimal resultWeight = Convert.ToDecimal(weight.Value);
				decimal resultTheoreticalWeight = Convert.ToDecimal(theoreticalWeight.Value);
				decimal resultTolerance = Convert.ToDecimal(logisticCarrierTolerance);

				decimal differece = Math.Abs(resultWeight - resultTheoreticalWeight);
				decimal acceptedDifference = resultTheoreticalWeight * (resultTolerance / 100);

				if (differece > acceptedDifference)
				{
					BackRequested.Set(true);

					// Inform the user with the error
					session.ShowScreen(typeof(Produmex.Foundation.SlimScreen.Interfaces.IShowMessageScreen),
						DefaultCultureInfo.Get(), BuildParamCollection(
							"MessageKey", $"Expected weight is {resultTheoreticalWeight} {uom} with {resultTolerance}% tolerance, but the actual weight is {resultWeight} {uom}",
							"NoTranslationOfMessageKey", false,
							"IsError", true,
							"ShowButton", true));
					WaitForMessage();
				}
			}
		}

		private double? GetDataFromQuery(ISboProviderService sboProviderService, string weightQuery)
		{
			double? result = null;

			DataSet dsWeightQuery = sboProviderService.RunView(false, null, null, weightQuery);

			if (GetNumberOfRows(dsWeightQuery) == 0)
			{
				return 0.0;
			}
			else
			{
				result = Convertor.ConvertToDoubleNullable(dsWeightQuery.Tables[0].Rows[0][0]);
			}
			return result;
		}

		private int GetNumberOfRows(DataSet ds)
		{
			if (ds != null &&
				ds.Tables.Count > 0)
			{
				return ds.Tables[0].Rows.Count;
			}

			return 0;
		}
		#endregion
	}

	#region BUILD QUERY

	/// <summary>
	/// Static class grouping our native queries
	/// </summary>
	internal static class BuildQuery
	{

		public static string GetQuerySumTheoreticalWeigtOfItemsOnLuid(int luid,  DbTool dbTool)
		{
			DbQueryBuilder strBuilder = dbTool.GetQueryBuilder();

			strBuilder.Append("SELECT SUM(");
			strBuilder.Append(SqlCommandHelper.EscapeTableColumnName(PmxItemMeasurementViewDefinition.NAME, PmxItemMeasurementViewDefinition.Columns.SalesWeight1.NAME));
			strBuilder.Append(" * ");
			strBuilder.Append(SqlCommandHelper.EscapeTableColumnName(SboWeightUnitTable.NAME, SboWeightUnitTable.Columns.WeightInMG.NAME));
			strBuilder.Append(" * ");
			//strBuilder.Append(" * 1E-6 * ");
			strBuilder.Append(SqlCommandHelper.EscapeTableColumnName(PmxInventoryTotalTable.NAME, PmxInventoryTotalTable.Columns.Quantity.NAME));
			strBuilder.Append(") AS ");
			strBuilder.Append(SqlCommandHelper.EscapeColumnName("Weight"));

			strBuilder.Append(" FROM ");
			strBuilder.Append(SqlCommandHelper.EscapeTableName(PmxInventoryTotalTable.NAME));
			strBuilder.Append(strBuilder.WITH_NOLOCK);

			strBuilder.Append(" INNER JOIN ");
			strBuilder.Append(SqlCommandHelper.EscapeTableName(PmxItemMeasurementViewDefinition.NAME));
			strBuilder.Append(strBuilder.WITH_NOLOCK);
			strBuilder.Append(" ON ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxItemMeasurementViewDefinition.NAME, PmxItemMeasurementViewDefinition.Columns.ItemCode.NAME);
			strBuilder.Append(" = ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxInventoryTotalTable.NAME, PmxInventoryTotalTable.Columns.ItemCode.NAME);
			strBuilder.Append(" AND CASE WHEN  ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxItemMeasurementViewDefinition.NAME, PmxItemMeasurementViewDefinition.Columns.UomEntry.NAME);
			strBuilder.Append(" = -1 THEN '' ELSE ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxItemMeasurementViewDefinition.NAME, PmxItemMeasurementViewDefinition.Columns.UomCode.NAME);
			strBuilder.Append(" END = CASE WHEN ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxItemMeasurementViewDefinition.NAME, PmxItemMeasurementViewDefinition.Columns.UomEntry.NAME);
			strBuilder.Append(" = -1 THEN '' ELSE ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxInventoryTotalTable.NAME, PmxInventoryTotalTable.Columns.Uom.NAME);
			strBuilder.Append(" END ");

			strBuilder.Append(" INNER JOIN ");
			strBuilder.Append(SqlCommandHelper.EscapeTableName(SboWeightUnitTable.NAME));
			strBuilder.Append(strBuilder.WITH_NOLOCK);
			strBuilder.Append(" ON ");
			SqlCommandHelper.AddTableColumnName(strBuilder, SboWeightUnitTable.NAME, SboWeightUnitTable.Columns.UnitCode.NAME);
			strBuilder.Append(" = ");
			SqlCommandHelper.AddTableColumnName(strBuilder, PmxItemMeasurementViewDefinition.NAME, PmxItemMeasurementViewDefinition.Columns.SalesWght1Unit.NAME);


			strBuilder.Append(" WHERE ");

			strBuilder.Append(SqlCommandHelper.EscapeColumnName(PmxInventoryTotalTable.Columns.LogUnitIdentKey.NAME));
			strBuilder.Append(" = ");
			strBuilder.Append(SqlCommandHelper.FormatAndEscapeColumnValue(luid, CultureInfo.InvariantCulture));

			return strBuilder.ToString();
		}

		public static string GetQueryLuidDetails(int luid, DbTool dbTool)
		{
			DbQueryBuilder qb = dbTool.GetQueryBuilder();

			qb = qb
				.Select
				.EscColumnName(PmxLogisticUnitIDTable.Columns.Weight.NAME)
				.Comma
				.EscColumnName(PmxLogisticUnitIDTable.Columns.WeightUom.NAME)
				.Comma
				.EscColumnName(PmxLogisticUnitIDTable.Columns.LogisticCarrierCode.NAME)
				.From
				.EscTableName(PmxLogisticUnitIDTable.NAME)
				.WithNoLock
				.Where
				.EscColumnName(PmxLogisticUnitIDTable.Columns.InternalKey.NAME)
				.Eq
				.Append(SqlCommandHelper.FormatAndEscapeColumnValue(luid, CultureInfo.InvariantCulture));

			return qb.ToString();
		}

		public static string GetConversionToMiligrams(string uom, DbTool dbTool)
		{
			DbQueryBuilder qb = dbTool.GetQueryBuilder();

			qb = qb.Select.
					EscTableColumnName(SboWeightUnitTable.NAME, SboWeightUnitTable.Columns.WeightInMG.NAME).
				From.
					EscTableName(SboWeightUnitTable.NAME).
					CRLF.
				Where.
					AppendFormat(SqlCommandHelper.FormatAndEscapeColumnValue(uom, dbTool.CultureInfo)).Eq.
					EscTableColumnName(SboWeightUnitTable.NAME, SboWeightUnitTable.Columns.UnitDisply.NAME);

			return qb.ToString();
		}

		internal static string GetToleranceOfLogisticCarrer(string logisticCarrierCode, DbTool dbTool)
		{
			DbQueryBuilder qb = dbTool.GetQueryBuilder();

			qb = qb.Select.
					EscTableColumnName(PmxItemsTablePL.NAME, PmxItemsTablePL.Columns.WeightTolerancePercentage.NAME).
				From.
					EscTableName(PmxItemsTablePL.NAME).
					CRLF.
				Where.
					AppendFormat(SqlCommandHelper.FormatAndEscapeColumnValue(logisticCarrierCode, dbTool.CultureInfo)).Eq.
					EscTableColumnName(PmxItemsTablePL.NAME, PmxItemsTablePL.Columns.ItemCode.NAME);

			return qb.ToString();
		}

		#endregion
	}
}