-- =============================================
-- Author:	PMX
-- Description:	Produmex extension on the SBO_SP_TransactionNotification
-- =============================================
ALTER PROCEDURE [dbo].[PMX_SP_TransactionNotificationChecks]
	-- Add the parameters for the stored procedure here
	@object_type nvarchar(30), 			-- SBO Object Type
	@transaction_type nchar(1),			-- [A]dd, [U]pdate, [D]elete, [C]ancel, C[L]ose
	@num_of_cols_in_key int,
	@list_of_key_cols_tab_del nvarchar(255),
	@list_of_cols_val_tab_del nvarchar(255),
	@extraDB NVARCHAR(MAX),
	@tsNow DATETIME,
	@updateDate DATETIME,
	@updateTime INT,
	@error int OUTPUT, -- Result (0 for no error)
	@error_message nvarchar (255) OUTPUT -- Error string to be displayed

AS

BEGIN

	--**********************************************************************************************************
	--Return and handling errors:
	--If a problem is found the function sets the parameters @error and @error_message
	--and logs the error in the windows event logging
	--**********************************************************************************************************
	
	--**********************************************************************************************************
	--Logging errors:
	--If there are errors and this procedure is called when using the DIAPI then the user does not see the error
	--Because of this problem the errors are also logged in the windows event logging
	--To log you have to use the funciton: EXEC xp_logevent @error, @error_message, ERROR
	--**********************************************************************************************************

	--Logging
	DECLARE @MSG NVARCHAR (255);
	DECLARE @errorList  NVARCHAR(255);
	DECLARE @errorList2 VARCHAR(255);
	DECLARE @count  INT;

	--At this moment the function supports: purchase delivery note, sales delivery note, move, item and BOM
	
	DECLARE @isPurchaseOrderDocument  SMALLINT;
	--DECLARE @isPurchaseDeliveryDocument  SMALLINT;
	--DECLARE @isPurchaseReturnDocument  SMALLINT;
	--DECLARE @isPurchaseInvoice  SMALLINT;
	--DECLARE @isPurchaseCreditNote  SMALLINT;
	--DECLARE @isSalesDeliveryDocument  SMALLINT;
	--DECLARE @isSalesReturnDocument  SMALLINT;
	DECLARE @isSalesOrderDocument  SMALLINT;
	--DECLARE @isSalesInvoice  SMALLINT;
	--DECLARE @isSalesCreditNote  SMALLINT;
	--DECLARE @isInventoryEntry  SMALLINT;
	--DECLARE @isInventoryExit  SMALLINT;
	--DECLARE @isMoveDoc  SMALLINT;
	DECLARE @isItem  SMALLINT;
	DECLARE @isTransactionTypeAdd  SMALLINT;
	DECLARE @isTransactionTypeUpdate  SMALLINT;
	DECLARE @isTransactionTypeDelete  SMALLINT;
	DECLARE @isTransactionTypeCancel  SMALLINT;
	DECLARE @isBOM  SMALLINT;
	DECLARE @isUser  SMALLINT;
	--DECLARE @isDocument  SMALLINT;
	DECLARE @isProductionOrder  SMALLINT;
	DECLARE @isItemBatchNumber  SMALLINT;
	DECLARE @isUomGroup  SMALLINT;
	--DECLARE @isStockUpdate  SMALLINT;
	DECLARE @isBusinessPartner  SMALLINT;
	--DECLARE @isStockTransfer  SMALLINT;
	DECLARE @isItemPackagingType  SMALLINT;
	DECLARE @isPickListType  SMALLINT;
	DECLARE @isPriority  SMALLINT;
	DECLARE @isFreightChargeDefinition  SMALLINT;
	DECLARE @isWarehouseTransferRequestDocument  SMALLINT;
	DECLARE @isWarehouse  SMALLINT;
	DECLARE @isBarcode  SMALLINT;
	--DECLARE @isInventoryOpeningBalance SMALLINT;
	--DECLARE @isInventoryPosting SMALLINT;
	--DECLARE @isInventoryCounting SMALLINT;

	--Declaration of variables used for locking stock based on batches in sales order
	DECLARE @batchToLock nvarchar (36);
	DECLARE @batchQuantityToLock numeric (19,6);
	DECLARE @expDateToLock DateTime
	DECLARE @shippingQualityOptionToLock nvarchar(30)
	DECLARE @lineNumToLock int
	DECLARE @docEntryToLock int
	DECLARE @typeToLock nvarchar (30);
	DECLARE @itemCodeToLock nvarchar (20);	
	DECLARE @catchWeightRatioToLock numeric(19,6);	
	DECLARE @itriToLock int	
	DECLARE @qualityStatusToLock nvarchar (8);
	DECLARE @pmxWarehouseToLock nvarchar (50);	

	DECLARE @freeQuantityToLock numeric (19,6);
	DECLARE @currentQuantityToLock numeric (19,6);
	DECLARE @remainingQuantityToLock numeric (19,6);
	DECLARE @lockedQuantity numeric (19,6);

	DECLARE @currentLockInternalKey int;
	--END: Declaration of variables used for locking stock based on batches in sales order


	--Set default values
	SET @error = 0;
	SET @error_message = N'Ok';

	SET @tsNow = CURRENT_TIMESTAMP;
	SET @updateDate = CONVERT( DATETIME, CONVERT( VARCHAR, @tsNow, 112 ), 112 );
	SET @updateTime = DATEPART( hour, @tsNow ) * 100 + DATEPART( minute, @tsNow );

	
	SET @isPurchaseOrderDocument = 0;
	SET @isSalesOrderDocument = 0;
	
	SET @isTransactionTypeAdd = 0;
	SET @isTransactionTypeUpdate = 0;
	SET @isTransactionTypeDelete = 0;
	SET @isTransactionTypeCancel = 0;
	SET @isItem = 0;
	SET @isBOM = 0;
	SET @isUser = 0;
	SET @isProductionOrder = 0;
	SET @isItemBatchNumber = 0;
	SET @isUomGroup = 0;
	SET @isBusinessPartner = 0;
	SET @isItemPackagingType = 0;
	SET @isPickListType = 0;
	SET @isPriority = 0;
	SET @isFreightChargeDefinition = 0;
	SET @isWarehouseTransferRequestDocument = 0;
	SET @isWarehouse = 0;
	SET @isBarcode = 0;


	--Test object type (this can be found in the SDK 'BoObjectTypes Enumeration'
	IF (@object_type = N'4') BEGIN
		SET @isItem = 1;
	END ELSE IF (@object_type = N'22') BEGIN
		SET @isPurchaseOrderDocument = 1;
	END ELSE IF (@object_type = N'66') BEGIN
		SET @isBOM = 1;
	END ELSE IF (@object_type = N'12') BEGIN
		SET @isUser = 1;
	END ELSE IF (@object_type = N'17') BEGIN
		SET @isSalesOrderDocument = 1;
	END ELSE IF (@object_type = N'202') BEGIN
		SET @isProductionOrder = 1;
	END ELSE IF (@object_type = N'106') BEGIN
		SET @isItemBatchNumber = 1;
	END ELSE IF (@object_type = N'10000197') BEGIN
		SET @isUomGroup = 1;
	END ELSE IF (@object_type = N'PMX_PRIO') BEGIN
		SET @isPriority = 1;
	END ELSE IF (@object_type = N'2') BEGIN
		SET @isBusinessPartner = 1;
	END ELSE IF (@object_type = N'64') BEGIN
		SET @isWarehouse = 1;
	END ELSE IF (@object_type = N'PMX_IPTY') BEGIN
		SET @isItemPackagingType = 1;
	END ELSE IF (@object_type = N'PMX_PLTY') BEGIN
		SET @isPickListType = 1;
	END ELSE IF (@object_type = N'PMX_FCDE') BEGIN
		SET @isFreightChargeDefinition = 1;
	END ELSE IF (@object_type = N'1250000001') BEGIN
		SET @isWarehouseTransferRequestDocument = 1;
	END ELSE IF (@object_type = N'1470000062') BEGIN
		SET @isBarcode = 1;
	END


	--Test transaction type
	IF (@transaction_type = N'A') BEGIN
		SET @isTransactionTypeAdd = 1;
	END ELSE IF (@transaction_type = N'U') BEGIN
		SET @isTransactionTypeUpdate = 1;
	END ELSE IF (@transaction_type = N'D') BEGIN
		SET @isTransactionTypeDelete = 1;
	END ELSE IF (@transaction_type = N'C') BEGIN
		SET @isTransactionTypeCancel = 1;
	END


	--Check if we have to do something. If nothing to do then exit.	
	IF (NOT(
		(@isItem=1)	OR																--All operations on an item are supported
		(@isBOM=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on a BOM
		(@isUser=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1))	OR	--Add/update on a User
		(@isSalesOrderDocument=1) OR												--All operations on a sales order are supported
		(@isPurchaseOrderDocument=1) OR												--All operations on a purchase order are supported
		(@isProductionOrder=1) OR 	--Add/update of a production order is supported
		--(@isItemBatchNumber=1 AND @isTransactionTypeUpdate=1) OR					--Update of item batch numbers
		(@isUomGroup=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on UomGroup
		(@isFreightChargeDefinition=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on a Pmx Uom
		(@isPriority=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on a Pmx priority
		(@isItemPackagingType=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR 	--Add/update on a Pmx item packaging type
		(@isWarehouse=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR 	--Add/update on SAP warehouse
		(@isPickListType=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR 	--Add/update on a Pmx pick list type
		(@isBusinessPartner=1) OR	--Add/update on a business partner
		(@isBarcode=1) OR	--Add/update on a barcode
		(@isWarehouseTransferRequestDocument=1)
		)) BEGIN
			RETURN;
	END;


	BEGIN TRY
		--General variables
		DECLARE @keyAsInteger AS INT
		
		--The number of columns of the primary key must always be one. 
		IF @num_of_cols_in_key <> 1 BEGIN
			SET @error = 50001;
			SET @error_message = 'Number of keys must be 1! Contact Produmex Support.'
			SELECT @error, @error_message
			EXEC xp_logevent @error, @error_message, ERROR
			RETURN;
		END


		--**********************************************************************************************************
		--Do checks on an item
		--**********************************************************************************************************
		IF (@isItem=1) BEGIN
				

			--Check if this is a delete. 
			IF (@isTransactionTypeDelete=1) BEGIN
				
				--Check if the item is not used for the bin items
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_SLIT."ParentCode" + ' ' 
				FROM PMX_SLIT
				WHERE PMX_SLIT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50250;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item is used for the BIN items ' + @errorList + '.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item has no zone types linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_ITZT.ZoneTypeCode + ' ' 
				FROM PMX_ITZT
				WHERE PMX_ITZT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50251;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has zone types linked to it for zone types: ' + @errorList + '. Delete these zone types from the item.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item has no shelflifes linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_ICSL."CardCode" + ' ' 
				FROM PMX_ICSL
				WHERE PMX_ICSL."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50252;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has shelf lifes linked to it for customers: ' + @errorList + '. Delete these customers from the item.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item is not used in a steplist
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_PSLH."ParentItemCode" + ' ' 
				FROM PMX_PSLH
				WHERE PMX_PSLH."ParentItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50253;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has production order step list linked to it.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item is not used in a steplist lines
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_PSLH."ParentItemCode" + ' ' 
				FROM PMX_PSLH
				INNER JOIN PMX_PSLL ON PMX_PSLL."ParentKey" = PMX_PSLH."InternalKey"
				WHERE PMX_PSLL."ChildItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50254;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item is used in production order step list for items: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item is not used as default logistic carrier
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM."ItemCode" + ' ' 
				FROM OITM
				WHERE OITM.U_PMX_DLCP = @list_of_cols_val_tab_del OR OITM.U_PMX_DLCS = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50255;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item is used as default logistic carrier for items: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item does not have any batch-attributes linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IBAT.BatchAttributeCode + ' ' 
				FROM PMX_IBAT
				WHERE PMX_IBAT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50256;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has batch-attributes linked to it:' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END


				--Check if the item does not have any packaging types linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IPTY."PackagingTypeCode" + ' ' 
				FROM PMX_IPTY
				WHERE PMX_IPTY."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50257;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has packaging types linked to it: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END


				--No other checks must be done for the item delete
				RETURN;
			END

			--Get data of the item
			DECLARE @storageLocationCode NVARCHAR(50);
			DECLARE @isReturnableItem NVARCHAR(1);
			DECLARE @itemCodeReturnableItemNotInventory NVARCHAR(20);
			DECLARE @isInventoryItem CHAR(1);
			DECLARE @barcode NVARCHAR(254);
			DECLARE @purchaseBarcode NVARCHAR(254);
			DECLARE @salesBarcode NVARCHAR(254);
			DECLARE @qualityStatusQuarantinedAfterReception NVARCHAR(8);
			DECLARE @qualityStatusAfterSalesReturn NVARCHAR(8);
			DECLARE @barcodeType NVARCHAR(1);
			DECLARE @purchaseBarcodeType NVARCHAR(1);
			DECLARE @salesBarcodeType NVARCHAR(1);
			DECLARE @isSerialNumber NVARCHAR(1);
			DECLARE @hasBestBefore NVARCHAR(1);
			DECLARE @expiryDefinitionCodeProduction NVARCHAR(30);
			DECLARE @numberOfDecimalsForUOM INT;
			DECLARE @numberOfDecimalsForUOM2 INT;
			DECLARE @gs1MeasureTypeForUOM NVARCHAR(30);
			DECLARE @gs1MeasureTypeForUOM2 NVARCHAR(30);
			DECLARE @itemUOM NVARCHAR(100);
			DECLARE @itemUOM2 NVARCHAR(100);
			DECLARE @defaultQuantityUOM NUMERIC(19, 6);
			DECLARE @defaultQuantityUOM2 NUMERIC(19, 6);
			DECLARE @UOMToUseForSales NVARCHAR(1);
			DECLARE @purchaseQuantity NUMERIC(19, 6);
			DECLARE @salesQuantity NUMERIC(19, 6);
			DECLARE @purchaseUOM NVARCHAR(100);
			DECLARE @salesUOM NVARCHAR(100);
			DECLARE @correctStockForUOM NVARCHAR(1);
			DECLARE @correctStockForUOM2 NVARCHAR(1);
			DECLARE @UOMToUseForInventoryTransitions NVARCHAR(1);
			DECLARE @UOMToUseForPurchase NVARCHAR(1);
			DECLARE @isItemLogisticCarrier  NVARCHAR(1);
			DECLARE @sevesoClass  NVARCHAR(30);
			DECLARE @qualityStatusReleasedAfterReception NVARCHAR(8);
			DECLARE @qualityStatusReceptionController NVARCHAR(8);
			DECLARE @hasSecondBatchNumber NVARCHAR(1);
			DECLARE @defaultLogisticCarrierProduction NVARCHAR(20);
			DECLARE @defaultLogisticCarrierPicking NVARCHAR(20);
			DECLARE @hasPmxSerialNumber NVARCHAR(1);
			DECLARE @qualityStatusProduction NVARCHAR(8);
			DECLARE @pickType NVARCHAR(30);
			DECLARE @trackLocationSerialNumber NVARCHAR(1);
			DECLARE @formatSerialNumber NVARCHAR(50);
			DECLARE @createSsccOnReception NVARCHAR(1);
			DECLARE @isCatchWeight NVARCHAR(1);
			DECLARE @addNonInventoryItemToPickList CHAR(1);


			SELECT @storageLocationCode=ISNULL( U_PMX_DSLC,''), @isReturnableItem=ISNULL( U_PMX_RETR,'N'), @itemCodeReturnableItemNotInventory=ISNULL( U_PMX_ICRI,''),
					@isInventoryItem=ISNULL( "InvntItem",'N'),@barcode=ISNULL( "CodeBars",''),@barcodeType=ISNULL( U_PMX_BCTY,'G'), 
					@qualityStatusQuarantinedAfterReception=U_PMX_QSCR, @qualityStatusAfterSalesReturn=U_PMX_QSSR,@isSerialNumber=ISNULL( "ManSerNum",'N'),
					@hasBestBefore=ISNULL( U_PMX_HBBD,'N'), @purchaseBarcode=ISNULL( U_PMX_PBCO,''), @purchaseBarcodeType=ISNULL( U_PMX_PBCT,'G'),
					@salesBarcode=ISNULL( U_PMX_SBCO,''), @salesBarcodeType=ISNULL( U_PMX_SBCT,'G'), @expiryDefinitionCodeProduction=U_PMX_EXDP,
					@numberOfDecimalsForUOM = U_PMX_UOMD, @numberOfDecimalsForUOM2 = U_PMX_UM2D, @gs1MeasureTypeForUOM = ISNULL( U_PMX_UMMT,'') , @gs1MeasureTypeForUOM2 = ISNULL( U_PMX_U2MT,''),
					@itemUOM = ISNULL( UOM_INV."UomCode", ISNULL( "InvntryUom",'')) , @itemUOM2 = ISNULL( U_PMX_UOM2,''), @defaultQuantityUOM = U_PMX_DQUM, @defaultQuantityUOM2 = U_PMX_DQU2,
					@UOMToUseForSales = U_PMX_UMSA, @purchaseQuantity = ISNULL( "NumInBuy",0), @salesQuantity = ISNULL( "NumInSale",0),
					@purchaseUOM = ISNULL( UOM_PUR."UomCode", ISNULL( "BuyUnitMsr",'')), @salesUOM = ISNULL( UOM_SAL."UomCode",ISNULL( "SalUnitMsr",'')), @correctStockForUOM = ISNULL( U_PMX_CSUM, 'N'), @correctStockForUOM2 = ISNULL( U_PMX_CSU2, 'N'),
					@UOMToUseForPurchase = U_PMX_UMPU, @UOMToUseForInventoryTransitions = U_PMX_UMIT, @isItemLogisticCarrier = U_PMX_LOCA, @sevesoClass = U_PMX_SEVE,
					@qualityStatusReleasedAfterReception = U_PMX_RQSR, @qualityStatusReceptionController = U_PMX_QSRC,
					@hasSecondBatchNumber = ISNULL( U_PMX_HBN2, 'N'), @defaultLogisticCarrierProduction = U_PMX_DLCP, @defaultLogisticCarrierPicking = U_PMX_DLCS,
					@hasPmxSerialNumber=ISNULL( U_PMX_HSER,'N'), @qualityStatusProduction = U_PMX_QSPR, @pickType=U_PMX_PITY, @trackLocationSerialNumber=ISNULL( U_PMX_TLSN,'N'), @formatSerialNumber = U_PMX_SNFO , @createSsccOnReception=ISNULL( U_PMX_CSOR,'N'),
					@isCatchWeight = U_PMX_ICAW, @addNonInventoryItemToPickList = U_PMX_NIOP
			FROM OITM
			LEFT JOIN OUOM AS UOM_INV ON UOM_INV."UomEntry" = OITM."IUoMEntry"
			LEFT JOIN OUOM AS UOM_PUR ON UOM_PUR."UomEntry" = OITM."PUoMEntry"
			LEFT JOIN OUOM AS UOM_SAL ON UOM_SAL."UomEntry" = OITM."SUoMEntry"
			WHERE "ItemCode" = @list_of_cols_val_tab_del;

	--		print '@isReturnableItem='  + @isReturnableItem
	--		print '@itemCodeReturnableItemNotInventory=' + @itemCodeReturnableItemNotInventory
	--		print '@isInventoryItem=' + @isInventoryItem
	--		print '@storageLocationCode=' + @storageLocationCode
	--		print '@barcode=' + @barcode

			--Check if the default storage location filled in is an existing storage location or a zone
			IF (LEN(@storageLocationCode)>0) BEGIN
				IF (SELECT COUNT("Code") FROM PMX_OSSL WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS=@storageLocationCode AND "IsActive"='Y') <> 1 BEGIN
					IF (SELECT COUNT("Code") FROM PMX_OSZO WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS=@storageLocationCode) <> 1 BEGIN
						SET @error = 50200;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The storage location code or zone code ''' + @storageLocationCode + ''' could not be found in the system or the storage location is not active!'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END
				END
			END

			--The property ItemCodeReturnableItemNotInventory on an item must be filled in if the item is inventory and IsReturnableItem=True
			IF (@isReturnableItem = 'Y') AND (@isInventoryItem='Y') AND (LEN(@itemCodeReturnableItemNotInventory)=0) BEGIN
				SET @error = 50201;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item code returnable item not inventory is not filled. This must be filled in if the item is an inventory item and is returnable. Every inventory returnable item must have a not inventory returnable item linked to it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--The property ItemCodeReturnableItemNotInventory on an item can only be filled in if the item is an inventory returnable item.
			IF NOT((@isReturnableItem = 'Y' OR @isItemLogisticCarrier = 'Y') AND (@isInventoryItem='Y')) AND (LEN(@itemCodeReturnableItemNotInventory)>0) BEGIN
				SET @error = 50202;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The field ItemCodeReturnableItemNotInventory may only be filled in if the item is an inventory returnable item or logistic carrier!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the item code that is filled in in @itemCodeReturnableItemNotInventory is an existing not inventory returnable item.
			IF (LEN(@itemCodeReturnableItemNotInventory)>0) AND ((SELECT COUNT("ItemCode") FROM OITM WHERE "ItemCode" = @itemCodeReturnableItemNotInventory AND "InvntItem"='N' AND (U_PMX_RETR='Y' OR U_PMX_LOCA='Y'))=0) BEGIN
				SET @error = 50203;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The not inventory returnable item with code '''+@itemCodeReturnableItemNotInventory+''' could not be found!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check the GTINs of the item
			DECLARE @sqlStatementBegin AS NVARCHAR(max)
			DECLARE @sqlStatementEnd AS NVARCHAR(max)
			DECLARE @sqlStatement AS NVARCHAR(max)
			DECLARE @parameterList AS NVARCHAR(max)

			SET @sqlStatementBegin = 'SELECT @error=Error, @error_message=Message FROM ' + @extraDb + '.dbo.PMX_FN_CheckGTIN(''';
			SET @sqlStatementEnd = ''')';
			SET @parameterList = '@error INT output, @error_message NVARCHAR(200) output';
			
			
			--Check the GTIN of the item
			IF ((LEN(@barcode) <> 0) AND (@barcodeType IN ( 'G', 'V'))) BEGIN
				SET @sqlStatement = @sqlStatementBegin + @barcode + ''', ''' + @barcodeType + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcode + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			
			
			--Check the purchase GTIN of the item
			IF ((LEN(@purchaseBarcode) <> 0) AND (@purchaseBarcodeType IN ( 'G', 'V'))) BEGIN
				SET @sqlStatement = @sqlStatementBegin + @purchaseBarcode + ''', ''' + @barcodeType + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = 'Purchase barcode '''+ @purchaseBarcode + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			--Check the sales GTIN of the item
			IF ((LEN(@salesBarcode) <> 0) AND (@salesBarcodeType IN ( 'G', 'V'))) BEGIN
				SET @sqlStatement = @sqlStatementBegin + @salesBarcode + ''', ''' + @barcodeType + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = 'Sales barcode '''+ @salesBarcode + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			

			--Check the GTIN of the item packaging types
			DECLARE @barcodePackagingType NVARCHAR(16);
			DECLARE barCodePackagingTypeCursor CURSOR LOCAL FOR SELECT "Barcode" FROM PMX_IPTY WHERE PMX_IPTY."ItemCode" = @list_of_cols_val_tab_del AND LEN("Barcode") <> 0 AND ("BarcodeType" IN ( 'G', 'V'));
			OPEN barCodePackagingTypeCursor;
			FETCH NEXT FROM barCodePackagingTypeCursor INTO @barcodePackagingType;
			WHILE @@FETCH_STATUS >= 0 BEGIN
				SET @sqlStatement = @sqlStatementBegin + @barcodePackagingType + ''', ''' + @barcodeType + @sqlStatementEnd;
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcodePackagingType + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					CLOSE barCodePackagingTypeCursor;
					DEALLOCATE barCodePackagingTypeCursor

					RETURN;
				END

				FETCH NEXT FROM barCodePackagingTypeCursor INTO @barcodePackagingType;
			
			END
			CLOSE barCodePackagingTypeCursor;
			DEALLOCATE barCodePackagingTypeCursor
			
			
			--Check the GTIN of the uom barcodes
			DECLARE @barcodeForUom NVARCHAR(254);
			DECLARE @barcodeUom NVARCHAR(100);
			DECLARE barCodeUomCursor CURSOR LOCAL FOR SELECT "BcdCode", OUOM."UomCode" 
														FROM OBCD 
														INNER JOIN OUOM ON OUOM."UomEntry" = OBCD."UomEntry" 
														WHERE OBCD."ItemCode" = @list_of_cols_val_tab_del AND LEN("BcdCode") <> 0 
														AND @barcodeType IN ('G', 'V');
			OPEN barCodeUomCursor;
			FETCH NEXT FROM barCodeUomCursor INTO @barcodeForUom, @barcodeUom;
			WHILE @@FETCH_STATUS >= 0 BEGIN
				SET @sqlStatement = @sqlStatementBegin + @barcodeForUom + ''', ''' + @barcodeType + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcodeForUom + ''' for uom '''+ @barcodeUom + '''  of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message;
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					CLOSE barCodeUomCursor ;
					DEALLOCATE barCodeUomCursor

					RETURN;
				END

				FETCH NEXT FROM barCodeUomCursor INTO @barcodeForUom, @barcodeUom;
			
			END
			CLOSE barCodeUomCursor;
			DEALLOCATE barCodeUomCursor
			
			

			--Check if the quality status code that is filled in in @qualityStatusQuarantinedAfterReception is an existing quality status code.
			IF (@qualityStatusQuarantinedAfterReception IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusQuarantinedAfterReception COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50207;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status code after purchase return ''' + @qualityStatusQuarantinedAfterReception + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status code that is filled in in @qualityStatusAfterSalesReturn is an existing quality status code.
			IF (@qualityStatusAfterSalesReturn IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusAfterSalesReturn COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50208;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status code after sales return ''' + @qualityStatusAfterSalesReturn + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if barcode does not occur more than once for inventory items
			IF (@isInventoryItem = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM."ItemCode" + ' ' 
				FROM OITM
				LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				WHERE OITM."ItemCode" <> @list_of_cols_val_tab_del AND 
				((LEN(ISNULL( OITM."CodeBars",'')) <> 0 AND OITM."CodeBars" IN(@barcode,@purchaseBarcode,@salesBarcode)) OR
				(LEN(ISNULL( OITM.U_PMX_PBCO,'')) <> 0 AND OITM.U_PMX_PBCO IN(@barcode,@purchaseBarcode,@salesBarcode))OR
				(LEN(ISNULL( OITM.U_PMX_SBCO,'')) <> 0 AND OITM.U_PMX_SBCO IN(@barcode,@purchaseBarcode,@salesBarcode))OR
				(LEN(ISNULL( PMX_IPTY."BarCode",'')) <> 0 AND PMX_IPTY."BarCode" IN(@barcode,@purchaseBarcode,@salesBarcode))) AND
				OITM."InvntItem" = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50209;
					SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': The item ' + @errorList + 'has the same barcode.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--The property "ManSerNum" on an item cannot be true because the addon does not support it
--			IF (@isSerialNumber = 'Y') BEGIN
--				SET @error = 50210;
--				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item cannot be managed by serial numbers.'
--				SELECT @error, @error_message
--				EXEC xp_logevent @error, @error_message, ERROR
--				RETURN;
--			END

			--When the item has bestBeforeDate, there cannot be an inventory item with bestBeforeDate is null
			IF (@hasBestBefore = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."BestBeforeDate" IS NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50211;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has a best before date, but in the inventory the best before date is not set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item has no bestBeforeDate, there cannot be an inventory item with a bestBeforeDate
			IF (@hasBestBefore = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."BestBeforeDate" IS NOT NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50212;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no best before date, but in the inventory the best before date is set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			-- Check if the barcode in purchase and sales are not the same as the base barcode
			IF (LEN(@barcode)>0 AND @barcode = @purchasebarCode) OR (LEN(@barcode)>0 AND @barcode = @salesBarcode) OR (LEN(@salesBarcode)>0 AND @salesBarcode=@purchaseBarcode) BEGIN
				SET @error = 50213;
				SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': The barcode or the purchase barcode or the sales barcode of item ' + @list_of_cols_val_tab_del + ' is the same. This is not allowed.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the expiry definition for production is filled in in @expiryDefinitionCodeProduction is an existing expiry definition code.
			IF (@expiryDefinitionCodeProduction IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_EXDE] WHERE "Code"=@expiryDefinitionCodeProduction)=0) BEGIN
				SET @error = 50214;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The expiry defintion code for production ''' + @expiryDefinitionCodeProduction + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the seveso class is filled in and if @sevesoClass is an existing seveso class code.
			IF (@sevesoClass IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_SEVE] WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS=@sevesoClass)=0) BEGIN
				SET @error = 50215;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The seveso class code ''' + @sevesoClass + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status code that is filled in in @qualityStatusReleasedAfterReception is an existing quality status code.
			IF (@qualityStatusReleasedAfterReception IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusReleasedAfterReception COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50216;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The released quality status code after reception ''' + @qualityStatusReleasedAfterReception + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status controller exists
			IF (@qualityStatusReceptionController IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_EXTE WHERE "Code"=@qualityStatusReceptionController)=0) BEGIN
				SET @error = 50217;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status controller ''' + @qualityStatusReceptionController + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--When the item has second batchnumber, there cannot be an inventory item with second batchnumber is null
			IF (@hasSecondBatchNumber = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."InternalBatchNumber" IS NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50218;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has a second batch number, but in the inventory the second batch number is not set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item has no second batchnumber, there cannot be an inventory item with a second batchnumber
			IF (@hasSecondBatchNumber = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."InternalBatchNumber" IS NOT NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50219;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no second batch number, but in the inventory the second batch number is set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--If no batchnumber or best before date is set, user cannot ask for packaging type quantity during receipt
			IF (@hasBestBefore = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IPTY."PackagingTypeCode" + ' ' 
				FROM PMX_IPTY INNER JOIN OITM
				ON OITM."ItemCode" = PMX_IPTY."ItemCode"
				WHERE PMX_IPTY."ItemCode" = @list_of_cols_val_tab_del
				AND OITM."ManBtchNum" = 'N'
				AND PMX_IPTY."AskDuringReception" = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50220;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no batch or BBD, so packaging quantity cannot be asked during receipt for packaging types: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END


			--Check if the uom does not exists in item packaging types
			IF((SELECT COUNT("ItemCode") FROM PMX_IPTY WHERE "PackagingTypeCode"=@itemUOM AND "ItemCode" = @list_of_cols_val_tab_del)>0 AND LEN(@itemUOM) > 0) BEGIN
				SET @error = 50221;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The UOM with code ''' + @itemUOM + ''' is used in packaging types for the item'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Batchnumber cannot be set if item is logistic carrier
			SET @errorList = ''
			SELECT @errorList = @errorList + OITM."ItemCode" + ' ' 
			FROM OITM
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			AND OITM.U_PMX_LOCA = 'Y'
			AND (OITM."ManBtchNum" = 'Y' OR OITM.U_PMX_HBBD = 'Y' OR OITM.U_PMX_HBN2 = 'Y');
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50222;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': An item defined as logistic carrier cannot have a batch or BBD.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Default logistic carrier for production should be valid item and logistic carrier
			IF ISNULL( @defaultLogisticCarrierProduction, '') <> ''
			BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM.U_PMX_DLCP + ' ' 
				FROM OITM
				WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT ItemCode FROM OITM WHERE "ItemCode" = @defaultLogisticCarrierProduction
				AND OITM.U_PMX_LOCA = 'Y');
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50223;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The default logistic carrier for Production ' + @defaultLogisticCarrierProduction + ' is not a valid Logistic Carrier.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--Default logistic carrier for picking should be valid item and logistic carrier
			IF ISNULL( @defaultLogisticCarrierProduction, '') <> ''
			BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM.U_PMX_DLCS + ' ' 
				FROM OITM
				WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT ItemCode FROM OITM WHERE "ItemCode" = @defaultLogisticCarrierPicking
				AND OITM.U_PMX_LOCA = 'Y');
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50224;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The default logistic carrier for Picking ' + @defaultLogisticCarrierPicking + ' is not a valid Logistic Carrier.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--If item has Pmx serial number, the decimals for uom should be 0
			IF ((@hasPmxSerialNumber = 'Y' OR @isSerialNumber = 'Y') AND
				@numberOfDecimalsForUOM > 0) BEGIN
				SET @error = 50225;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has serial numbers, so the number of decimals for UOM should be 0.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status code that is filled in in @qualityStatusProduction is an existing quality status code.
			IF (@qualityStatusProduction IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusProduction COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50226;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status code production ''' + @qualityStatusProduction + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the pick type that is filled in in @pickType is an existing pick type.
			IF (@pickType IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_IPIT] WHERE "Code"=@picktype)=0) BEGIN
				SET @error = 50227;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The pick type ''' + @pickType + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--When the item tracks location for serial numbers, there cannot be one without LUID
			IF (@trackLocationSerialNumber = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_SENU."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_SENU
				WHERE PMX_SENU."InStock" = 'Y' 
				AND PMX_SENU.LUID IS NULL
				AND PMX_SENU."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50228;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item needs to track locations of serial nr, but there are serial nrs in stock without LUID: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item tracks location for serial numbers, there cannot be inventory totals without LUID (needed for "on release only" setting)
			IF (@trackLocationSerialNumber = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				WHERE PMX_INVT."Quantity" > 0
				AND PMX_INVT.LogUnitIdentKey IS NULL
				AND PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50238;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item needs to track locations of serial nr, but there are inventory totals without LUID: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item does not tracks location for serial numbers, there cannot be one with LUID
			IF (@trackLocationSerialNumber = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_SENU."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_SENU
				WHERE PMX_SENU."InStock" = 'Y' 
				AND PMX_SENU.LUID IS NOT NULL
				AND PMX_SENU."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50229;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item does not need to track locations of ser. nr, but there are ser. nrs are in stock with LUID: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END			

			--When the item does track location for serial numbers, either SAP or PMX serial numbers needs to be filled
			IF (@trackLocationSerialNumber = 'Y' AND @hasPmxSerialNumber = 'N' AND @isSerialNumber = 'N') BEGIN
				SET @error = 50230;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item tracks locations of ser. nr, but the item is not defined as serial number'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--When the item does track location for serial numbers, either SAP or PMX serial numbers needs to be filled
			IF (@hasPmxSerialNumber = 'Y' AND @isSerialNumber = 'Y') BEGIN
				SET @error = 50231;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has SAP serial numbers and PMX serial numbers. Please choose only one of them.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the serial format is a correct one
			IF (@formatSerialNumber IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_ISFT] WHERE "Code"=@formatSerialNumber)=0) BEGIN
				SET @error = 50232;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The serial number format ''' + @formatSerialNumber + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			IF (@isInventoryItem = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + T0."ItemCode" + ' ' 
				FROM (
					SELECT DISTINCT Barcode2."ItemCode"				
					FROM OBCD AS Barcode1
					LEFT JOIN OBCD Barcode2 ON Barcode1."BcdCode" = Barcode2."BcdCode"
					WHERE Barcode1."ItemCode" = @list_of_cols_val_tab_del
					AND Barcode1."BcdEntry" <> Barcode2."BcdEntry"
				) AS T0;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50233;
					SET @error_message = 'Item '''+ @errorList +''' contains the same barcode '
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			
			IF (@isInventoryItem = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + T0."ItemCode" + ' ' 
				FROM (
					SELECT OBCD."ItemCode"
					FROM OBCD 
					WHERE OBCD."ItemCode" <> @list_of_cols_val_tab_del AND 
					(LEN(ISNULL( OBCD."BcdCode",'')) <> 0 AND OBCD."BcdCode" IN(@barcode,@purchaseBarcode,@salesBarcode)) 
				) AS T0;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50234;
					SET @error_message = 'Item ''' + @list_of_cols_val_tab_del + ''': The item ''' + @errorList + ''' has the same barcode.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			
			-- Check if item has serial that needs to be tracked -> need to have SSCC's at reception
			IF (@hasPmxSerialNumber = 'Y' OR @isSerialNumber = 'Y') AND @trackLocationSerialNumber ='Y' AND @createSsccOnReception = 'N' BEGIN
				SET @error = 50235;
				SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + '''. Cannot add an item with SAP or PMX serial numbers and tracking location without create SSCC on reception.' + @errorList + '.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			
			--When the item has no Serialnumber, there cannot be an inventory item with Serialnumber
			IF (@hasPmxSerialNumber = 'N' AND @isSerialNumber = 'N' ) BEGIN
				SET @errorList = ''
				SELECT DISTINCT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_SENU ON PMX_SENU."ItemCode" = PMX_INVT."ItemCode"
				WHERE PMX_SENU."InStock" ='Y' AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50236;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no serial number, but serial number(s) are set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			-- Check if item has serial that needs to be tracked -> need to have SSCC's at reception
			IF (@isItemLogisticCarrier = 'Y' AND EXISTS(SELECT "ItemCode" FROM OITM WHERE "ItemCode" = @list_of_cols_val_tab_del AND OITM."UgpEntry" <> -1)) BEGIN
			    SET @error = 50237;
				SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + '''. A logistic carrier cannot have an uom group.' 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			IF (@isInventoryItem = 'Y' AND @addNonInventoryItemToPickList = 'Y') BEGIN
			    SET @error = 50239;
				SET @error_message = 'Add non-inventory item to pick list is not allowed for inventory item. Item with code ''' + @list_of_cols_val_tab_del + ''''
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			

			--******************************
			--Catch weight
			--******************************

			--Check if the number of decimals for uom is between 0 and the number of decimals for quantities
			IF(@numberOfDecimalsForUOM < 0 OR @numberOfDecimalsForUOM > (SELECT TOP (1) OADM.QtyDec FROM OADM)) BEGIN
				SET @error = 50350;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The number of decimals for uom should be between 0 and ' + CAST( (SELECT TOP (1) OADM.QtyDec FROM OADM) AS NVARCHAR (10))
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the number of decimals for uom2 is between 0 and the number of decimals for quantities
			IF(@numberOfDecimalsForUOM2 < 0 OR @numberOfDecimalsForUOM2 > (SELECT TOP (1) OADM.QtyDec FROM OADM)) BEGIN
				SET @error = 50351;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The number of decimals for uom2 should be between 0 and ' + CAST( (SELECT TOP (1) OADM.QtyDec FROM OADM) AS NVARCHAR (10))
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the gs1 measure types are not the same
			IF( LEN(@gs1MeasureTypeForUOM) > 0 AND LEN(@gs1MeasureTypeForUOM2) > 0 AND @gs1MeasureTypeForUOM = @gs1MeasureTypeForUOM2) BEGIN
				SET @error = 50352;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The GS1 Measure Type cannot be the same for uom and uom2'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
		
			

			--Check if the uom exists in UOM
			IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@itemUOM)=0 AND LEN(@itemUOM) > 0) BEGIN
				SET @error = 50355;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The UOM with code ''' + @itemUOM + ''' does not exist in SAP Uom'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the purchase uom exists in UOM
			IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@purchaseUOM)=0 AND LEN(@purchaseUOM) > 0) BEGIN
				SET @error = 50356;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The purchase UOM with code ''' + @purchaseUOM + ''' does not exist in SAP Uom'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the sales uom exists in UOM
			IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@salesUOM)=0 AND LEN(@salesUOM) > 0) BEGIN
				SET @error = 50357;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The sales UOM with code ''' + @salesUOM + ''' does not exist in SAP Uom'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Checks for catch weight
			IF( @isCatchWeight = 'Y' ) BEGIN

				--Check if uom decimals for UOM = 0 when it should be **DISABLE FOR NOW **
				--IF( @numberOfDecimalsForUOM <> 0 ) BEGIN
				--	SET @error = 50371;
				--	SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The number of decimals for UOM should be 0 for inventory uom.'
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END
				
				----Check if uom decimals for UOM = 0 when it should be
				--IF(EXISTS(SELECT "ItemCode" FROM [OITM] WHERE "ItemCode"=@list_of_cols_val_tab_del AND ISNULL( "IUoMEntry", -1) <> -1 )) BEGIN
				--	SET @error = 50372;
				--	SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': UOM groups are not allowed for catch weight items';
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END
				
				--IF (@salesQuantity <> 1) BEGIN
				--	SET @error = 50373;
				--	SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': Items per sales unit should be 1 for catch weight items';
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END
				
				--IF (@purchaseQuantity <> 1) BEGIN
				--	SET @error = 50374;
				--	SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': Items per purchase unit should be 1 for catch weight items';
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END		
				
				IF ISNULL( @defaultQuantityUOM, 0 ) = 0 BEGIN
					SET @error = 50375;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': A default quantity for uom should be filled in for catch weight items';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
				--Only allow catch weight items if no stock, or stock has quantity for Uom2
				IF EXISTS(SELECT TOP(1)[PMX_INVT]."InternalKey" FROM [PMX_INVT] WHERE [PMX_INVT]."ItemCode"=@list_of_cols_val_tab_del AND [PMX_INVT].QuantityUom2 IS NULL ) BEGIN
					SET @error = 50376;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Cannot change item to catch weight item if stock exists';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
				--Update the default quantity for uom2
				UPDATE OITM SET U_PMX_DQU2 = 1 / @defaultQuantityUOM WHERE "ItemCode" = @list_of_cols_val_tab_del;
				
			END
			ELSE
			BEGIN
			
				--Only allow NON catch weight items if no stock, or stock has no quantity for Uom2
				IF EXISTS(SELECT TOP(1)[PMX_INVT]."InternalKey" FROM [PMX_INVT] WHERE [PMX_INVT]."ItemCode"=@list_of_cols_val_tab_del AND [PMX_INVT].QuantityUom2 IS NOT NULL ) BEGIN
					SET @error = 50401;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Cannot change item to NON-catch weight item if stock exists';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
			
			END


			--Do checks when UOM2 is filled in
			IF(LEN(@itemUOM2) > 0) BEGIN
				--Check if the uom2 exists in UOM
				IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@itemUOM2)=0) BEGIN
					SET @error = 50358;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The UOM2 with code ''' + @itemUOM2 + ''' does not exist in SAP Uom'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
                /*
				IF ISNULL( @defaultQuantityUOM , 0 ) = 0 BEGIN
					UPDATE OITM SET U_PMX_DQUM = 1 WHERE "ItemCode" = @list_of_cols_val_tab_del
					SET @defaultQuantityUOM = 1;
				END
				IF ISNULL( @defaultQuantityUOM2, 0 ) = 0 BEGIN
					UPDATE OITM SET U_PMX_DQU2 = 1 WHERE "ItemCode" = @list_of_cols_val_tab_del
					SET @defaultQuantityUOM2 = 1;
				END		
				IF @defaultQuantityUOM <> 1 AND @defaultQuantityUOM2 <> 1 BEGIN
					SET @error = 50359;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': At least one of the default quantities for UOM has to be 1'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the uom to use for orders is not 'A'
				IF( @UOMToUseForSales = 'A') BEGIN
					SET @error = 50360;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Uom to use for orders cannot be ''UomAndUOM2'''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
                */
                
                IF(@isCatchWeight = 'N') BEGIN

					--Check if purchase quantity = 1
					IF(@purchaseQuantity <> 1) BEGIN
						SET @error = 50361;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The items in purchase unit should be 1'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					--Check if sales quantity = 1
					IF(@purchaseQuantity <> 1) BEGIN
						SET @error = 50362;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The items in sales unit should be 1'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					--Check if purchase, sales and inventory uom is the same
					IF ( NULLIF( @purchaseUOM, '' ) <> @itemUOM OR NULLIF( @salesUOM, '' ) <> @itemUOM ) BEGIN
						SET @error = 50363;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The sales and purchase uom should be the same as the inventory uom'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					/*
					IF(@correctStockForUOM = 'Y' AND @correctStockForUOM2 = 'Y') BEGIN
						--Check if the uom to use for inventory transitions is 'A'
						IF( @UOMToUseForInventoryTransitions <> 'A') BEGIN
							SET @error = 50364;
							SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Uom to use for inventory transitions should be ''UomAndUOM2'' if correct stock for uom and uom2 is set to true'
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END
						--Check if the uom to use for documents is 'A'
						IF( @UOMToUseForPurchase <> 'A') BEGIN
							SET @error = 50365;
							SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Uom to use for documents should be ''UomAndUOM2'' if correct stock for uom and uom2 is set to true'
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END			
					END
					*/
				END
				
				-- Verify that UOM2 prices are possible (convert to UOM1 and back)
				DECLARE curPriceLists CURSOR LOCAL FOR
				SELECT U_PMX_UOM2PRICEUDF, CAST( "PriceDec" AS NVARCHAR ) FROM OPLN, OADM WHERE ISNULL( U_PMX_UOM2PRICEUDF, '' ) <> '';

				DECLARE @priceUDF NVARCHAR(20);
				DECLARE @priceDecimals NVARCHAR(20);
				DECLARE @getPriceUDFs NVARCHAR(500);
				DECLARE @badPrices NVARCHAR(500);

				DECLARE @tmp TABLE ( priceFromUDF DECIMAL(19,6), flooredPrice DECIMAL(19,6), ceiledPrice DECIMAL(19,6) )

				OPEN curPriceLists;
				FETCH NEXT FROM curPriceLists INTO @priceUDF, @priceDecimals;
				WHILE @@FETCH_STATUS >= 0
				BEGIN
					SET @getPriceUDFs = 'SELECT U_' + @priceUDF + ',
						CEILING( FLOOR  ( U_' + @priceUDF + ' / U_PMX_DQU2 * POWER(10,' + @priceDecimals + ') ) * U_PMX_DQU2 ) / POWER(10,' + @priceDecimals + '),
						CEILING( CEILING( U_' + @priceUDF + ' / U_PMX_DQU2 * POWER(10,' + @priceDecimals + ') ) * U_PMX_DQU2 ) / POWER(10,' + @priceDecimals + ')
					FROM OITM WHERE "ItemCode" = ''' + @list_of_cols_val_tab_del + '''';
					
					INSERT INTO @tmp
					EXEC( @getPriceUDFs )

					FETCH NEXT FROM curPriceLists INTO @priceUDF, @priceDecimals;
				END
				CLOSE curPriceLists;

				SELECT @badPrices = ISNULL( @badPrices + '. ', '' ) + 'UOM2 price ' + CONVERT( NVARCHAR, priceFromUDF ) + ' not possible, choose ' + CONVERT( NVARCHAR, flooredPrice ) + ' or ' + CONVERT( NVARCHAR, ceiledPrice )
				FROM @tmp
				WHERE priceFromUDF <> flooredPrice AND flooredPrice <> ceiledPrice;
				
				IF @badPrices IS NOT NULL
				BEGIN
					SET @error = 50366;
					SET @error_message = @badPrices
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END					
					



			--******************************
			--Updates
			--******************************

			-- * Set the custom description *

			UPDATE OITM SET U_PMX_CUDE = OITM."ItemCode" + ISNULL( ' - ' + OITM."ItemName", '') + ISNULL( ' - ' + OITM."CodeBars", '')
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del;

			-- * SBO UomGroup details to PMX_IPTY *

			DELETE FROM PMX_IPTY
			FROM OITM
			INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
			LEFT JOIN (
				UGP1
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
			) ON UGP1."UgpEntry" = OITM."UgpEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			  AND OITM."UgpEntry" <> -1
			  AND OUOM."UomCode" IS NULL

			UPDATE PMX_IPTY SET "Quantity" = UGP1."BaseQty" / UGP1."AltQty", "UserSign" = ISNULL( OUGP."UserSign2", 1 ), "UpdateDateTime" = @tsNow, "Version" = "Version" + 1
			FROM OITM
			INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
			INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
			INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
			INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			  AND OITM."UgpEntry" <> -1
			  AND "Quantity" <> CAST( UGP1."BaseQty" / UGP1."AltQty" AS DECIMAL(19,6) );

			INSERT INTO PMX_IPTY ( "UserSign", "CreateDateTime", "Version", "ItemCode", "PackagingTypeCode", "Quantity", "BarcodeType", "Canceled", "HideDuringEnteringQuantity", "AskDuringReception" )
			SELECT COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 ), @tsNow, 1, OITM."ItemCode", OUOM."UomCode", UGP1."BaseQty" / UGP1."AltQty", 'G', 'N', 'N', 'N'
			FROM OITM
			INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
			INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
			INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
			LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode" AND PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			  AND UGP1."UomEntry" <> OITM."IUoMEntry"
			  AND OITM."UgpEntry" <> -1
			  AND PMX_IPTY."PackagingTypeCode" IS NULL;


			--All checks are done so we can exit the checks.
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on BOM
		--**********************************************************************************************************
		IF (@isBOM=1) BEGIN
			--The lines of BOM may not contain a returnable item that is not an inventory item
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."ChildNum" AS VARCHAR) + ' ' 
			FROM ITT1 AS Lines INNER JOIN OITM AS Item ON Lines."Code" = Item."ItemCode"
			WHERE Lines."Father" = @list_of_cols_val_tab_del AND Item.U_PMX_RETR = 'Y' AND Item."InvntItem" = 'N';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50300;
				SET @error_message = 'BOM ''' + @list_of_cols_val_tab_del + ''': The line(s) ' + @errorList + ' contain returnable items that are NOT inventory. All items can be added to a BOM except not inventory returnable items.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--The inventory items on the lines of a BOM may not have backflush selected.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."ChildNum" AS VARCHAR) + ' ' 
			FROM ITT1 AS Lines INNER JOIN OITM AS Item ON Lines."Code" = Item."ItemCode"
			INNER JOIN OWHS ON Lines."Warehouse" = OWHS."Whscode"
			WHERE Lines."Father" = @list_of_cols_val_tab_del AND Item."InvntItem" = 'Y' AND Lines."IssueMthd" = 'B' AND OWHS."U_PMX_IMBP"='Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50301;
				SET @error_message = 'BOM ''' + @list_of_cols_val_tab_del + ''': The line(s) ' + @errorList + ' have ''Backflush'' selected as issue method. If using the Produmex add-on this is not allowed. This must be set to ''Manual''.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--All checks are done so we can exit the checks.
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on ProductionOrder
		--**********************************************************************************************************
		IF (@isProductionOrder=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--The inventory items on the lines of a production order may not have backflush selected.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM WOR1 AS Lines INNER JOIN OITM AS Item ON Lines."ItemCode" = Item."ItemCode"
			INNER JOIN OWOR AS ProductionOrder on ProductionOrder."DocEntry" = Lines."DocEntry"
			INNER JOIN OWHS ON Lines."Warehouse" = OWHS."Whscode"
			WHERE Lines."DocEntry" = @keyAsInteger AND Item."InvntItem" = 'Y' AND Lines."IssueType" = 'B' AND OWHS."U_PMX_IMBP"='Y' 
			AND ProductionOrder."Status" <> 'L';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50900;
				SET @error_message = 'Production order ''' + @list_of_cols_val_tab_del + ''': The line(s) ' + @errorList + ' have ''Backflush'' selected as issue method. If using the Produmex add-on this is not allowed. This must be set to ''Manual''.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			DECLARE @parentDocEntry INT;
			DECLARE @parentDocNum INT;
			DECLARE @parentFound INT;
			
			SELECT @parentDocEntry = this.U_PMX_PPRD, @parentDocNum = this.U_PMX_PPRN, @parentFound = parent."DocEntry"
			FROM OWOR this
			LEFT JOIN OWOR parent ON this.U_PMX_PPRD = parent."DocEntry" AND this.U_PMX_PPRN = parent."DocNum"
			WHERE this."DocEntry" = @keyAsInteger;
			IF ( @parentDocEntry IS NULL AND @parentDocNum IS NOT NULL )
				OR ( @parentDocEntry IS NOT NULL AND @parentDocNum IS NULL )
				OR ( @parentDocEntry IS NOT NULL AND @parentDocNum IS NOT NULL AND @parentFound IS NULL )
			BEGIN
				SET @error = 50901;
				SET @error_message = 'Production order ''' + @list_of_cols_val_tab_del + ''': The "DocEntry" ''' + ISNULL( CONVERT(VARCHAR,@parentDocEntry), '' ) + ''' and "DocNum" ''' + ISNULL( CONVERT(VARCHAR,@parentDocNum), '' ) + ''' referencing the parent production order do not match.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			IF @parentDocEntry = @keyAsInteger BEGIN
				SET @error = 50902;
				SET @error_message = 'Production order ''' + @list_of_cols_val_tab_del + ''': a production order cannot have itself as parent.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Closed production orders cannot have the pmx status 'started'
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + ' ' 
			FROM OWOR										--Cancelled, Closed
			WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" IN ('C', 'L') AND OWOR.U_PMX_PRST = 'S'
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50903;
				SET @error_message = 'Production order ''' + @errorList + ''': The production order cannot be closed when pmx status is started!';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Planned qty cannot be lowered if open proposals exist
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + '.' + CAST(WOR1."LineNum" AS VARCHAR) + ' ' 
			FROM OWOR
			INNER JOIN WOR1 ON WOR1."DocEntry" = OWOR."DocEntry"
			AND WOR1."PlannedQty" > 0
			LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseEntry" = WOR1."DocEntry"
			AND PMX_PLPL."BaseLine" = WOR1."LineNum"
			AND PMX_PLPL."BaseType" = '202'
			AND PMX_PLPL."LineStatus" = 'O'
			WHERE OWOR."DocEntry" = @keyAsInteger
			GROUP BY OWOR."DocNum", WOR1."DocEntry", WOR1."LineNum", WOR1."PlannedQty"
			HAVING (SUM(ISNULL( PMX_PLPL."Quantity", 0)) > WOR1."PlannedQty");
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50904;
				SET @error_message = 'Production order line ''' + @errorList + ''': The production order planned quantity cannot be lowered because there are open proposal(s)!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Productionline cannot be changed if open proposals exist
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) 
			FROM OWOR
			INNER JOIN WOR1 ON WOR1."DocEntry" = OWOR."DocEntry"
			INNER JOIN PMX_PLPL ON PMX_PLPL."BaseEntry" = WOR1."DocEntry"
			AND PMX_PLPL."BaseLine" = WOR1."LineNum"
			AND PMX_PLPL."BaseType" = '202'
			INNER JOIN PMX_PLPH ON PMX_PLPH."DocEntry" = PMX_PLPL."DocEntry"
			INNER JOIN PMX_OSPL ON PMX_OSPL."Code" = OWOR.U_PMX_PLCD
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND PMX_PLPH."DestStorLocCode" <> ISNULL(PMX_OSPL."PickToLocationCode", PMX_OSPL."InputLocationCode")
			AND PMX_PLPH."DocStatus" = 'O';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50905;
				SET @error_message = 'Production order ''' + @errorList + ''': The production order has a different production line than the proposal(s)!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

		
			--Only one production order allowed for a production line with the pmx status 'started'
			IF(EXISTS(SELECT OWOR."DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" NOT IN ('C', 'L') 
				AND OWOR.U_PMX_PRST = 'S')) BEGIN

				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + ' ' 
				FROM OWOR	
				INNER JOIN OWOR AS currentOWOR
				ON OWOR.U_PMX_PLCD = currentOWOR.U_PMX_PLCD	
				INNER JOIN PMX_OSPL ON PMX_OSPL."Code" = currentOWOR.U_PMX_PLCD
										
				WHERE OWOR."DocEntry" <> @keyAsInteger 
				AND currentOWOR."DocEntry" = @keyAsInteger
									--Cancelled, Closed
				AND OWOR."Status" NOT IN ('C', 'L') 
				AND OWOR.U_PMX_PRST = 'S'
				AND PMX_OSPL."OnlyOneStartedProductionOrderAllowed" = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50904;
					SET @error_message = 'Production order(s) ''' + @errorList + ''' started: There can only be 1 production order for a certain production line with the pmx status started!'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			--All batchnumbers must be entered in upper case
			--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."U_PMX_BTCH" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND OWOR."U_PMX_BTCH" COLLATE SQL_Latin1_General_Cp850_CS_AS <> UPPER(OWOR."U_PMX_BTCH") COLLATE SQL_Latin1_General_Cp850_CS_AS
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50906;
				SET @error_message = 'Batch numbers must be upper case. This is not the case for the batchnumber: ' + @errorList
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."U_PMX_BTC2" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND OWOR."U_PMX_BTC2" COLLATE SQL_Latin1_General_Cp850_CS_AS <> UPPER(OWOR."U_PMX_BTC2") COLLATE SQL_Latin1_General_Cp850_CS_AS
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50907;
				SET @error_message = 'Batch numbers must be upper case. This is not the case for the batchnumber 2: ' + @errorList
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."U_PMX_NRPB" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND OWOR."U_PMX_NRPB" = 0
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50908;
				SET @error_message = 'The number of production batches cannot be 0 for production order ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Add check for removing time registration items
			IF (@isTransactionTypeUpdate=1) BEGIN
				SET @errorList = ''	
				SELECT @errorList =	@errorList + CAST(PMX_TRHE."ItemCode" AS VARCHAR) + ' '
				FROM PMX_TRHE 
				LEFT JOIN WOR1 ON PMX_TRHE."BaseEntry" = WOR1."DocEntry" AND PMX_TRHE."ItemCode" = WOR1."ItemCode"
				AND PMX_TRHE."BaseType" = '202'
				WHERE WOR1."ItemCode" IS NULL
				AND WOR1."DocEntry" = @keyAsInteger 

				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50909;
					SET @error_message = 'Cannot delete item codes because time has already been registered: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--All checks are done so we can exit the checks.

			--Perform actions when adding production order

			--When a production order is duplicated, UDF's are also copied
			IF (@isTransactionTypeAdd=1) BEGIN
				--The pmx status is also copied.
				--But the status should be (P)lanned when a production order is added.
				UPDATE OWOR SET OWOR.U_PMX_PRST = 'P' WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR.U_PMX_PRST <> 'P';
				--When production order is duplicated, the CurrentProdBatch should NOT be copied and set to default 1			
				UPDATE OWOR SET OWOR.U_PMX_CUPB = 1  WHERE OWOR."DocEntry" = @keyAsInteger; --AND OWOR.U_PMX_PRST <> 'P';
				--Set picked quantity to 0
				UPDATE WOR1 SET WOR1.U_PMX_QTYP = 0 WHERE WOR1."DocEntry" = @keyAsInteger AND U_PMX_QTYP <> 0;
			END

            IF (@isTransactionTypeUpdate=1) BEGIN
            
            SET @errorList = ''
				SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + ' ' 
				FROM OWOR													
				WHERE OWOR."DocEntry" = @keyAsInteger 				
				AND  OWOR.U_PMX_NRPB <= ( SELECT MAX(PMX_POBA."SeqNr") FROM PMX_POBA WHERE PMX_POBA."DocEntry" = 220 );
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50906;
					SET @error_message = 'Already components weighed for production order ''' + @errorList + '''. Cannot alter number of production batches lower than current production batch.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			    --Remove locking when production order is closed
			    DELETE FROM PMX_INLD WHERE "BaseType" = '202' 
			    AND "BaseEntry" IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" IN ('C', 'L'));
			    
			      --Remove dispended SSCC when production order is closed
			    DELETE FROM PMX_POPC WHERE "ProdOrderDocEntry"
			     IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger AND 
						( OWOR."Status" IN ('C', 'L') OR OWOR.U_PMX_PRST IN ('C')));

                --Remove lined up locations when order is closed and production line is not used in another started production order
                DELETE FROM PMX_LIUP WHERE "ProductionLineCode" IN
                (SELECT U_PMX_PLCD FROM OWOR WHERE
                 OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" IN ('C', 'L') 
                 AND U_PMX_PLCD NOT IN 
                 (SELECT U_PMX_PLCD FROM OWOR WHERE
					OWOR."Status" = 'R' AND U_PMX_PRST = 'S'));

             END
             
             IF(@isTransactionTypeDelete=1 OR @isTransactionTypeCancel=1) BEGIN
             
                --Remove dispended SSCC when production order is closed
			    DELETE FROM PMX_POPC WHERE "ProdOrderDocEntry"
			     IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger );
             
             END

			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on a user
		--**********************************************************************************************************
		IF (@isUser=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--Check if the language key filled in at the user is an existing language key
			SELECT @count = COUNT(OUSR.INTERNAL_K) 
			FROM OUSR
            LEFT JOIN OLNG ON OUSR.U_PMX_LANG = OLNG."Code"
			WHERE OUSR.INTERNAL_K = @keyAsInteger AND OLNG."Code" IS NULL AND OUSR.U_PMX_LANG IS NOT NULL;
			IF @count <> 0 BEGIN
				SET @error = 50400;
				SET @error_message = 'The language key filled in for user with key ' + @list_of_cols_val_tab_del + ' does not exist.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
		
			--All checks are done so we can exit the checks.
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on a business partner
		--**********************************************************************************************************
		IF (@isBusinessPartner=1) BEGIN
			--Check if this is a delete. 
			IF (@isTransactionTypeDelete=1) BEGIN
				
				--Check if the customer is not used in a route template
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_RTTH."Name" + ' ' 
				FROM PMX_RTTH
				INNER JOIN PMX_RTTL ON PMX_RTTL."ParentCode" = PMX_RTTH."Code"
				WHERE PMX_RTTL."CardCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50500;
					SET @error_message = 'Business partner with card code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the customer is used in route templates with name: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the customer is not used in a picklist proposal
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_PLPH."DocEntry" + ' ' 
				FROM PMX_PLPH
				WHERE PMX_PLPH."CardCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50501;
					SET @error_message = 'Business partner with card code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the customer is used in picklist proposals with doc number: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--No other checks must be done for the item delete
				RETURN;
			END

			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1 ) BEGIN
			
				--Check if linked business partner exists
				SET @errorList = ''
				SELECT @errorList = @errorList + OCRD.U_PMX_LBPA + ' ' 
				FROM OCRD
				WHERE OCRD."CardCode" = @list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT CardCode FROM OCRD AS InnerOCRD WHERE InnerOCRD."CardCode" = OCRD.U_PMX_LBPA)
				AND ISNULL( OCRD.U_PMX_LBPA, '') <> '';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50502;
					SET @error_message = 'Business partner with card code ''' + @list_of_cols_val_tab_del + ''' has an incorrect linked business partner: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--No other checks must be done for the item delete
				RETURN;
			END

			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on a sales order
		--**********************************************************************************************************
		IF (@isSalesOrderDocument=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--Check if the sales order is closed that there are no open pick list proposals of this sales order
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM ORDR 
			INNER JOIN PMX_PLPL ON ORDR."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = ORDR."ObjType"
			INNER JOIN RDR1 ON ORDR."DocEntry" = RDR1."DocEntry" AND RDR1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE ORDR."DocEntry" = @keyAsInteger AND (ORDR."DocStatus" = 'C' OR RDR1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50600;
				SET @error_message = 'The sales order ''' + @list_of_cols_val_tab_del + ''' has open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before closing the sales order (line).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the full quantity is selected when entering batch numbers
--			SET @errorList = ''
--			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
--			FROM RDR1 AS Lines
--			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
--			LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesOrder."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
--			WHERE SalesOrder."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O'
--			GROUP BY Lines."LineNum", Lines."Quantity", Lines."NumPerMsr"
--			HAVING SUM(IBT1."Quantity") <> (Lines."Quantity" * Lines."NumPerMsr") AND SUM(IBT1."Quantity") > 0
--			IF LEN(@errorList) <> 0 BEGIN
--				SET @error = 50601;
--				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
--				SELECT @error, @error_message
--				EXEC xp_logevent @error, @error_message, ERROR
--				RETURN;
--			END

			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM RDR1 AS Lines
            INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
            INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
            INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
            INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
            AND ITL1."SysNumber" = OBTN."SysNumber"
            WHERE OITL."AllocatEnt" = @keyAsInteger 
			AND Lines."LineStatus" = 'O'
            AND "StockEff" = 2
            GROUP BY Lines."LineNum", Lines."OpenQty", Lines."NumPerMsr"
            HAVING SUM("AllocQty") <> (Lines."OpenQty" * Lines."NumPerMsr") AND SUM("AllocQty") > 0;
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50601;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			--Check if sales order lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the sales order line is lower than on the open pick list proposal line / open pick list line
			SET @errorList = ''
			SELECT @errorList  = @errorList  + CASE WHEN        RDR1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN RDR1 ON PMX_PLPL."BaseEntry" = RDR1."DocEntry" AND PMX_PLPL."BaseLine" = RDR1."LineNum"  AND PMX_PLPL."BaseType" = RDR1."ObjType"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = 17 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR ISNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50603;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' The item code on a sales order line cannot be changed if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList2 = ''
			SELECT @errorList2 = @errorList2 + CASE WHEN ISNULL( RDR1."OpenInvQty",0) < SUM(ROUND( PMX_PLPL."OpenQty" * PMX_PLPL."QuantityPerUom", ISNULL( OITM.U_PMX_UOMD, 6)) ) THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' '  ) ELSE '' END
			FROM PMX_PLPL
			INNER JOIN OITM ON OITM."ItemCode" = PMX_PLPL."ItemCode"
			LEFT JOIN RDR1 ON PMX_PLPL."BaseEntry" = RDR1."DocEntry" AND PMX_PLPL."BaseLine" = RDR1."LineNum"  AND PMX_PLPL."BaseType" = RDR1."ObjType"
			WHERE PMX_PLPL."BaseType" = 17 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C') 
			GROUP BY PMX_PLPL."BaseLine", RDR1."OpenQty", RDR1."OpenInvQty";
			IF LEN(@errorList2) <> 0 BEGIN
				SET @error = 50602;
				SET @error_message = 'Error on line(s):  ' + @errorList2 + ' The quantity on a sales order line cannot be lowered if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM RDR1 AS Lines
			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesOrder."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."AllocatEnt" = Lines."DocEntry" AND BATCH."AllocateLn" = Lines."LineNum" AND BATCH."AllocateTp" = Lines."ObjType"
						
			WHERE SalesOrder."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O' AND Lines.[U_PMX_AMBA] = 'N'
			GROUP BY Lines."LineNum"
			HAVING (Count(*)) > 1;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50604;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			-- change of batch only allowed when not linked in DOLL
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '		
			FROM RDR1 AS Lines
			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
			INNER JOIN PMX_DOLL ON PMX_DOLL."ObjType" = Lines."ObjType" 
			AND PMX_DOLL."DocEntry" = Lines."DocEntry" 
			AND PMX_DOLL."LineNum" = Lines."LineNum" 

			LEFT JOIN 
			(SELECT "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp"
				HAVING SUM("AllocQty") > 0) AS BATCH

			ON BATCH."AllocatEnt" = Lines."DocEntry" AND BATCH."AllocateLn" = Lines."LineNum" AND BATCH."AllocateTp" = Lines."ObjType"
			AND BATCH."DistNumber" = PMX_DOLL."BatchNum"

			WHERE Lines."DocEntry" = @keyAsInteger 
			AND Lines."LineStatus" = 'O'
			AND (ISNULL( BATCH."AllocQty", PMX_DOLL."LockedQuantity") < PMX_DOLL."LockedQuantity"
				OR (PMX_DOLL."BatchNum" IS NOT NULL AND BATCH."DistNumber" IS NULL )
				);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50605;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' Proposal exists: Changing the batch number(s) not allowed'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Multiple batches on serial numbers are only allowed when multiple batches on line level are allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(RDR1."LineNum" AS VARCHAR) + ' '	
			FROM PMX_SELD	
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN RDR1 on RDR1."DocEntry" = PMX_SELD."BaseEntry"
			AND RDR1."LineNum" = PMX_SELD."BaseLine"
			AND RDR1."ObjType" = PMX_SELD."BaseType"
			WHERE RDR1.U_PMX_AMBA = 'N'
			AND RDR1."DocEntry" = @keyAsInteger	
			GROUP BY RDR1."LineNum", PMX_SENU."ItriKey"
			HAVING COUNT(PMX_SENU."ItriKey") > 1;

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50606;
				SET @error_message = 'Serial numbers linked with multiple batches, but multiple batches are not allowed. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Linked serial numbers cannot be more than open quantity
			SET @errorList = ''
			SELECT @errorList = @errorList + 'Line ' + CAST(RDR1."LineNum" AS VARCHAR) + ', Link qty:' + CAST(COUNT(PMX_SELD."InternalKey") AS VARCHAR) + '; '	
			FROM PMX_SELD
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN RDR1 on RDR1."DocEntry" = PMX_SELD."BaseEntry"
			AND RDR1."LineNum" = PMX_SELD."BaseLine"
			AND RDR1."ObjType" = PMX_SELD."BaseType"
			WHERE RDR1."DocEntry" = @keyAsInteger
			AND PMX_SENU."InStock" = 'Y'
			AND RDR1."LineStatus" = 'O'
			GROUP BY RDR1."LineNum", RDR1."OpenQty", RDR1."NumPerMsr"
			HAVING COUNT(PMX_SELD."InternalKey") > RDR1."OpenQty" * RDR1."NumPerMsr";

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50607;
				SET @error_message = 'Number of linked serial numbers cannot be more than open quantity. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date minutes
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(ORDR.U_PMX_DDTM AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = @keyAsInteger
			AND (ISNULL( ORDR.U_PMX_DDTM, 0) > 59 OR
				ISNULL( ORDR.U_PMX_DDTM, 0) < 0);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50608;
				SET @error_message = 'The due date minutes: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date Hour
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(ORDR.U_PMX_DDTH AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = @keyAsInteger
			AND ( ISNULL( ORDR.U_PMX_DDTH, 0) < 0
					OR ( ISNULL( ORDR.U_PMX_DDTH, 0) > 23 AND ISNULL( ORDR.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR ( ISNULL( ORDR.U_PMX_DDTH, 0) > 11 AND ISNULL( ORDR.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50609;
				SET @error_message = 'The due date hours: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE
			DELETE LOCKING FROM PMX_INLD LOCKING INNER JOIN ORDR ON ORDR."DocEntry" = LOCKING."BaseEntry"
			AND LOCKING."BaseType" = '17'
			WHERE ORDR."DocStatus" = 'C' AND ORDR."DocEntry" = @keyAsInteger


			IF(@isTransactionTypeAdd=0 AND
				EXISTS(SELECT DocEntry FROM RDR1 WHERE RDR1."LineStatus" = 'C' AND RDR1."DocEntry" = @keyAsInteger)) 
			BEGIN

				--Close container (lines) if needed
				UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
				FROM PMX_COLI
				INNER JOIN RDR1
				ON PMX_COLI."BaseEntry" = RDR1."DocEntry"
				AND PMX_COLI."BaseLine" = RDR1."LineNum"
				AND PMX_COLI."BaseType" = RDR1."ObjType"
				WHERE PMX_COLI."LineStatus" <> 'C'
				AND RDR1."DocEntry" = @keyAsInteger
				AND RDR1."LineStatus" = 'C';

				--Close header if no open lines are found
				UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
				PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
				PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM ORDR  WHERE ORDR."DocEntry" = @keyAsInteger)
				WHERE PMX_COHE."DocStatus" = 'O'
				AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

			END


			--**********************************************************************************************************
			--START: Add/update locking based on batch numbers
			--**********************************************************************************************************



			--Start with deleting the lockings that are not correct
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" NOT IN 
			(--Query to get the correct lockings
				SELECT 	PMX_INLD."InternalKey"
				FROM RDR1 AS Lines
				INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
				INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')
				--AND ISNULL( PMX_ITRI."BestBeforeDate", 1) = ISNULL( "ExpDate", 1)

				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"
				AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

				WHERE OITL."AllocatEnt" = @keyAsInteger 
				AND Lines."LineStatus" = 'O'
				AND "StockEff" = 2

				AND PMX_INLD."InternalKey" IS NOT NULL


				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				HAVING SUM("AllocQty") > 0

				-- Exclude locked stock created on goods receipt.
				UNION SELECT PMX_INLD."InternalKey"
				FROM RDR1 Lines 
				INNER JOIN POR1 ON 
				POR1."BaseType" = Lines."ObjType"
				AND POR1."BaseEntry" = Lines."DocEntry"
				AND POR1."BaseLine" = Lines."LineNum"

				INNER JOIN PDN1 ON 
				PDN1."BaseType" = POR1."ObjType"
				AND PDN1."BaseEntry" = POR1."DocEntry"
				AND PDN1."BaseLine" = POR1."LineNum"

				INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
				LEFT JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
				LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL(  PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')


				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"

				WHERE Lines."DocEntry" = @keyAsInteger				

				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				--Only allow detailed locking from purchase order when no batch has been linked
				HAVING SUM( ISNULL( "AllocQty", 0)) = 0 

				--Exclude manual lockings on LUID level when no batch has been selected on sales order
				UNION SELECT PMX_INLD."InternalKey"
				FROM PMX_INLD 
				LEFT JOIN
				(
					SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
					FROM RDR1 AS Lines
									INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
									LEFT JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
									LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
									LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
									AND ITL1."SysNumber" = OBTN."SysNumber"

					WHERE OITL."AllocatEnt"  = @keyAsInteger 
									AND Lines."LineStatus" = 'O'
									AND ISNULL( "StockEff", 2) = 2

						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
									Lines."NumPerMsr", "DistNumber", "ExpDate"

					HAVING SUM( ISNULL( "AllocQty", 0)) > 0
				) AS ORDR

				ON ORDR."LineNum" = PMX_INLD."BaseLine"
				AND ORDR."DocEntry" = PMX_INLD."BaseEntry"
				AND ORDR."ObjType" = PMX_INLD."BaseType"


				WHERE PMX_INLD."InvLockLevel" = 'L'
				AND PMX_INLD."BaseEntry" = @keyAsInteger
				AND ORDR."DocEntry" IS NULL

				

			) 

			AND PMX_INLD."BaseType" = '17'
			AND PMX_INLD."BaseEntry" = @keyAsInteger;



			BEGIN TRY

				DECLARE salesOrderBatchCursor CURSOR LOCAL FOR 
						SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", Lines."ObjType", 
						Lines."ItemCode", PMX_OSWH."Code", CASE WHEN OITM."U_PMX_ICAW" = 'Y' THEN OITM."U_PMX_DQUM" ELSE NULL END, Lines."U_PMX_SQOP"
						FROM RDR1 AS Lines
						INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
						INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = Lines."WhsCode"

						LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType"
						AND PMX_PLPL."BaseEntry" = Lines."DocEntry"
						AND PMX_PLPL."BaseLine" = Lines."LineNum"
						AND PMX_PLPL."LineStatus" = 'O'

						WHERE OITL."AllocatEnt" = @keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2
						--Only when no openpick list proposal is found
						AND PMX_PLPL."DocEntry" IS NULL
						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code", OITM."U_PMX_ICAW", OITM."U_PMX_DQUM", Lines."U_PMX_SQOP"

						HAVING SUM("AllocQty") > 0;


				OPEN salesOrderBatchCursor;
				FETCH NEXT FROM salesOrderBatchCursor 
				INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock,@pmxWarehouseToLock, @catchWeightRatioToLock, @shippingQualityOptionToLock;
				WHILE @@FETCH_STATUS >= 0 BEGIN

					--Check for each batch if we have a corresponding locking line

					IF NOT EXISTS (SELECT TOP(1) PMX_INLD.InternalKey FROM PMX_INLD 
					INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
					WHERE PMX_INLD."BaseType" = @typeToLock
					AND PMX_INLD."BaseEntry" = @docEntryToLock
					AND PMX_INLD."BaseLine" = @lineNumToLock
					AND PMX_ITRI."BatchNumber" = @batchToLock
					--AND ISNULL( PMX_ITRI."BestBeforeDate", '') = ISNULL( @expDateToLock, '');
					)
					BEGIN

						--Entry does not exist, so we need to lock the stock
						--First try to get the itri key to lock
						SET @remainingQuantityToLock = @batchQuantityToLock;

						BEGIN TRY
							DECLARE itriCursor CURSOR LOCAL FOR 
								SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" 
								FROM PMX_ITRI 
										LEFT JOIN (
											SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" 
											FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (
											SELECT "Code" 
											FROM PMX_QYST 
											WHERE  (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
													)
											AND PMX_INVT.StorLocCode NOT IN (SELECT Code FROM PMX_DISALLOWED_LOCATIONS_FOR_PICKING) 
											GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
										) AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (
											SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" 
											FROM PMX_INLD 
											WHERE "QualityStatusCode" IN (
												SELECT "Code" FROM PMX_QYST 
												WHERE (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
													)
											AND ISNULL(PMX_INLD."StorLocCode", '') NOT IN (SELECT Code FROM PMX_DISALLOWED_LOCATIONS_FOR_PICKING)
											GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;


							OPEN itriCursor;
							FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
							WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

								IF(@remainingQuantityToLock < @freeQuantityToLock)
								BEGIN
									SET @currentQuantityToLock = @remainingQuantityToLock;
								END
								ELSE
								BEGIN
									SET @currentQuantityToLock = @freeQuantityToLock;
								END

								--Insert INTO the PMX_INLD table
								INSERT INTO PMX_INLD ("BaseType","BaseEntry","BaseLine","CardCode","CardName",
									"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
									"UserSign","CreateDate",
									"CreateTime","LogUnitIdentKey","QualityStatusCode", 
									"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
								SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(ORDR."CardCode"),MAX(ORDR."CardName"),
									@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
									MAX(ORDR."UserSign"),@updateDate,
									@updateTime,NULL, @qualityStatusToLock,
									'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, @currentQuantityToLock * @catchWeightRatioToLock
								FROM ORDR 
								INNER JOIN RDR1 ON RDR1."DocEntry" = ORDR."DocEntry"
								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = RDR1."WhsCode"
								WHERE ORDR."DocEntry" = @docEntryToLock
								AND RDR1."LineNum" = @lineNumToLock;

								--Reduce remaining
								SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


								FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;


							END
							CLOSE itriCursor;
							DEALLOCATE itriCursor

							--If there is remaining QTY, there is not enough free stock, so throw error
							IF(@remainingQuantityToLock > 0)
							BEGIN
								SET @error = 50650;
								SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock + '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END

						END TRY
						BEGIN CATCH
							SET @error = 50690;
							SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
							+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
							BEGIN TRY
								CLOSE itriCursor 
								DEALLOCATE itriCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END TRY
							BEGIN CATCH
								DEALLOCATE itriCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END CATCH
						END CATCH

					END
					ELSE 
					BEGIN
						--The line exists, so now we need to check if the quantity is the same/more/less
						SET @remainingQuantityToLock = (SELECT SUM(PMX_INLD."Quantity") -  @batchQuantityToLock AS "Quantity"
                        FROM PMX_INLD 
						INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
						WHERE PMX_INLD."BaseType" = @typeToLock
						AND PMX_INLD."BaseEntry" = @docEntryToLock
						AND PMX_INLD."BaseLine" = @lineNumToLock
						AND PMX_ITRI."BatchNumber" = @batchToLock
						GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine");
							
										
						IF @remainingQuantityToLock <> 0 BEGIN
						--Only if deviation, then we try to update/insert
						--Loop through all locking lines
							BEGIN TRY
								DECLARE inldCursor CURSOR LOCAL FOR
                                SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey"
                                FROM PMX_INLD
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = @typeToLock
								AND PMX_INLD."BaseEntry" = @docEntryToLock
								AND PMX_INLD."BaseLine" = @lineNumToLock
								AND PMX_ITRI."BatchNumber" = @batchToLock;

								OPEN inldCursor;
								FETCH NEXT FROM inldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock <> 0 BEGIN

									IF @remainingQuantityToLock > 0 BEGIN
										--Remaining is more than 0, so we can just remove the locking
										IF @remainingQuantityToLock >= @lockedQuantity BEGIN
											--Delete
											DELETE FROM PMX_INLD WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = @remainingQuantityToLock - @lockedQuantity;
										END
                                        ELSE
                                        BEGIN
											--update
											UPDATE PMX_INLD SET "Quantity" = "Quantity" - @remainingQuantityToLock, "QuantityUom2" = "QuantityUom2" - (@remainingQuantityToLock * @catchWeightRatioToLock) WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = 0;
										END
									END
                                    ELSE
                                    BEGIN

										--Remaining is less than 0, so we need to add locking
										--If there is still free stock for current itri, then take that one
										SELECT @freeQuantityToLock = ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0)
                                        FROM PMX_ITRI 
										LEFT JOIN (
                                            SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
                                            FROM PMX_INVT 
                                            INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
                                            WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
                                            GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
                                        ) AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (
                                            SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
                                            FROM PMX_INLD
                                            WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
                                            GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
                                        ) AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
                                            AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
                                            AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0
										AND PMX_ITRI."InternalKey" = @itriToLock;

										IF(@remainingQuantityToLock > @freeQuantityToLock) BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
										END
                                        ELSE
                                        BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END

										UPDATE PMX_INLD SET "Quantity" = "Quantity" + @currentQuantityToLock, "QuantityUom2" = "QuantityUom2" + (@currentQuantityToLock * @catchWeightRatioToLock) WHERE PMX_INLD."InternalKey" = @currentLockInternalKey;

										-- add qty, because @remainingQuantityToLock is negative
										SET @remainingQuantityToLock = @remainingQuantityToLock + @currentQuantityToLock;
										
									END

									FETCH NEXT FROM inldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								END
								CLOSE inldCursor;
								DEALLOCATE inldCursor
							END TRY
							BEGIN CATCH
								SET @error = 50691
								SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
								+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
								BEGIN TRY
									CLOSE inldCursor 
									DEALLOCATE inldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END TRY
								BEGIN CATCH
									DEALLOCATE inldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END CATCH
							END CATCH

							IF @remainingQuantityToLock <> 0 BEGIN
								-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
								--Set @remainingQuantityToLock as positive quanitty
								SET @remainingQuantityToLock = ABS(@remainingQuantityToLock);
								

								BEGIN TRY
									DECLARE itriCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
												INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
												AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
												WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
												AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
												AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;


									OPEN itriCursor;
									FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock
									WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

										IF (@remainingQuantityToLock < @freeQuantityToLock)
										BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END
										ELSE
										BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
										END

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ( "BaseType", "BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
										SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(ORDR."CardCode"),MAX(ORDR."CardName"),
											@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
											MAX(ORDR."UserSign"),@updateDate,
											@updateTime,NULL, @qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, @currentQuantityToLock * @catchWeightRatioToLock
										FROM ORDR 
										INNER JOIN RDR1 ON RDR1."DocEntry" = ORDR."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = RDR1."WhsCode"
										WHERE ORDR."DocEntry" = @docEntryToLock
										AND RDR1."LineNum" = @lineNumToLock;

										--Reduce remaining
										SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


										FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									END
									CLOSE itriCursor;
									DEALLOCATE itriCursor

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(@remainingQuantityToLock > 0)
									BEGIN
										SET @error = 50652;
										SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN;
									END
								END TRY
								BEGIN CATCH
									SET @error = 50691
									SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
									+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE itriCursor 
										DEALLOCATE itriCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE itriCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END CATCH
								END CATCH


							END


						END


						IF @remainingQuantityToLock <> 0 BEGIN
						-- if quantity is still different from 0, then there was not enough stock to reserve!
							CLOSE salesOrderBatchCursor 
							DEALLOCATE salesOrderBatchCursor

							SET @error = 50653;
							SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END

					END

					FETCH NEXT FROM salesOrderBatchCursor
					INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock, @pmxWarehouseToLock, @catchWeightRatioToLock, @shippingQualityOptionToLock;


				END
				CLOSE salesOrderBatchCursor;
				DEALLOCATE salesOrderBatchCursor
			END TRY
			BEGIN CATCH
				SET @error = 50699
				SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
				+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE salesOrderBatchCursor 
					DEALLOCATE salesOrderBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE salesOrderBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END CATCH
			END CATCH

			--**********************************************************************************************************
			--END: Add/update locking based on batch numbers
			--**********************************************************************************************************

			--**********************************************************************************************************
			--START: SAP Serial numbers
			--**********************************************************************************************************

			--Delete system serial links
			DELETE FROM PMX_SELD WHERE "LinkStatus" = 'S'
			AND PMX_SELD."BaseEntry" = @keyAsInteger
			AND PMX_SELD."BaseType" = '17';

			--Add to PMX serial numbers table

			INSERT INTO PMX_SELD ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine", "LinkStatus") 
			SELECT 'N', SERIAL."UserSign", @tsNow, @tsNow, 1, PMX_SENU."InternalKey", "ApplyType", "ApplyEntry", "ApplyLine", 'S'	
			FROM (
                SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty", "StockEff", ORDR."UserSign"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
				AND ITL1."SysNumber" = OSRN."SysNumber"
				INNER JOIN ORDR ON ORDR."DocEntry" = OITL."ApplyEntry"
				INNER JOIN RDR1 ON RDR1."DocEntry" = OITL."ApplyEntry"
				AND RDR1."LineNum" = OITL."ApplyLine"
				AND RDR1."ObjType" = OITL."ApplyType"

				WHERE "StockEff" = 2
				AND OITL."ApplyEntry" = @keyAsInteger
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", ORDR."UserSign"
				HAVING SUM("AllocQty") > 0 AND "DistNumber" IS NOT NULL
			) AS SERIAL

			INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = SERIAL."ItemCode"
			AND PMX_SENU."SerialNumber" = SERIAL."DistNumber"
			LEFT JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
			AND PMX_SELD."BaseEntry" = SERIAL."ApplyEntry"
			AND PMX_SELD."BaseLine" = SERIAL."ApplyLine"
			AND PMX_SELD."BaseType" = N'17'
			WHERE PMX_SELD."InternalKey" IS NULL;



			--**********************************************************************************************************
			--END: SAP Serial numbers
			--**********************************************************************************************************


			RETURN;

		END


		--**********************************************************************************************************
		--Do checks on a purchase order - ERROR 51300 - 51399
		--**********************************************************************************************************
		IF (@isPurchaseOrderDocument=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			IF(@isTransactionTypeAdd=0 AND
				EXISTS(SELECT DocEntry FROM POR1 WHERE POR1."LineStatus" = 'C' AND POR1."DocEntry" = @keyAsInteger)) 
			BEGIN

				--Close container (lines) if needed
				UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
				FROM PMX_COLI
				INNER JOIN POR1
				ON PMX_COLI."BaseEntry" = POR1."DocEntry"
				AND PMX_COLI."BaseLine" = POR1."LineNum"
				AND PMX_COLI."BaseType" = POR1."ObjType"
				WHERE PMX_COLI."LineStatus" <> 'C'
				AND POR1."DocEntry" = @keyAsInteger
				AND POR1."LineStatus" = 'C';

				--Close header is no open lines are found
				UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
				PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
				PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM OPOR  WHERE OPOR."DocEntry" = @keyAsInteger)
				WHERE PMX_COHE."DocStatus" = 'O'
				AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

			END		
			
			IF ( @isTransactionTypeAdd = 0 )
			BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(PMX_COLI."DocEntry" AS VARCHAR) + ' ' 
				FROM PMX_COLI 
				LEFT JOIN POR1 ON POR1."DocEntry" = PMX_COLI."BaseEntry" AND POR1."LineNum" = PMX_COLI."BaseLine" 
					 WHERE PMX_COLI."LineStatus" <> 'C' 
					 AND PMX_COLI."BaseEntry" = @keyAsInteger 
					 AND PMX_COLI."BaseType" = N'22'
					 AND POR1."ItemCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 51300;
					SET @error_message = 'Cannot delete purchase order line(s) because of container-document(s): ' + @errorList + '. These container-document(s) have links to the purchase order line(s).'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END			
	
			RETURN;
		END
		
		--**********************************************************************************************************
		--Do checks on a warehouse: ERROR 50850 - 50900
		--**********************************************************************************************************
		IF (@isWarehouse=1) BEGIN
			--Check if bin locations are not enabled for PMX managed whs
			SET @errorList = ''
			SELECT @errorList = @errorList + OWHS."WhsCode" + ' ' 
			FROM OWHS
			WHERE OWHS."WhsCode" = @list_of_cols_val_tab_del
			AND OWHS."BinActivat" = 'Y'
			AND OWHS.U_PMX_IMBP = 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50850;
				SET @error_message = 'Warehouse with code''' + @list_of_cols_val_tab_del + ''' cannot have BIN locations enabled ' 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			
			IF EXISTS (SELECT OWHS."WhsCode" FROM OWHS
			WHERE OWHS."WhsCode" = @list_of_cols_val_tab_del
			AND OWHS.U_PMX_IMBP = 'Y')  BEGIN
				--The warehouse needs to be managed by Produmex.
				--So now check if this is allowed
				
				IF(EXISTS (SELECT OITW."WhsCode" FROM OITW 
					LEFT JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OITW."WhsCode"
					LEFT JOIN PMX_OSEL ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
					LEFT JOIN PMX_INVT ON PMX_INVT."StorLocCode" = PMX_OSEL."Code"
					WHERE "WhsCode" = @list_of_cols_val_tab_del 
					AND "OnHand" > 0
					GROUP BY OITW."WhsCode"
					HAVING ISNULL( SUM(ABS(PMX_INVT."Quantity")), 0) = 0 )) BEGIN
					
					SET @error = 50851;
					SET @error_message = 'Warehouse with code''' + @list_of_cols_val_tab_del + ''' cannot be managed by Produmex, because there is stock in SAP, but not in PMX.' 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				
				END

			END
			
			IF EXISTS (SELECT OWHS."WhsCode" FROM OWHS
			WHERE OWHS."WhsCode" = @list_of_cols_val_tab_del
			AND OWHS.U_PMX_IMBP = 'N')  BEGIN
				--The warehouse does not need to be managed by Produmex.
				--So now check if this is allowed
				
				IF(EXISTS (SELECT TOP(1) PMX_INVT."InternalKey"  FROM PMX_INVT 
						LEFT JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode"
						LEFT JOIN PMX_OSWH ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
						WHERE "SboWhsCode" = @list_of_cols_val_tab_del )) BEGIN
					
					SET @error = 50852;
					SET @error_message = 'Warehouse with code''' + @list_of_cols_val_tab_del + ''' needs to be managed by Produmex, because there is stock in PMX.' 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			END
    
            RETURN;
		END


		--**********************************************************************************************************
		--Do checks on a barcode: ERROR 51900 - 51950
		--**********************************************************************************************************
		IF (@isBarcode=1) BEGIN
		
			DECLARE @sqlStatementBeginBarcode AS NVARCHAR(max)
			DECLARE @sqlStatementEndBarcode AS NVARCHAR(max)
			DECLARE @sqlStatementBarcode AS NVARCHAR(max)
			DECLARE @parameterListBarcode AS NVARCHAR(max)

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
		
			--Check the GTINs of the item
			SET @sqlStatementBeginBarcode = 'SELECT @error=Error, @error_message=Message FROM ' + @extraDb + '.dbo.PMX_FN_CheckGTIN(''';
			SET @sqlStatementEndBarcode = ''')';
			SET @parameterListBarcode = '@error INT output, @error_message NVARCHAR(200) output';
			
			
			--Check the GTIN of the uom barcodes
			DECLARE @barcodeForBarcode NVARCHAR(254);
			DECLARE @barcodeItemCode NVARCHAR(254);
			DECLARE @barcodeTypeForBarcode NVARCHAR(1);
			DECLARE barCodeCursor CURSOR LOCAL FOR SELECT "BcdCode", OBCD."ItemCode",ISNULL( U_PMX_BCTY,'G') 
														FROM OBCD 
														INNER JOIN OITM ON OITM."ItemCode" = OBCD."ItemCode"
														WHERE OBCD."BcdEntry" = @list_of_cols_val_tab_del AND LEN("BcdCode") <> 0 
														AND (OITM.U_PMX_BCTY IN ('G', 'V'));
			OPEN barCodeCursor;
			FETCH NEXT FROM barCodeCursor INTO @barcodeForBarcode, @barcodeItemCode, @barcodeTypeForBarcode;
			WHILE @@FETCH_STATUS >= 0 BEGIN
				SET @sqlStatementBarcode = @sqlStatementBeginBarcode + @barcodeForBarCode+ ''', ''' + @barcodeTypeForBarcode + @sqlStatementEndBarcode;
				EXECUTE sp_executesql @sqlStatementBarcode, @parameterListBarcode, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcodeForBarcode +  '''  of Item ' + @barcodeItemCode + ': ' + @error_message;
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					CLOSE barCodeCursor;
					DEALLOCATE barCodeCursor

					RETURN;
				END

				FETCH NEXT FROM barCodeCursor INTO @barcodeForBarcode, @barcodeItemCode, @barcodeTypeForBarcode
			END
			CLOSE barCodeCursor;
			DEALLOCATE barCodeCursor
			
            RETURN;
		END

		--**********************************************************************************************************
		--Do actions on change of UomGroup
		--**********************************************************************************************************
		IF @isUomGroup = 1 BEGIN

			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN

				DELETE FROM PMX_IPTY
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				LEFT JOIN (
					UGP1
					INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				) ON UGP1."UgpEntry" = OITM."UgpEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."UgpEntry" = @list_of_cols_val_tab_del 
				AND OUOM."UomCode" IS NULL

				UPDATE PMX_IPTY SET "Quantity" = UGP1."BaseQty" / UGP1."AltQty", "UserSign" = COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 )
                    , "UpdateDateTime" = @tsNow, "Version" = "Version" + 1
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
				INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."UgpEntry" = @list_of_cols_val_tab_del 
  				  AND "Quantity" <> CAST( UGP1."BaseQty" / UGP1."AltQty" AS DECIMAL(19,6) );

				INSERT INTO PMX_IPTY ( "UserSign", "CreateDateTime", "Version", "ItemCode", "PackagingTypeCode", "Quantity", "BarcodeType", "Canceled", "HideDuringEnteringQuantity", "AskDuringReception" )
				SELECT COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 ), @tsNow, 1, OITM."ItemCode", OUOM."UomCode", UGP1."BaseQty" / UGP1."AltQty", 'G', 'N', 'N', 'N'
				FROM OITM
				INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
				INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode" AND PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
				WHERE OITM."UgpEntry" = @list_of_cols_val_tab_del 
				  AND UGP1."UomEntry" <> OITM."IUoMEntry"
				  AND PMX_IPTY."PackagingTypeCode" IS NULL;

			END

	--This code is in comment because for some reason the delete is not captured!

	--		IF (@isTransactionTypeDelete=1 OR @isTransactionTypeUpdate = 1) BEGIN
	--			--Check if the uom is used on item
	--			SET @errorList = ''
	--			SELECT @errorList = @errorList + [OITM]."ItemCode" + ' ' 
	--			FROM [OITM]
	--			WHERE "BuyUnitMsr" = @list_of_cols_val_tab_del OR
	--					"SalUnitMsr" = @list_of_cols_val_tab_del OR
	--					"InvntryUom" = @list_of_cols_val_tab_del OR
	--					U_PMX_UOM2 = @list_of_cols_val_tab_del
	--			IF LEN(@errorList) <> 0 BEGIN
	--				SET @error = 80002
	--				SET @error_message = 'PmxUom with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted or updated because it is used in items with code ''' + @errorList + '''.'
	--				SELECT @error, @error_message
	--				EXEC xp_logevent @error, @error_message, ERROR
	--				RETURN
	--			END 
	--		END

            RETURN;
		END


		--**********************************************************************************************************
		--Do checks on an PmxPriority
		--**********************************************************************************************************
		IF (@isPriority=1) BEGIN
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN
                --Check if the Msequence is not null
				SET @errorList = ''
				SELECT @errorList = @errorList + [@PMX_PRIO]."Code" + ' ' 
				FROM "@PMX_PRIO"
				WHERE "@PMX_PRIO"."Code" = @list_of_cols_val_tab_del AND "@PMX_PRIO".U_PMX_SEQU IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80101;
					SET @error_message = 'PmxPriority with code ''' + @list_of_cols_val_tab_del + ''' can not be added or updated because the sequence is not filled in.';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

				--Check if the Sequence is unique
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PRIO"."Code" + ' ' 
				FROM "@PMX_PRIO"
				WHERE "@PMX_PRIO"."Code" <> @list_of_cols_val_tab_del 
				AND "@PMX_PRIO".U_PMX_SEQU = (SELECT TOP(1) "@PMX_PRIO".U_PMX_SEQU FROM "@PMX_PRIO" WHERE "@PMX_PRIO"."Code" = @list_of_cols_val_tab_del);
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80102;
					SET @error_message = 'PmxPriority with code ''' + @list_of_cols_val_tab_del + ''' can not be added or updated because the sequence is not unique';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

			END

            RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an @isFreightChargeDefinition
		--**********************************************************************************************************
		IF (@isFreightChargeDefinition = 1) BEGIN
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN
				--Check if shipping type exists in table
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_FCDE".U_PMX_SHTY + ' ' 
				FROM "@PMX_FCDE" LEFT JOIN "OSHP"
				ON OSHP."TrnspCode" = "@PMX_FCDE".U_PMX_SHTY
				WHERE "@PMX_FCDE"."Code" = @list_of_cols_val_tab_del AND "OSHP"."TrnspCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80201;
					SET @error_message = 'Shipping type with code ''' + @errorList + ''' does not exist.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

				--Check if freight exists in table
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_FCDE".U_PMX_FCDE + ' ' 
				FROM "@PMX_FCDE" LEFT JOIN "OEXD"
				ON OEXD."ExpnsCode" = "@PMX_FCDE".U_PMX_FCDE
				WHERE "@PMX_FCDE"."Code" = @list_of_cols_val_tab_del AND "OEXD"."ExpnsCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80202;
					SET @error_message = 'Freight charge with code ''' + @errorList + ''' does not exist.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

			END

            RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an ItemPackagingType
		--**********************************************************************************************************
		IF (@isItemPackagingType = 1) BEGIN
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN

				SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

				--Check if packaging type exists in table
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IPTY."PackagingTypeCode" + ' ' 
				FROM PMX_IPTY 
                LEFT JOIN OUOM
				ON PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
				WHERE PMX_IPTY."InternalKey" = @keyAsInteger AND OUOM."UomCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81001;
					SET @error_message = 'ItemPackagingType with code ''' + @errorList + ''' does not exist.';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

			END

            RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an PickListType
		--**********************************************************************************************************
		IF (@isPickListType = 1) BEGIN
			IF (@isTransactionTypeAdd = 1 OR @isTransactionTypeUpdate = 1) BEGIN

				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."U_PMX_ItemPickPLTY" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  "@PMX_PLTY" PLTY_ITEM
				ON PLTY_ITEM."Code" = "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				WHERE PLTY_ITEM.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81051;
					SET @error_message = 'Split up for for items ''' + @errorList + ''' is not of the same type as for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."U_PMX_FullPalletPLTY" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81052;
					SET @error_message = 'Split up type for full ''' + @errorList + ''' is not of the same type as for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if proposals exist with production that do not correspond to pick object type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  PMX_PLPH
				ON PMX_PLPH."PickListType" = "@PMX_PLTY"."Code"
				WHERE PMX_PLPH."PickObjType" = 'Production' 
				AND "@PMX_PLTY".U_PMX_PROD = 'N'
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del 
				AND PMX_PLPH."DocStatus" = 'O';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81053;
					SET @error_message = 'Proposals of pick object type ''production'' exist for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if proposals exist without production that do not correspond to pick object type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  PMX_PLPH
				ON PMX_PLPH."PickListType" = "@PMX_PLTY"."Code"
				WHERE PMX_PLPH."PickObjType" <> 'Production' 
				AND "@PMX_PLTY".U_PMX_PROD = 'Y'
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del 
				AND PMX_PLPH."DocStatus" = 'O';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81054;
					SET @error_message = 'Proposals of pick object type different from ''production'' exist for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."U_PMX_FullPalletPLTY" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81055;
					SET @error_message = 'The pick list type is used as split for full pallets, but the type does not correspond  on ''' + @errorList 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."U_PMX_ItemPickPLTY" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81056;
					SET @error_message = 'The pick list type is used as split for full pallets, but the type does not correspond  on ''' + @errorList 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
			END

            RETURN;
		END


		--**********************************************************************************************************
		--Do checks on an Warehouse transfer request
		--**********************************************************************************************************
		IF @isWarehouseTransferRequestDocument = 1 BEGIN

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--Check if the warehouse inventory request is closed that there are no open pick list proposals of this warehouse inventory request
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM OWTQ 
			INNER JOIN PMX_PLPL ON OWTQ."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = OWTQ."ObjType"
			INNER JOIN WTQ1 ON OWTQ."DocEntry" = WTQ1."DocEntry" AND WTQ1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE OWTQ."DocEntry" = @keyAsInteger AND (OWTQ."DocStatus" = 'C' OR WTQ1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51200;
				SET @error_message = 'The warehouse inventory request ''' + @list_of_cols_val_tab_del + ''' has open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before closing the warehouse inventory request (line).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the full quantity is selected when entering batch numbers
--			SET @errorList = ''
--			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
--			FROM WTQ1 AS Lines
--			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
--			LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = WhsInvRequest."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
--			WHERE WhsInvRequest."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O'
--			GROUP BY Lines."LineNum", Lines."Quantity", Lines."NumPerMsr"
--			HAVING SUM(IBT1."Quantity") <> (Lines."Quantity" * Lines."NumPerMsr") AND SUM(IBT1."Quantity") > 0
--			IF LEN(@errorList) <> 0 BEGIN
--				SET @error = 51201
--				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
--				SELECT @error, @error_message
--				EXEC xp_logevent @error, @error_message, ERROR
--				RETURN
--			END

			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM WTQ1 AS Lines
            INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
            INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocatEnt" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
            INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
            INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
            AND ITL1."SysNumber" = OBTN."SysNumber"
            WHERE OITL."AllocatEnt" = @keyAsInteger 
			AND Lines."LineStatus" = 'O'
            AND "StockEff" = 2
            GROUP BY Lines."LineNum", Lines."OpenQty", Lines."NumPerMsr"
            HAVING SUM("AllocQty") <> (Lines."OpenQty" * Lines."NumPerMsr") AND SUM("AllocQty") > 0;
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51201;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			--Check if warehouse inventory request lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the warehouse inventory request line is lower than on the open pick list proposal line / open pick list line
			SET @errorList = ''
			SELECT @errorList  = @errorList  + CASE WHEN        WTQ1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN WTQ1 ON PMX_PLPL."BaseEntry" = WTQ1."DocEntry" AND PMX_PLPL."BaseLine" = WTQ1."LineNum"  AND PMX_PLPL."BaseType" = WTQ1."ObjType"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = 1250000001 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR ISNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51203;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' The item code on a warehouse inventory request line cannot be changed if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList2 = ''
			SELECT @errorList2 = @errorList2 + CASE WHEN ISNULL( WTQ1."OpenQty",0) < SUM(PMX_PLPL."Quantity") THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN WTQ1 ON PMX_PLPL."BaseEntry" = WTQ1."DocEntry" AND PMX_PLPL."BaseLine" = WTQ1."LineNum"  AND PMX_PLPL."BaseType" = WTQ1."ObjType"
			WHERE PMX_PLPL."BaseType" = 1250000001 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C') 
			GROUP BY PMX_PLPL."BaseLine", WTQ1."OpenQty";
			IF LEN(@errorList2) <> 0 BEGIN
				SET @error = 51202;
				SET @error_message = 'Error on line(s):  ' + @errorList2 + ' The quantity on a warehouse inventory request line cannot be lowered if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM WTQ1 AS Lines
			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = WhsInvRequest."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"
						
			WHERE WhsInvRequest."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O' AND Lines."U_PMX_AMBA" = 'N'
			GROUP BY Lines."LineNum"
			HAVING (Count(*)) > 1;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51204;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			-- change of batch only allowed when not linked in PMX_PLPL and PMX_DOLL
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '		
			FROM WTQ1 AS Lines
			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"	
			
			INNER JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType" 
			AND PMX_PLPL."BaseEntry" = Lines."DocEntry" 
			AND PMX_PLPL."BaseLine" = Lines."LineNum" 
			
			INNER JOIN PMX_DOLL ON PMX_DOLL."ObjType" = Lines."ObjType" 
			AND PMX_DOLL."DocEntry" = Lines."DocEntry" 
			AND PMX_DOLL."LineNum" = Lines."LineNum" 
			
			INNER JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			
			LEFT JOIN 
			(	SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
								GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH

			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"		

			WHERE Lines."DocEntry" = @keyAsInteger
			AND Lines."LineStatus" = 'O'
			AND( ISNULL( BATCH."AllocQty", PMX_DOLL."LockedQuantity") < PMX_DOLL."LockedQuantity"
			OR BATCH."DistNumber" <> ISNULL( PMX_ITRI."BatchNumber", BATCH."DistNumber"));
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51205;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' Proposal exists: Changing the batch number(s) not allowed'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Multiple batches on serial numbers are only allowed when multiple batches on line level are allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(WTQ1."LineNum" AS VARCHAR) + ' '	
			FROM PMX_SELD	
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN WTQ1 on WTQ1."DocEntry" = PMX_SELD."BaseEntry"
			AND WTQ1."LineNum" = PMX_SELD."BaseLine"
			AND WTQ1."ObjType" = PMX_SELD."BaseType"
			WHERE WTQ1.U_PMX_AMBA = 'N'
			AND WTQ1."DocEntry" = @keyAsInteger	
			GROUP BY WTQ1."LineNum", PMX_SENU."ItriKey"
			HAVING COUNT(PMX_SENU."ItriKey") > 1;
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51206;
				SET @error_message = 'Serial numbers linked with multiple batches, but multiple batches are not allowed. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Linked serial numbers cannot be more than open quantity
			SET @errorList = ''
			SELECT @errorList = @errorList + 'Line ' + CAST(WTQ1."LineNum" AS VARCHAR) + ', Link qty:' + CAST(COUNT(PMX_SELD."InternalKey") AS VARCHAR) + '; '	
			FROM PMX_SELD
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN WTQ1 on WTQ1."DocEntry" = PMX_SELD."BaseEntry"
			AND WTQ1."LineNum" = PMX_SELD."BaseLine"
			AND WTQ1."ObjType" = PMX_SELD."BaseType"
			WHERE WTQ1."DocEntry" = @keyAsInteger
			AND PMX_SENU."InStock" = 'Y'
			AND WTQ1."LineStatus" = 'O'
			GROUP BY WTQ1."LineNum", WTQ1."OpenQty", WTQ1."NumPerMsr"
			HAVING COUNT(PMX_SELD."InternalKey") > WTQ1."OpenQty" * WTQ1."NumPerMsr";

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51207;
				SET @error_message = 'Number of linked serial numbers cannot be more than open quantity. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date minutes
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(OWTQ.U_PMX_DDTM AS VARCHAR) 
			FROM OWTQ
			WHERE OWTQ."DocEntry" = @keyAsInteger
			AND ( ISNULL( OWTQ.U_PMX_DDTM, 0) > 59 OR
				ISNULL( OWTQ.U_PMX_DDTM, 0) < 0 );

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51208;
				SET @error_message = 'The due date minutes: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date Hour
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(OWTQ.U_PMX_DDTH AS VARCHAR) 
			FROM OWTQ
			WHERE OWTQ."DocEntry" = @keyAsInteger
			AND (ISNULL( OWTQ.U_PMX_DDTH, 0) < 0
					OR (ISNULL( OWTQ.U_PMX_DDTH, 0) > 23 AND ISNULL( OWTQ.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR (ISNULL( OWTQ.U_PMX_DDTH, 0) > 11 AND ISNULL( OWTQ.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51209;
				SET @error_message = 'The due date hours: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE
			DELETE FROM LOCKING
            FROM PMX_INLD LOCKING
            INNER JOIN OWTQ ON OWTQ."DocEntry" = LOCKING."BaseEntry"
			AND OWTQ."ObjType" = LOCKING."BaseType"
			WHERE OWTQ."DocStatus" = 'C' AND OWTQ."DocEntry" = @keyAsInteger;


			--IF(@isTransactionTypeAdd=0 AND
			--	EXISTS(SELECT * FROM WTQ1 WHERE WTQ1."LineStatus" = 'C' AND WTQ1."DocEntry" = @keyAsInteger)) 
			--BEGIN

			--	--Close container (lines) if needed
			--	UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			--	FROM PMX_COLI
			--	INNER JOIN WTQ1
			--	ON PMX_COLI."BaseEntry" = WTQ1."DocEntry"
			--	AND PMX_COLI."BaseLine" = WTQ1."LineNum"
			--	AND PMX_COLI."BaseType" = WTQ1."ObjType"
			--	WHERE PMX_COLI."LineStatus" <> 'C'
			--	AND WTQ1."DocEntry" = @keyAsInteger
			--	AND WTQ1."LineStatus" = 'C'

			--	--Close header if no open lines are found
			--	UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
			--	PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
			--	PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM OWTQ  WHERE OWTQ."DocEntry" = @keyAsInteger)
			--	WHERE PMX_COHE."DocStatus" = 'O'
			--	AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O')

			--END


			--**********************************************************************************************************
			--START: Add/update locking based on batch numbers
			--**********************************************************************************************************



			--Start with deleting the lockings that are not correct
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" NOT IN 
			(--Query to get the correct lockings
				SELECT 	PMX_INLD."InternalKey"
				FROM WTQ1 AS Lines
				INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
				INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')
				--AND ISNULL( PMX_ITRI."BestBeforeDate", 1) = ISNULL( "ExpDate", 1)

				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"
				AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

				WHERE OITL."ApplyEntry" = @keyAsInteger 
				AND Lines."LineStatus" = 'O'
				AND "StockEff" = 2

				AND PMX_INLD."InternalKey" IS NOT NULL


				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				HAVING SUM("AllocQty") > 0

				-- Exclude locked stock created on goods receipt.
				UNION SELECT PMX_INLD."InternalKey"
				FROM WTQ1 AS Lines 
				INNER JOIN POR1 ON 
				POR1."BaseType" = Lines."ObjType"
				AND POR1."BaseEntry" = Lines."DocEntry"
				AND POR1."BaseLine" = Lines."LineNum"

				INNER JOIN PDN1 ON 
				PDN1."BaseType" = POR1."ObjType"
				AND PDN1."BaseEntry" = POR1."DocEntry"
				AND PDN1."BaseLine" = POR1."LineNum"

				INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
				LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
				LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')


				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"

				WHERE Lines."DocEntry" = @keyAsInteger				

				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				--Only allow detailed locking from purchase order when no batch has been linked
				HAVING SUM( ISNULL( "AllocQty", 0) ) = 0 

				--Exclude manual lockings on LUID level when no batch has been selected on warehouse inventory request
				UNION SELECT PMX_INLD."InternalKey"
				FROM PMX_INLD 
				LEFT JOIN
				(
					SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
					FROM WTQ1 AS Lines
									INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
									LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
									LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
									LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
									AND ITL1."SysNumber" = OBTN."SysNumber"

					WHERE OITL."ApplyEntry"  = @keyAsInteger 
									AND Lines."LineStatus" = 'O'
									AND ISNULL( "StockEff", 2) = 2

						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
									Lines."NumPerMsr", "DistNumber", "ExpDate"

					HAVING SUM( ISNULL( "AllocQty", 0) ) > 0
				) AS OWTQ

				ON OWTQ."LineNum" = PMX_INLD."BaseLine"
				AND OWTQ."DocEntry" = PMX_INLD."BaseEntry"
				AND OWTQ."ObjType" = PMX_INLD."BaseType"


				WHERE PMX_INLD."InvLockLevel" = 'L'
				AND PMX_INLD."BaseEntry" = @keyAsInteger
				AND OWTQ."DocEntry" IS NULL

				

			) 

			AND PMX_INLD."BaseType" = '1250000001'
			AND PMX_INLD."BaseEntry" = @keyAsInteger;



			BEGIN TRY

				DECLARE WhsInvRequestBatchCursor CURSOR LOCAL FOR 
						SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."ItemCode", PMX_OSWH."Code"
						FROM WTQ1 AS Lines
						INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
						INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = WhsInvRequest."Filler"

						LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType"
						AND PMX_PLPL."BaseEntry" = Lines."DocEntry"
						AND PMX_PLPL."BaseLine" = Lines."LineNum"
						AND PMX_PLPL."LineStatus" = 'O'

						WHERE OITL."ApplyEntry" = @keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2
						--Only when no openpick list proposal is found
						AND PMX_PLPL."DocEntry" IS NULL
						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code"

						HAVING SUM("AllocQty") > 0;


				OPEN WhsInvRequestBatchCursor;
				FETCH NEXT FROM WhsInvRequestBatchCursor 
				INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock,@pmxWarehouseToLock;
				WHILE @@FETCH_STATUS >= 0 BEGIN

					--Check for each batch if we have a corresponding locking line
					IF NOT EXISTS (SELECT TOP(1) PMX_INLD.InternalKey FROM PMX_INLD 
    					INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
    					WHERE PMX_INLD."BaseType" = @typeToLock
    					AND PMX_INLD."BaseEntry" = @docEntryToLock
    					AND PMX_INLD."BaseLine" = @lineNumToLock
    					AND PMX_ITRI."BatchNumber" = @batchToLock
    					--AND ISNULL( PMX_ITRI."BestBeforeDate", '') = ISNULL( @expDateToLock, '')
					)
					BEGIN

						--Entry does not exist, so we need to lock the stock
						--First try to get the itri key to lock
						SET @remainingQuantityToLock = @batchQuantityToLock;

						BEGIN TRY
							DECLARE WhsInvRequestinldCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;


							OPEN WhsInvRequestinldCursor;
							FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
							WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

								IF(@remainingQuantityToLock < @freeQuantityToLock)
								BEGIN
									SET @currentQuantityToLock = @remainingQuantityToLock
								END
								ELSE
								BEGIN
									SET @currentQuantityToLock = @freeQuantityToLock
								END

								--Insert INTO the PMX_INLD table
								INSERT INTO PMX_INLD ( "BaseType","BaseEntry","BaseLine","CardCode","CardName",
									"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
									"UserSign","CreateDate",
									"CreateTime","LogUnitIdentKey","QualityStatusCode", 
									"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
								SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(OWTQ."CardCode"),MAX(OWTQ."CardName"),
									@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
									MAX(OWTQ."UserSign"),@updateDate,
									@updateTime,NULL, @qualityStatusToLock,
									'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0
								FROM OWTQ 
								INNER JOIN WTQ1 ON WTQ1."DocEntry" = OWTQ."DocEntry"
								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OWTQ."Filler"
								WHERE OWTQ."DocEntry" = @docEntryToLock
								AND WTQ1."LineNum" = @lineNumToLock

								--Reduce remaining
								SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


								FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
							END
							CLOSE WhsInvRequestinldCursor;
							DEALLOCATE WhsInvRequestinldCursor

							--If there is remaining QTY, there is not enough free stock, so throw error
							IF(@remainingQuantityToLock > 0)
							BEGIN
								SET @error = 51250;
								SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock + '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;

							END

						END TRY
						BEGIN CATCH
							SET @error = 51290
							SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
							+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
							BEGIN TRY
								CLOSE WhsInvRequestinldCursor 
								DEALLOCATE WhsInvRequestinldCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN
							END TRY
							BEGIN CATCH
								DEALLOCATE WhsInvRequestinldCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN
							END CATCH
						END CATCH

					END
					ELSE 
					BEGIN
						--The line exists, so now we need to check if the quantity is the same/more/less
						SET @remainingQuantityToLock = (
                            SELECT SUM(PMX_INLD."Quantity") -  @batchQuantityToLock AS "Quantity"
                            FROM PMX_INLD 
							INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
							WHERE PMX_INLD."BaseType" = @typeToLock
							AND PMX_INLD."BaseEntry" = @docEntryToLock
							AND PMX_INLD."BaseLine" = @lineNumToLock
							AND PMX_ITRI."BatchNumber" = @batchToLock
							GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine"
                        );
							
										
						IF @remainingQuantityToLock <> 0 BEGIN
						--Only if deviation, then we try to update/insert
						--Loop through all locking lines
							BEGIN TRY
								DECLARE WhsInvRequestinldCursor CURSOR LOCAL FOR SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey" FROM PMX_INLD 
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = @typeToLock
								AND PMX_INLD."BaseEntry" = @docEntryToLock
								AND PMX_INLD."BaseLine" = @lineNumToLock
								AND PMX_ITRI."BatchNumber" = @batchToLock;

								OPEN WhsInvRequestinldCursor;
								FETCH NEXT FROM WhsInvRequestinldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock <> 0 BEGIN

									IF @remainingQuantityToLock > 0 BEGIN

										--Remaining is more than 0, so we can just remove the locking
										IF @remainingQuantityToLock >= @lockedQuantity BEGIN
											--Delete
											DELETE FROM PMX_INLD WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = @remainingQuantityToLock - @lockedQuantity;
										END
                                        ELSE
                                        BEGIN
											--update
											UPDATE PMX_INLD SET "Quantity" = "Quantity" - @remainingQuantityToLock WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = 0;
										END
									END
                                    ELSE
                                    BEGIN
										--Remaining is less than 0, so we need to add locking
										--If there is still free stock for current itri, then take that one
										SELECT @freeQuantityToLock = ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0
										AND PMX_ITRI."InternalKey" = @itriToLock;

										IF @remainingQuantityToLock > @freeQuantityToLock BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
    									END
                                        ELSE
                                        BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END

										UPDATE PMX_INLD SET "Quantity" = "Quantity" + @currentQuantityToLock WHERE PMX_INLD."InternalKey" = @currentLockInternalKey;

										-- add qty, because @remainingQuantityToLock is negative
										SET @remainingQuantityToLock = @remainingQuantityToLock + @currentQuantityToLock;
										
									END

									FETCH NEXT FROM WhsInvRequestinldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								END
								CLOSE WhsInvRequestinldCursor;
								DEALLOCATE WhsInvRequestinldCursor

							END TRY
							BEGIN CATCH
								SET @error = 51291
								SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
								+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
								BEGIN TRY
									CLOSE WhsInvRequestinldCursor 
									DEALLOCATE WhsInvRequestinldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END TRY
								BEGIN CATCH
									DEALLOCATE WhsInvRequestinldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END CATCH
							END CATCH

							IF @remainingQuantityToLock <> 0 BEGIN
								-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
								--Set @remainingQuantityToLock as positive quanitty
								SET @remainingQuantityToLock = ABS( @remainingQuantityToLock );
								

								BEGIN TRY
									DECLARE WhsInvRequestinldCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
									LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
									INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
									WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
									LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
									WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
									AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
									WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
									AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
									AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;

									OPEN WhsInvRequestinldCursor;
									FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

										IF @remainingQuantityToLock < @freeQuantityToLock
											SET @currentQuantityToLock = @remainingQuantityToLock;
										ELSE
											SET @currentQuantityToLock = @freeQuantityToLock;

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ( "BaseType","BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
										SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(OWTQ."CardCode"),MAX(OWTQ."CardName"),
											@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
											MAX(OWTQ."UserSign"),@updateDate,
											@updateTime,NULL, @qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0
										FROM OWTQ 
										INNER JOIN WTQ1 ON WTQ1."DocEntry" = OWTQ."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OWTQ."Filler"
										WHERE OWTQ."DocEntry" = @docEntryToLock
										AND WTQ1."LineNum" = @lineNumToLock

										--Reduce remaining
										SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


										FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									END
									CLOSE WhsInvRequestinldCursor;
									DEALLOCATE WhsInvRequestinldCursor

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(@remainingQuantityToLock > 0)
									BEGIN
										SET @error = 51252;
										SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN;
									END

								END TRY
								BEGIN CATCH
									SET @error = 51291
									SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
									+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE WhsInvRequestinldCursor 
										DEALLOCATE WhsInvRequestinldCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE WhsInvRequestinldCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END CATCH
								END CATCH

							END

						END

						IF @remainingQuantityToLock <> 0 BEGIN
						-- if quantity is still different from 0, then there was not enough stock to reserve!
							CLOSE WhsInvRequestBatchCursor;
							DEALLOCATE WhsInvRequestBatchCursor;
							SET @error = 51253;
							SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END

					END


					FETCH NEXT FROM WhsInvRequestBatchCursor
					INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock, @pmxWarehouseToLock;
				END
				CLOSE WhsInvRequestBatchCursor;
				DEALLOCATE WhsInvRequestBatchCursor;
			END TRY
			BEGIN CATCH
				SET @error = 51299
				SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
				+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE WhsInvRequestBatchCursor 
					DEALLOCATE WhsInvRequestBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE WhsInvRequestBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END CATCH
			END CATCH

			--**********************************************************************************************************
			--END: Add/update locking based on batch numbers
			--**********************************************************************************************************

			--**********************************************************************************************************
			--START: SAP Serial numbers
			--**********************************************************************************************************

			/*--Delete system serial links
			DELETE FROM PMX_SELD WHERE "LinkStatus" = 'S'
			AND PMX_SELD."BaseEntry" = @keyAsInteger
			AND PMX_SELD."BaseType" = '1250000001';

			--Add to PMX serial numbers table

			INSERT INTO PMX_SELD ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine", "LinkStatus") 
			SELECT 'N', SERIAL."UserSign", @tsNow, @tsNow, 1, PMX_SENU."InternalKey", "ApplyType", "ApplyEntry", "ApplyLine", 'S'	
			FROM 
				(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty", "StockEff", OWTQ."UserSign"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
				AND ITL1."SysNumber" = OSRN."SysNumber"
				INNER JOIN OWTQ ON OWTQ."DocEntry" = OITL."ApplyEntry"
				INNER JOIN WTQ1 ON WTQ1."DocEntry" = OITL."ApplyEntry"
				AND WTQ1."LineNum" = OITL."ApplyLine"
				AND WTQ1."ObjType" = OITL."ApplyType"

				WHERE "StockEff" = 2
				AND OITL."ApplyEntry" = @keyAsInteger
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", OWTQ."UserSign"
				HAVING SUM("AllocQty") > 0 AND "DistNumber" IS NOT NULL
				) AS SERIAL

			INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = SERIAL."ItemCode"
			AND PMX_SENU."SerialNumber" = SERIAL."DistNumber"
			LEFT JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
			AND PMX_SELD."BaseEntry" = SERIAL."ApplyEntry"
			AND PMX_SELD."BaseLine" = SERIAL."ApplyLine"
			AND PMX_SELD."BaseType" = SERIAL."ApplyType"
			WHERE PMX_SELD."InternalKey" IS NULL;*/



			--**********************************************************************************************************
			--END: SAP Serial numbers
			--**********************************************************************************************************


			RETURN;

		END


		--**********************************************************************************************************
		--Do general checks
		--**********************************************************************************************************
		

	END TRY
	BEGIN CATCH
		SET @error = 60005
		SET @error_message = 'PMX_SP: sql error ' + CAST(ERROR_NUMBER() AS NVARCHAR) + ' : ' + ERROR_MESSAGE()
		   + ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
		SELECT @error, @error_message --> done at the end of SBO_SP_...
		EXEC xp_logevent @error, @error_message, ERROR
	END CATCH
	
END


