- =============================================
-- Author:	PMX
-- Description:	Produmex extension on the SBO_SP_TransactionNotification
-- =============================================
ALTER PROCEDURE [dbo].[PMX_SP_TransactionNotification]
	-- Add the parameters for the stored procedure here
	@object_type nvarchar(20), 			-- SBO Object Type
	@transaction_type nchar(1),			-- [A]dd, [U]pdate, [D]elete, [C]ancel, C[L]ose
	@num_of_cols_in_key int,
	@list_of_key_cols_tab_del nvarchar(255),
	@list_of_cols_val_tab_del nvarchar(255),
	@error int OUTPUT, -- Result (0 for no error)
	@error_message nvarchar (255) OUTPUT -- Error string to be displayed

AS

BEGIN

	--**********************************************************************************************************
	--Return and handling errors:
	--If a problem is found the function sets the parameters @error and @error_message
	--and logs the error in the windows event logging
	--**********************************************************************************************************
	
	--**********************************************************************************************************
	--Logging errors:
	--If there are errors and this procedure is called when using the DIAPI then the user does not see the error
	--Because of this problem the errors are also logged in the windows event logging
	--To log you have to use the funciton: EXEC xp_logevent @error, @error_message, ERROR
	--**********************************************************************************************************

	--Logging
	DECLARE @MSG NVARCHAR (255);
	DECLARE @errorList  NVARCHAR(255);
	DECLARE @errorList2 VARCHAR(255);
	DECLARE @count  INT;

	DECLARE @tsNow DATETIME;
	DECLARE @updateDate DATETIME;
	DECLARE @updateTime INT;

	--At this moment the function supports: purchase delivery note, sales delivery note, move, item and BOM
	DECLARE @isPurchaseOrderDocument  SMALLINT;
	DECLARE @isPurchaseDeliveryDocument  SMALLINT;
	DECLARE @isPurchaseReturnDocument  SMALLINT;
	DECLARE @isPurchaseInvoice  SMALLINT;
	DECLARE @isPurchaseCreditNote  SMALLINT;
	DECLARE @isSalesDeliveryDocument  SMALLINT;
	DECLARE @isSalesReturnDocument  SMALLINT;
	DECLARE @isSalesOrderDocument  SMALLINT;
	DECLARE @isSalesInvoice  SMALLINT;
	DECLARE @isSalesCreditNote  SMALLINT;
	DECLARE @isInventoryEntry  SMALLINT;
	DECLARE @isInventoryExit  SMALLINT;
	DECLARE @isMoveDoc  SMALLINT;
	DECLARE @isItem  SMALLINT;
	DECLARE @isTransactionTypeAdd  SMALLINT;
	DECLARE @isTransactionTypeUpdate  SMALLINT;
	DECLARE @isTransactionTypeDelete  SMALLINT;
	DECLARE @isTransactionTypeCancel  SMALLINT;
	DECLARE @isBOM  SMALLINT;
	DECLARE @isUser  SMALLINT;
	DECLARE @isDocument  SMALLINT;
	DECLARE @isProductionOrder  SMALLINT;
	DECLARE @isItemBatchNumber  SMALLINT;
	DECLARE @isUomGroup  SMALLINT;
	DECLARE @isStockUpdate  SMALLINT;
	DECLARE @isBusinessPartner  SMALLINT;
	DECLARE @isStockTransfer  SMALLINT;
	DECLARE @isItemPackagingType  SMALLINT;
	DECLARE @isPickListType  SMALLINT;
	DECLARE @isPriority  SMALLINT;
	DECLARE @isFreightChargeDefinition  SMALLINT;
	DECLARE @isWarehouseTransferRequestDocument  SMALLINT;
	DECLARE @isWarehouse  SMALLINT;
	DECLARE @isBarcode  SMALLINT;
	DECLARE @isInventoryOpeningBalance SMALLINT;
	DECLARE @isInventoryPosting SMALLINT;
	DECLARE @isInventoryCounting SMALLINT;

	--Declaration of variables used for locking stock based on batches in sales order
	DECLARE @batchToLock nvarchar (36);
	DECLARE @batchQuantityToLock numeric (19,6);
	DECLARE @expDateToLock DateTime
	DECLARE @shippingQualityOptionToLock nvarchar(30)
	DECLARE @lineNumToLock int
	DECLARE @docEntryToLock int
	DECLARE @typeToLock nvarchar (20);
	DECLARE @itemCodeToLock nvarchar (20);	
	DECLARE @catchWeightRatioToLock numeric(19,6);	
	DECLARE @itriToLock int	
	DECLARE @qualityStatusToLock nvarchar (8);
	DECLARE @pmxWarehouseToLock nvarchar (50);	

	DECLARE @freeQuantityToLock numeric (19,6);
	DECLARE @currentQuantityToLock numeric (19,6);
	DECLARE @remainingQuantityToLock numeric (19,6);
	DECLARE @lockedQuantity numeric (19,6);

	DECLARE @currentLockInternalKey int;
	--END: Declaration of variables used for locking stock based on batches in sales order


	--Set default values
	SET @error = 0;
	SET @error_message = N'Ok';

	SET @tsNow = CURRENT_TIMESTAMP;
	SET @updateDate = CONVERT( DATETIME, CONVERT( VARCHAR, @tsNow, 112 ), 112 );
	SET @updateTime = DATEPART( hour, @tsNow ) * 100 + DATEPART( minute, @tsNow );

	SET @isPurchaseOrderDocument = 0;
	SET @isPurchaseDeliveryDocument = 0;
	SET @isPurchaseReturnDocument = 0;
	SET @isPurchaseInvoice = 0;
	SET @isPurchaseCreditNote = 0;
	SET @isSalesDeliveryDocument = 0;
	SET @isSalesReturnDocument = 0;
	SET @isSalesOrderDocument = 0;
	SET @isSalesInvoice = 0;
	SET @isSalesCreditNote = 0;
	SET @isInventoryEntry = 0;
	SET @isInventoryExit = 0;
	SET @isMoveDoc = 0;
	SET @isTransactionTypeAdd = 0;
	SET @isTransactionTypeUpdate = 0;
	SET @isTransactionTypeDelete = 0;
	SET @isTransactionTypeCancel = 0;
	SET @isItem = 0;
	SET @isBOM = 0;
	SET @isUser = 0;
	SET @isDocument = 0;
	SET @isProductionOrder = 0;
	SET @isItemBatchNumber = 0;
	SET @isUomGroup = 0;
	SET @isStockUpdate = 0;
	SET @isBusinessPartner = 0;
	SET @isStockTransfer = 0;
	SET @isItemPackagingType = 0;
	SET @isPickListType = 0;
	SET @isPriority = 0;
	SET @isFreightChargeDefinition = 0;
	SET @isWarehouseTransferRequestDocument = 0;
	SET @isWarehouse = 0;
	SET @isBarcode = 0;
	SET @isInventoryOpeningBalance = 0;
	SET @isInventoryPosting = 0;
	SET @isInventoryCounting = 0;


	--Test object type (this can be found in the SDK 'BoObjectTypes Enumeration'
	IF (@object_type = N'20') BEGIN
		SET @isPurchaseDeliveryDocument = 1;
	END ELSE IF (@object_type = N'PMX_MVHE') BEGIN
		SET @isMoveDoc = 1;
	END ELSE IF (@object_type = N'22') BEGIN
		SET @isPurchaseOrderDocument = 1;
	END ELSE IF (@object_type = N'4') BEGIN
		SET @isItem = 1;
	END ELSE IF (@object_type = N'66') BEGIN
		SET @isBOM = 1;
	END ELSE IF (@object_type = N'12') BEGIN
		SET @isUser = 1;
	END ELSE IF (@object_type = N'15') BEGIN
		SET @isSalesDeliveryDocument = 1;
	END ELSE IF (@object_type = N'59') BEGIN
		SET @isInventoryEntry = 1;
	END ELSE IF (@object_type = N'60') BEGIN
		SET @isInventoryExit = 1;
	END ELSE IF (@object_type = N'21') BEGIN
		SET @isPurchaseReturnDocument = 1;
	END ELSE IF (@object_type = N'16') BEGIN
		SET @isSalesReturnDocument = 1;
	END ELSE IF (@object_type = N'17') BEGIN
		SET @isSalesOrderDocument = 1;
	END ELSE IF (@object_type = N'202') BEGIN
		SET @isProductionOrder = 1;
	END ELSE IF (@object_type = N'106') BEGIN
		SET @isItemBatchNumber = 1;
	END ELSE IF (@object_type = N'13') BEGIN
		SET @isSalesInvoice = 1;
	END ELSE IF (@object_type = N'14') BEGIN
		SET @isSalesCreditNote = 1;
	END ELSE IF (@object_type = N'18') BEGIN
		SET @isPurchaseInvoice = 1;
	END ELSE IF (@object_type = N'19') BEGIN
		SET @isPurchaseCreditNote = 1;
	END ELSE IF (@object_type = N'10000197') BEGIN
		SET @isUomGroup = 1;
	END ELSE IF (@object_type = N'PMX_PRIO') BEGIN
		SET @isPriority = 1;
	END ELSE IF (@object_type = N'58') BEGIN
		SET @isStockUpdate = 1;
	END ELSE IF (@object_type = N'2') BEGIN
		SET @isBusinessPartner = 1;
	END ELSE IF (@object_type = N'67') BEGIN
		SET @isStockTransfer = 1;
	END ELSE IF (@object_type = N'64') BEGIN
		SET @isWarehouse = 1;
	END ELSE IF (@object_type = N'PMX_IPTY') BEGIN
		SET @isItemPackagingType = 1;
	END ELSE IF (@object_type = N'PMX_PLTY') BEGIN
		SET @isPickListType = 1;
	END ELSE IF (@object_type = N'PMX_FCDE') BEGIN
		SET @isFreightChargeDefinition = 1;
	END ELSE IF (@object_type = N'1250000001') BEGIN
		SET @isWarehouseTransferRequestDocument = 1;
	END ELSE IF (@object_type = N'1470000062') BEGIN
		SET @isBarcode = 1;
	END ELSE IF (@object_type = N'310000001') BEGIN
		SET @isInventoryOpeningBalance = 1;
	END ELSE IF (@object_type = N'10000071') BEGIN
		SET @isInventoryPosting = 1;
	END ELSE IF (@object_type = N'1470000065') BEGIN
		SET @isInventoryCounting = 1;
	END


	--Test transaction type
	IF (@transaction_type = N'A') BEGIN
		SET @isTransactionTypeAdd = 1;
	END ELSE IF (@transaction_type = N'U') BEGIN
		SET @isTransactionTypeUpdate = 1;
	END ELSE IF (@transaction_type = N'D') BEGIN
		SET @isTransactionTypeDelete = 1;
	END ELSE IF (@transaction_type = N'C') BEGIN
		SET @isTransactionTypeCancel = 1;
	END

	--Set if it is a document
	IF ((@isPurchaseDeliveryDocument = 1) OR
		(@isSalesDeliveryDocument = 1) OR
		(@isInventoryEntry = 1) OR
		(@isInventoryExit = 1) OR
		(@isPurchaseReturnDocument = 1) OR
		(@isSalesReturnDocument = 1) OR
		(@isSalesInvoice = 1) OR
		(@isSalesCreditNote = 1) OR
		(@isPurchaseInvoice = 1) OR
		(@isPurchaseCreditNote = 1) OR
		(@isStockTransfer = 1) OR
		(@isWarehouseTransferRequestDocument = 1) --OR
		--(@isSalesOrderDocument = 1) --Sales order is a document but not a document that has changes the stock. With document we mean here documents that change stock
		) BEGIN
		SET @isDocument = 1;
	END

	--Check if we have to do something. If nothing to do then exit.	
	IF (NOT(
		(@isPurchaseDeliveryDocument=1 AND @isTransactionTypeAdd=1) OR				--Add of goods receipt(purchase delivery note) is supported
		(@isMoveDoc=1 AND @isTransactionTypeAdd=1) OR								--Add of move is supported
		(@isItem=1)	OR																--All operations on an item are supported
		(@isBOM=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on a BOM
		(@isUser=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1))	OR	--Add/update on a User
		(@isSalesDeliveryDocument=1 AND @transaction_type IN (N'A',N'P',N'C')) OR		--Add of delivery (sales delivery note) is supported
		(@isInventoryEntry=1 AND @isTransactionTypeAdd=1) OR						--Add of inventory entry is supported
		(@isInventoryExit=1 AND @isTransactionTypeAdd=1) OR							--Add of inventory exit is supported
		(@isPurchaseReturnDocument=1 AND @isTransactionTypeAdd=1) OR				--Add of purchase return is supported
		(@isSalesReturnDocument=1 AND @isTransactionTypeAdd=1) OR					--Add of sales return is supported
		(@isSalesOrderDocument=1) OR												--All operations on a sales order are supported
		(@isPurchaseOrderDocument=1) OR												--All operations on a purchase order are supported
		(@isProductionOrder=1) OR 	--Add/update of a production order is supported
		--(@isItemBatchNumber=1 AND @isTransactionTypeUpdate=1) OR					--Update of item batch numbers
		(@isUomGroup=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on UomGroup
		(@isFreightChargeDefinition=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on a Pmx Uom
		(@isPriority=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR	--Add/update on a Pmx priority
		(@isStockUpdate=1) OR														--Add/update of stock
		(@isSalesInvoice=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR							--Add/update of sales invoice is supported
		(@isSalesCreditNote=1 AND @isTransactionTypeAdd=1) OR						--Add of sales credit note is supported
		(@isPurchaseInvoice=1 AND @isTransactionTypeAdd=1) OR						--Add of purchase invoice is supported
		(@isPurchaseCreditNote=1 AND @isTransactionTypeAdd=1) OR					--Add of purchase credit note is supported
		(@isStockTransfer=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeCancel=1 OR @transaction_type = N'P')) OR						--Add of stock transfer is supported
		(@isItemPackagingType=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR 	--Add/update on a Pmx item packaging type
		(@isWarehouse=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR 	--Add/update on SAP warehouse
		(@isPickListType=1 AND (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1)) OR 	--Add/update on a Pmx pick list type
		(@isBusinessPartner=1) OR	--Add/update on a business partner
		(@isBarcode=1) OR	--Add/update on a barcode
		(@isWarehouseTransferRequestDocument=1)OR
		(@isInventoryCounting = 1) OR
		(@isInventoryPosting = 1) OR
		(@isInventoryOpeningBalance = 1)
		)) BEGIN
			RETURN;
	END;


	BEGIN TRY
		--General variables
		DECLARE @keyAsInteger AS INT
		DECLARE @extraDB AS NVARCHAR(MAX)
		
		--Check if the object type is supported by Produmex
		IF (@isInventoryPosting=1) BEGIN
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
		
			IF ('Y' IN (SELECT OWHS.U_PMX_IMBP FROM IQR1 JOIN OWHS ON OWHS.WhsCode = IQR1.WhsCode WHERE IQR1.DocEntry = @keyAsInteger)) BEGIN	
				SET @error = 50097;
				SET @error_message = 'Inventory posting documents (ObjType: ' + CAST(@object_type AS NVARCHAR) + ') cannot reference warehouses that are managed by Produmex!';
			
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR;
				RETURN;
			END	
			RETURN;
		END
		
		IF (@isInventoryCounting=1) BEGIN
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
		
			IF ('Y' IN (SELECT OWHS.U_PMX_IMBP FROM INC1 JOIN OWHS ON OWHS.WhsCode = INC1.WhsCode WHERE INC1.DocEntry = @keyAsInteger)) BEGIN	
				SET @error = 50098;
				SET @error_message = 'Inventory counting documents (ObjType: ' + CAST(@object_type AS NVARCHAR) + ') cannot reference warehouses that are managed by Produmex!';
			
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR;
				RETURN;
			END	
			RETURN;
		END
		
		IF (@isInventoryOpeningBalance=1) BEGIN
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
		
			IF ('Y' IN (SELECT OWHS.U_PMX_IMBP FROM IQI1 JOIN OWHS ON OWHS.WhsCode = IQI1.WhsCode WHERE IQI1.DocEntry = @keyAsInteger)) BEGIN	
				SET @error = 50099;
				SET @error_message = 'Opening balance documents (ObjType: ' + CAST(@object_type AS NVARCHAR) + ') cannot reference warehouses that are managed by Produmex!';
			
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR;
				RETURN;
			END	
			RETURN;
		END
		
		--**********************************************************************************************************
		--Do that the stock update is allowed by the Produmex Add-On
		--Only stock changes that are supported by Produmex are allowed.
		--This line is put on top because this transaction notification contains 2 keys instead of one
		--**********************************************************************************************************
		IF (@isStockUpdate=1) BEGIN
			DECLARE @position INT;
			DECLARE @transNum INT;
			DECLARE @instance INT;
			DECLARE @transType  INT;
			DECLARE @inQty  NUMERIC(19, 6);
			DECLARE @outQty  NUMERIC(19, 6);
			--The number of columns must be always 2.
			IF (@num_of_cols_in_key <> 2) BEGIN
				SET @error = 50260;
				SET @error_message = 'The number of columns for a stock update is always 2. In this case it is '+  CAST(@num_of_cols_in_key AS VARCHAR) + ' AND the columns are ' + @list_of_key_cols_tab_del + '. This is not supported!';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END

			--Stock updates are always adds and never updates.
			IF (@isTransactionTypeAdd <> 1) BEGIN
				SET @error = 50261;
				SET @error_message = 'A stock update is always an ADD. In this case it is ''' + @transaction_type +'''. This is not supported!';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END

			--Get the position of the second column value
			--If the returned position is zero then it can not find the second column. What is not possible.
			SET @position = CHARINDEX(CHAR(9),@list_of_cols_val_tab_del)
			IF (@position = 0) BEGIN
				SET @error = 50262;
				SET @error_message = 'No second column value could be found in ''' + @list_of_cols_val_tab_del  + '''';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END

			--Get the columns values of "TransNum" and "Instance"
			SET @transNum = CAST(SUBSTRING(@list_of_cols_val_tab_del,1,@position-1) AS INT)
			SET @instance = CAST(SUBSTRING(@list_of_cols_val_tab_del,@position+1,LEN(@list_of_cols_val_tab_del)-@position-1) AS INT)

			--Get the stock update row
			SELECT @transType = "TransType", @inQty = "InQty", @outQty = "OutQty" FROM OINM WHERE "TransNum" = @transNum AND "Instance" = @instance;
			
			--Check if "TransType" is filled in
			IF (@transType IS NULL) BEGIN
				SET @error = 50263;
				SET @error_message = 'TransType IS NULL or could not be found for TransNum ' + CAST(@transNum AS NVARCHAR) + ' and Instance ' + CAST(@instance AS NVARCHAR);
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if there is quantity booked. If not then no stock update
			IF (@inQty-@outQty=0) BEGIN
				RETURN;
			END

			--Check if the "TransType" is supported by Produmex
			IF (@transType NOT IN (13,14,15,16,18,19,20,21,59,60,67)) BEGIN --,67)) BEGIN
				SET @error = 50264;
				SET @error_message = 'TransType '+ CAST(@transType AS NVARCHAR) + ' is not supported by the Produmex add-on!';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--If we are here everything is OK for the stock update
			RETURN;
		END


	--	--**********************************************************************************************************
	--	--Do checks on an item batchnumbers
	--	--This line is put on top because this transaction notification contains 2 keys instead of one
	--	--**********************************************************************************************************
	--	IF (@isItemBatchNumber=1 AND @isTransactionTypeUpdate=1) BEGIN
	--		SET @error = 50270
	--		SET @error_message = 'Batchnumber can not be updated when using the Produmex Add-On.'
	--		SELECT @error, @error_message
	--		EXEC xp_logevent @error, @error_message, ERROR
	--		RETURN
	--	END

		--The number of columns of the primary key must always be one. 
		IF @num_of_cols_in_key <> 1 BEGIN
			SET @error = 50001;
			SET @error_message = 'Number of keys must be 1! Contact Produmex Support.'
			SELECT @error, @error_message
			EXEC xp_logevent @error, @error_message, ERROR
			RETURN;
		END

		--**********************************************************************************************************
		--Get "Name" of extra DB
		--**********************************************************************************************************
		SELECT TOP 1 @extraDB= QUOTENAME(ExtraDb) FROM PMX_EXDB;
		IF @extraDB IS NULL BEGIN
			SET @error = 50002;
			SET @error_message = 'The extra database name is not filled in in the table PMX_EXDB. This must be filled in!'
			SELECT @error, @error_message
			EXEC xp_logevent @error, @error_message, ERROR
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an item
		--**********************************************************************************************************
		IF (@isItem=1) BEGIN
				
			--Check if this is a delete. 
			IF (@isTransactionTypeDelete=1) BEGIN
				
				--Check if the item is not used for the bin items
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_SLIT."ParentCode" + ' ' 
				FROM PMX_SLIT
				WHERE PMX_SLIT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50250;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item is used for the BIN items ' + @errorList + '.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item has no zone types linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_ITZT.ZoneTypeCode + ' ' 
				FROM PMX_ITZT
				WHERE PMX_ITZT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50251;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has zone types linked to it for zone types: ' + @errorList + '. Delete these zone types from the item.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item has no shelflifes linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_ICSL."CardCode" + ' ' 
				FROM PMX_ICSL
				WHERE PMX_ICSL."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50252;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has shelf lifes linked to it for customers: ' + @errorList + '. Delete these customers from the item.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item is not used in a steplist
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_PSLH."ParentItemCode" + ' ' 
				FROM PMX_PSLH
				WHERE PMX_PSLH."ParentItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50253;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has production order step list linked to it.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item is not used in a steplist lines
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_PSLH."ParentItemCode" + ' ' 
				FROM PMX_PSLH
				INNER JOIN PMX_PSLL ON PMX_PSLL."ParentKey" = PMX_PSLH."InternalKey"
				WHERE PMX_PSLL."ChildItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50254;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item is used in production order step list for items: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item is not used as default logistic carrier
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM."ItemCode" + ' ' 
				FROM OITM
				WHERE OITM.U_PMX_DLCP = @list_of_cols_val_tab_del OR OITM.U_PMX_DLCS = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50255;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item is used as default logistic carrier for items: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the item does not have any batch-attributes linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IBAT.BatchAttributeCode + ' ' 
				FROM PMX_IBAT
				WHERE PMX_IBAT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50256;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has batch-attributes linked to it:' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END


				--Check if the item does not have any packaging types linked to it
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IPTY."PackagingTypeCode" + ' ' 
				FROM PMX_IPTY
				WHERE PMX_IPTY."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50257;
					SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the item has packaging types linked to it: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END


				--No other checks must be done for the item delete
				RETURN;
			END

			--Get data of the item
			DECLARE @storageLocationCode NVARCHAR(50);
			DECLARE @isReturnableItem NVARCHAR(1);
			DECLARE @itemCodeReturnableItemNotInventory NVARCHAR(20);
			DECLARE @isInventoryItem CHAR(1);
			DECLARE @barcode NVARCHAR(254);
			DECLARE @purchaseBarcode NVARCHAR(254);
			DECLARE @salesBarcode NVARCHAR(254);
			DECLARE @qualityStatusQuarantinedAfterReception NVARCHAR(8);
			DECLARE @qualityStatusAfterSalesReturn NVARCHAR(8);
			DECLARE @barcodeType NVARCHAR(1);
			DECLARE @purchaseBarcodeType NVARCHAR(1);
			DECLARE @salesBarcodeType NVARCHAR(1);
			DECLARE @isSerialNumber NVARCHAR(1);
			DECLARE @hasBestBefore NVARCHAR(1);
			DECLARE @expiryDefinitionCodeProduction NVARCHAR(30);
			DECLARE @numberOfDecimalsForUOM INT;
			DECLARE @numberOfDecimalsForUOM2 INT;
			DECLARE @gs1MeasureTypeForUOM NVARCHAR(30);
			DECLARE @gs1MeasureTypeForUOM2 NVARCHAR(30);
			DECLARE @itemUOM NVARCHAR(100);
			DECLARE @itemUOM2 NVARCHAR(100);
			DECLARE @defaultQuantityUOM NUMERIC(19, 6);
			DECLARE @defaultQuantityUOM2 NUMERIC(19, 6);
			DECLARE @UOMToUseForSales NVARCHAR(1);
			DECLARE @purchaseQuantity NUMERIC(19, 6);
			DECLARE @salesQuantity NUMERIC(19, 6);
			DECLARE @purchaseUOM NVARCHAR(100);
			DECLARE @salesUOM NVARCHAR(100);
			DECLARE @correctStockForUOM NVARCHAR(1);
			DECLARE @correctStockForUOM2 NVARCHAR(1);
			DECLARE @UOMToUseForInventoryTransitions NVARCHAR(1);
			DECLARE @UOMToUseForPurchase NVARCHAR(1);
			DECLARE @isItemLogisticCarrier  NVARCHAR(1);
			DECLARE @sevesoClass  NVARCHAR(30);
			DECLARE @qualityStatusReleasedAfterReception NVARCHAR(8);
			DECLARE @qualityStatusReceptionController NVARCHAR(8);
			DECLARE @hasSecondBatchNumber NVARCHAR(1);
			DECLARE @defaultLogisticCarrierProduction NVARCHAR(20);
			DECLARE @defaultLogisticCarrierPicking NVARCHAR(20);
			DECLARE @hasPmxSerialNumber NVARCHAR(1);
			DECLARE @qualityStatusProduction NVARCHAR(8);
			DECLARE @pickType NVARCHAR(30);
			DECLARE @trackLocationSerialNumber NVARCHAR(1);
			DECLARE @formatSerialNumber NVARCHAR(50);
			DECLARE @createSsccOnReception NVARCHAR(1);
			DECLARE @isCatchWeight NVARCHAR(1);
			DECLARE @addNonInventoryItemToPickList CHAR(1);


			SELECT @storageLocationCode=ISNULL( U_PMX_DSLC,''), @isReturnableItem=ISNULL( U_PMX_RETR,'N'), @itemCodeReturnableItemNotInventory=ISNULL( U_PMX_ICRI,''),
					@isInventoryItem=ISNULL( "InvntItem",'N'),@barcode=ISNULL( "CodeBars",''),@barcodeType=ISNULL( U_PMX_BCTY,'G'), 
					@qualityStatusQuarantinedAfterReception=U_PMX_QSCR, @qualityStatusAfterSalesReturn=U_PMX_QSSR,@isSerialNumber=ISNULL( "ManSerNum",'N'),
					@hasBestBefore=ISNULL( U_PMX_HBBD,'N'), @purchaseBarcode=ISNULL( U_PMX_PBCO,''), @purchaseBarcodeType=ISNULL( U_PMX_PBCT,'G'),
					@salesBarcode=ISNULL( U_PMX_SBCO,''), @salesBarcodeType=ISNULL( U_PMX_SBCT,'G'), @expiryDefinitionCodeProduction=U_PMX_EXDP,
					@numberOfDecimalsForUOM = U_PMX_UOMD, @numberOfDecimalsForUOM2 = U_PMX_UM2D, @gs1MeasureTypeForUOM = ISNULL( U_PMX_UMMT,'') , @gs1MeasureTypeForUOM2 = ISNULL( U_PMX_U2MT,''),
					@itemUOM = ISNULL( UOM_INV."UomCode", ISNULL( "InvntryUom",'')) , @itemUOM2 = ISNULL( U_PMX_UOM2,''), @defaultQuantityUOM = U_PMX_DQUM, @defaultQuantityUOM2 = U_PMX_DQU2,
					@UOMToUseForSales = U_PMX_UMSA, @purchaseQuantity = ISNULL( "NumInBuy",0), @salesQuantity = ISNULL( "NumInSale",0),
					@purchaseUOM = ISNULL( UOM_PUR."UomCode", ISNULL( "BuyUnitMsr",'')), @salesUOM = ISNULL( UOM_SAL."UomCode",ISNULL( "SalUnitMsr",'')), @correctStockForUOM = ISNULL( U_PMX_CSUM, 'N'), @correctStockForUOM2 = ISNULL( U_PMX_CSU2, 'N'),
					@UOMToUseForPurchase = U_PMX_UMPU, @UOMToUseForInventoryTransitions = U_PMX_UMIT, @isItemLogisticCarrier = U_PMX_LOCA, @sevesoClass = U_PMX_SEVE,
					@qualityStatusReleasedAfterReception = U_PMX_RQSR, @qualityStatusReceptionController = U_PMX_QSRC,
					@hasSecondBatchNumber = ISNULL( U_PMX_HBN2, 'N'), @defaultLogisticCarrierProduction = U_PMX_DLCP, @defaultLogisticCarrierPicking = U_PMX_DLCS,
					@hasPmxSerialNumber=ISNULL( U_PMX_HSER,'N'), @qualityStatusProduction = U_PMX_QSPR, @pickType=U_PMX_PITY, @trackLocationSerialNumber=ISNULL( U_PMX_TLSN,'N'), @formatSerialNumber = U_PMX_SNFO , @createSsccOnReception=ISNULL( U_PMX_CSOR,'N'),
					@isCatchWeight = U_PMX_ICAW, @addNonInventoryItemToPickList = U_PMX_NIOP
			FROM OITM
			LEFT JOIN OUOM AS UOM_INV ON UOM_INV."UomEntry" = OITM."IUoMEntry"
			LEFT JOIN OUOM AS UOM_PUR ON UOM_PUR."UomEntry" = OITM."PUoMEntry"
			LEFT JOIN OUOM AS UOM_SAL ON UOM_SAL."UomEntry" = OITM."SUoMEntry"
			WHERE "ItemCode" = @list_of_cols_val_tab_del;

	--		print '@isReturnableItem='  + @isReturnableItem
	--		print '@itemCodeReturnableItemNotInventory=' + @itemCodeReturnableItemNotInventory
	--		print '@isInventoryItem=' + @isInventoryItem
	--		print '@storageLocationCode=' + @storageLocationCode
	--		print '@barcode=' + @barcode

			--Check if the default storage location filled in is an existing storage location or a zone
			IF (LEN(@storageLocationCode)>0) BEGIN
				IF (SELECT COUNT("Code") FROM PMX_OSSL WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS=@storageLocationCode AND "IsActive"='Y') <> 1 BEGIN
					IF (SELECT COUNT("Code") FROM PMX_OSZO WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS=@storageLocationCode) <> 1 BEGIN
						SET @error = 50200;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The storage location code or zone code ''' + @storageLocationCode + ''' could not be found in the system or the storage location is not active!'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END
				END
			END

			--The property ItemCodeReturnableItemNotInventory on an item must be filled in if the item is inventory and IsReturnableItem=True
			IF (@isReturnableItem = 'Y') AND (@isInventoryItem='Y') AND (LEN(@itemCodeReturnableItemNotInventory)=0) BEGIN
				SET @error = 50201;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item code returnable item not inventory is not filled. This must be filled in if the item is an inventory item and is returnable. Every inventory returnable item must have a not inventory returnable item linked to it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--The property ItemCodeReturnableItemNotInventory on an item can only be filled in if the item is an inventory returnable item.
			IF NOT((@isReturnableItem = 'Y' OR @isItemLogisticCarrier = 'Y') AND (@isInventoryItem='Y')) AND (LEN(@itemCodeReturnableItemNotInventory)>0) BEGIN
				SET @error = 50202;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The field ItemCodeReturnableItemNotInventory may only be filled in if the item is an inventory returnable item or logistic carrier!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the item code that is filled in in @itemCodeReturnableItemNotInventory is an existing not inventory returnable item.
			IF (LEN(@itemCodeReturnableItemNotInventory)>0) AND ((SELECT COUNT("ItemCode") FROM OITM WHERE "ItemCode" = @itemCodeReturnableItemNotInventory AND "InvntItem"='N' AND (U_PMX_RETR='Y' OR U_PMX_LOCA='Y'))=0) BEGIN
				SET @error = 50203;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The not inventory returnable item with code '''+@itemCodeReturnableItemNotInventory+''' could not be found!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check the GTINs of the item
			DECLARE @sqlStatementBegin AS NVARCHAR(max)
			DECLARE @sqlStatementEnd AS NVARCHAR(max)
			DECLARE @sqlStatement AS NVARCHAR(max)
			DECLARE @parameterList AS NVARCHAR(max)

			SET @sqlStatementBegin = 'SELECT @error=Error, @error_message=Message FROM ' + @extraDb + '.dbo.PMX_FN_CheckGTIN(''';
			SET @sqlStatementEnd = ''')';
			SET @parameterList = '@error INT output, @error_message NVARCHAR(200) output';
			
			--Check the GTIN of the item
			IF ((LEN(@barcode) <> 0) AND (@barcodeType = 'G')) BEGIN
				SET @sqlStatement = @sqlStatementBegin + @barcode + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcode + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			--Check the purchase GTIN of the item
			IF ((LEN(@purchaseBarcode) <> 0) AND (@purchaseBarcodeType = 'G')) BEGIN
				SET @sqlStatement = @sqlStatementBegin + @purchaseBarcode + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = 'Purchase barcode '''+ @purchaseBarcode + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			--Check the sales GTIN of the item
			IF ((LEN(@salesBarcode) <> 0) AND (@salesBarcodeType = 'G')) BEGIN
				SET @sqlStatement = @sqlStatementBegin + @salesBarcode + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = 'Sales barcode '''+ @salesBarcode + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			

			--Check the GTIN of the item packaging types
			DECLARE @barcodePackagingType NVARCHAR(16);
			DECLARE barCodePackagingTypeCursor CURSOR LOCAL FOR SELECT "Barcode" FROM PMX_IPTY WHERE PMX_IPTY."ItemCode" = @list_of_cols_val_tab_del AND LEN("Barcode") <> 0 AND ("BarcodeType" = 'G');
			OPEN barCodePackagingTypeCursor;
			FETCH NEXT FROM barCodePackagingTypeCursor INTO @barcodePackagingType;
			WHILE @@FETCH_STATUS >= 0 BEGIN
				SET @sqlStatement = @sqlStatementBegin + @barcodePackagingType + @sqlStatementEnd;
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcodePackagingType + ''' of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					CLOSE barCodePackagingTypeCursor;
					DEALLOCATE barCodePackagingTypeCursor

					RETURN;
				END

				FETCH NEXT FROM barCodePackagingTypeCursor INTO @barcodePackagingType;
			
			END
			CLOSE barCodePackagingTypeCursor;
			DEALLOCATE barCodePackagingTypeCursor
			
			
			--Check the GTIN of the uom barcodes
			DECLARE @barcodeForUom NVARCHAR(254);
			DECLARE @barcodeUom NVARCHAR(100);
			DECLARE barCodeUomCursor CURSOR LOCAL FOR SELECT "BcdCode", OUOM."UomCode" 
														FROM OBCD 
														INNER JOIN OUOM ON OUOM."UomEntry" = OBCD."UomEntry" 
														WHERE OBCD."ItemCode" = @list_of_cols_val_tab_del AND LEN("BcdCode") <> 0 
														AND @barcodeType = 'G';
			OPEN barCodeUomCursor;
			FETCH NEXT FROM barCodeUomCursor INTO @barcodeForUom, @barcodeUom;
			WHILE @@FETCH_STATUS >= 0 BEGIN
				SET @sqlStatement = @sqlStatementBegin + @barcodeForUom + @sqlStatementEnd
				EXECUTE sp_executesql @sqlStatement, @parameterList, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcodeForUom + ''' for uom '''+ @barcodeUom + '''  of Item ' + @list_of_cols_val_tab_del + ': ' + @error_message;
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					CLOSE barCodeUomCursor ;
					DEALLOCATE barCodeUomCursor

					RETURN;
				END

				FETCH NEXT FROM barCodeUomCursor INTO @barcodeForUom, @barcodeUom;
			
			END
			CLOSE barCodeUomCursor;
			DEALLOCATE barCodeUomCursor
			
			

			--Check if the quality status code that is filled in in @qualityStatusQuarantinedAfterReception is an existing quality status code.
			IF (@qualityStatusQuarantinedAfterReception IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusQuarantinedAfterReception COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50207;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status code after purchase return ''' + @qualityStatusQuarantinedAfterReception + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status code that is filled in in @qualityStatusAfterSalesReturn is an existing quality status code.
			IF (@qualityStatusAfterSalesReturn IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusAfterSalesReturn COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50208;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status code after sales return ''' + @qualityStatusAfterSalesReturn + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if barcode does not occur more than once for inventory items
			IF (@isInventoryItem = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM."ItemCode" + ' ' 
				FROM OITM
				LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				WHERE OITM."ItemCode" <> @list_of_cols_val_tab_del AND 
				((LEN(ISNULL( OITM."CodeBars",'')) <> 0 AND OITM."CodeBars" IN(@barcode,@purchaseBarcode,@salesBarcode)) OR
				(LEN(ISNULL( OITM.U_PMX_PBCO,'')) <> 0 AND OITM.U_PMX_PBCO IN(@barcode,@purchaseBarcode,@salesBarcode))OR
				(LEN(ISNULL( OITM.U_PMX_SBCO,'')) <> 0 AND OITM.U_PMX_SBCO IN(@barcode,@purchaseBarcode,@salesBarcode))OR
				(LEN(ISNULL( PMX_IPTY."BarCode",'')) <> 0 AND PMX_IPTY."BarCode" IN(@barcode,@purchaseBarcode,@salesBarcode))) AND
				OITM."InvntItem" = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50209;
					SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': The item ' + @errorList + 'has the same barcode.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--The property "ManSerNum" on an item cannot be true because the addon does not support it
--			IF (@isSerialNumber = 'Y') BEGIN
--				SET @error = 50210;
--				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item cannot be managed by serial numbers.'
--				SELECT @error, @error_message
--				EXEC xp_logevent @error, @error_message, ERROR
--				RETURN;
--			END

			--When the item has bestBeforeDate, there cannot be an inventory item with bestBeforeDate is null
			IF (@hasBestBefore = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."BestBeforeDate" IS NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50211;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has a best before date, but in the inventory the best before date is not set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item has no bestBeforeDate, there cannot be an inventory item with a bestBeforeDate
			IF (@hasBestBefore = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."BestBeforeDate" IS NOT NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50212;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no best before date, but in the inventory the best before date is set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			-- Check if the barcode in purchase and sales are not the same as the base barcode
			IF (LEN(@barcode)>0 AND @barcode = @purchasebarCode) OR (LEN(@barcode)>0 AND @barcode = @salesBarcode) OR (LEN(@salesBarcode)>0 AND @salesBarcode=@purchaseBarcode) BEGIN
				SET @error = 50213;
				SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': The barcode or the purchase barcode or the sales barcode of item ' + @list_of_cols_val_tab_del + ' is the same. This is not allowed.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the expiry definition for production is filled in in @expiryDefinitionCodeProduction is an existing expiry definition code.
			IF (@expiryDefinitionCodeProduction IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_EXDE] WHERE "Code"=@expiryDefinitionCodeProduction)=0) BEGIN
				SET @error = 50214;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The expiry defintion code for production ''' + @expiryDefinitionCodeProduction + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the seveso class is filled in and if @sevesoClass is an existing seveso class code.
			IF (@sevesoClass IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_SEVE] WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS=@sevesoClass)=0) BEGIN
				SET @error = 50215;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The seveso class code ''' + @sevesoClass + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status code that is filled in in @qualityStatusReleasedAfterReception is an existing quality status code.
			IF (@qualityStatusReleasedAfterReception IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusReleasedAfterReception COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50216;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The released quality status code after reception ''' + @qualityStatusReleasedAfterReception + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status controller exists
			IF (@qualityStatusReceptionController IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_EXTE WHERE "Code"=@qualityStatusReceptionController)=0) BEGIN
				SET @error = 50217;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status controller ''' + @qualityStatusReceptionController + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--When the item has second batchnumber, there cannot be an inventory item with second batchnumber is null
			IF (@hasSecondBatchNumber = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."InternalBatchNumber" IS NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50218;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has a second batch number, but in the inventory the second batch number is not set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item has no second batchnumber, there cannot be an inventory item with a second batchnumber
			IF (@hasSecondBatchNumber = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."InternalBatchNumber" IS NOT NULL AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50219;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no second batch number, but in the inventory the second batch number is set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--If no batchnumber or best before date is set, user cannot ask for packaging type quantity during receipt
			IF (@hasBestBefore = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IPTY."PackagingTypeCode" + ' ' 
				FROM PMX_IPTY INNER JOIN OITM
				ON OITM."ItemCode" = PMX_IPTY."ItemCode"
				WHERE PMX_IPTY."ItemCode" = @list_of_cols_val_tab_del
				AND OITM."ManBtchNum" = 'N'
				AND PMX_IPTY."AskDuringReception" = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50220;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no batch or BBD, so packaging quantity cannot be asked during receipt for packaging types: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END


			--Check if the uom does not exists in item packaging types
			IF((SELECT COUNT("ItemCode") FROM PMX_IPTY WHERE "PackagingTypeCode"=@itemUOM AND "ItemCode" = @list_of_cols_val_tab_del)>0 AND LEN(@itemUOM) > 0) BEGIN
				SET @error = 50221;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The UOM with code ''' + @itemUOM + ''' is used in packaging types for the item'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Batchnumber cannot be set if item is logistic carrier
			SET @errorList = ''
			SELECT @errorList = @errorList + OITM."ItemCode" + ' ' 
			FROM OITM
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			AND OITM.U_PMX_LOCA = 'Y'
			AND (OITM."ManBtchNum" = 'Y' OR OITM.U_PMX_HBBD = 'Y' OR OITM.U_PMX_HBN2 = 'Y');
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50222;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': An item defined as logistic carrier cannot have a batch or BBD.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Default logistic carrier for production should be valid item and logistic carrier
			IF ISNULL( @defaultLogisticCarrierProduction, '') <> ''
			BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM.U_PMX_DLCP + ' ' 
				FROM OITM
				WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT ItemCode FROM OITM WHERE "ItemCode" = @defaultLogisticCarrierProduction
				AND OITM.U_PMX_LOCA = 'Y');
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50223;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The default logistic carrier for Production ' + @defaultLogisticCarrierProduction + ' is not a valid Logistic Carrier.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--Default logistic carrier for picking should be valid item and logistic carrier
			IF ISNULL( @defaultLogisticCarrierProduction, '') <> ''
			BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + OITM.U_PMX_DLCS + ' ' 
				FROM OITM
				WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT ItemCode FROM OITM WHERE "ItemCode" = @defaultLogisticCarrierPicking
				AND OITM.U_PMX_LOCA = 'Y');
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50224;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The default logistic carrier for Picking ' + @defaultLogisticCarrierPicking + ' is not a valid Logistic Carrier.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--If item has Pmx serial number, the decimals for uom should be 0
			IF ((@hasPmxSerialNumber = 'Y' OR @isSerialNumber = 'Y') AND
				@numberOfDecimalsForUOM > 0) BEGIN
				SET @error = 50225;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has serial numbers, so the number of decimals for UOM should be 0.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the quality status code that is filled in in @qualityStatusProduction is an existing quality status code.
			IF (@qualityStatusProduction IS NOT NULL) AND ((SELECT COUNT("Code") FROM PMX_QYST WHERE "Code" COLLATE SQL_Latin1_General_Cp850_CS_AS = @qualityStatusProduction COLLATE SQL_Latin1_General_Cp850_CS_AS)=0) BEGIN
				SET @error = 50226;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The quality status code production ''' + @qualityStatusProduction + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the pick type that is filled in in @pickType is an existing pick type.
			IF (@pickType IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_IPIT] WHERE "Code"=@picktype)=0) BEGIN
				SET @error = 50227;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The pick type ''' + @pickType + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--When the item tracks location for serial numbers, there cannot be one without LUID
			IF (@trackLocationSerialNumber = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_SENU."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_SENU
				WHERE PMX_SENU."InStock" = 'Y' 
				AND PMX_SENU.LUID IS NULL
				AND PMX_SENU."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50228;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item needs to track locations of serial nr, but there are serial nrs in stock without LUID: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item tracks location for serial numbers, there cannot be inventory totals without LUID (needed for "on release only" setting)
			IF (@trackLocationSerialNumber = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				WHERE PMX_INVT."Quantity" > 0
				AND PMX_INVT.LogUnitIdentKey IS NULL
				AND PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50238;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item needs to track locations of serial nr, but there are inventory totals without LUID: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--When the item does not tracks location for serial numbers, there cannot be one with LUID
			IF (@trackLocationSerialNumber = 'N') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST( PMX_SENU."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_SENU
				WHERE PMX_SENU."InStock" = 'Y' 
				AND PMX_SENU.LUID IS NOT NULL
				AND PMX_SENU."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50229;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item does not need to track locations of ser. nr, but there are ser. nrs are in stock with LUID: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END			

			--When the item does track location for serial numbers, either SAP or PMX serial numbers needs to be filled
			IF (@trackLocationSerialNumber = 'Y' AND @hasPmxSerialNumber = 'N' AND @isSerialNumber = 'N') BEGIN
				SET @error = 50230;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item tracks locations of ser. nr, but the item is not defined as serial number'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--When the item does track location for serial numbers, either SAP or PMX serial numbers needs to be filled
			IF (@hasPmxSerialNumber = 'Y' AND @isSerialNumber = 'Y') BEGIN
				SET @error = 50231;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has SAP serial numbers and PMX serial numbers. Please choose only one of them.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the serial format is a correct one
			IF (@formatSerialNumber IS NOT NULL) AND ((SELECT COUNT("Code") FROM [@PMX_ISFT] WHERE "Code"=@formatSerialNumber)=0) BEGIN
				SET @error = 50232;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The serial number format ''' + @formatSerialNumber + ''' could not be found in the system!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			IF (@isInventoryItem = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + T0."ItemCode" + ' ' 
				FROM (
					SELECT DISTINCT Barcode2."ItemCode"				
					FROM OBCD AS Barcode1
					LEFT JOIN OBCD Barcode2 ON Barcode1."BcdCode" = Barcode2."BcdCode"
					WHERE Barcode1."ItemCode" = @list_of_cols_val_tab_del
					AND Barcode1."BcdEntry" <> Barcode2."BcdEntry"
				) AS T0;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50233;
					SET @error_message = 'Item '''+ @errorList +''' contains the same barcode '
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			
			IF (@isInventoryItem = 'Y') BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + T0."ItemCode" + ' ' 
				FROM (
					SELECT OBCD."ItemCode"
					FROM OBCD 
					WHERE OBCD."ItemCode" <> @list_of_cols_val_tab_del AND 
					(LEN(ISNULL( OBCD."BcdCode",'')) <> 0 AND OBCD."BcdCode" IN(@barcode,@purchaseBarcode,@salesBarcode)) 
				) AS T0;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50234;
					SET @error_message = 'Item ''' + @list_of_cols_val_tab_del + ''': The item ''' + @errorList + ''' has the same barcode.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			
			-- Check if item has serial that needs to be tracked -> need to have SSCC's at reception
			IF (@hasPmxSerialNumber = 'Y' OR @isSerialNumber = 'Y') AND @trackLocationSerialNumber ='Y' AND @createSsccOnReception = 'N' BEGIN
				SET @error = 50235;
				SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + '''. Cannot add an item with SAP or PMX serial numbers and tracking location without create SSCC on reception.' + @errorList + '.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			
			--When the item has no Serialnumber, there cannot be an inventory item with Serialnumber
			IF (@hasPmxSerialNumber = 'N' AND @isSerialNumber = 'N' ) BEGIN
				SET @errorList = ''
				SELECT DISTINCT @errorList = @errorList + CAST( PMX_INVT."InternalKey" AS NVARCHAR (11)) + ' ' 
				FROM PMX_INVT
				LEFT JOIN PMX_SENU ON PMX_SENU."ItemCode" = PMX_INVT."ItemCode"
				WHERE PMX_SENU."InStock" ='Y' AND 
				PMX_INVT."ItemCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50236;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The item has no serial number, but serial number(s) are set for the inventory total with keys: '+@errorList+'.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			-- Check if item has serial that needs to be tracked -> need to have SSCC's at reception
			IF (@isItemLogisticCarrier = 'Y' AND EXISTS(SELECT "ItemCode" FROM OITM WHERE "ItemCode" = @list_of_cols_val_tab_del AND OITM."UgpEntry" <> -1)) BEGIN
			    SET @error = 50237;
				SET @error_message = 'Item with code ''' + @list_of_cols_val_tab_del + '''. A logistic carrier cannot have an uom group.' 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			IF (@isInventoryItem = 'Y' AND @addNonInventoryItemToPickList = 'Y') BEGIN
			    SET @error = 50239;
				SET @error_message = 'Add non-inventory item to pick list is not allowed for inventory item. Item with code ''' + @list_of_cols_val_tab_del + ''''
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			

			--******************************
			--Catch weight
			--******************************

			--Check if the number of decimals for uom is between 0 and the number of decimals for quantities
			IF(@numberOfDecimalsForUOM < 0 OR @numberOfDecimalsForUOM > (SELECT TOP (1) OADM.QtyDec FROM OADM)) BEGIN
				SET @error = 50350;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The number of decimals for uom should be between 0 and ' + CAST( (SELECT TOP (1) OADM.QtyDec FROM OADM) AS NVARCHAR (10))
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the number of decimals for uom2 is between 0 and the number of decimals for quantities
			IF(@numberOfDecimalsForUOM2 < 0 OR @numberOfDecimalsForUOM2 > (SELECT TOP (1) OADM.QtyDec FROM OADM)) BEGIN
				SET @error = 50351;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The number of decimals for uom2 should be between 0 and ' + CAST( (SELECT TOP (1) OADM.QtyDec FROM OADM) AS NVARCHAR (10))
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the gs1 measure types are not the same
			IF( LEN(@gs1MeasureTypeForUOM) > 0 AND LEN(@gs1MeasureTypeForUOM2) > 0 AND @gs1MeasureTypeForUOM = @gs1MeasureTypeForUOM2) BEGIN
				SET @error = 50352;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The GS1 Measure Type cannot be the same for uom and uom2'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
		
			

			--Check if the uom exists in UOM
			IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@itemUOM)=0 AND LEN(@itemUOM) > 0) BEGIN
				SET @error = 50355;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The UOM with code ''' + @itemUOM + ''' does not exist in SAP Uom'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the purchase uom exists in UOM
			IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@purchaseUOM)=0 AND LEN(@purchaseUOM) > 0) BEGIN
				SET @error = 50356;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The purchase UOM with code ''' + @purchaseUOM + ''' does not exist in SAP Uom'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the sales uom exists in UOM
			IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@salesUOM)=0 AND LEN(@salesUOM) > 0) BEGIN
				SET @error = 50357;
				SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The sales UOM with code ''' + @salesUOM + ''' does not exist in SAP Uom'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Checks for catch weight
			IF( @isCatchWeight = 'Y' ) BEGIN

				--Check if uom decimals for UOM = 0 when it should be **DISABLE FOR NOW **
				--IF( @numberOfDecimalsForUOM <> 0 ) BEGIN
				--	SET @error = 50371;
				--	SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The number of decimals for UOM should be 0 for inventory uom.'
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END
				
				----Check if uom decimals for UOM = 0 when it should be
				--IF(EXISTS(SELECT "ItemCode" FROM [OITM] WHERE "ItemCode"=@list_of_cols_val_tab_del AND ISNULL( "IUoMEntry", -1) <> -1 )) BEGIN
				--	SET @error = 50372;
				--	SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': UOM groups are not allowed for catch weight items';
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END
				
				--IF (@salesQuantity <> 1) BEGIN
				--	SET @error = 50373;
				--	SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': Items per sales unit should be 1 for catch weight items';
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END
				
				--IF (@purchaseQuantity <> 1) BEGIN
				--	SET @error = 50374;
				--	SET @error_message = 'Item ' + @list_of_cols_val_tab_del + ': Items per purchase unit should be 1 for catch weight items';
				--	SELECT @error, @error_message
				--	EXEC xp_logevent @error, @error_message, ERROR
				--	RETURN;
				--END		
				
				IF ISNULL( @defaultQuantityUOM, 0 ) = 0 BEGIN
					SET @error = 50375;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': A default quantity for uom should be filled in for catch weight items';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
				--Only allow catch weight items if no stock, or stock has quantity for Uom2
				IF EXISTS(SELECT TOP(1)[PMX_INVT]."InternalKey" FROM [PMX_INVT] WHERE [PMX_INVT]."ItemCode"=@list_of_cols_val_tab_del AND [PMX_INVT].QuantityUom2 IS NULL ) BEGIN
					SET @error = 50376;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Cannot change item to catch weight item if stock exists';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
				--Update the default quantity for uom2
				UPDATE OITM SET U_PMX_DQU2 = 1 / @defaultQuantityUOM WHERE "ItemCode" = @list_of_cols_val_tab_del;
				
			END
			ELSE
			BEGIN
			
				--Only allow NON catch weight items if no stock, or stock has no quantity for Uom2
				IF EXISTS(SELECT TOP(1)[PMX_INVT]."InternalKey" FROM [PMX_INVT] WHERE [PMX_INVT]."ItemCode"=@list_of_cols_val_tab_del AND [PMX_INVT].QuantityUom2 IS NOT NULL ) BEGIN
					SET @error = 50401;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Cannot change item to NON-catch weight item if stock exists';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
			
			END


			--Do checks when UOM2 is filled in
			IF(LEN(@itemUOM2) > 0) BEGIN
				--Check if the uom2 exists in UOM
				IF((SELECT COUNT("UomCode") FROM [OUOM] WHERE "UomCode"=@itemUOM2)=0) BEGIN
					SET @error = 50358;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The UOM2 with code ''' + @itemUOM2 + ''' does not exist in SAP Uom'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
                /*
				IF ISNULL( @defaultQuantityUOM , 0 ) = 0 BEGIN
					UPDATE OITM SET U_PMX_DQUM = 1 WHERE "ItemCode" = @list_of_cols_val_tab_del
					SET @defaultQuantityUOM = 1;
				END
				IF ISNULL( @defaultQuantityUOM2, 0 ) = 0 BEGIN
					UPDATE OITM SET U_PMX_DQU2 = 1 WHERE "ItemCode" = @list_of_cols_val_tab_del
					SET @defaultQuantityUOM2 = 1;
				END		
				IF @defaultQuantityUOM <> 1 AND @defaultQuantityUOM2 <> 1 BEGIN
					SET @error = 50359;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': At least one of the default quantities for UOM has to be 1'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the uom to use for orders is not 'A'
				IF( @UOMToUseForSales = 'A') BEGIN
					SET @error = 50360;
					SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Uom to use for orders cannot be ''UomAndUOM2'''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
                */
                
                IF(@isCatchWeight = 'N') BEGIN

					--Check if purchase quantity = 1
					IF(@purchaseQuantity <> 1) BEGIN
						SET @error = 50361;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The items in purchase unit should be 1'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					--Check if sales quantity = 1
					IF(@purchaseQuantity <> 1) BEGIN
						SET @error = 50362;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The items in sales unit should be 1'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					--Check if purchase, sales and inventory uom is the same
					IF ( NULLIF( @purchaseUOM, '' ) <> @itemUOM OR NULLIF( @salesUOM, '' ) <> @itemUOM ) BEGIN
						SET @error = 50363;
						SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': The sales and purchase uom should be the same as the inventory uom'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					/*
					IF(@correctStockForUOM = 'Y' AND @correctStockForUOM2 = 'Y') BEGIN
						--Check if the uom to use for inventory transitions is 'A'
						IF( @UOMToUseForInventoryTransitions <> 'A') BEGIN
							SET @error = 50364;
							SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Uom to use for inventory transitions should be ''UomAndUOM2'' if correct stock for uom and uom2 is set to true'
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END
						--Check if the uom to use for documents is 'A'
						IF( @UOMToUseForPurchase <> 'A') BEGIN
							SET @error = 50365;
							SET @error_message = 'Item '+ @list_of_cols_val_tab_del +': Uom to use for documents should be ''UomAndUOM2'' if correct stock for uom and uom2 is set to true'
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END			
					END
					*/
				END
				
				-- Verify that UOM2 prices are possible (convert to UOM1 and back)
				DECLARE curPriceLists CURSOR LOCAL FOR
				SELECT U_PMX_UOM2PRICEUDF, CAST( "PriceDec" AS NVARCHAR ) FROM OPLN, OADM WHERE ISNULL( U_PMX_UOM2PRICEUDF, '' ) <> '';

				DECLARE @priceUDF NVARCHAR(20);
				DECLARE @priceDecimals NVARCHAR(20);
				DECLARE @getPriceUDFs NVARCHAR(500);
				DECLARE @badPrices NVARCHAR(500);

				DECLARE @tmp TABLE ( priceFromUDF DECIMAL(19,6), flooredPrice DECIMAL(19,6), ceiledPrice DECIMAL(19,6) )

				OPEN curPriceLists;
				FETCH NEXT FROM curPriceLists INTO @priceUDF, @priceDecimals;
				WHILE @@FETCH_STATUS >= 0
				BEGIN
					SET @getPriceUDFs = 'SELECT U_' + @priceUDF + ',
						CEILING( FLOOR  ( U_' + @priceUDF + ' / U_PMX_DQU2 * POWER(10,' + @priceDecimals + ') ) * U_PMX_DQU2 ) / POWER(10,' + @priceDecimals + '),
						CEILING( CEILING( U_' + @priceUDF + ' / U_PMX_DQU2 * POWER(10,' + @priceDecimals + ') ) * U_PMX_DQU2 ) / POWER(10,' + @priceDecimals + ')
					FROM OITM WHERE "ItemCode" = ''' + @list_of_cols_val_tab_del + '''';
					
					INSERT INTO @tmp
					EXEC( @getPriceUDFs )

					FETCH NEXT FROM curPriceLists INTO @priceUDF, @priceDecimals;
				END
				CLOSE curPriceLists;

				SELECT @badPrices = ISNULL( @badPrices + '. ', '' ) + 'UOM2 price ' + CONVERT( NVARCHAR, priceFromUDF ) + ' not possible, choose ' + CONVERT( NVARCHAR, flooredPrice ) + ' or ' + CONVERT( NVARCHAR, ceiledPrice )
				FROM @tmp
				WHERE priceFromUDF <> flooredPrice AND flooredPrice <> ceiledPrice;
				
				IF @badPrices IS NOT NULL
				BEGIN
					SET @error = 50366;
					SET @error_message = @badPrices
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END					
					



			--******************************
			--Updates
			--******************************

			-- * Set the custom description *

			UPDATE OITM SET U_PMX_CUDE = OITM."ItemCode" + ISNULL( ' - ' + OITM."ItemName", '') + ISNULL( ' - ' + OITM."CodeBars", '')
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del;

			-- * SBO UomGroup details to PMX_IPTY *

			DELETE FROM PMX_IPTY
			FROM OITM
			INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
			LEFT JOIN (
				UGP1
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
			) ON UGP1."UgpEntry" = OITM."UgpEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			  AND OITM."UgpEntry" <> -1
			  AND OUOM."UomCode" IS NULL

			UPDATE PMX_IPTY SET "Quantity" = UGP1."BaseQty" / UGP1."AltQty", "UserSign" = ISNULL( OUGP."UserSign2", 1 ), "UpdateDateTime" = @tsNow, "Version" = "Version" + 1
			FROM OITM
			INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
			INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
			INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
			INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			  AND OITM."UgpEntry" <> -1
			  AND "Quantity" <> CAST( UGP1."BaseQty" / UGP1."AltQty" AS DECIMAL(19,6) );

			INSERT INTO PMX_IPTY ( "UserSign", "CreateDateTime", "Version", "ItemCode", "PackagingTypeCode", "Quantity", "BarcodeType", "Canceled", "HideDuringEnteringQuantity", "AskDuringReception" )
			SELECT COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 ), @tsNow, 1, OITM."ItemCode", OUOM."UomCode", UGP1."BaseQty" / UGP1."AltQty", 'G', 'N', 'N', 'N'
			FROM OITM
			INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
			INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
			INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
			LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode" AND PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
			WHERE OITM."ItemCode" = @list_of_cols_val_tab_del
			  AND UGP1."UomEntry" <> OITM."IUoMEntry"
			  AND OITM."UgpEntry" <> -1
			  AND PMX_IPTY."PackagingTypeCode" IS NULL;


			--All checks are done so we can exit the checks.
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on BOM
		--**********************************************************************************************************
		IF (@isBOM=1) BEGIN
			--The lines of BOM may not contain a returnable item that is not an inventory item
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."ChildNum" AS VARCHAR) + ' ' 
			FROM ITT1 AS Lines INNER JOIN OITM AS Item ON Lines."Code" = Item."ItemCode"
			WHERE Lines."Father" = @list_of_cols_val_tab_del AND Item.U_PMX_RETR = 'Y' AND Item."InvntItem" = 'N';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50300;
				SET @error_message = 'BOM ''' + @list_of_cols_val_tab_del + ''': The line(s) ' + @errorList + ' contain returnable items that are NOT inventory. All items can be added to a BOM except not inventory returnable items.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--The inventory items on the lines of a BOM may not have backflush selected.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."ChildNum" AS VARCHAR) + ' ' 
			FROM ITT1 AS Lines INNER JOIN OITM AS Item ON Lines."Code" = Item."ItemCode"
			INNER JOIN OWHS ON Lines."Warehouse" = OWHS."Whscode"
			WHERE Lines."Father" = @list_of_cols_val_tab_del AND Item."InvntItem" = 'Y' AND Lines."IssueMthd" = 'B' AND OWHS."U_PMX_IMBP"='Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50301;
				SET @error_message = 'BOM ''' + @list_of_cols_val_tab_del + ''': The line(s) ' + @errorList + ' have ''Backflush'' selected as issue method. If using the Produmex add-on this is not allowed. This must be set to ''Manual''.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--All checks are done so we can exit the checks.
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on ProductionOrder
		--**********************************************************************************************************
		IF (@isProductionOrder=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--The inventory items on the lines of a production order may not have backflush selected.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM WOR1 AS Lines INNER JOIN OITM AS Item ON Lines."ItemCode" = Item."ItemCode"
			INNER JOIN OWOR AS ProductionOrder on ProductionOrder."DocEntry" = Lines."DocEntry"
			INNER JOIN OWHS ON Lines."Warehouse" = OWHS."Whscode"
			WHERE Lines."DocEntry" = @keyAsInteger AND Item."InvntItem" = 'Y' AND Lines."IssueType" = 'B' AND OWHS."U_PMX_IMBP"='Y' 
			AND ProductionOrder."Status" <> 'L';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50900;
				SET @error_message = 'Production order ''' + @list_of_cols_val_tab_del + ''': The line(s) ' + @errorList + ' have ''Backflush'' selected as issue method. If using the Produmex add-on this is not allowed. This must be set to ''Manual''.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			DECLARE @parentDocEntry INT;
			DECLARE @parentDocNum INT;
			DECLARE @parentFound INT;
			
			SELECT @parentDocEntry = this.U_PMX_PPRD, @parentDocNum = this.U_PMX_PPRN, @parentFound = parent."DocEntry"
			FROM OWOR this
			LEFT JOIN OWOR parent ON this.U_PMX_PPRD = parent."DocEntry" AND this.U_PMX_PPRN = parent."DocNum"
			WHERE this."DocEntry" = @keyAsInteger;
			IF ( @parentDocEntry IS NULL AND @parentDocNum IS NOT NULL )
				OR ( @parentDocEntry IS NOT NULL AND @parentDocNum IS NULL )
				OR ( @parentDocEntry IS NOT NULL AND @parentDocNum IS NOT NULL AND @parentFound IS NULL )
			BEGIN
				SET @error = 50901;
				SET @error_message = 'Production order ''' + @list_of_cols_val_tab_del + ''': The "DocEntry" ''' + ISNULL( CONVERT(VARCHAR,@parentDocEntry), '' ) + ''' and "DocNum" ''' + ISNULL( CONVERT(VARCHAR,@parentDocNum), '' ) + ''' referencing the parent production order do not match.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			IF @parentDocEntry = @keyAsInteger BEGIN
				SET @error = 50902;
				SET @error_message = 'Production order ''' + @list_of_cols_val_tab_del + ''': a production order cannot have itself as parent.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Closed production orders cannot have the pmx status 'started'
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + ' ' 
			FROM OWOR										--Cancelled, Closed
			WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" IN ('C', 'L') AND OWOR.U_PMX_PRST = 'S'
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50903;
				SET @error_message = 'Production order ''' + @errorList + ''': The production order cannot be closed when pmx status is started!';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Planned qty cannot be lowered if open proposals exist
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + '.' + CAST(WOR1."LineNum" AS VARCHAR) + ' ' 
			FROM OWOR
			INNER JOIN WOR1 ON WOR1."DocEntry" = OWOR."DocEntry"
			AND WOR1."PlannedQty" > 0
			LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseEntry" = WOR1."DocEntry"
			AND PMX_PLPL."BaseLine" = WOR1."LineNum"
			AND PMX_PLPL."BaseType" = '202'
			AND PMX_PLPL."LineStatus" = 'O'
			WHERE OWOR."DocEntry" = @keyAsInteger
			GROUP BY OWOR."DocNum", WOR1."DocEntry", WOR1."LineNum", WOR1."PlannedQty"
			HAVING (SUM(ISNULL( PMX_PLPL."Quantity", 0)) > WOR1."PlannedQty");
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50904;
				SET @error_message = 'Production order line ''' + @errorList + ''': The production order planned quantity cannot be lowered because there are open proposal(s)!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Productionline cannot be changed if open proposals exist
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) 
			FROM OWOR
			INNER JOIN WOR1 ON WOR1."DocEntry" = OWOR."DocEntry"
			INNER JOIN PMX_PLPL ON PMX_PLPL."BaseEntry" = WOR1."DocEntry"
			AND PMX_PLPL."BaseLine" = WOR1."LineNum"
			AND PMX_PLPL."BaseType" = '202'
			INNER JOIN PMX_PLPH ON PMX_PLPH."DocEntry" = PMX_PLPL."DocEntry"
			INNER JOIN PMX_OSPL ON PMX_OSPL."Code" = OWOR.U_PMX_PLCD
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND PMX_PLPH."DestStorLocCode" <> ISNULL(PMX_OSPL."PickToLocationCode", PMX_OSPL."InputLocationCode")
			AND PMX_PLPH."DocStatus" = 'O';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50905;
				SET @error_message = 'Production order ''' + @errorList + ''': The production order has a different production line than the proposal(s)!'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

		
			--Only one production order allowed for a production line with the pmx status 'started'
			IF(EXISTS(SELECT OWOR."DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" NOT IN ('C', 'L') 
				AND OWOR.U_PMX_PRST = 'S')) BEGIN

				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + ' ' 
				FROM OWOR	
				INNER JOIN OWOR AS currentOWOR
				ON OWOR.U_PMX_PLCD = currentOWOR.U_PMX_PLCD	
				INNER JOIN PMX_OSPL ON PMX_OSPL."Code" = currentOWOR.U_PMX_PLCD
										
				WHERE OWOR."DocEntry" <> @keyAsInteger 
				AND currentOWOR."DocEntry" = @keyAsInteger
									--Cancelled, Closed
				AND OWOR."Status" NOT IN ('C', 'L') 
				AND OWOR.U_PMX_PRST = 'S'
				AND PMX_OSPL."OnlyOneStartedProductionOrderAllowed" = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50904;
					SET @error_message = 'Production order(s) ''' + @errorList + ''' started: There can only be 1 production order for a certain production line with the pmx status started!'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END
			
			--All batchnumbers must be entered in upper case
			--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."U_PMX_BTCH" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND OWOR."U_PMX_BTCH" COLLATE SQL_Latin1_General_Cp850_CS_AS <> UPPER(OWOR."U_PMX_BTCH") COLLATE SQL_Latin1_General_Cp850_CS_AS
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50906;
				SET @error_message = 'Batch numbers must be upper case. This is not the case for the batchnumber: ' + @errorList
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."U_PMX_BTC2" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND OWOR."U_PMX_BTC2" COLLATE SQL_Latin1_General_Cp850_CS_AS <> UPPER(OWOR."U_PMX_BTC2") COLLATE SQL_Latin1_General_Cp850_CS_AS
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50907;
				SET @error_message = 'Batch numbers must be upper case. This is not the case for the batchnumber 2: ' + @errorList
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(OWOR."U_PMX_NRPB" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = @keyAsInteger 
			AND OWOR."U_PMX_NRPB" = 0
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50908;
				SET @error_message = 'The number of production batches cannot be 0 for production order ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Add check for removing time registration items
			IF (@isTransactionTypeUpdate=1) BEGIN
				SET @errorList = ''	
				SELECT @errorList =	@errorList + CAST(PMX_TRHE."ItemCode" AS VARCHAR) + ' '
				FROM PMX_TRHE 
				LEFT JOIN WOR1 ON PMX_TRHE."BaseEntry" = WOR1."DocEntry" AND PMX_TRHE."ItemCode" = WOR1."ItemCode"
				AND PMX_TRHE."BaseType" = '202'
				WHERE WOR1."ItemCode" IS NULL
				AND WOR1."DocEntry" = @keyAsInteger 

				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50909;
					SET @error_message = 'Cannot delete item codes because time has already been registered: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END

			--All checks are done so we can exit the checks.

			--Perform actions when adding production order

			--When a production order is duplicated, UDF's are also copied
			IF (@isTransactionTypeAdd=1) BEGIN
				--The pmx status is also copied.
				--But the status should be (P)lanned when a production order is added.
				UPDATE OWOR SET OWOR.U_PMX_PRST = 'P' WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR.U_PMX_PRST <> 'P';
				--When production order is duplicated, the CurrentProdBatch should NOT be copied and set to default 1			
				UPDATE OWOR SET OWOR.U_PMX_CUPB = 1  WHERE OWOR."DocEntry" = @keyAsInteger; --AND OWOR.U_PMX_PRST <> 'P';
				--Set picked quantity to 0
				UPDATE WOR1 SET WOR1.U_PMX_QTYP = 0 WHERE WOR1."DocEntry" = @keyAsInteger AND U_PMX_QTYP <> 0;
			END

            IF (@isTransactionTypeUpdate=1) BEGIN
            
            SET @errorList = ''
				SELECT @errorList = @errorList + CAST(OWOR."DocNum" AS VARCHAR) + ' ' 
				FROM OWOR													
				WHERE OWOR."DocEntry" = @keyAsInteger 				
				AND  OWOR.U_PMX_NRPB <= ( SELECT MAX(PMX_POBA."SeqNr") FROM PMX_POBA WHERE PMX_POBA."DocEntry" = 220 );
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50906;
					SET @error_message = 'Already components weighed for production order ''' + @errorList + '''. Cannot alter number of production batches lower than current production batch.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			    --Remove locking when production order is closed
			    DELETE FROM PMX_INLD WHERE "BaseType" = '202' 
			    AND "BaseEntry" IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" IN ('C', 'L'));
			    
			      --Remove dispended SSCC when production order is closed
			    DELETE FROM PMX_POPC WHERE "ProdOrderDocEntry"
			     IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger AND 
						( OWOR."Status" IN ('C', 'L') OR OWOR.U_PMX_PRST IN ('C')));

                --Remove lined up locations when order is closed and production line is not used in another started production order
                DELETE FROM PMX_LIUP WHERE "ProductionLineCode" IN
                (SELECT U_PMX_PLCD FROM OWOR WHERE
                 OWOR."DocEntry" = @keyAsInteger AND OWOR."Status" IN ('C', 'L') 
                 AND U_PMX_PLCD NOT IN 
                 (SELECT U_PMX_PLCD FROM OWOR WHERE
					OWOR."Status" = 'R' AND U_PMX_PRST = 'S'));

             END
             
             IF(@isTransactionTypeDelete=1 OR @isTransactionTypeCancel=1) BEGIN
             
                --Remove dispended SSCC when production order is closed
			    DELETE FROM PMX_POPC WHERE "ProdOrderDocEntry"
			     IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = @keyAsInteger );
             
             END

			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on a user
		--**********************************************************************************************************
		IF (@isUser=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--Check if the language key filled in at the user is an existing language key
			SELECT @count = COUNT(OUSR.INTERNAL_K) 
			FROM OUSR
            LEFT JOIN OLNG ON OUSR.U_PMX_LANG = OLNG."Code"
			WHERE OUSR.INTERNAL_K = @keyAsInteger AND OLNG."Code" IS NULL AND OUSR.U_PMX_LANG IS NOT NULL;
			IF @count <> 0 BEGIN
				SET @error = 50400;
				SET @error_message = 'The language key filled in for user with key ' + @list_of_cols_val_tab_del + ' does not exist.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
		
			--All checks are done so we can exit the checks.
			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on a business partner
		--**********************************************************************************************************
		IF (@isBusinessPartner=1) BEGIN
			--Check if this is a delete. 
			IF (@isTransactionTypeDelete=1) BEGIN
				
				--Check if the customer is not used in a route template
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_RTTH."Name" + ' ' 
				FROM PMX_RTTH
				INNER JOIN PMX_RTTL ON PMX_RTTL."ParentCode" = PMX_RTTH."Code"
				WHERE PMX_RTTL."CardCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50500;
					SET @error_message = 'Business partner with card code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the customer is used in route templates with name: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check if the customer is not used in a picklist proposal
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_PLPH."DocEntry" + ' ' 
				FROM PMX_PLPH
				WHERE PMX_PLPH."CardCode" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50501;
					SET @error_message = 'Business partner with card code ''' + @list_of_cols_val_tab_del + ''' can not be deleted because the customer is used in picklist proposals with doc number: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--No other checks must be done for the item delete
				RETURN;
			END

			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate=1 ) BEGIN
			
				--Check if linked business partner exists
				SET @errorList = ''
				SELECT @errorList = @errorList + OCRD.U_PMX_LBPA + ' ' 
				FROM OCRD
				WHERE OCRD."CardCode" = @list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT CardCode FROM OCRD AS InnerOCRD WHERE InnerOCRD."CardCode" = OCRD.U_PMX_LBPA)
				AND ISNULL( OCRD.U_PMX_LBPA, '') <> '';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50502;
					SET @error_message = 'Business partner with card code ''' + @list_of_cols_val_tab_del + ''' has an incorrect linked business partner: ' + @errorList
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--No other checks must be done for the item delete
				RETURN;
			END

			RETURN;
		END

		--**********************************************************************************************************
		--Do checks on a sales order
		--**********************************************************************************************************
		IF (@isSalesOrderDocument=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--Check if the sales order is closed that there are no open pick list proposals of this sales order
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM ORDR 
			INNER JOIN PMX_PLPL ON ORDR."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = ORDR."ObjType"
			INNER JOIN RDR1 ON ORDR."DocEntry" = RDR1."DocEntry" AND RDR1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE ORDR."DocEntry" = @keyAsInteger AND (ORDR."DocStatus" = 'C' OR RDR1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50600;
				SET @error_message = 'The sales order ''' + @list_of_cols_val_tab_del + ''' has open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before closing the sales order (line).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the full quantity is selected when entering batch numbers
--			SET @errorList = ''
--			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
--			FROM RDR1 AS Lines
--			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
--			LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesOrder."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
--			WHERE SalesOrder."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O'
--			GROUP BY Lines."LineNum", Lines."Quantity", Lines."NumPerMsr"
--			HAVING SUM(IBT1."Quantity") <> (Lines."Quantity" * Lines."NumPerMsr") AND SUM(IBT1."Quantity") > 0
--			IF LEN(@errorList) <> 0 BEGIN
--				SET @error = 50601;
--				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
--				SELECT @error, @error_message
--				EXEC xp_logevent @error, @error_message, ERROR
--				RETURN;
--			END

			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM RDR1 AS Lines
            INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
            INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
            INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
            INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
            AND ITL1."SysNumber" = OBTN."SysNumber"
            WHERE OITL."AllocatEnt" = @keyAsInteger 
			AND Lines."LineStatus" = 'O'
            AND "StockEff" = 2
            GROUP BY Lines."LineNum", Lines."OpenQty", Lines."NumPerMsr"
            HAVING SUM("AllocQty") <> (Lines."OpenQty" * Lines."NumPerMsr") AND SUM("AllocQty") > 0;
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50601;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			--Check if sales order lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the sales order line is lower than on the open pick list proposal line / open pick list line
			SET @errorList = ''
			SELECT @errorList  = @errorList  + CASE WHEN        RDR1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN RDR1 ON PMX_PLPL."BaseEntry" = RDR1."DocEntry" AND PMX_PLPL."BaseLine" = RDR1."LineNum"  AND PMX_PLPL."BaseType" = RDR1."ObjType"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = 17 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR ISNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50603;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' The item code on a sales order line cannot be changed if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList2 = ''
			SELECT @errorList2 = @errorList2 + CASE WHEN ISNULL( RDR1."OpenInvQty",0) < SUM(ROUND( PMX_PLPL."OpenQty" * PMX_PLPL."QuantityPerUom", ISNULL( OITM.U_PMX_UOMD, 6)) ) THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' '  ) ELSE '' END
			FROM PMX_PLPL
			INNER JOIN OITM ON OITM."ItemCode" = PMX_PLPL."ItemCode"
			LEFT JOIN RDR1 ON PMX_PLPL."BaseEntry" = RDR1."DocEntry" AND PMX_PLPL."BaseLine" = RDR1."LineNum"  AND PMX_PLPL."BaseType" = RDR1."ObjType"
			WHERE PMX_PLPL."BaseType" = 17 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C') 
			GROUP BY PMX_PLPL."BaseLine", RDR1."OpenQty", RDR1."OpenInvQty";
			IF LEN(@errorList2) <> 0 BEGIN
				SET @error = 50602;
				SET @error_message = 'Error on line(s):  ' + @errorList2 + ' The quantity on a sales order line cannot be lowered if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM RDR1 AS Lines
			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesOrder."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."AllocatEnt" = Lines."DocEntry" AND BATCH."AllocateLn" = Lines."LineNum" AND BATCH."AllocateTp" = Lines."ObjType"
						
			WHERE SalesOrder."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O' AND Lines.[U_PMX_AMBA] = 'N'
			GROUP BY Lines."LineNum"
			HAVING (Count(*)) > 1;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50604;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			-- change of batch only allowed when not linked in DOLL
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '		
			FROM RDR1 AS Lines
			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
			INNER JOIN PMX_DOLL ON PMX_DOLL."ObjType" = Lines."ObjType" 
			AND PMX_DOLL."DocEntry" = Lines."DocEntry" 
			AND PMX_DOLL."LineNum" = Lines."LineNum" 

			LEFT JOIN 
			(SELECT "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp"
				HAVING SUM("AllocQty") > 0) AS BATCH

			ON BATCH."AllocatEnt" = Lines."DocEntry" AND BATCH."AllocateLn" = Lines."LineNum" AND BATCH."AllocateTp" = Lines."ObjType"
			AND BATCH."DistNumber" = PMX_DOLL."BatchNum"

			WHERE Lines."DocEntry" = @keyAsInteger 
			AND Lines."LineStatus" = 'O'
			AND (ISNULL( BATCH."AllocQty", PMX_DOLL."LockedQuantity") < PMX_DOLL."LockedQuantity"
				OR (PMX_DOLL."BatchNum" IS NOT NULL AND BATCH."DistNumber" IS NULL )
				);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50605;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' Proposal exists: Changing the batch number(s) not allowed'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Multiple batches on serial numbers are only allowed when multiple batches on line level are allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(RDR1."LineNum" AS VARCHAR) + ' '	
			FROM PMX_SELD	
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN RDR1 on RDR1."DocEntry" = PMX_SELD."BaseEntry"
			AND RDR1."LineNum" = PMX_SELD."BaseLine"
			AND RDR1."ObjType" = PMX_SELD."BaseType"
			WHERE RDR1.U_PMX_AMBA = 'N'
			AND RDR1."DocEntry" = @keyAsInteger	
			GROUP BY RDR1."LineNum", PMX_SENU."ItriKey"
			HAVING COUNT(PMX_SENU."ItriKey") > 1;

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50606;
				SET @error_message = 'Serial numbers linked with multiple batches, but multiple batches are not allowed. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Linked serial numbers cannot be more than open quantity
			SET @errorList = ''
			SELECT @errorList = @errorList + 'Line ' + CAST(RDR1."LineNum" AS VARCHAR) + ', Link qty:' + CAST(COUNT(PMX_SELD."InternalKey") AS VARCHAR) + '; '	
			FROM PMX_SELD
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN RDR1 on RDR1."DocEntry" = PMX_SELD."BaseEntry"
			AND RDR1."LineNum" = PMX_SELD."BaseLine"
			AND RDR1."ObjType" = PMX_SELD."BaseType"
			WHERE RDR1."DocEntry" = @keyAsInteger
			AND PMX_SENU."InStock" = 'Y'
			AND RDR1."LineStatus" = 'O'
			GROUP BY RDR1."LineNum", RDR1."OpenQty", RDR1."NumPerMsr"
			HAVING COUNT(PMX_SELD."InternalKey") > RDR1."OpenQty" * RDR1."NumPerMsr";

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50607;
				SET @error_message = 'Number of linked serial numbers cannot be more than open quantity. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date minutes
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(ORDR.U_PMX_DDTM AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = @keyAsInteger
			AND (ISNULL( ORDR.U_PMX_DDTM, 0) > 59 OR
				ISNULL( ORDR.U_PMX_DDTM, 0) < 0);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50608;
				SET @error_message = 'The due date minutes: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date Hour
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(ORDR.U_PMX_DDTH AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = @keyAsInteger
			AND ( ISNULL( ORDR.U_PMX_DDTH, 0) < 0
					OR ( ISNULL( ORDR.U_PMX_DDTH, 0) > 23 AND ISNULL( ORDR.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR ( ISNULL( ORDR.U_PMX_DDTH, 0) > 11 AND ISNULL( ORDR.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50609;
				SET @error_message = 'The due date hours: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE
			DELETE LOCKING FROM PMX_INLD LOCKING INNER JOIN ORDR ON ORDR."DocEntry" = LOCKING."BaseEntry"
			AND LOCKING."BaseType" = '17'
			WHERE ORDR."DocStatus" = 'C' AND ORDR."DocEntry" = @keyAsInteger


			IF(@isTransactionTypeAdd=0 AND
				EXISTS(SELECT DocEntry FROM RDR1 WHERE RDR1."LineStatus" = 'C' AND RDR1."DocEntry" = @keyAsInteger)) 
			BEGIN

				--Close container (lines) if needed
				UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
				FROM PMX_COLI
				INNER JOIN RDR1
				ON PMX_COLI."BaseEntry" = RDR1."DocEntry"
				AND PMX_COLI."BaseLine" = RDR1."LineNum"
				AND PMX_COLI."BaseType" = RDR1."ObjType"
				WHERE PMX_COLI."LineStatus" <> 'C'
				AND RDR1."DocEntry" = @keyAsInteger
				AND RDR1."LineStatus" = 'C';

				--Close header if no open lines are found
				UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
				PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
				PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM ORDR  WHERE ORDR."DocEntry" = @keyAsInteger)
				WHERE PMX_COHE."DocStatus" = 'O'
				AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

			END


			--**********************************************************************************************************
			--START: Add/update locking based on batch numbers
			--**********************************************************************************************************



			--Start with deleting the lockings that are not correct
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" NOT IN 
			(--Query to get the correct lockings
				SELECT 	PMX_INLD."InternalKey"
				FROM RDR1 AS Lines
				INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
				INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')
				--AND ISNULL( PMX_ITRI."BestBeforeDate", 1) = ISNULL( "ExpDate", 1)

				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"
				AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

				WHERE OITL."AllocatEnt" = @keyAsInteger 
				AND Lines."LineStatus" = 'O'
				AND "StockEff" = 2

				AND PMX_INLD."InternalKey" IS NOT NULL


				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				HAVING SUM("AllocQty") > 0

				-- Exclude locked stock created on goods receipt.
				UNION SELECT PMX_INLD."InternalKey"
				FROM RDR1 Lines 
				INNER JOIN POR1 ON 
				POR1."BaseType" = Lines."ObjType"
				AND POR1."BaseEntry" = Lines."DocEntry"
				AND POR1."BaseLine" = Lines."LineNum"

				INNER JOIN PDN1 ON 
				PDN1."BaseType" = POR1."ObjType"
				AND PDN1."BaseEntry" = POR1."DocEntry"
				AND PDN1."BaseLine" = POR1."LineNum"

				INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
				LEFT JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
				LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL(  PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')


				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"

				WHERE Lines."DocEntry" = @keyAsInteger				

				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				--Only allow detailed locking from purchase order when no batch has been linked
				HAVING SUM( ISNULL( "AllocQty", 0)) = 0 

				--Exclude manual lockings on LUID level when no batch has been selected on sales order
				UNION SELECT PMX_INLD."InternalKey"
				FROM PMX_INLD 
				LEFT JOIN
				(
					SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
					FROM RDR1 AS Lines
									INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
									LEFT JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
									LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
									LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
									AND ITL1."SysNumber" = OBTN."SysNumber"

					WHERE OITL."AllocatEnt"  = @keyAsInteger 
									AND Lines."LineStatus" = 'O'
									AND ISNULL( "StockEff", 2) = 2

						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
									Lines."NumPerMsr", "DistNumber", "ExpDate"

					HAVING SUM( ISNULL( "AllocQty", 0)) > 0
				) AS ORDR

				ON ORDR."LineNum" = PMX_INLD."BaseLine"
				AND ORDR."DocEntry" = PMX_INLD."BaseEntry"
				AND ORDR."ObjType" = PMX_INLD."BaseType"


				WHERE PMX_INLD."InvLockLevel" = 'L'
				AND PMX_INLD."BaseEntry" = @keyAsInteger
				AND ORDR."DocEntry" IS NULL

				

			) 

			AND PMX_INLD."BaseType" = '17'
			AND PMX_INLD."BaseEntry" = @keyAsInteger;



			BEGIN TRY

				DECLARE salesOrderBatchCursor CURSOR LOCAL FOR 
						SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", Lines."ObjType", 
						Lines."ItemCode", PMX_OSWH."Code", CASE WHEN OITM."U_PMX_ICAW" = 'Y' THEN OITM."U_PMX_DQUM" ELSE NULL END, Lines."U_PMX_SQOP"
						FROM RDR1 AS Lines
						INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
						INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = Lines."WhsCode"

						LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType"
						AND PMX_PLPL."BaseEntry" = Lines."DocEntry"
						AND PMX_PLPL."BaseLine" = Lines."LineNum"
						AND PMX_PLPL."LineStatus" = 'O'

						WHERE OITL."AllocatEnt" = @keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2
						--Only when no openpick list proposal is found
						AND PMX_PLPL."DocEntry" IS NULL
						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code", OITM."U_PMX_ICAW", OITM."U_PMX_DQUM", Lines."U_PMX_SQOP"

						HAVING SUM("AllocQty") > 0;


				OPEN salesOrderBatchCursor;
				FETCH NEXT FROM salesOrderBatchCursor 
				INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock,@pmxWarehouseToLock, @catchWeightRatioToLock, @shippingQualityOptionToLock;
				WHILE @@FETCH_STATUS >= 0 BEGIN

					--Check for each batch if we have a corresponding locking line

					IF NOT EXISTS (SELECT TOP(1) PMX_INLD.InternalKey FROM PMX_INLD 
					INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
					WHERE PMX_INLD."BaseType" = @typeToLock
					AND PMX_INLD."BaseEntry" = @docEntryToLock
					AND PMX_INLD."BaseLine" = @lineNumToLock
					AND PMX_ITRI."BatchNumber" = @batchToLock
					--AND ISNULL( PMX_ITRI."BestBeforeDate", '') = ISNULL( @expDateToLock, '');
					)
					BEGIN

						--Entry does not exist, so we need to lock the stock
						--First try to get the itri key to lock
						SET @remainingQuantityToLock = @batchQuantityToLock;

						BEGIN TRY
							DECLARE itriCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (
											SELECT "Code" 
											FROM PMX_QYST 
											WHERE  (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
													)
										AND PMX_INVT.StorLocCode NOT IN (
											SELECT Code FROM PMX_DISALLOWED_LOCATIONS_FOR_PICKING) GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (
											SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" 
											FROM PMX_INLD 
											WHERE "QualityStatusCode" IN (
												SELECT "Code" FROM PMX_QYST 
												WHERE (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
													)
											GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;


							OPEN itriCursor;
							FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
							WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

								IF(@remainingQuantityToLock < @freeQuantityToLock)
								BEGIN
									SET @currentQuantityToLock = @remainingQuantityToLock;
								END
								ELSE
								BEGIN
									SET @currentQuantityToLock = @freeQuantityToLock;
								END

								--Insert INTO the PMX_INLD table
								INSERT INTO PMX_INLD ("BaseType","BaseEntry","BaseLine","CardCode","CardName",
									"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
									"UserSign","CreateDate",
									"CreateTime","LogUnitIdentKey","QualityStatusCode", 
									"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
								SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(ORDR."CardCode"),MAX(ORDR."CardName"),
									@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
									MAX(ORDR."UserSign"),@updateDate,
									@updateTime,NULL, @qualityStatusToLock,
									'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, @currentQuantityToLock * @catchWeightRatioToLock
								FROM ORDR 
								INNER JOIN RDR1 ON RDR1."DocEntry" = ORDR."DocEntry"
								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = RDR1."WhsCode"
								WHERE ORDR."DocEntry" = @docEntryToLock
								AND RDR1."LineNum" = @lineNumToLock;

								--Reduce remaining
								SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


								FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;


							END
							CLOSE itriCursor;
							DEALLOCATE itriCursor

							--If there is remaining QTY, there is not enough free stock, so throw error
							IF(@remainingQuantityToLock > 0)
							BEGIN
								SET @error = 50650;
								SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock + '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END

						END TRY
						BEGIN CATCH
							SET @error = 50690;
							SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
							+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
							BEGIN TRY
								CLOSE itriCursor 
								DEALLOCATE itriCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END TRY
							BEGIN CATCH
								DEALLOCATE itriCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END CATCH
						END CATCH

					END
					ELSE 
					BEGIN
						--The line exists, so now we need to check if the quantity is the same/more/less
						SET @remainingQuantityToLock = (SELECT SUM(PMX_INLD."Quantity") -  @batchQuantityToLock AS "Quantity"
                        FROM PMX_INLD 
						INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
						WHERE PMX_INLD."BaseType" = @typeToLock
						AND PMX_INLD."BaseEntry" = @docEntryToLock
						AND PMX_INLD."BaseLine" = @lineNumToLock
						AND PMX_ITRI."BatchNumber" = @batchToLock
						GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine");
							
										
						IF @remainingQuantityToLock <> 0 BEGIN
						--Only if deviation, then we try to update/insert
						--Loop through all locking lines
							BEGIN TRY
								DECLARE inldCursor CURSOR LOCAL FOR
                                SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey"
                                FROM PMX_INLD
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = @typeToLock
								AND PMX_INLD."BaseEntry" = @docEntryToLock
								AND PMX_INLD."BaseLine" = @lineNumToLock
								AND PMX_ITRI."BatchNumber" = @batchToLock;

								OPEN inldCursor;
								FETCH NEXT FROM inldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock <> 0 BEGIN

									IF @remainingQuantityToLock > 0 BEGIN
										--Remaining is more than 0, so we can just remove the locking
										IF @remainingQuantityToLock >= @lockedQuantity BEGIN
											--Delete
											DELETE FROM PMX_INLD WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = @remainingQuantityToLock - @lockedQuantity;
										END
                                        ELSE
                                        BEGIN
											--update
											UPDATE PMX_INLD SET "Quantity" = "Quantity" - @remainingQuantityToLock, "QuantityUom2" = "QuantityUom2" - (@remainingQuantityToLock * @catchWeightRatioToLock) WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = 0;
										END
									END
                                    ELSE
                                    BEGIN

										--Remaining is less than 0, so we need to add locking
										--If there is still free stock for current itri, then take that one
										SELECT @freeQuantityToLock = ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0)
                                        FROM PMX_ITRI 
										LEFT JOIN (
                                            SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
                                            FROM PMX_INVT 
                                            INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
                                            WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
                                            GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
                                        ) AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (
                                            SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
                                            FROM PMX_INLD
                                            WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
                                            GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
                                        ) AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
                                            AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
                                            AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0
										AND PMX_ITRI."InternalKey" = @itriToLock;

										IF(@remainingQuantityToLock > @freeQuantityToLock) BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
										END
                                        ELSE
                                        BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END

										UPDATE PMX_INLD SET "Quantity" = "Quantity" + @currentQuantityToLock, "QuantityUom2" = "QuantityUom2" + (@currentQuantityToLock * @catchWeightRatioToLock) WHERE PMX_INLD."InternalKey" = @currentLockInternalKey;

										-- add qty, because @remainingQuantityToLock is negative
										SET @remainingQuantityToLock = @remainingQuantityToLock + @currentQuantityToLock;
										
									END

									FETCH NEXT FROM inldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								END
								CLOSE inldCursor;
								DEALLOCATE inldCursor
							END TRY
							BEGIN CATCH
								SET @error = 50691
								SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
								+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
								BEGIN TRY
									CLOSE inldCursor 
									DEALLOCATE inldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END TRY
								BEGIN CATCH
									DEALLOCATE inldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END CATCH
							END CATCH

							IF @remainingQuantityToLock <> 0 BEGIN
								-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
								--Set @remainingQuantityToLock as positive quanitty
								SET @remainingQuantityToLock = ABS(@remainingQuantityToLock);
								

								BEGIN TRY
									DECLARE itriCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
												INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
												AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
												WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
												AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
												AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;


									OPEN itriCursor;
									FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock
									WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

										IF (@remainingQuantityToLock < @freeQuantityToLock)
										BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END
										ELSE
										BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
										END

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ( "BaseType", "BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
										SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(ORDR."CardCode"),MAX(ORDR."CardName"),
											@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
											MAX(ORDR."UserSign"),@updateDate,
											@updateTime,NULL, @qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, @currentQuantityToLock * @catchWeightRatioToLock
										FROM ORDR 
										INNER JOIN RDR1 ON RDR1."DocEntry" = ORDR."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = RDR1."WhsCode"
										WHERE ORDR."DocEntry" = @docEntryToLock
										AND RDR1."LineNum" = @lineNumToLock;

										--Reduce remaining
										SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


										FETCH NEXT FROM itriCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									END
									CLOSE itriCursor;
									DEALLOCATE itriCursor

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(@remainingQuantityToLock > 0)
									BEGIN
										SET @error = 50652;
										SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN;
									END
								END TRY
								BEGIN CATCH
									SET @error = 50691
									SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
									+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE itriCursor 
										DEALLOCATE itriCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE itriCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END CATCH
								END CATCH


							END


						END


						IF @remainingQuantityToLock <> 0 BEGIN
						-- if quantity is still different from 0, then there was not enough stock to reserve!
							CLOSE salesOrderBatchCursor 
							DEALLOCATE salesOrderBatchCursor

							SET @error = 50653;
							SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END

					END

					FETCH NEXT FROM salesOrderBatchCursor
					INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock, @pmxWarehouseToLock, @catchWeightRatioToLock, @shippingQualityOptionToLock;


				END
				CLOSE salesOrderBatchCursor;
				DEALLOCATE salesOrderBatchCursor
			END TRY
			BEGIN CATCH
				SET @error = 50699
				SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
				+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE salesOrderBatchCursor 
					DEALLOCATE salesOrderBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE salesOrderBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END CATCH
			END CATCH

			--**********************************************************************************************************
			--END: Add/update locking based on batch numbers
			--**********************************************************************************************************

			--**********************************************************************************************************
			--START: SAP Serial numbers
			--**********************************************************************************************************

			--Delete system serial links
			DELETE FROM PMX_SELD WHERE "LinkStatus" = 'S'
			AND PMX_SELD."BaseEntry" = @keyAsInteger
			AND PMX_SELD."BaseType" = '17';

			--Add to PMX serial numbers table

			INSERT INTO PMX_SELD ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine", "LinkStatus") 
			SELECT 'N', SERIAL."UserSign", @tsNow, @tsNow, 1, PMX_SENU."InternalKey", "ApplyType", "ApplyEntry", "ApplyLine", 'S'	
			FROM (
                SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty", "StockEff", ORDR."UserSign"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
				AND ITL1."SysNumber" = OSRN."SysNumber"
				INNER JOIN ORDR ON ORDR."DocEntry" = OITL."ApplyEntry"
				INNER JOIN RDR1 ON RDR1."DocEntry" = OITL."ApplyEntry"
				AND RDR1."LineNum" = OITL."ApplyLine"
				AND RDR1."ObjType" = OITL."ApplyType"

				WHERE "StockEff" = 2
				AND OITL."ApplyEntry" = @keyAsInteger
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", ORDR."UserSign"
				HAVING SUM("AllocQty") > 0 AND "DistNumber" IS NOT NULL
			) AS SERIAL

			INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = SERIAL."ItemCode"
			AND PMX_SENU."SerialNumber" = SERIAL."DistNumber"
			LEFT JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
			AND PMX_SELD."BaseEntry" = SERIAL."ApplyEntry"
			AND PMX_SELD."BaseLine" = SERIAL."ApplyLine"
			AND PMX_SELD."BaseType" = N'17'
			WHERE PMX_SELD."InternalKey" IS NULL;



			--**********************************************************************************************************
			--END: SAP Serial numbers
			--**********************************************************************************************************


			RETURN;

		END


		--**********************************************************************************************************
		--Do checks on a purchase order - ERROR 51300 - 51399
		--**********************************************************************************************************
		IF (@isPurchaseOrderDocument=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			IF(@isTransactionTypeAdd=0 AND
				EXISTS(SELECT DocEntry FROM POR1 WHERE POR1."LineStatus" = 'C' AND POR1."DocEntry" = @keyAsInteger)) 
			BEGIN

				--Close container (lines) if needed
				UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
				FROM PMX_COLI
				INNER JOIN POR1
				ON PMX_COLI."BaseEntry" = POR1."DocEntry"
				AND PMX_COLI."BaseLine" = POR1."LineNum"
				AND PMX_COLI."BaseType" = POR1."ObjType"
				WHERE PMX_COLI."LineStatus" <> 'C'
				AND POR1."DocEntry" = @keyAsInteger
				AND POR1."LineStatus" = 'C';

				--Close header is no open lines are found
				UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
				PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
				PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM OPOR  WHERE OPOR."DocEntry" = @keyAsInteger)
				WHERE PMX_COHE."DocStatus" = 'O'
				AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

			END		
			
			IF ( @isTransactionTypeAdd = 0 )
			BEGIN
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(PMX_COLI."DocEntry" AS VARCHAR) + ' ' 
				FROM PMX_COLI 
				LEFT JOIN POR1 ON POR1."DocEntry" = PMX_COLI."BaseEntry" AND POR1."LineNum" = PMX_COLI."BaseLine" 
					 WHERE PMX_COLI."LineStatus" <> 'C' 
					 AND PMX_COLI."BaseEntry" = @keyAsInteger 
					 AND PMX_COLI."BaseType" = N'22'
					 AND POR1."ItemCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 51300;
					SET @error_message = 'Cannot delete purchase order line(s) because of container-document(s): ' + @errorList + '. These container-document(s) have links to the purchase order line(s).'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
			END			
	
			RETURN;
		END
		
		--**********************************************************************************************************
		--Do checks on a warehouse: ERROR 50850 - 50900
		--**********************************************************************************************************
		IF (@isWarehouse=1) BEGIN
			--Check if bin locations are not enabled for PMX managed whs
			SET @errorList = ''
			SELECT @errorList = @errorList + OWHS."WhsCode" + ' ' 
			FROM OWHS
			WHERE OWHS."WhsCode" = @list_of_cols_val_tab_del
			AND OWHS."BinActivat" = 'Y'
			AND OWHS.U_PMX_IMBP = 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50850;
				SET @error_message = 'Warehouse with code''' + @list_of_cols_val_tab_del + ''' cannot have BIN locations enabled ' 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			
			IF EXISTS (SELECT OWHS."WhsCode" FROM OWHS
			WHERE OWHS."WhsCode" = @list_of_cols_val_tab_del
			AND OWHS.U_PMX_IMBP = 'Y')  BEGIN
				--The warehouse needs to be managed by Produmex.
				--So now check if this is allowed
				
				IF(EXISTS (SELECT OITW."WhsCode" FROM OITW 
					LEFT JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OITW."WhsCode"
					LEFT JOIN PMX_OSEL ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
					LEFT JOIN PMX_INVT ON PMX_INVT."StorLocCode" = PMX_OSEL."Code"
					WHERE "WhsCode" = @list_of_cols_val_tab_del 
					AND "OnHand" > 0
					GROUP BY OITW."WhsCode"
					HAVING ISNULL( SUM(ABS(PMX_INVT."Quantity")), 0) = 0 )) BEGIN
					
					SET @error = 50851;
					SET @error_message = 'Warehouse with code''' + @list_of_cols_val_tab_del + ''' cannot be managed by Produmex, because there is stock in SAP, but not in PMX.' 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				
				END

			END
			
			IF EXISTS (SELECT OWHS."WhsCode" FROM OWHS
			WHERE OWHS."WhsCode" = @list_of_cols_val_tab_del
			AND OWHS.U_PMX_IMBP = 'N')  BEGIN
				--The warehouse does not need to be managed by Produmex.
				--So now check if this is allowed
				
				IF(EXISTS (SELECT TOP(1) PMX_INVT."InternalKey"  FROM PMX_INVT 
						LEFT JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode"
						LEFT JOIN PMX_OSWH ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
						WHERE "SboWhsCode" = @list_of_cols_val_tab_del )) BEGIN
					
					SET @error = 50852;
					SET @error_message = 'Warehouse with code''' + @list_of_cols_val_tab_del + ''' needs to be managed by Produmex, because there is stock in PMX.' 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			END
    
            RETURN;
		END


		--**********************************************************************************************************
		--Do checks on a barcode: ERROR 51900 - 51950
		--**********************************************************************************************************
		IF (@isBarcode=1) BEGIN
		
			DECLARE @sqlStatementBeginBarcode AS NVARCHAR(max)
			DECLARE @sqlStatementEndBarcode AS NVARCHAR(max)
			DECLARE @sqlStatementBarcode AS NVARCHAR(max)
			DECLARE @parameterListBarcode AS NVARCHAR(max)

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
		
			--Check the GTINs of the item
			SET @sqlStatementBeginBarcode = 'SELECT @error=Error, @error_message=Message FROM ' + @extraDb + '.dbo.PMX_FN_CheckGTIN(''';
			SET @sqlStatementEndBarcode = ''')';
			SET @parameterListBarcode = '@error INT output, @error_message NVARCHAR(200) output';
			
			
			--Check the GTIN of the uom barcodes
			DECLARE @barcodeForBarcode NVARCHAR(254);
			DECLARE @barcodeItemCode NVARCHAR(254);
			DECLARE barCodeCursor CURSOR LOCAL FOR SELECT "BcdCode", OBCD."ItemCode" 
														FROM OBCD 
														INNER JOIN OITM ON OITM."ItemCode" = OBCD."ItemCode"
														WHERE OBCD."BcdEntry" = @list_of_cols_val_tab_del AND LEN("BcdCode") <> 0 
														AND (OITM.U_PMX_BCTY = 'G');
			OPEN barCodeCursor;
			FETCH NEXT FROM barCodeCursor INTO @barcodeForBarcode, @barcodeItemCode;
			WHILE @@FETCH_STATUS >= 0 BEGIN
				SET @sqlStatementBarcode = @sqlStatementBeginBarcode + @barcodeForBarCode + @sqlStatementEndBarcode;
				EXECUTE sp_executesql @sqlStatementBarcode, @parameterListBarcode, @error output, @error_message output
				IF (@error <> 0) BEGIN
					SET @error_message = '"Barcode" '''+ @barcodeForBarcode +  '''  of Item ' + @barcodeItemCode + ': ' + @error_message;
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					CLOSE barCodeCursor;
					DEALLOCATE barCodeCursor

					RETURN;
				END

				FETCH NEXT FROM barCodeCursor INTO @barcodeForBarcode, @barcodeItemCode
			END
			CLOSE barCodeCursor;
			DEALLOCATE barCodeCursor
			
            RETURN;
		END

		--**********************************************************************************************************
		--Do actions on change of UomGroup
		--**********************************************************************************************************
		IF @isUomGroup = 1 BEGIN

			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN

				DELETE FROM PMX_IPTY
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				LEFT JOIN (
					UGP1
					INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				) ON UGP1."UgpEntry" = OITM."UgpEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."UgpEntry" = @list_of_cols_val_tab_del 
				AND OUOM."UomCode" IS NULL

				UPDATE PMX_IPTY SET "Quantity" = UGP1."BaseQty" / UGP1."AltQty", "UserSign" = COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 )
                    , "UpdateDateTime" = @tsNow, "Version" = "Version" + 1
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
				INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."UgpEntry" = @list_of_cols_val_tab_del 
  				  AND "Quantity" <> CAST( UGP1."BaseQty" / UGP1."AltQty" AS DECIMAL(19,6) );

				INSERT INTO PMX_IPTY ( "UserSign", "CreateDateTime", "Version", "ItemCode", "PackagingTypeCode", "Quantity", "BarcodeType", "Canceled", "HideDuringEnteringQuantity", "AskDuringReception" )
				SELECT COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 ), @tsNow, 1, OITM."ItemCode", OUOM."UomCode", UGP1."BaseQty" / UGP1."AltQty", 'G', 'N', 'N', 'N'
				FROM OITM
				INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
				INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode" AND PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
				WHERE OITM."UgpEntry" = @list_of_cols_val_tab_del 
				  AND UGP1."UomEntry" <> OITM."IUoMEntry"
				  AND PMX_IPTY."PackagingTypeCode" IS NULL;

			END

	--This code is in comment because for some reason the delete is not captured!

	--		IF (@isTransactionTypeDelete=1 OR @isTransactionTypeUpdate = 1) BEGIN
	--			--Check if the uom is used on item
	--			SET @errorList = ''
	--			SELECT @errorList = @errorList + [OITM]."ItemCode" + ' ' 
	--			FROM [OITM]
	--			WHERE "BuyUnitMsr" = @list_of_cols_val_tab_del OR
	--					"SalUnitMsr" = @list_of_cols_val_tab_del OR
	--					"InvntryUom" = @list_of_cols_val_tab_del OR
	--					U_PMX_UOM2 = @list_of_cols_val_tab_del
	--			IF LEN(@errorList) <> 0 BEGIN
	--				SET @error = 80002
	--				SET @error_message = 'PmxUom with code ''' + @list_of_cols_val_tab_del + ''' can not be deleted or updated because it is used in items with code ''' + @errorList + '''.'
	--				SELECT @error, @error_message
	--				EXEC xp_logevent @error, @error_message, ERROR
	--				RETURN
	--			END 
	--		END

            RETURN;
		END


		--**********************************************************************************************************
		--Do checks on an PmxPriority
		--**********************************************************************************************************
		IF (@isPriority=1) BEGIN
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN
                --Check if the Msequence is not null
				SET @errorList = ''
				SELECT @errorList = @errorList + [@PMX_PRIO]."Code" + ' ' 
				FROM "@PMX_PRIO"
				WHERE "@PMX_PRIO"."Code" = @list_of_cols_val_tab_del AND "@PMX_PRIO".U_PMX_SEQU IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80101;
					SET @error_message = 'PmxPriority with code ''' + @list_of_cols_val_tab_del + ''' can not be added or updated because the sequence is not filled in.';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

				--Check if the Sequence is unique
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PRIO"."Code" + ' ' 
				FROM "@PMX_PRIO"
				WHERE "@PMX_PRIO"."Code" <> @list_of_cols_val_tab_del 
				AND "@PMX_PRIO".U_PMX_SEQU = (SELECT TOP(1) "@PMX_PRIO".U_PMX_SEQU FROM "@PMX_PRIO" WHERE "@PMX_PRIO"."Code" = @list_of_cols_val_tab_del);
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80102;
					SET @error_message = 'PmxPriority with code ''' + @list_of_cols_val_tab_del + ''' can not be added or updated because the sequence is not unique';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

			END

            RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an @isFreightChargeDefinition
		--**********************************************************************************************************
		IF (@isFreightChargeDefinition = 1) BEGIN
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN
				--Check if shipping type exists in table
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_FCDE".U_PMX_SHTY + ' ' 
				FROM "@PMX_FCDE" LEFT JOIN "OSHP"
				ON OSHP."TrnspCode" = "@PMX_FCDE".U_PMX_SHTY
				WHERE "@PMX_FCDE"."Code" = @list_of_cols_val_tab_del AND "OSHP"."TrnspCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80201;
					SET @error_message = 'Shipping type with code ''' + @errorList + ''' does not exist.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

				--Check if freight exists in table
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_FCDE".U_PMX_FCDE + ' ' 
				FROM "@PMX_FCDE" LEFT JOIN "OEXD"
				ON OEXD."ExpnsCode" = "@PMX_FCDE".U_PMX_FCDE
				WHERE "@PMX_FCDE"."Code" = @list_of_cols_val_tab_del AND "OEXD"."ExpnsCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 80202;
					SET @error_message = 'Freight charge with code ''' + @errorList + ''' does not exist.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

			END

            RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an ItemPackagingType
		--**********************************************************************************************************
		IF (@isItemPackagingType = 1) BEGIN
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeUpdate = 1) BEGIN

				SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

				--Check if packaging type exists in table
				SET @errorList = ''
				SELECT @errorList = @errorList + PMX_IPTY."PackagingTypeCode" + ' ' 
				FROM PMX_IPTY 
                LEFT JOIN OUOM
				ON PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
				WHERE PMX_IPTY."InternalKey" = @keyAsInteger AND OUOM."UomCode" IS NULL;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81001;
					SET @error_message = 'ItemPackagingType with code ''' + @errorList + ''' does not exist.';
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 

			END

            RETURN;
		END

		--**********************************************************************************************************
		--Do checks on an PickListType
		--**********************************************************************************************************
		IF (@isPickListType = 1) BEGIN
			IF (@isTransactionTypeAdd = 1 OR @isTransactionTypeUpdate = 1) BEGIN

				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."U_PMX_ItemPickPLTY" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  "@PMX_PLTY" PLTY_ITEM
				ON PLTY_ITEM."Code" = "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				WHERE PLTY_ITEM.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81051;
					SET @error_message = 'Split up for for items ''' + @errorList + ''' is not of the same type as for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."U_PMX_FullPalletPLTY" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81052;
					SET @error_message = 'Split up type for full ''' + @errorList + ''' is not of the same type as for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if proposals exist with production that do not correspond to pick object type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  PMX_PLPH
				ON PMX_PLPH."PickListType" = "@PMX_PLTY"."Code"
				WHERE PMX_PLPH."PickObjType" = 'Production' 
				AND "@PMX_PLTY".U_PMX_PROD = 'N'
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del 
				AND PMX_PLPH."DocStatus" = 'O';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81053;
					SET @error_message = 'Proposals of pick object type ''production'' exist for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if proposals exist without production that do not correspond to pick object type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN  PMX_PLPH
				ON PMX_PLPH."PickListType" = "@PMX_PLTY"."Code"
				WHERE PMX_PLPH."PickObjType" <> 'Production' 
				AND "@PMX_PLTY".U_PMX_PROD = 'Y'
				AND "@PMX_PLTY"."Code" = @list_of_cols_val_tab_del 
				AND PMX_PLPH."DocStatus" = 'O';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81054;
					SET @error_message = 'Proposals of pick object type different from ''production'' exist for pick list type with code ''' + @list_of_cols_val_tab_del + ''''
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."U_PMX_FullPalletPLTY" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81055;
					SET @error_message = 'The pick list type is used as split for full pallets, but the type does not correspond  on ''' + @errorList 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
				--Check if split up pick list types are for the same type
				SET @errorList = ''
				SELECT @errorList = @errorList + "@PMX_PLTY"."Code" + ' ' 
				FROM "@PMX_PLTY" 
				INNER JOIN "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."U_PMX_ItemPickPLTY" = @list_of_cols_val_tab_del;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 81056;
					SET @error_message = 'The pick list type is used as split for full pallets, but the type does not correspond  on ''' + @errorList 
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END 
				
			END

            RETURN;
		END


		--*******************************************************************
		--*																	*
		--*Code below is for maintaining inventory between SBO and Produmex	*
		--*Check on documents												*
		--*																	*
		--*******************************************************************

		--Variables used for the document and document lines
		DECLARE @lineNums NVARCHAR(4000);
		DECLARE @objType NVARCHAR(20);
		DECLARE @docEntry INT;
		DECLARE @docNum INT;
		DECLARE @docDate DATETIME;
		DECLARE @cardCode NVARCHAR(15);
		DECLARE @cardName NVARCHAR(100);
		DECLARE @userSign INT;
		DECLARE @createDate DATETIME;
		DECLARE @createTime INT;
		DECLARE @stockDirection INT;
		DECLARE @lineNum INT;
		DECLARE @baseType NVARCHAR(20);
		DECLARE @baseEntry INT;
		DECLARE @BaseLine INT;
		DECLARE @itemCode NVARCHAR(20);
		DECLARE @description NVARCHAR(100);
		DECLARE @quantity numeric(19, 6);
		DECLARE @inventoryQuantity numeric(19, 6);
		DECLARE @uom NVARCHAR(20);
		DECLARE @inventoryUom NVARCHAR(20);
		DECLARE @quantityPerUom numeric(19, 6);
		DECLARE @whsCode nvarchar(8);
		DECLARE @batchNumbers1 NVARCHAR(MAX);
		DECLARE @batchNumbers2 NVARCHAR(MAX);
		DECLARE @bestBeforeDates NVARCHAR(MAX);
		DECLARE @quantities NVARCHAR(MAX);
		DECLARE @storLocs NVARCHAR(MAX);
		DECLARE @luids NVARCHAR(MAX);
		DECLARE @ssccs NVARCHAR(MAX);
		DECLARE @qualityStatuses NVARCHAR(MAX);
		DECLARE @isLogisticCarrier NVARCHAR(1);
		DECLARE @reasonCode NVARCHAR(MAX);
		DECLARE @quantityUom2 numeric(19, 6);
		DECLARE @uom2 NVARCHAR(100);
		DECLARE @quantitiesUom2 NVARCHAR(MAX);
		DECLARE @dropShip NVARCHAR(1);
		--DECLARE	@destinationStorLocCode NVARCHAR(50);
		DECLARE	@destinationStorLocCodes NVARCHAR(MAX);
		DECLARE	@destinationWhsCode NVARCHAR(8);
		DECLARE	@destinationQualityStatusCode NVARCHAR(8);
		DECLARE	@destinationQualityStatusCodes NVARCHAR(MAX);
		--DECLARE	@destinationLUID INT;
		DECLARE @destinationLuids NVARCHAR(MAX);
		DECLARE @trackLocationForSerial NVARCHAR(1);
		DECLARE @isSAPSerial NVARCHAR(1);
		DECLARE @isCatchWeightForDocument NVARCHAR(1);
		DECLARE @masterLuids NVARCHAR(MAX);

		--Declare memory tables for changing the raw document lines (with piping) to normal document lines (without piping)
		DECLARE @tblBatchNumbers1 AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)
		DECLARE @tblBatchNumbers2 AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)
		DECLARE @tblBestBeforeDates AS TABLE (listpos INT, nstr DateTime)
		DECLARE @tblQuantities AS TABLE (listpos INT, nstr numeric(19,6))
		DECLARE @tblQuantitiesUom2 AS TABLE (listpos INT, nstr numeric(19,6))
		DECLARE @tblStorLocs AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)
		DECLARE @tblLuids AS TABLE (listpos INT, nstr INT)
		DECLARE @tblSSCCs AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)
		DECLARE @tblQualityStatuses AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)

		DECLARE @tblDestStorLocs AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)
		DECLARE @tblDestLuids AS TABLE (listpos INT, nstr INT)
		DECLARE @tblDestQualityStatuses AS TABLE (listpos INT, nstr nvarchar(2000) COLLATE database_default)
		DECLARE @tblMasterLuids AS TABLE (listpos INT, nstr INT)

		--Declare memory table to store the luid's created within 1 run of the SP
		DECLARE @tblCurrentLuids AS TABLE (luid INT, sscc nvarchar(18))

		--Declare memory table for storing the details of the inventory
		DECLARE @inventoryDetailTable TABLE (
			"TransType" NVARCHAR(20) NULL ,
			"DocEntry" INT NULL ,
			"DocNum" INT NULL ,
			"DocDate" DATETIME  NULL ,
			"DocLineNum" INT NULL ,
			"BaseType" NVARCHAR(20) NULL ,
			"BaseEntry" INT NULL ,
			"BaseLine" INT NULL ,
			"CardCode" NVARCHAR (15)  NULL ,
			"CardName" NVARCHAR (100)  NULL ,
			"ItemCode" NVARCHAR (20) NOT NULL ,
			"ItemName" NVARCHAR (100)  NULL ,
			"Quantity" NUMERIC(19, 6) NOT NULL ,
			"StorLocCode" NVARCHAR (50) NOT NULL ,
			"ItemTransactionalInfoKey" INT  NULL ,
			"SerialNum" NVARCHAR (17)  NULL ,
			"LogUnitIdentKey" INT NULL ,
			"CreatedBy" INT NULL ,
			"CreateDate" DATETIME NULL ,
			"CreateTime" INT NULL ,
			"SSCC" NVARCHAR(18) NULL ,
			"QualityStatusCode" NVARCHAR(8) NULL,
			"Uom" NVARCHAR(20) NULL,
			"InventoryUom" NVARCHAR(20) NULL,
			"QuantityPerUom" NUMERIC(19,6) NULL,
			"Uom2" NVARCHAR(20) NULL,
			"QuantityUom2" NUMERIC(19,6) NULL,
			"InventoryQuantity" NUMERIC(19, 6) NOT NULL

		);

		--Declare memory table for storing the document lines in raw format. We have to store them in a nicer format
		DECLARE @documentLineRawTable TABLE (
			"DocEntry" INT NOT NULL ,
			"ObjType" NVARCHAR(20) NOT NULL,

			"DocNum" INT NULL ,
			"DocDate" DATETIME  NULL ,
			"CardCode" NVARCHAR (15)  NULL ,
			"CardName" NVARCHAR (100)  NULL ,
			"CreatedBy" INT NULL ,
			"CreateDate" DATETIME NULL ,
			"CreateTime" INT NULL ,

			"LineNum" INT NOT NULL ,
			"BaseType" NVARCHAR(20) NULL,
			"BaseEntry" INT NULL,
			"BaseLine" INT NULL,
			"ItemCode" NVARCHAR(20) NULL ,
			"Dscription" NVARCHAR(100) NULL,
			"Quantity" NUMERIC(19,6) NULL,
			"InventoryQuantity" NUMERIC(19,6) NULL,
			"Uom" NVARCHAR(20) NULL,
			"InventoryUom" NVARCHAR(20) NULL,
			"QuantityPerUom" NUMERIC(19,6) NULL,
			"WhsCode" NVARCHAR(8) NULL,
			"U_PMX_BATC" NVARCHAR(max) NULL,
			"U_PMX_BAT2" NVARCHAR(max) NULL,
			"U_PMX_BBDT" NVARCHAR(max) NULL,
			"U_PMX_QUTY" NVARCHAR(max) NULL,
			"U_PMX_LOCO" NVARCHAR(max) NULL,
			"U_PMX_LUID" NVARCHAR(max) NULL,
			"U_PMX_SSCC" NVARCHAR(max) NULL,
			"U_PMX_QYSC" NVARCHAR(max) NULL,
			"U_PMX_ISLC" NVARCHAR(max) NULL,
			"U_PMX_RSNC" NVARCHAR(max) NULL,
			"U_PMX_PQY2" NVARCHAR(max) NULL,
			"Uom2" NVARCHAR(20) NULL,
			"QuantityUom2" NUMERIC(19,6) NULL,
			"DropShip" NVARCHAR(1) NULL,
			"DestinationStorLocCode" NVARCHAR(max) NULL,
			"DestinationWhsCode" NVARCHAR(8) NULL,
			"DestinationQualityStatusCode" NVARCHAR(max) NULL,
			"DestinationLUID" NVARCHAR(max) NULL,
			"TrackLocationForSerial" NVARCHAR (1) NULL,
			"IsSAPSerial" NVARCHAR (1) NULL,
			"IsCatchWeight" NVARCHAR (1) NULL,
			"MasterLUID" NVARCHAR(max) NULL
		);

		--Declare memory table for storing the document lines of a purchase delivery note, sales delivery note, ...
		DECLARE @documentLineTable TABLE (
			"DocEntry" INT NOT NULL ,
			"ObjType" NVARCHAR(20) NOT NULL,

			"DocNum" INT NULL ,
			"DocDate" DATETIME  NULL ,
			"CardCode" NVARCHAR (15)  NULL ,
			"CardName" NVARCHAR (100)  NULL ,
			"CreatedBy" INT NULL ,
			"CreateDate" DATETIME NULL ,
			"CreateTime" INT NULL ,

			"LineNum" INT NOT NULL ,
			"BaseType" NVARCHAR(20) NULL,
			"BaseEntry" INT NULL,
			"BaseLine" INT NULL,
			"ItemCode" NVARCHAR(20) NULL ,
			"Dscription" NVARCHAR(100) NULL,
			"Quantity" NUMERIC(19,9) NULL,
			"InventoryQuantity" NUMERIC(19,9) NULL,
			"Uom" NVARCHAR(20) NULL,
			"InventoryUom" NVARCHAR(20) NULL,
			"QuantityPerUom" NUMERIC(19,6) NULL,
			"WhsCode" NVARCHAR(8) NULL,
			"StorLocCode" NVARCHAR(50) NULL,
			"ItemTransactionalInfoKey" INT  NULL ,
	--		"BatchNumber" NVARCHAR(36) NULL,
	--		"BestBeforeDate" DATETIME NULL,
			"LogUnitIdentKey" INT NULL,

			"SSCC" NVARCHAR (18) NULL ,

			"QualityStatusCode" NVARCHAR(8) NULL,
			"IsLogisticCarrier" NVARCHAR(1) NULL,
			"ReasonCode" NVARCHAR(8) NULL,
			"Uom2" NVARCHAR(20) NULL,
			"QuantityUom2" NUMERIC(19,6) NULL,
			"DestinationStorLocCode" NVARCHAR(50) NULL,
			"DestinationWhsCode" NVARCHAR(8) NULL,
			"DestinationQualityStatusCode" NVARCHAR(8) NULL,
			"DestinationLUID" INT NULL,
			"MasterLUID" INT NULL
		);

		--**********************************************************************************************************
		--If the object that is changed is of the type document then
		--copy the document lines in a memory table
		--This memory table is used because it is easier and faster to do all the checks on this memory table
		--instead of getting the lines again from the database.
		--Checks are also done on the document lines
		--If a document line is indicated that it is a logistic carrier then this logistic carrier is put on the
		--logistic unit.
		--**********************************************************************************************************
		IF (@isDocument=1) AND (@isTransactionTypeAdd=1 OR (@isTransactionTypeCancel=1 AND @isStockTransfer=1)) BEGIN
			DECLARE @docLineTable VARCHAR(20);

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--**********************************************************************************************************
			--Copy document lines in the Raw memory table
			--**********************************************************************************************************
			--Copy the good receipt lines (purchase delivery note lines) to the memory table.
			IF (@isPurchaseDeliveryDocument = 1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OPDN."DocEntry", OPDN."ObjType", OPDN."DocNum", OPDN."DocDate",OPDN."CardCode", OPDN."CardName",OPDN."UserSign", OPDN."CreateDate", OPDN."DocTime", PDN1."LineNum", PDN1."BaseType", PDN1."BaseEntry", PDN1."BaseLine", PDN1."ItemCode", PDN1."Dscription", PDN1."Quantity", PDN1."InvQty", ISNULL( PDN1."UomCode", PDN1."unitMsr"), CASE WHEN NULLIF(PDN1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( PDN1."UomCode2", PDN1."unitMsr2") END, PDN1."NumPerMsr", PDN1."WhsCode", 
						  PDN1.U_PMX_BATC, PDN1.U_PMX_BAT2, PDN1.U_PMX_BBDT, PDN1.U_PMX_QUAN, PDN1.U_PMX_LOCO, PDN1.U_PMX_LUID, PDN1.U_PMX_SSCC, PDN1.U_PMX_QYSC,
						  PDN1.U_PMX_ISLC, PDN1.U_PMX_RSNC, PDN1.U_PMX_PQY2, PDN1.U_PMX_UOM2, PDN1.U_PMX_QTY2, PDN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  PDN1 INNER JOIN OPDN ON PDN1."DocEntry" = OPDN."DocEntry" 
								INNER JOIN OITM ON OITM."ItemCode" = PDN1."ItemCode"
								INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = PDN1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE OPDN."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND PDN1."DropShip" = 'N';
					
				SET @docLineTable = 'PDN1';
				SET @stockDirection = +1;
				
				IF EXISTS (SELECT "DocEntry" FROM OPDN WHERE OPDN."DocEntry" = @keyAsInteger AND OPDN."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = -1;
				END
				

			END

			--Copy the delivery lines (sales delivery note lines) to the memory table.
			IF (@isSalesDeliveryDocument = 1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT ODLN."DocEntry", ODLN."ObjType", ODLN."DocNum", ODLN."DocDate",ODLN."CardCode", ODLN."CardName",ODLN."UserSign", ODLN."CreateDate", ODLN."DocTime", DLN1."LineNum", DLN1."BaseType", DLN1."BaseEntry", DLN1."BaseLine", DLN1."ItemCode", DLN1."Dscription", DLN1."Quantity", DLN1."InvQty", ISNULL( DLN1."UomCode", DLN1."unitMsr"), CASE WHEN NULLIF(DLN1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( DLN1."UomCode2", DLN1."unitMsr2") END, DLN1."NumPerMsr", DLN1."WhsCode", 
						  DLN1.U_PMX_BATC, DLN1.U_PMX_BAT2, DLN1.U_PMX_BBDT, DLN1.U_PMX_QUAN, DLN1.U_PMX_LOCO, DLN1.U_PMX_LUID, DLN1.U_PMX_SSCC, DLN1.U_PMX_QYSC,
						  DLN1.U_PMX_ISLC, DLN1.U_PMX_RSNC, DLN1.U_PMX_PQY2, DLN1.U_PMX_UOM2, DLN1.U_PMX_QTY2, DLN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  DLN1 INNER JOIN ODLN ON DLN1."DocEntry" = ODLN."DocEntry" 
								INNER JOIN OITM ON OITM."ItemCode" = DLN1."ItemCode"
								INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = DLN1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE ODLN."DocEntry" = @keyAsInteger AND DLN1."Quantity" > 0
					AND OITM."InvntItem" = 'Y'
					AND DLN1."DropShip" = 'N';
					

				SET @stockDirection = -1;
				
				IF EXISTS (SELECT "DocEntry" FROM ODLN WHERE ODLN."DocEntry" = @keyAsInteger AND ODLN."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = +1;
				END
			
				
			END


			--Copy the inventory entry lines to the memory table.
			IF (@isInventoryEntry = 1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OIGN."DocEntry", OIGN."ObjType", OIGN."DocNum", OIGN."DocDate",OIGN."CardCode", OIGN."CardName",OIGN."UserSign", OIGN."CreateDate", OIGN."DocTime", IGN1."LineNum", IGN1."BaseType", IGN1."BaseEntry", IGN1."BaseLine", IGN1."ItemCode", IGN1."Dscription", IGN1."Quantity", IGN1."InvQty", ISNULL( OUOM."UomCode", OITM."InvntryUom"), ISNULL( OUOM."UomCode", OITM."InvntryUom") AS "unitMsr", 1 AS "NumPerMsr", IGN1."WhsCode", 
						  IGN1.U_PMX_BATC, IGN1.U_PMX_BAT2, IGN1.U_PMX_BBDT, IGN1.U_PMX_QUAN, IGN1.U_PMX_LOCO, IGN1.U_PMX_LUID, IGN1.U_PMX_SSCC, IGN1.U_PMX_QYSC,
						  IGN1.U_PMX_ISLC, IGN1.U_PMX_RSNC, IGN1.U_PMX_PQY2, IGN1.U_PMX_UOM2, IGN1.U_PMX_QTY2, IGN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  IGN1 INNER JOIN OIGN ON IGN1."DocEntry" = OIGN."DocEntry"
						  INNER JOIN OITM ON IGN1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = IGN1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE OIGN."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND IGN1."DropShip" = 'N';
	
				SET @docLineTable = 'IGN1';
				SET @stockDirection = +1;
				
				IF EXISTS (SELECT "DocEntry" FROM OIGN WHERE OIGN."DocEntry" = @keyAsInteger AND OIGN."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = -1;
				END
				
			END

			--Copy the inventory exit lines to the memory table.
			IF (@isInventoryExit=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OIGE."DocEntry", OIGE."ObjType", OIGE."DocNum", OIGE."DocDate", OIGE."CardCode", OIGE."CardName", OIGE."UserSign", OIGE."CreateDate", OIGE."DocTime", IGE1."LineNum", IGE1."BaseType", IGE1."BaseEntry", IGE1."BaseLine", IGE1."ItemCode", IGE1."Dscription", IGE1."Quantity", IGE1."InvQty", ISNULL( OUOM."UomCode", OITM."InvntryUom"), ISNULL( OUOM."UomCode", OITM."InvntryUom") AS "unitMsr", 1 AS "NumPerMsr", IGE1."WhsCode", 
						  IGE1.U_PMX_BATC, IGE1.U_PMX_BAT2, IGE1.U_PMX_BBDT, IGE1.U_PMX_QUAN, IGE1.U_PMX_LOCO, IGE1.U_PMX_LUID, IGE1.U_PMX_SSCC, IGE1.U_PMX_QYSC,
						  IGE1.U_PMX_ISLC, IGE1.U_PMX_RSNC, IGE1.U_PMX_PQY2, IGE1.U_PMX_UOM2,IGE1.U_PMX_QTY2, IGE1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  IGE1 INNER JOIN OIGE ON IGE1."DocEntry" = OIGE."DocEntry"
						  INNER JOIN OITM ON IGE1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = IGE1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE OIGE."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND IGE1."DropShip" = 'N';
	
				SET @stockDirection = -1;
				
				IF EXISTS (SELECT "DocEntry" FROM OIGE WHERE OIGE."DocEntry" = @keyAsInteger AND OIGE."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = +1;
				END

			END

			--Copy the sales return lines to the memory table
			IF (@isSalesReturnDocument=1) BEGIN
				INSERT INTO @documentLineRawTable
					SELECT ORDN."DocEntry", ORDN."ObjType", ORDN."DocNum", ORDN."DocDate", ORDN."CardCode", ORDN."CardName", ORDN."UserSign", ORDN."CreateDate", ORDN."DocTime", RDN1."LineNum", RDN1."BaseType", RDN1."BaseEntry", RDN1."BaseLine", RDN1."ItemCode", RDN1."Dscription", RDN1."Quantity", RDN1."InvQty", ISNULL( RDN1."UomCode", RDN1."unitMsr"), CASE WHEN NULLIF(RDN1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( RDN1."UomCode2", RDN1."unitMsr2") END, RDN1."NumPerMsr", RDN1."WhsCode", 
						  RDN1.U_PMX_BATC, RDN1.U_PMX_BAT2, RDN1.U_PMX_BBDT, RDN1.U_PMX_QUAN, RDN1.U_PMX_LOCO, RDN1.U_PMX_LUID, RDN1.U_PMX_SSCC, RDN1.U_PMX_QYSC,
						  RDN1.U_PMX_ISLC, RDN1.U_PMX_RSNC, RDN1.U_PMX_PQY2, RDN1.U_PMX_UOM2, RDN1.U_PMX_QTY2, RDN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RDN1 INNER JOIN ORDN ON RDN1."DocEntry" = ORDN."DocEntry"
						INNER JOIN OITM ON RDN1."ItemCode" = OITM."ItemCode"
						INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = RDN1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE ORDN."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND RDN1."DropShip" = 'N';
		
				SET @docLineTable = 'RDN1';
				SET @stockDirection = +1;
				
				IF EXISTS (SELECT "DocEntry" FROM ORDN WHERE ORDN."DocEntry" = @keyAsInteger AND ORDN."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = -1;
				END

			END

			--Copy the purchase return lines to the memory table
			IF (@isPurchaseReturnDocument=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT ORPD."DocEntry", ORPD."ObjType", ORPD."DocNum", ORPD."DocDate", ORPD."CardCode", ORPD."CardName", ORPD."UserSign", ORPD."CreateDate", ORPD."DocTime", RPD1."LineNum", RPD1."BaseType", RPD1."BaseEntry", RPD1."BaseLine", RPD1."ItemCode", RPD1."Dscription", RPD1."Quantity", RPD1."InvQty", ISNULL( RPD1."UomCode", RPD1."unitMsr"), CASE WHEN NULLIF(RPD1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( RPD1."UomCode2", RPD1."unitMsr2") END, RPD1."NumPerMsr", RPD1."WhsCode", 
						  RPD1.U_PMX_BATC, RPD1.U_PMX_BAT2, RPD1.U_PMX_BBDT, RPD1.U_PMX_QUAN, RPD1.U_PMX_LOCO, RPD1.U_PMX_LUID, RPD1.U_PMX_SSCC, RPD1.U_PMX_QYSC,
						  RPD1.U_PMX_ISLC, RPD1.U_PMX_RSNC, RPD1.U_PMX_PQY2, RPD1.U_PMX_UOM2, RPD1.U_PMX_QTY2, RPD1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RPD1 INNER JOIN ORPD ON RPD1."DocEntry" = ORPD."DocEntry"
						  INNER JOIN OITM ON RPD1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = RPD1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE ORPD."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND RPD1."DropShip" = 'N';

				SET @stockDirection = -1;
				
				
				IF EXISTS (SELECT "DocEntry" FROM ORPD WHERE ORPD."DocEntry" = @keyAsInteger AND ORPD."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = +1;
				END

			END

			--Copy the purchase invoice lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a good receipt
			IF (@isPurchaseInvoice=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OPCH."DocEntry", OPCH."ObjType", OPCH."DocNum", OPCH."DocDate", OPCH."CardCode", OPCH."CardName", OPCH."UserSign", OPCH."CreateDate", OPCH."DocTime", PCH1."LineNum", PCH1."BaseType", PCH1."BaseEntry", PCH1."BaseLine", PCH1."ItemCode", PCH1."Dscription", PCH1."Quantity", PCH1."InvQty", ISNULL( PCH1."UomCode", PCH1."unitMsr"), CASE WHEN NULLIF(PCH1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( PCH1."UomCode2", PCH1."unitMsr2") END, PCH1."NumPerMsr", PCH1."WhsCode", 
						  PCH1.U_PMX_BATC, PCH1.U_PMX_BAT2, PCH1.U_PMX_BBDT, PCH1.U_PMX_QUAN, PCH1.U_PMX_LOCO, PCH1.U_PMX_LUID, PCH1.U_PMX_SSCC, PCH1.U_PMX_QYSC,
						  PCH1.U_PMX_ISLC, PCH1.U_PMX_RSNC, PCH1.U_PMX_PQY2, PCH1.U_PMX_UOM2, PCH1.U_PMX_QTY2, PCH1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  PCH1 
						INNER JOIN OPCH ON PCH1."DocEntry" = OPCH."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = @keyAsInteger AND "TransType"=18 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON OPCH."DocEntry" = OINM."CreatedBy" AND PCH1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = OPCH."ObjType"
						  INNER JOIN OITM ON PCH1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = PCH1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE OPCH."DocEntry" = @keyAsInteger AND OINM."InQty" <> OINM."OutQty"
					AND OITM."InvntItem" = 'Y'
					AND PCH1."DropShip" = 'N';

				SET @stockDirection = +1;
				
				
				IF EXISTS (SELECT "DocEntry" FROM OPCH WHERE OPCH."DocEntry" = @keyAsInteger AND OPCH."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = -1;
				END

			END

			--Copy the purchase credit note lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a goods return
			IF (@isPurchaseCreditNote=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT ORPC."DocEntry", ORPC."ObjType", ORPC."DocNum", ORPC."DocDate", ORPC."CardCode", ORPC."CardName", ORPC."UserSign", ORPC."CreateDate", ORPC."DocTime", RPC1."LineNum", RPC1."BaseType", RPC1."BaseEntry", RPC1."BaseLine", RPC1."ItemCode", RPC1."Dscription", RPC1."Quantity", RPC1."InvQty", ISNULL( RPC1."UomCode", RPC1."unitMsr"), CASE WHEN NULLIF(RPC1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( RPC1."UomCode2", RPC1."unitMsr2") END, RPC1."NumPerMsr", RPC1."WhsCode", 
						  RPC1.U_PMX_BATC, RPC1.U_PMX_BAT2, RPC1.U_PMX_BBDT, RPC1.U_PMX_QUAN, RPC1.U_PMX_LOCO, RPC1.U_PMX_LUID, RPC1.U_PMX_SSCC, RPC1.U_PMX_QYSC,
						  RPC1.U_PMX_ISLC, RPC1.U_PMX_RSNC, RPC1.U_PMX_PQY2, RPC1.U_PMX_UOM2, RPC1.U_PMX_QTY2, RPC1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RPC1 
						INNER JOIN ORPC ON RPC1."DocEntry" = ORPC."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = @keyAsInteger AND "TransType"=19 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON ORPC."DocEntry" = OINM."CreatedBy" AND RPC1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = ORPC."ObjType"
						  INNER JOIN OITM ON RPC1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = RPC1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE ORPC."DocEntry" = @keyAsInteger AND OINM."InQty" <> OINM."OutQty"
					AND OITM."InvntItem" = 'Y'
					AND RPC1."DropShip" = 'N';

				SET @stockDirection = -1;
				
				IF EXISTS (SELECT "DocEntry" FROM ORPC WHERE ORPC."DocEntry" = @keyAsInteger AND ORPC."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = +1;
				END

			END

			--Copy the sales invoice lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a delivery
			IF (@isSalesInvoice=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OINV."DocEntry", OINV."ObjType", OINV."DocNum", OINV."DocDate", OINV."CardCode", OINV."CardName", OINV."UserSign", OINV."CreateDate", OINV."DocTime", INV1."LineNum", INV1."BaseType", INV1."BaseEntry", INV1."BaseLine", INV1."ItemCode", INV1."Dscription", INV1."Quantity", INV1."InvQty", ISNULL( INV1."UomCode", INV1."unitMsr"), CASE WHEN NULLIF(INV1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( INV1."UomCode2", INV1."unitMsr2") END, INV1."NumPerMsr", INV1."WhsCode", 
						  INV1.U_PMX_BATC, INV1.U_PMX_BAT2, INV1.U_PMX_BBDT, INV1.U_PMX_QUAN, INV1.U_PMX_LOCO, INV1.U_PMX_LUID, INV1.U_PMX_SSCC, INV1.U_PMX_QYSC,
						  INV1.U_PMX_ISLC, INV1.U_PMX_RSNC, INV1.U_PMX_PQY2, INV1.U_PMX_UOM2, INV1.U_PMX_QTY2, INV1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  INV1 
						INNER JOIN OINV ON INV1."DocEntry" = OINV."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = @keyAsInteger AND "TransType"=13 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON OINV."DocEntry" = OINM."CreatedBy" AND INV1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = OINV."ObjType"
						  INNER JOIN OITM ON INV1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = INV1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE OINV."DocEntry" = @keyAsInteger AND OINM."InQty" <> OINM."OutQty"
					AND OITM."InvntItem" = 'Y'
					AND INV1."DropShip" = 'N';

				SET @stockDirection = -1;
				
				IF EXISTS (SELECT "DocEntry" FROM OINV WHERE OINV."DocEntry" = @keyAsInteger AND OINV."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = +1;
				END
				
			END

			--Copy the sales credit note lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a goods return
			IF (@isSalesCreditNote=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT ORIN."DocEntry", ORIN."ObjType", ORIN."DocNum", ORIN."DocDate", ORIN."CardCode", ORIN."CardName", ORIN."UserSign", ORIN."CreateDate", ORIN."DocTime", RIN1."LineNum", RIN1."BaseType", RIN1."BaseEntry", RIN1."BaseLine", RIN1."ItemCode", RIN1."Dscription", RIN1."Quantity", RIN1."InvQty", ISNULL( RIN1."UomCode", RIN1."unitMsr"), CASE WHEN NULLIF(RIN1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( RIN1."UomCode2", RIN1."unitMsr2") END, RIN1."NumPerMsr", RIN1."WhsCode", 
						  RIN1.U_PMX_BATC, RIN1.U_PMX_BAT2, RIN1.U_PMX_BBDT, RIN1.U_PMX_QUAN, RIN1.U_PMX_LOCO, RIN1.U_PMX_LUID, RIN1.U_PMX_SSCC, RIN1.U_PMX_QYSC,
						  RIN1.U_PMX_ISLC, RIN1.U_PMX_RSNC, RIN1.U_PMX_PQY2, RIN1.U_PMX_UOM2, RIN1.U_PMX_QTY2, RIN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RIN1 
						INNER JOIN ORIN ON RIN1."DocEntry" = ORIN."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = @keyAsInteger AND "TransType"=14 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON ORIN."DocEntry" = OINM."CreatedBy" AND RIN1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = ORIN."ObjType"
						  INNER JOIN OITM ON RIN1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = RIN1."WhsCode"
								AND SrcWHS.U_PMX_IMBP = 'Y'
					WHERE ORIN."DocEntry" = @keyAsInteger AND OINM."InQty" <> OINM."OutQty"
					AND OITM."InvntItem" = 'Y'
					AND RIN1."DropShip" = 'N';

				SET @stockDirection = +1;
				
				IF EXISTS (SELECT "DocEntry" FROM ORIN WHERE ORIN."DocEntry" = @keyAsInteger AND ORIN."Canceled" = 'C') BEGIN
					--Cancellation of document
					SET @stockDirection = -1;
				END

			END

			--Copy the stock transfer lines 
			IF (@isStockTransfer=1 AND @isTransactionTypeCancel=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OWTR."DocEntry", OWTR."ObjType", OWTR."DocNum", OWTR."DocDate", OWTR."CardCode", OWTR."CardName", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", WTR1."LineNum", ISNULL( WTR1."BaseType", WTR1.U_PMX_BATY), ISNULL( WTR1."BaseEntry", WTR1.U_PMX_BAEN), ISNULL( WTR1."BaseLine", WTR1.U_PMX_BALI), WTR1."ItemCode", WTR1."Dscription", WTR1."Quantity", WTR1."InvQty", ISNULL( WTR1."UomCode", WTR1."unitMsr"), CASE WHEN NULLIF(WTR1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( WTR1."UomCode2", WTR1."unitMsr2") END, WTR1."NumPerMsr", WTR1."WhsCode", 
						  WTR1.U_PMX_BATC, WTR1.U_PMX_BAT2, WTR1.U_PMX_BBDT, WTR1.U_PMX_QUAN, WTR1.U_PMX_LOCO, WTR1.U_PMX_LUID, NULL, WTR1.U_PMX_QYSC,
						  WTR1.U_PMX_ISLC, WTR1.U_PMX_RSNC, WTR1.U_PMX_PQY2, WTR1.U_PMX_UOM2, WTR1.U_PMX_QTY2, WTR1."DropShip", WTR1.U_PMX_SLOC, WTR1."FromWhsCod", WTR1.U_PMX_SQSC, WTR1.U_PMX_SLUI,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, REPLACE(CAST(WTR1.U_PMX_MLUI AS NVARCHAR),'0123456789','')
					FROM  WTR1 
						INNER JOIN OWTR ON WTR1."DocEntry" = OWTR."DocEntry"
						  INNER JOIN OITM ON WTR1."ItemCode" = OITM."ItemCode"
						INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = WTR1."WhsCode"
						INNER JOIN OWHS DstWHS ON DstWHS."WhsCode" = WTR1."FromWhsCod"

					WHERE OWTR."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND (SrcWHS.U_PMX_IMBP = 'Y' OR DstWHS.U_PMX_IMBP = 'Y')
					AND WTR1."DropShip" = 'N';

				SET @stockDirection = +1;
			
			END

			--Copy the stock transfer lines 
			IF (@isStockTransfer=1 AND @isTransactionTypeAdd=1) BEGIN

				INSERT INTO @documentLineRawTable
					SELECT OWTR."DocEntry", OWTR."ObjType", OWTR."DocNum", OWTR."DocDate", OWTR."CardCode", OWTR."CardName", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", WTR1."LineNum", ISNULL( WTR1."BaseType", WTR1.U_PMX_BATY), ISNULL( WTR1."BaseEntry", WTR1.U_PMX_BAEN), ISNULL( WTR1."BaseLine", WTR1.U_PMX_BALI), WTR1."ItemCode", WTR1."Dscription", WTR1."Quantity", WTR1."InvQty", ISNULL( WTR1."UomCode", WTR1."unitMsr"), CASE WHEN NULLIF(WTR1.UomEntry2, -1) IS NULL THEN OITM."InvntryUom" ELSE ISNULL( WTR1."UomCode2", WTR1."unitMsr2") END, WTR1."NumPerMsr", WTR1."FromWhsCod", 
						  WTR1.U_PMX_BATC, WTR1.U_PMX_BAT2, WTR1.U_PMX_BBDT, WTR1.U_PMX_QUAN, WTR1.U_PMX_SLOC, WTR1.U_PMX_SLUI, NULL, WTR1.U_PMX_SQSC,
						  WTR1.U_PMX_ISLC, WTR1.U_PMX_RSNC, WTR1.U_PMX_PQY2, WTR1.U_PMX_UOM2, WTR1.U_PMX_QTY2, WTR1."DropShip", WTR1.U_PMX_LOCO, WTR1."WhsCode", WTR1.U_PMX_QYSC, WTR1.U_PMX_LUID,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, WTR1.U_PMX_MLUI
					FROM  WTR1 
						INNER JOIN OWTR ON WTR1."DocEntry" = OWTR."DocEntry"
						  INNER JOIN OITM ON WTR1."ItemCode" = OITM."ItemCode"
						  INNER JOIN OWHS SrcWHS ON SrcWHS."WhsCode" = WTR1."WhsCode"
						INNER JOIN OWHS DstWHS ON DstWHS."WhsCode" = WTR1."FromWhsCod"
					WHERE OWTR."DocEntry" = @keyAsInteger
					AND OITM."InvntItem" = 'Y'
					AND (SrcWHS.U_PMX_IMBP = 'Y' OR DstWHS.U_PMX_IMBP = 'Y')
					AND WTR1."DropShip" = 'N';

				SET @stockDirection = -1;

			END


			
			--**********************************************************************************************************
			--Convert Raw document lines to normal document lines
			--Remove the pipes
			--**********************************************************************************************************
			BEGIN TRY
				DECLARE documentLineRawCursor CURSOR LOCAL FOR SELECT * FROM @documentLineRawTable;
				OPEN documentLineRawCursor;
				FETCH NEXT FROM documentLineRawCursor INTO @docEntry,@objType,@docNum,@docDate,@cardCode,@cardName,@userSign,@createDate,@createTime,@lineNum,@baseType,@baseEntry,@BaseLine,@itemCode,@description,@quantity,@inventoryQuantity,@uom,@inventoryUom,@quantityPerUom,@whsCode,@batchNumbers1,@batchNumbers2,@bestBeforeDates,@quantities,@storLocs,@luids,@ssccs,@qualityStatuses,@isLogisticCarrier,@reasonCode,@quantitiesUom2,@uom2,@quantityUom2, @dropShip, @destinationStorLocCodes, @destinationWhsCode, @destinationQualityStatusCodes, @destinationLuids, @trackLocationForSerial, @isSAPSerial, @isCatchWeightForDocument, @masterLuids;
				WHILE @@FETCH_STATUS >= 0 BEGIN
					--Delete all temp tables
					DELETE FROM @tblBatchNumbers1;
					DELETE FROM @tblBatchNumbers2;
					DELETE FROM @tblBestBeforeDates;
					DELETE FROM @tblQuantities;
					DELETE FROM @tblQuantitiesUom2;
					DELETE FROM @tblStorLocs;
					DELETE FROM @tblLuids;
					DELETE FROM @tblQualityStatuses;
					DELETE FROM @tblSSCCs;

					DELETE FROM @tblDestStorLocs;
					DELETE FROM @tblDestLuids;
					DELETE FROM @tblDestQualityStatuses;
					DELETE FROM @tblMasterLuids;
					
					--Cleanup quotes in batch numbers
					SET @batchNumbers1 = REPLACE( @batchNumbers1, '''', '''''')
					SET @batchNumbers2 = REPLACE( @batchNumbers2, '''', '''''')


					--Parse all the concatenated columns
					--The function that is used is stored in the extra database
					INSERT INTO @tblBatchNumbers1 EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @batchNumbers1 + ''', ''|'',1,0)')
					INSERT INTO @tblBatchNumbers2 EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @batchNumbers2 + ''', ''|'',1,0)')
					INSERT INTO @tblBestBeforeDates EXEC('SELECT listPos, CAST(nstr AS DATETIME) FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @bestBeforeDates+ ''', ''|'',1,0)')
					--INSERT INTO @tblQuantities EXEC('SELECT listPos, CAST(nstr AS numeric(19,6)) FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @quantities+ ''', ''|'',1,0)')
					INSERT INTO @tblQuantities EXEC('SELECT listPos, CASE PATINDEX(''%[eE]%'', nstr) WHEN 0 THEN CAST(nstr AS numeric(19,6)) ELSE CONVERT(numeric(19,6), cast( nstr as real)) END FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @quantities+ ''', ''|'',1,0)')
					INSERT INTO @tblQuantitiesUom2 EXEC('SELECT listPos, CAST(nstr AS numeric(19,6)) FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @quantitiesUom2+ ''', ''|'',1,0)')
					INSERT INTO @tblStorLocs EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @storLocs+ ''', ''|'',1,0)')
					INSERT INTO @tblLuids EXEC('SELECT listPos, CAST(nstr AS INT) FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @luids+ ''', ''|'',1,0)')
					INSERT INTO @tblSSCCs           EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @ssccs           + ''', ''|'',1,0)')
					INSERT INTO @tblQualityStatuses EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @qualityStatuses+ ''', ''|'',1,0)')

					INSERT INTO @tblDestStorLocs EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @destinationStorLocCodes+ ''', ''|'',1,0)')
					
					INSERT INTO @tblDestLuids EXEC('SELECT listPos, CAST(nstr AS INT) FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @destinationLuids+ ''', ''|'',1,0)')
					INSERT INTO @tblDestQualityStatuses EXEC('SELECT listPos, nstr FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @destinationQualityStatusCodes+ ''', ''|'',1,0)')
					INSERT INTO @tblMasterLuids EXEC('SELECT listPos, CAST(nstr AS INT) FROM '+@extraDb +'.dbo.PMX_FN_IterateCharListToTable('''+ @masterLuids+ ''', ''|'',1,0)')

					--Sometimes SBO set "BaseType" to -1 if there is no base document
					--We don't want this we want that the baseType is NULL then
					IF @baseType = '-1' BEGIN
						SET @baseType = NULL;
					END

               -- Auto-fill PMX BatchNumber1, "BestBeforeDate" and "Quantity"
               IF @batchNumbers1 IS NULL OR @bestBeforeDates IS NULL OR @quantities IS NULL
               BEGIN
                  DECLARE @tmpITL TABLE ( rowNum INT, BatchNum1 NVARCHAR(36), "BestBeforeDate" DATETIME, "Quantity" DECIMAL(19,6) )
                  DECLARE @tmpITLRows INT
                  
                  DELETE FROM @tmpITL

                  INSERT INTO @tmpITL
                  SELECT ROW_NUMBER() OVER ( ORDER BY OBTN."DistNumber" ), OBTN."DistNumber", OBTN."ExpDate", SUM( ITL1."Quantity" ) * @stockDirection * CASE WHEN @isStockTransfer = 1 THEN -1 ELSE +1 END
                  FROM OITL 
                  INNER JOIN (   ITL1
                     INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
                     AND ITL1."SysNumber" = OBTN."SysNumber"
                  ) ON ITL1."LogEntry" = OITL."LogEntry"
                  WHERE OITL."DocType" = @objType AND OITL."DocEntry" = @docEntry AND OITL."DocLine" = @lineNum AND OITL."StockEff" = 1  -- Affects "OnHand"
                  AND ( OITL."LocCode" = @destinationWhsCode OR @destinationWhsCode IS NULL )
                  GROUP BY OBTN."DistNumber", OBTN."ExpDate"
                  HAVING SUM( ITL1."Quantity" ) <> 0
 
                  SET @tmpITLRows = @@ROWCOUNT
                  
                  IF @tmpITLRows = 0  -- No batch numbers
                  BEGIN
					      IF @quantities IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblQuantities

						      INSERT INTO @tblQuantities ( listPos, nstr )
							   --VALUES ( 1, @quantity * @quantityPerUom )
							   VALUES ( 1, @inventoryQuantity)
							END
                  END
					   ELSE IF @tmpITLRows = 1 OR ( @tmpITLRows > 1 AND @batchNumbers1 IS NULL AND @bestBeforeDates IS NULL AND @quantities IS NULL )
					   BEGIN
					      IF @batchNumbers1 IS NULL
					      BEGIN
					         --Delete the NULL row inserted before
					         DELETE FROM @tblBatchNumbers1

						      INSERT INTO @tblBatchNumbers1 ( listPos, nstr )
                        SELECT rowNum, BatchNum1  FROM @tmpITL  ORDER BY rowNum
					      END
					      
					      IF @bestBeforeDates IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblBestBeforeDates

						      INSERT INTO @tblBestBeforeDates ( listPos, nstr )
                        SELECT rowNum, "BestBeforeDate"  FROM @tmpITL  ORDER BY rowNum
					      END

					      IF @quantities IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblQuantities

						      INSERT INTO @tblQuantities ( listPos, nstr )
                        SELECT rowNum, "Quantity"  FROM @tmpITL  ORDER BY rowNum
					      END
                  END
                  
                  IF @tmpITLRows > 1
                  BEGIN
					      IF @batchNumbers2 IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblBatchNumbers2

						      INSERT INTO @tblBatchNumbers2 ( listPos, nstr )
                        SELECT rowNum, NULL  FROM @tmpITL  ORDER BY rowNum
                     END
                     
                     IF @storLocs IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblStorLocs

						      INSERT INTO @tblStorLocs ( listPos, nstr )
                        SELECT rowNum, NULL  FROM @tmpITL  ORDER BY rowNum
                     END
                     
                     IF @luids IS NULL AND @ssccs IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblLuids

						      INSERT INTO @tblLuids ( listPos, nstr )
                        SELECT rowNum, NULL  FROM @tmpITL  ORDER BY rowNum

						      --Delete the NULL row inserted before
						      DELETE FROM @tblSSCCs

						      INSERT INTO @tblSSCCs ( listPos, nstr )
                        SELECT rowNum, NULL  FROM @tmpITL  ORDER BY rowNum
                     END

                     IF @qualityStatuses IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblQualityStatuses

						      INSERT INTO @tblQualityStatuses ( listPos, nstr )
                        SELECT rowNum, NULL  FROM @tmpITL  ORDER BY rowNum
                     END
                     
                     IF @quantitiesUom2 IS NULL
					      BEGIN
						      --Delete the NULL row inserted before
						      DELETE FROM @tblQuantitiesUom2

						      INSERT INTO @tblQuantitiesUom2 ( listPos, nstr )
                        SELECT rowNum, NULL  FROM @tmpITL  ORDER BY rowNum
                     END
                  END
               END

					IF @isStockTransfer = 0 BEGIN

                        IF @docLineTable IS NOT NULL BEGIN

						    DECLARE @pos  INT;
						    DECLARE @sscc VARCHAR(2000);
						    DECLARE @len  INT;
						    DECLARE @luid INT;

                            DECLARE @query NVARCHAR(4000);
						    DECLARE @pos_1 INT;
						    DECLARE @pos_2 INT;
						    SET @pos_1 = PATINDEX( '%[0-9]%', @luids );
						    SET @pos_2 = PATINDEX( '%[0-9]%', @ssccs );

					        IF ( @luids IS NULL OR @pos_1 = 0 ) AND ( @pos_2 > 0 )
					        BEGIN

							    -- Fill in LUID's from SSCC's
							    DECLARE curSSCC CURSOR LOCAL FOR
							    SELECT T0.listPos, T0.nstr, LEN(T0.nstr), CASE WHEN CurrentLuids.LUID IS NULL THEN PMX_INVT."LogUnitIdentKey" ELSE CurrentLuids.LUID END AS "LogUnitIdentKey"
							    FROM @tblSSCCs T0
							    LEFT JOIN PMX_INVT ON PMX_INVT.SSCC = T0.nstr
							    LEFT JOIN @tblCurrentLuids as CurrentLuids ON CurrentLuids.SSCC = T0.nstr
							    GROUP BY T0.listPos, T0.nstr, LEN(T0.nstr), "LogUnitIdentKey", CurrentLuids.LUID;

							    DELETE FROM @tblLuids;

							    OPEN curSSCC;
							    FETCH NEXT FROM curSSCC INTO @pos, @sscc, @len, @luid;
							    WHILE @@FETCH_STATUS = 0
							    BEGIN

								    IF @luid IS NULL BEGIN
									    IF @len <> 18 BEGIN
										    SET @error = 50031;
										    SET @error_message = 'Error on line ' + CAST(@lineNum AS NVARCHAR(255)) + ' : SSCC must have 18 digits.'

										    SELECT @error, @error_message
										    EXEC xp_logevent @error, @error_message, ERROR
										    CLOSE curSSCC;
										    DEALLOCATE curSSCC
										    CLOSE documentLineRawCursor;
										    DEALLOCATE documentLineRawCursor;
										    RETURN;
									    END

									    DECLARE @multiplier INT;
									    DECLARE @checkSum INT;
									    SET @multiplier = 1;
									    SET @checkSum = 0;
									    WHILE @len > 0 BEGIN
										    SET @checkSum = @checkSum + @multiplier * CAST( SUBSTRING( @sscc, @len, 1 ) AS INT );
										    SET @multiplier = 4 - @multiplier;
										    SET @len = @len - 1;
									    END
									    IF ( @checkSum % 10 ) <> 0 BEGIN
										    SET @error = 50032;
										    SET @error_message = 'Error on line ' + CAST(@lineNum AS NVARCHAR(255)) + ' SSCC ' + @sscc + ' : check digit invalid.'

										    SELECT @error, @error_message
										    EXEC xp_logevent @error, @error_message, ERROR
										    CLOSE curSSCC;
										    DEALLOCATE curSSCC
										    CLOSE documentLineRawCursor;
										    DEALLOCATE documentLineRawCursor;
										    RETURN;
									    END
    									
									    INSERT INTO PMX_LUID ( "Version", SSCC, "IsGenerated", "IsFullPallet", CreateDate, CreateTime, UserSign )
                                        VALUES ( 1, @sscc, 'N', 'Y', @updateDate, @updateTime, @userSign );
									    SET @luid = SCOPE_IDENTITY()
									    
									    INSERT INTO @tblCurrentLuids (luid, sscc) VALUES (@luid, @sscc)
								    END
    								
								    INSERT INTO @tblLuids ( listPos, nstr ) VALUES ( @pos, @luid );
    								
								    FETCH NEXT FROM curSSCC INTO @pos, @sscc, @len, @luid;
							    END
							    CLOSE curSSCC;
							    DEALLOCATE curSSCC
    							
							    SET @luids = NULL;
							    SELECT @luids = ISNULL( @luids + '|', '' ) + CONVERT( VARCHAR, nstr )
							    FROM @tblLUIDs
							    ORDER BY listPos;
							
							    SET @query = 'UPDATE ' + @docLineTable + ' SET U_PMX_LUID = ' + QUOTENAME( @luids, '''' )
							   	           + ' WHERE ' + @docLineTable + '."DocEntry" = ' + CAST( @docEntry AS NVARCHAR )
								           + ' AND '   + @docLineTable + '."LineNum"  = ' + CAST( @lineNum  AS NVARCHAR );
							    EXEC( @query );

						    END
						    ELSE IF ( @ssccs IS NULL OR @pos_2 = 0 ) AND ( @pos_1 > 0 )
						    BEGIN
							    -- Fill in SSCC's from LUID's
							    DECLARE @currentSSCC BIGINT;
							    DECLARE @newLuidCreated INT;

							    DELETE FROM @tblSSCCs;
							    INSERT INTO @tblSSCCs ( listPos, nstr )
							    SELECT T0.listPos, PMX_LUID.SSCC
							    FROM @tblLUIDs T0
							    LEFT JOIN PMX_LUID ON PMX_LUID."InternalKey" = T0.nstr;
							    
								
								-- Generate new SSCC if luid = -1
							    SET @newLuidCreated = 0;

							    DECLARE curSSCC CURSOR LOCAL FOR
								    SELECT listPos, nstr FROM @tblSSCCs
							    OPEN curSSCC;
							    FETCH NEXT FROM curSSCC INTO @pos, @sscc;
							    WHILE @@FETCH_STATUS = 0
							    BEGIN
									-- if no SSCC was found for the current LUID, check if the LUID was -1
									IF @sscc IS NULL BEGIN
										SELECT @luid = nstr FROM @tblLUIDs WHERE listpos = @pos
									END
									
									-- if the LUID was -1, generate a new SSCC
									IF @sscc IS NULL AND @luid = -1 BEGIN
										-- generate new SSCC
										SELECT TOP 1 @currentSSCC = CAST(CurrentSSCC AS bigint) + 1 FROM PMX_SSCC;
										UPDATE PMX_SSCC SET CurrentSSCC = CAST(CurrentSSCC AS bigint) + 1, "Version" = "Version" + 1;

										-- format new SSCC with checksum into variable
										DECLARE @sql NVARCHAR(4000)
										SET @sql = 'SELECT @sscc = ' + @extraDb + '.dbo.PMX_FN_GenerateSSCC(' + CAST(@currentSSCC AS NVARCHAR(20)) + ')';
										exec sp_executesql @sql, N'@sscc NVARCHAR(20) out', @sscc OUT

										-- insert new LUID
									    INSERT INTO PMX_LUID ( "Version", SSCC, "IsGenerated", "IsFullPallet", CreateDate, CreateTime, UserSign )
                                        VALUES ( 1, @sscc, 'N', 'Y', @updateDate, @updateTime, @userSign );
									    SET @luid = SCOPE_IDENTITY()

									    INSERT INTO @tblCurrentLuids (luid, sscc) VALUES (@luid, @sscc)

										-- update temp tables with newly generated LUID and SSCC
										UPDATE @tblSSCCs SET nstr = @sscc WHERE listPos = @pos;
										UPDATE @tblLUIDs SET nstr = @luid WHERE listPos = @pos;

										SET @newLuidCreated = 1;
									END
								    FETCH NEXT FROM curSSCC INTO @pos, @sscc;
							    END
							    CLOSE curSSCC;
							    DEALLOCATE curSSCC
							    
    							
							    SET @ssccs = NULL;
							    SELECT @ssccs = ISNULL( @ssccs + '|', '' ) + nstr
							    FROM @tblSSCCs
							    ORDER BY listPos;
							    
								-- if a new LUID was generated, rebuild the list of LUID to replace -1 with new values
							    IF @newLuidCreated=1 BEGIN
									SET @luids = NULL;
									SELECT @luids = ISNULL( @luids + '|', '' ) + CONVERT( VARCHAR, nstr )
									FROM @tblLUIDs
									ORDER BY listPos;
							    END
    						
							    SET @query = 'UPDATE ' + @docLineTable + ' SET U_PMX_SSCC = ' + QUOTENAME( @ssccs, '''' );
							    IF @newLuidCreated=1 BEGIN
									SET @query = @query + ' , U_PMX_LUID = ' + QUOTENAME( @luids, '''' );
							    END
								SET @query = @query
											+ ' WHERE ' + @docLineTable + '."DocEntry" = ' + CAST( @docEntry AS NVARCHAR )
											+ ' AND '   + @docLineTable + '."LineNum"  = ' + CAST( @lineNum  AS NVARCHAR );
							    EXEC( @query );

						    END
						
						END
						
					END

					--All the tables must have the same number of records
					DECLARE @numberOfRecords INT
					SELECT @numberOfRecords = COUNT(listPos) FROM @tblBatchNumbers1
					IF NOT(
						(SELECT COUNT(listPos) FROM @tblBatchNumbers2) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblBestBeforeDates) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblQuantities) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblStorLocs) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblLuids) = @numberOfRecords AND
						(@isStockTransfer = 1 OR (SELECT COUNT(listPos) FROM @tblSSCCs) = @numberOfRecords) AND
						(SELECT COUNT(listPos) FROM @tblQualityStatuses) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblQuantitiesUom2) = @numberOfRecords
					) BEGIN
						SET @error = 50014;
						--SET @error_message = 'Error on line ' + CAST(@lineNum AS NVARCHAR(MAX)) + ': One or more columns (BatchNumbers1,BatchNumbers2,BestBeforeDates,Quantities,StorLocs,Luids or QualityStatus) have more or less pipes(|) than the other columns.'
						
						SET @error_message = 'Line ' + CAST(@lineNum AS NVARCHAR(MAX)) +  ': Values for BatchNumbers1 = '+ CAST((SELECT COUNT(listPos) FROM @tblBatchNumbers1) AS NVARCHAR(MAX)) +
						',BatchNumbers2 = '+CAST((SELECT COUNT(listPos) FROM @tblBatchNumbers2) AS NVARCHAR(MAX))+
						',BestBeforeDates = '+CAST((SELECT COUNT(listPos) FROM @tblBestBeforeDates) AS NVARCHAR(MAX)) +
						',Quantities = '+CAST((SELECT COUNT(listPos) FROM @tblQuantities) AS NVARCHAR(MAX)) +
						',StorLocs = '+CAST((SELECT COUNT(listPos) FROM @tblStorLocs) AS NVARCHAR(MAX)) +
						',LUIDs = '+CAST((SELECT COUNT(listPos) FROM @tblLuids) AS NVARCHAR(MAX)) +
						',QualityStatus = ' +CAST((SELECT COUNT(listPos) FROM @tblQualityStatuses) AS NVARCHAR(MAX))+ 
						',Quantities UOM2 = ' +CAST((SELECT COUNT(listPos) FROM @tblQuantitiesUom2) AS NVARCHAR(MAX))+ 
						',SSCCs = ' +CAST((SELECT COUNT(listPos) FROM @tblSSCCs) AS NVARCHAR(MAX))

						
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						CLOSE documentLineRawCursor;
						DEALLOCATE documentLineRawCursor
						RETURN;
					END
					
					
					IF @isStockTransfer = 1 AND
						NOT(
						(SELECT COUNT(listPos) FROM @tblStorLocs) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblDestStorLocs) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblDestLuids) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblDestQualityStatuses) = @numberOfRecords AND
						(SELECT COUNT(listPos) FROM @tblMasterLuids) = @numberOfRecords
					) BEGIN
						SET @error = 50015;
						--SET @error_message = 'Error on line ' + CAST(@lineNum AS NVARCHAR(MAX)) + ': One or more columns (BatchNumbers1,BatchNumbers2,BestBeforeDates,Quantities,StorLocs,Luids or QualityStatus) have more or less pipes(|) than the other columns.'
						
						SET @error_message = 'Line ' + CAST(@lineNum AS NVARCHAR(MAX)) +  ': Values for DestStorLocs = '+ CAST((SELECT COUNT(listPos) FROM @tblDestStorLocs) AS NVARCHAR(MAX)) +
						',DestLUIDs = '+CAST((SELECT COUNT(listPos) FROM @tblDestLuids) AS NVARCHAR(MAX))+
						',DestQualityStatus = '+CAST((SELECT COUNT(listPos) FROM @tblDestQualityStatuses) AS NVARCHAR(MAX)) +
						',Quantities = '+CAST((SELECT COUNT(listPos) FROM @tblQuantities) AS NVARCHAR(MAX)) + 
						',MasterLUIDs = '+CAST((SELECT COUNT(listPos) FROM @tblMasterLuids) AS NVARCHAR(MAX)) 

						
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						CLOSE documentLineRawCursor;
						DEALLOCATE documentLineRawCursor
						RETURN;
					END


					--Check if luid corresponds to SSCC
					IF @isStockTransfer = 0
					BEGIN
						DECLARE @checksscc VARCHAR(2000)
						DECLARE @checkluid VARCHAR(2000)
						SET @checksscc = ''
						SET @checkluid = ''
						SELECT @checkluid = @checkluid + ISNULL(CAST(LUIDs.nstr AS NVARCHAR(MAX)), '') + ' ', @checksscc = @checksscc + SSCCs.nstr + ' '
						FROM @tblLuids AS LUIDs INNER JOIN
							@tblSSCCs AS SSCCs ON LUIDs.ListPos = SSCCs.ListPos
						WHERE LUIDs.nstr NOT IN (SELECT "InternalKey" FROM PMX_LUID WHERE "InternalKey" = LUIDs.nstr AND SSCC = SSCCs.nstr)
						IF LEN(@checksscc) <> 0 BEGIN
							SET @error = 50033;
							SET @error_message = 'Error: luid '+ @checkluid + ' does not correspond to sscc ' + @checksscc
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							CLOSE documentLineRawCursor;
							DEALLOCATE documentLineRawCursor
							RETURN;
						END
					END

					DECLARE @uomDecimals AS INT;
					SELECT @uomDecimals = ISNULL( U_PMX_UOMD,6) FROM OITM WHERE "ItemCode" = @itemCode;

					--The sum of the quantities must be the same as the quantity on the line
					--IF (SELECT SUM(nstr) FROM @tblQuantities) <> ROUND ( ROUND((@quantity * @quantityPerUom),6), @uomDecimals) BEGIN
					IF (SELECT SUM(nstr) FROM @tblQuantities) <> ROUND ( ROUND((@inventoryQuantity),6), @uomDecimals) BEGIN
						SET @error = 50034;
						SET @error_message = 'Error for item: ' + CAST(@itemCode AS NVARCHAR(MAX)) + ' The total of the quantities in the column  (Pmx Quantities) '+CAST((SELECT SUM(nstr) FROM @tblQuantities)AS NVARCHAR(MAX))+
												' is not the same as the SAP quantity ' + CAST(ROUND((@inventoryQuantity), @uomDecimals) AS NVARCHAR(MAX))+' fields on the line.'
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						CLOSE documentLineRawCursor ;
						DEALLOCATE documentLineRawCursor
						RETURN;
					END

					--All batchnumbers1 must be entered in upper case
					--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
					SET @batchNumbers1 = '';
					SET @lineNums = '';
					SELECT @batchNumbers1 = @batchNumbers1 + BatchNumbers.nstr + ' ', @lineNums = CAST(@lineNum AS NVARCHAR(MAX)) + ' '
					FROM @tblBatchNumbers1 AS BatchNumbers
					WHERE BatchNumbers.nstr COLLATE SQL_Latin1_General_Cp850_CS_AS <> UPPER(BatchNumbers.nstr) COLLATE SQL_Latin1_General_Cp850_CS_AS
					IF LEN(@lineNums) <> 0 BEGIN
						SET @error = 50018;
						SET @error_message = 'Error on line(s):  ' + @lineNums + ' All batchNumbers1 must be upper case. This is not the case for the batchnumbers: ' + @batchNumbers1
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END

					--All batchnumbers2 must be entered in upper case
					--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
					SET @batchNumbers2 = '';
					SET @lineNums = '';
					SELECT @batchNumbers2 = @batchNumbers2 + BatchNumbers.nstr + ' ', @lineNums = CAST(@lineNum AS NVARCHAR(MAX)) + ' '
					FROM @tblBatchNumbers2 AS BatchNumbers
					WHERE BatchNumbers.nstr COLLATE SQL_Latin1_General_Cp850_CS_AS <> UPPER(BatchNumbers.nstr) COLLATE SQL_Latin1_General_Cp850_CS_AS
					IF LEN(@lineNums) <> 0 BEGIN
						SET @error = 50024;
						SET @error_message = 'Error on line(s):  ' + @lineNums + ' All batchNumbers2 must be upper case. This is not the case for the batchnumbers: ' + @batchNumbers2
						SELECT @error, @error_message
						EXEC xp_logevent @error, @error_message, ERROR
						RETURN;
					END
					
					--When this is a catch weight item, the quantity for UOM2 should be filled in
					IF(@isCatchWeightForDocument = 'Y' ) BEGIN
					
						SET @lineNums = ''
						SELECT @lineNums = CAST(@lineNum AS NVARCHAR(MAX)) + ' '
							FROM @tblQuantitiesUom2 AS QuantitiesUom2 
							WHERE QuantitiesUom2.nstr IS NULL
							
						IF LEN(@lineNums) <> 0 BEGIN
							SET @error = 50039;
							SET @error_message = 'Error on line(s):  ' + @lineNums + ' "Quantity" for UOM 2 needs to be filled for catch weight items.' 
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END
						
						--Check if quantity is within tolerance
						--For goods issue/receipt, the check is not done, to make sure adjustments can be made.
						SET @lineNums = ''
						SELECT @lineNums = CAST(@lineNum AS NVARCHAR(MAX)) + ' ' 
						FROM @tblQuantitiesUom2 AS QuantitiesUom2 
						INNER JOIN @tblQuantities AS Quantities ON Quantities.ListPos = QuantitiesUom2.ListPos
						INNER JOIN OITM ON OITM.ItemCode = @itemCode
						WHERE OITM.U_PMX_DQUM < 100
						AND @objType NOT IN ('59', '60')
						AND (QuantitiesUom2.nstr > ((Quantities.nstr * OITM.U_PMX_DQUM) * (1 + (OITM.U_PMX_CWTO/100))) OR
						QuantitiesUom2.nstr < ((Quantities.nstr * OITM.U_PMX_DQUM) * (1 - (OITM.U_PMX_CWTO/100))))
						
														
						IF LEN(@lineNums) <> 0 BEGIN
							SET @error = 50040;
							SET @error_message = 'Error on line(s):  ' + @lineNums + ' The weight for UOM 2 is not within the tolerance.' 
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END
					
					END

					--Insert new item transactional info where this line does not exist
					--The line is only inserted if a batchnumber or best before date is filled in
					--TODO: Add function to external database to get the date and time
					INSERT INTO PMX_ITRI( "Canceled", "UserSign", "CreateDate", "CreateTime", "UpdateDate", "UpdateTime", "Version", "ItemCode", "BatchNumber","InternalBatchNumber","BestBeforeDate")
					SELECT DISTINCT 'N', @userSign, @createDate,@createTime,@createDate,@createTime, 1, @itemCode, BatchNumbers1.nstr, BatchNumbers2.nstr, BestBeforeDates.nstr
					FROM @tblBatchNumbers1 AS BatchNumbers1 
						 INNER JOIN @tblBestBeforeDates AS BestBeforeDates ON BatchNumbers1.listPos = BestBeforeDates.listPos
						 INNER JOIN @tblBatchNumbers2 AS BatchNumbers2 ON BatchNumbers1.listPos = BatchNumbers2.listPos
						 LEFT JOIN PMX_ITRI ON 
							ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( BatchNumbers1.nstr, '') AND 
							ISNULL( PMX_ITRI."InternalBatchNumber", '') = ISNULL( BatchNumbers2.nstr, '') AND 
							(PMX_ITRI."BestBeforeDate" = BestBeforeDates.nstr OR (PMX_ITRI."BestBeforeDate" IS NULL AND BestBeforeDates.nstr IS NULL)) AND
							(PMX_ITRI."ItemCode" = @itemCode)
					WHERE PMX_ITRI."InternalKey" IS NULL AND (NOT(BatchNumbers1.nstr IS NULL AND BatchNumbers2.nstr IS NULL AND BestBeforeDates.nstr IS NULL))


					--Check if we already have a link between serial number and document.
					--If not, only 1 itri key can be used, because otherwise, we cannot decide what the serial numbers are linked to the different itri keys
					IF( @isSAPSerial = 'Y' AND
						NOT EXISTS(SELECT InternalKey FROM PMX_SELD WHERE "BaseEntry" = @docEntry AND "BaseLine" = @lineNum AND "BaseType" = @objType))
					BEGIN

						--Check for only 1 itri key
						IF ((SELECT COUNT(listPos) FROM @tblBatchNumbers1) +
						(SELECT COUNT(listPos) FROM @tblBatchNumbers2) +
						(SELECT COUNT(listPos) FROM @tblBestBeforeDates) > 3)
						BEGIN
							SET @error = 50060;
							SET @error_message = 'Only 1 ITRI allowed by line. Error on line ' + CAST(@lineNum AS NVARCHAR(MAX)) + ' and docentry ' + CAST(@docEntry AS NVARCHAR(MAX))
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END

						--Get the itri key
						DECLARE @itriForSerial int;
						SELECT TOP 1 @itriForSerial = PMX_ITRI."InternalKey"
						FROM @tblBatchNumbers1 AS BatchNumbers1 
						 INNER JOIN @tblBestBeforeDates AS BestBeforeDates ON BatchNumbers1.listPos = BestBeforeDates.listPos
						 INNER JOIN @tblBatchNumbers2 AS BatchNumbers2 ON BatchNumbers1.listPos = BatchNumbers2.listPos
						 LEFT JOIN PMX_ITRI ON 
								ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( BatchNumbers1.nstr, '') AND 
								ISNULL( PMX_ITRI."InternalBatchNumber", '') = ISNULL( BatchNumbers2.nstr, '') AND 
							(PMX_ITRI."BestBeforeDate" = BestBeforeDates.nstr OR (PMX_ITRI."BestBeforeDate" IS NULL AND BestBeforeDates.nstr IS NULL)) AND
							(PMX_ITRI."ItemCode" = @itemCode);


						--If we need to keep track of locations of the serial number, the LUID needs to be filled, and there can only be 1 LUID!
						DECLARE @luidForSerial  INT;
						DECLARE @destLuidForSerial  INT;

						IF (@trackLocationForSerial = 'Y'
                            AND (  (SELECT COUNT(*) FROM @tblLuids) > 1
                                OR (SELECT TOP 1 nstr FROM @tblLuids) = 0)) BEGIN
							SET @error = 50061;
							SET @error_message = 'Serial location tracked for item ' + @itemCode + ' and docentry ' + CAST(@docEntry AS NVARCHAR(MAX)) + ' requires 1 LUID'
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END


						IF (@trackLocationForSerial = 'Y') BEGIN
							SET @luidForSerial = (SELECT TOP 1 nstr FROM @tblLuids);
							SET @destLuidForSerial = (SELECT TOP 1 nstr FROM @tblDestLuids);
						END								
						ELSE
                        BEGIN
							SET @luidForSerial = NULL;
							SET @destLuidForSerial = NULL;
						END
							
						--Add to PMX serial numbers table
						DECLARE @outputSerialTable TABLE ("InternalKey" int, "SerialNumber" nvarchar(36)) 

						INSERT INTO PMX_SENU ("Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialNumber", "InStock", "ItriKey", "ItemCode", "LUID", "PmxWhsCode") 
						OUTPUT INSERTED."InternalKey", INSERTED."SerialNumber" INTO @outputSerialTable
						SELECT 'N', @userSign, @tsNow, @tsNow, 1, "DistNumber", CASE WHEN @stockDirection > 0 OR @isStockTransfer = 1 THEN 'Y' ELSE 'N' END, @itriForSerial, OSRN."ItemCode", ISNULL( @destLuidForSerial, @luidForSerial), PMX_OSWH."Code"
						FROM OITL 
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
						AND ITL1."SysNumber" = OSRN."SysNumber"
						INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OITL."LocCode"
						LEFT JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
						AND PMX_SENU."SerialNumber" = "DistNumber"
						WHERE OITL."ApplyEntry" = @docEntry
						AND OITL."ApplyLine" = @lineNum
						AND OITL."ApplyType" = @objType
						AND OITM."ManSerNum" = 'Y'
						AND OITL."LocCode" = @whsCode
						AND PMX_SENU."InternalKey" IS NULL
						GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN.itemcode, "StockEff", PMX_OSWH."Code"
						HAVING SUM(ITL1."Quantity") <> 0
						AND "DistNumber" IS NOT NULL;


						--Update the 'in stock' values
						UPDATE PMX_SENU SET "InStock" = CASE WHEN @stockDirection > 0  OR @isStockTransfer = 1  THEN 'Y' ELSE 'N' END, "PmxWhsCode" = CASE WHEN @isStockTransfer = 1 THEN PMX_OSWH."Code" ELSE "PmxWhsCode" END
						FROM PMX_SENU 
						INNER JOIN OITM ON OITM."ItemCode" = PMX_SENU."ItemCode"
						INNER JOIN
							(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM(ITL1."Quantity") * @StockDirection AS "Quantity", "StockEff", "LocCode"

							FROM OSRN
							INNER JOIN ITL1 ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITL ON ITL1."LogEntry" = OITL."LogEntry"
							WHERE OITL."ApplyEntry" = @docEntry
							AND OITL."ApplyLine" = @lineNum
							AND OITL."ApplyType" = @objType
							AND OITL."LocCode" = CASE WHEN @isStockTransfer = 1 THEN @destinationWhsCode ELSE @whsCode END 
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", "LocCode"
							HAVING SUM(ITL1."Quantity") <> 0
							) AS SerialTable ON SerialTable."ItemCode" = PMX_SENU."ItemCode"
							AND SerialTable."DistNumber" = PMX_SENU."SerialNumber"
						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = SerialTable."LocCode"
						WHERE OITM."ManSerNum" = 'Y'
						AND ISNULL(PMX_SENU."ItriKey", -1) = ISNULL(@itriForSerial, -1);

						IF (@trackLocationForSerial = 'Y'
                            AND ISNULL( @luidForSerial, 0) <> ISNULL( @destLuidForSerial, 0)) BEGIN

							--Check if move for track location serial number does not split up LUID when linked to document
							SET @errorList = ''
							SELECT @errorList = @errorList + CAST(OSRN."ItemCode" AS VARCHAR) + ' ' 
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
							AND PMX_SENU."SerialNumber" = "DistNumber"
							AND PMX_SENU.LUID = @luidForSerial
							INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
							INNER JOIN RDR1 ON RDR1."ObjType" = PMX_SELD."BaseType"
							AND RDR1."DocEntry" = PMX_SELD."BaseEntry"
							AND RDR1."LineNum" = PMX_SELD."BaseLine"
							AND RDR1."LineStatus" = 'O'
							WHERE OITL."ApplyEntry" = @docEntry
							AND OITL."ApplyLine" = @lineNum
							AND OITL."ApplyType" = @objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = @whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0;
							IF LEN(@errorList) <> 0 BEGIN
								SET @error = 50062;
								SET @error_message = 'LUID cannot be split up when serial numbers are linked to sales order for items: ' + @errorList 
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END

							--Check if move for track location serial number does not split up LUID when linked to document
							SET @errorList = ''
							SELECT @errorList = @errorList + CAST(OSRN."ItemCode" AS VARCHAR) + ' ' 
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
							AND PMX_SENU."SerialNumber" = "DistNumber"
							AND PMX_SENU.LUID = @luidForSerial
							INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
							INNER JOIN PMX_PLPL ON PMX_PLPL."DocEntry" = PMX_SELD."BaseEntry"
							AND PMX_PLPL."LineNum" = PMX_SELD."BaseLine"
							AND PMX_PLPL."LineStatus" = 'O'
							AND PMX_SELD."BaseType" = 'PMX_PLPH'
							WHERE OITL."ApplyEntry" = @docEntry
							AND OITL."ApplyLine" = @lineNum
							AND OITL."ApplyType" = @objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = @whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0;
							IF LEN(@errorList) <> 0 BEGIN
								SET @error = 50063;
								SET @error_message = 'LUID cannot be split up when serial numbers are linked to pick list proposal for items: ' + @errorList 
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END

							--Check if move for track location serial number does not split up LUID when linked to document
							SET @errorList = ''
							SELECT @errorList = @errorList + CAST(OSRN."ItemCode" AS VARCHAR) + ' ' 
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
							AND PMX_SENU."SerialNumber" = "DistNumber"
							AND PMX_SENU.LUID = @luidForSerial
							INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
							INNER JOIN PMX_PLLI ON PMX_PLLI."DocEntry" = PMX_SELD."BaseEntry"
							AND PMX_PLLI."LineNum" = PMX_SELD."BaseLine"
							AND PMX_PLLI."LineStatus" = 'O'
							AND PMX_SELD."BaseType" = 'PMX_PLHE'
							WHERE OITL."ApplyEntry" = @docEntry
							AND OITL."ApplyLine" = @lineNum
							AND OITL."ApplyType" = @objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = @whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0;
							IF LEN(@errorList) <> 0 BEGIN
								SET @error = 50064;
								SET @error_message = 'LUID cannot be split up when serial numbers are linked to pick list for items: ' + @errorList 
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;
							END

						END


						--Add link to serial number document link table
						INSERT INTO PMX_SELD ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine") 
						SELECT 'N', @userSign, @tsNow, @tsNow, 1, "InternalKey", @objType, @docEntry, @lineNum
						FROM @outputSerialTable

						


					END
					ELSE IF (@isSAPSerial = 'Y' AND @isStockTransfer = 1) BEGIN
						--Adjust serial number whs code
						UPDATE PMX_SENU
						SET PMX_SENU.PmxWhsCode = PMX_OSWH."Code",
							PMX_SENU."Version" = PMX_SENU."Version" + 1
						FROM PMX_SENU
						INNER JOIN OITM ON OITM."ItemCode" = PMX_SENU."ItemCode"
						INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
						INNER JOIN
							(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM(ITL1."Quantity") * @StockDirection AS "Quantity", "StockEff", "LocCode"

							FROM OSRN
							INNER JOIN ITL1 ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITL ON ITL1."LogEntry" = OITL."LogEntry"
							WHERE OITL."ApplyEntry" = @keyAsInteger
							AND OITL."ApplyType" = '67'
							AND OITL."LocCode" = @destinationWhsCode 
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", "LocCode"
							HAVING SUM(ITL1."Quantity") <> 0
							) AS SerialTable ON SerialTable."ItemCode" = PMX_SENU."ItemCode"
							AND SerialTable."DistNumber" = PMX_SENU."SerialNumber"
							AND SerialTable."ApplyLine" = PMX_SELD."BaseLine"
							AND SerialTable."ApplyEntry" = PMX_SELD."BaseEntry"
							AND SerialTable."ApplyType" = PMX_SELD."BaseType"
						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = SerialTable."LocCode"
						WHERE OITM."ManSerNum" = 'Y'
						AND PMX_SELD."BaseEntry" = @keyAsInteger
						AND PMX_SELD."BaseType" = '67';  --Warehouse transfer
					END


					
					--Insert the document lines in a nice format in the table
					--Quantities.nstr is the number of items in inventory and not in sales/purchase UOM. Therefore we devide the quantity again by the quantityPerUom
					INSERT INTO @documentLineTable 
						SELECT @docEntry,@objType,@docNum,@docDate,@cardCode,@cardName,@userSign,@createDate,@createTime,@lineNum, @baseType ,@baseEntry,@BaseLine,@itemCode,@description, Quantities.nstr/@quantityPerUom, Quantities.nstr, @uom,@inventoryUom, @quantityPerUom, @whsCode, StorLocs.nstr, PMX_ITRI."InternalKey", Luids.nstr, PMX_LUID.SSCC, Qualities.nstr, @isLogisticCarrier, @reasonCode, @uom2, QuantitiesUom2.nstr, 
						DestStorLocs.nstr, @destinationWhsCode, DestQualities.nstr, DestLuids.nstr, MasterLuids.nstr
						FROM @tblBatchNumbers1 AS BatchNumbers1 
							 INNER JOIN @tblBatchNumbers2 AS BatchNumbers2 ON BatchNumbers1.listPos = BatchNumbers2.listPos
							 INNER JOIN @tblBestBeforeDates AS BestBeforeDates ON BatchNumbers1.listPos = BestBeforeDates.listPos
							 INNER JOIN @tblQuantities AS Quantities ON BatchNumbers1.listPos = Quantities.listPos
							 INNER JOIN @tblQuantitiesUom2 AS QuantitiesUom2 ON BatchNumbers1.listPos = QuantitiesUom2.listPos
							 INNER JOIN @tblStorLocs AS StorLocs ON BatchNumbers1.listPos = StorLocs.listPos
							 INNER JOIN @tblLuids AS Luids ON BatchNumbers1.listPos = Luids.listPos
							 INNER JOIN @tblQualityStatuses AS Qualities ON BatchNumbers1.listPos = Qualities.listPos
		 					 INNER JOIN OITM ON OITM."ItemCode" = @itemCode
							 LEFT JOIN @tblDestStorLocs AS DestStorLocs ON BatchNumbers1.listPos = DestStorLocs.listPos
							 LEFT JOIN @tblDestLuids AS DestLuids ON BatchNumbers1.listPos = DestLuids.listPos
							 LEFT JOIN @tblDestQualityStatuses AS DestQualities ON BatchNumbers1.listPos = DestQualities.listPos
							 LEFT JOIN @tblMasterLuids AS MasterLuids ON BatchNumbers1.listPos = MasterLuids.listPos
							 LEFT JOIN PMX_ITRI ON 
								ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( BatchNumbers1.nstr, '') AND 
								ISNULL( PMX_ITRI."InternalBatchNumber", '') = ISNULL( BatchNumbers2.nstr, '') AND 
								(PMX_ITRI."BestBeforeDate" = BestBeforeDates.nstr OR (PMX_ITRI."BestBeforeDate" IS NULL AND BestBeforeDates.nstr IS NULL)) AND
								(PMX_ITRI."ItemCode" = @itemCode)
							 LEFT JOIN PMX_LUID ON PMX_LUID."InternalKey" = Luids.nstr
						WHERE OITM."InvntItem" = 'Y';
						
					FETCH NEXT FROM documentLineRawCursor INTO @docEntry,@objType,@docNum,@docDate,@cardCode,@cardName,@userSign,@createDate,@createTime,@lineNum,@baseType,@baseEntry,@BaseLine,@itemCode,@description,@quantity, @inventoryQuantity,@uom,@inventoryUom,@quantityPerUom,@whsCode,@batchNumbers1,@batchNumbers2,@bestBeforeDates,@quantities,@storLocs,@luids,@ssccs,@qualityStatuses,@isLogisticCarrier,@reasonCode, @quantitiesUom2, @uom2, @quantityUom2, @dropShip, @destinationStorLocCodes, @destinationWhsCode, @destinationQualityStatusCodes, @destinationLuids, @trackLocationForSerial, @isSAPSerial, @isCatchWeightForDocument, @masterLuids;
				END

				CLOSE documentLineRawCursor 
				DEALLOCATE documentLineRawCursor
			END TRY
			BEGIN CATCH
				SET @error = 50016
				SET @error_message = 'Error raw lines->normal lines: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
				+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE documentLineRawCursor 
					DEALLOCATE documentLineRawCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE documentLineRawCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END CATCH
			END CATCH


			--**********************************************************************************************************
			--Check if the converted lines are the same as the SAP lines
			--**********************************************************************************************************

			--The batchnumbers and the quantities must be the same as what is saved in SAP
			--The batchnumbers are checked using case sensitive and not like default case insensitive
			--This means if in SAP batch is saved as b1 and in extra fields as B1 that this is not the same
			SET @lineNums = '';
			SET @batchNumbers1 = '';
			SET @quantities = '';
			SELECT TOP 1 @objType = "ObjType" FROM @documentLineTable --Get the object type of the document
			SELECT @lineNums = @lineNums + CAST(ISNULL( Lines."LineNum",TranLog."DocLine") AS NVARCHAR) + ' '
                 , @batchNumbers1 = @batchNumbers1 + ISNULL( ISNULL( Lines."BatchNumber", TranLog."DistNumber"),'<NULL>') + ' '
                 , @quantities = @quantities + 'PMX:'+ CAST(CAST(ISNULL( Lines."Quantity", 0) AS DECIMAL(19,6) ) AS NVARCHAR) + '-SAP:'+ CAST(CAST(ISNULL( TranLog."Quantity", 0) AS DECIMAL(19,6) ) AS NVARCHAR)+ ' '
			FROM (
				SELECT LinesTable."ObjType", LinesTable."DocEntry", LinesTable."LineNum", PMX_ITRI."BatchNumber", SUM(LinesTable."Quantity"*LinesTable."QuantityPerUom") AS "Quantity" , OITM.U_PMX_UOMD AS UOMDecimals, LinesTable."WhsCode"
   			    FROM @documentLineTable AS LinesTable 
				INNER JOIN OITM ON LinesTable."ItemCode" = OITM."ItemCode" 
				LEFT JOIN  PMX_ITRI ON LinesTable."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
				WHERE OITM."ManBtchNum" = 'Y' 
				GROUP BY LinesTable."ObjType", LinesTable."DocEntry", LinesTable."LineNum", PMX_ITRI."BatchNumber", OITM.U_PMX_UOMD, LinesTable."WhsCode"
   		    ) AS Lines 
			LEFT JOIN (
                SELECT OITL."DocType", OITL."DocEntry", OITL."DocLine", OBTN."DistNumber", SUM( ITL1."Quantity" ) * @stockDirection AS "Quantity", OITL."LocCode"
                FROM OITL 
                LEFT JOIN (   ITL1  -- or INNER JOIN?
                   INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
                   AND ITL1."SysNumber" = OBTN."SysNumber"
                ) ON ITL1."LogEntry" = OITL."LogEntry"
                WHERE OITL."DocType" = @objType AND OITL."DocEntry" = @keyAsInteger AND "StockEff" = 1  -- Affects "OnHand"
                GROUP BY OITL."DocType", OITL."DocEntry", OITL."DocLine", OBTN."DistNumber", OITL."LocCode"
                HAVING SUM( ITL1."Quantity" ) <> 0
			) AS TranLog ON TranLog."DocType" = Lines."ObjType" 
			AND TranLog."DocEntry" = Lines."DocEntry" 
			AND TranLog."DocLine" = Lines."LineNum"
			AND TranLog."LocCode" = Lines."WhsCode"
			AND TranLog."DistNumber" COLLATE SQL_Latin1_General_Cp850_CS_AS = Lines."BatchNumber" COLLATE SQL_Latin1_General_Cp850_CS_AS
			--WHERE ISNULL( ROUND(TranLog."Quantity" * (CASE WHEN @isTransactionTypeCancel=0 THEN 1 ELSE -1 END), ISNULL( Lines.UOMDecimals, 0)), 0) <> ISNULL( ROUND(Lines."Quantity" , ISNULL( Lines.UOMDecimals, 0)), 0)
			WHERE ISNULL( ROUND(TranLog."Quantity", ISNULL( Lines.UOMDecimals, 0)), 0) <> ISNULL( ROUND(Lines."Quantity" , ISNULL( Lines.UOMDecimals, 0)), 0)
			-- OR (TranLog."Quantity" IS NULL AND Lines."Quantity" > 0) 
			-- OR Lines."Quantity" IS NULL
            ;
			
			IF LEN(@lineNums) <> 0 BEGIN
				SET @error = 50017;
				SET @error_message = 'Line(s) ' + @lineNums + ': BatchNumbers ( '+@batchNumbers1+') and quantities ( '+@quantities+' ) are not equal for SAP and Produmex fields'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--**********************************************************************************************************
			--Do checks on document lines
			--**********************************************************************************************************
			-- Check if storage location code exists and is filled in in the organizational structure
			-- This check is only done for the inventory lines.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
				 INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
				 INNER JOIN OWHS ON OWHS."WhsCode" = Lines."WhsCode"
				 LEFT JOIN [PMX_OSEL] AS OSE ON Lines."StorLocCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = OSE."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS AND OSE."IsStorageLocation" = 'Y'
			WHERE Items."InvntItem" = 'Y' AND OSE."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y'
			GROUP BY Lines."LineNum";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50003;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The storage location code is not filled in or does not exist in the organizational structure.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if batch number 2 is filled in for items with a second batch number
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
				INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."InternalBatchNumber" IS NULL AND Items.U_PMX_HBN2 = 'Y'
			GROUP BY Lines."LineNum";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50022;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The second batch number must be filled in for items that have a second batch number.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if batch number 2 is NOT filled in for items that have no second batch number
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
				INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."InternalBatchNumber" IS NOT NULL AND Items.U_PMX_HBN2 = 'N'
			GROUP BY Lines."LineNum";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50023;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The second batch number cannot be filled in for items that have NO second batch number.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if best before date is filled in for expirable goods
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
				INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."BestBeforeDate" IS NULL AND Items.U_PMX_HBBD = 'Y'
			GROUP BY Lines."LineNum";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50004;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The best before date must be filled in for expirable item.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if best before date is NOT filled in for NON expirable goods
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines
			INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
			LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."BestBeforeDate" IS NOT NULL AND Items.U_PMX_HBBD = 'N'
			GROUP BY Lines."LineNum";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50005;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The best before date cannot be filled in for a non-expirable item.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if the entered LUID exists for the inventory items
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			LEFT JOIN PMX_LUID ON Lines."LogUnitIdentKey" = PMX_LUID."InternalKey" 
			WHERE Lines."LogUnitIdentKey" IS NOT NULL AND OITM."InvntItem" = 'Y' AND PMX_LUID."InternalKey" IS NULL
			GROUP BY Lines."LineNum";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50006;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The entered LUID does not exist.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

	--		-- Check if the entered SSCC matches the stored SSCC for the specified LUID
	--		SET @errorList = ''
	--		SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
	--		FROM @documentLineTable AS Lines LEFT JOIN PMX_LUID AS LogicalUnitID ON Lines.U_PMX_LUID = LogicalUnitID."InternalKey" 
	--		WHERE Lines.U_PMX_SSCC <> LogicalUnitID.SSCC
	--		IF LEN(@errorList) <> 0 BEGIN
	--			SET @error = 50007
	--			SET @error_message = 'Error on line(s): ' + @errorList + ' The entered SSCC does not match the stored SSCC for the specified LUID.'
	--			SELECT @error, @error_message
	--			EXEC xp_logevent @error, @error_message, ERROR
	--			RETURN
	--		END

			-- Check if the entered quality statuses exists and is filled in for inventory items
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR)  
			FROM @documentLineTable AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			INNER JOIN OWHS ON OWHS."WhsCode" = Lines."WhsCode"
			LEFT JOIN PMX_QYST ON Lines."QualityStatusCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = PMX_QYST."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE OITM."InvntItem" = 'Y' AND PMX_QYST."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50008;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The entered quality status does not exist or is not filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			-- Check if the entered quality statuses exists and is filled in for inventory items
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR)  
			FROM @documentLineTable AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			INNER JOIN OWHS ON OWHS."WhsCode" = Lines."DestinationWhsCode"
			LEFT JOIN PMX_QYST ON Lines.DestinationQualityStatusCode COLLATE SQL_Latin1_General_Cp850_CS_AS = PMX_QYST."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE OITM."InvntItem" = 'Y' AND PMX_QYST."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50008;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The entered destination quality status does not exist or is not filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if the warehouse of the storage location on the document line is the same
			-- warehouse as the one that is stored on the document line itself.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			INNER JOIN [PMX_OSEL] AS OSE ON Lines."StorLocCode" = OSE."Code"
			INNER JOIN [PMX_OSWH] AS WHS ON OSE."PmxWhsCode" = WHS."Code"
			WHERE Lines."WhsCode" <> WHS."SboWhsCode";
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50009;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The storage location code is of a different warehouse than the warehouse defined on the line.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- If a line is set as logisic carrier then the LUID must be filled in
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Lines."LogUnitIdentKey" IS NULL;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50010;
				SET @error_message = 'Error on line(s): ' + @errorList + ' If the flag is set that this line is the logistic carrier of the logistic unit then the logistic unit ID must be filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if logistic carrier is filled in on the logistic unit and that there are no lines
			-- indicated as logistic carrier.
			-- Also check that not 2 logistic carriers are defined on the same logistic unit
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LogUnitIdentKey" AS VARCHAR) + ' '
			FROM @documentLineTable AS Lines 
			INNER JOIN PMX_LUID ON Lines."LogUnitIdentKey" = PMX_LUID."InternalKey"
			WHERE Lines."IsLogisticCarrier" = 'Y'
			GROUP BY lines."DocEntry", lines."LogUnitIdentKey", PMX_LUID."LogisticCarrierCode"
			HAVING (count(lines."DocEntry") > 1) or (count(lines."DocEntry") = 1 AND PMX_LUID."LogisticCarrierCode" IS NOT NULL);
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50011;
				SET @error_message = 'Error for logistic unit(s): ' + @errorList + ' There is already a logistic carrier defined or there is more then 1 line is defined as logistic carrier for these logistic unit(s).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- If a line is set as logisic carrier then the item must be a logistic carrier
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			WHERE Lines."IsLogisticCarrier" = 'Y' AND OITM.U_PMX_LOCA <> 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50012;
				SET @error_message = 'Error on line(s): ' + @errorList + ' If the flag is set that this line is the logistic carrier of the logistic unit then the item must be a logistic carrier.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- If a line is set as logisic carrier then its quantity must be 1
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			WHERE Lines."IsLogisticCarrier" = 'Y' AND ( Lines."Quantity" * Lines."QuantityPerUom" ) <> 1;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50013;
				SET @error_message = 'Error on line(s): ' + @errorList + ' If a line is a logistic carrier then its quantity must be 1.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Does the item have zone types and is it allowed by the location
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode" 
			INNER JOIN PMX_OSEL AS Ose ON Lines."StorLocCode" = Ose."Code"
			LEFT OUTER JOIN PMX_ITZT AS ItemZoneTypes ON OITM."ItemCode" = ItemZoneTypes."ItemCode"
			LEFT OUTER JOIN PMX_LZZT AS LinkedZoneTypes ON ItemZoneTypes.ZoneTypeCode = LinkedZoneTypes.ZoneTypeCode
				AND Ose."PmxZoneCode" = LinkedZoneTypes."ParentCode"
			WHERE ItemZoneTypes."InternalKey" IS NOT NULL AND LinkedZoneTypes.ZoneTypeCode IS NULL;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50019;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The location does not have a zone type that corresponds to a zone type on the item.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if the filled in reason code exist
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			LEFT JOIN PMX_REAS ON Lines."ReasonCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = PMX_REAS."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE NULLIF( Lines."ReasonCode", '' ) IS NOT NULL AND PMX_REAS."Code" IS NULL;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50020;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The filled in reason code does not exist.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if the filled in uom2 is the same as the uom2 on an item
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			LEFT JOIN OITM ON Lines."ItemCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = OITM."ItemCode" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE Lines."Uom2" COLLATE SQL_Latin1_General_Cp850_CS_AS <> OITM.U_PMX_UOM2 COLLATE SQL_Latin1_General_Cp850_CS_AS
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50021;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The filled in uom2 is not the same as the one defined on the item.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if batch number is filled in for items with a batch number
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode" 
			LEFT JOIN PMX_ITRI ON Lines."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			WHERE OITM."ManBtchNum" = 'Y' AND NULLIF( PMX_ITRI."BatchNumber", '' ) IS NULL;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50025;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The batch number must be filled in for items that have a batch number.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if batch number is NOT filled in for items that have batch number
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode" 
			LEFT JOIN PMX_ITRI ON Lines."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			WHERE OITM."ManBtchNum" = 'N' AND NULLIF( PMX_ITRI."BatchNumber", '' ) IS NOT NULL;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50026;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The batch number cannot be filled in for items that have NO batch number.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- check if stock is stored on logistic carrier location for non logistic carriers (only for stock receipts)
			IF ((@isInventoryEntry=1 OR 
				@isPurchaseDeliveryDocument=1 OR		
				@isPurchaseInvoice=1 OR 
				@isSalesReturnDocument=1 OR		
				@isSalesCreditNote=1)
				AND (SELECT TOP 1 "LogCarOn1Loc" FROM PMX_OSCO) = 'Y' ) BEGIN

				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
					 INNER JOIN PMX_OSEL AS OSE ON Lines."StorLocCode" = OSE."Code"
					 INNER JOIN PMX_OSWH AS WHS ON OSE."PmxWhsCode" = WHS."Code"
					 INNER JOIN OITM on OITM."ItemCode" = Lines."ItemCode"
				WHERE Lines."StorLocCode" = WHS."StorLocLogCar" AND 
						OITM.U_PMX_LOCA = 'N';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50027;
					SET @error_message = 'Error on line(s): ' + @errorList + ' Items that are no logistic carrier cannot be stored on the location for logistic carriers.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END


			END

			IF (@isStockTransfer = 1) BEGIN

				-- Check if 'destination' storage location code exists and is filled in in the organizational structure
				-- This check is only done for the inventory lines.
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
					 INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
					 INNER JOIN OWHS ON OWHS."WhsCode" = Lines."DestinationWhsCode"
					 LEFT JOIN PMX_OSEL AS OSE ON Lines."DestinationStorLocCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = OSE."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS AND OSE."IsStorageLocation" = 'Y'
				WHERE Items."InvntItem" = 'Y' AND OSE."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y';
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50028;
					SET @error_message = 'Error on line(s): ' + @errorList + ' The destination storage location code is not filled in or does not exist in the organizational structure.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				-- Check if the warehouse of the 'destination' storage location on the document line is the same
				-- 'Destiniation' warehouse as the one that is stored on the document line itself.
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
				INNER JOIN PMX_OSEL AS OSE ON Lines."DestinationStorLocCode" = OSE."Code"
				INNER JOIN PMX_OSWH AS WHS ON OSE."PmxWhsCode" = WHS."Code"
				WHERE Lines."DestinationWhsCode" <> WHS."SboWhsCode";
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50029;
					SET @error_message = 'Error on line(s): ' + @errorList + ' The storage location code is of a different warehouse than the warehouse defined on the line.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END
				
				-- Check if 'destination' storage location code exists and is filled in in the organizational structure
				-- This check is only done for the inventory lines.
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."DestinationStorLocCode" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
					 INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
				WHERE Items."InvntItem" = 'Y'
				AND Lines."DestinationStorLocCode" IN (
									 SELECT "Code" FROM PMX_OSWA
									 UNION SELECT "InputLocationCode" FROM PMX_OSWA WHERE "InputLocationCode" IS NOT NULL
									 UNION SELECT "InputWithBoxLocationCode" FROM PMX_OSWA WHERE "InputWithBoxLocationCode" IS NOT NULL);
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50030;
					SET @error_message = 'Error for location(s): ' + @errorList + ' The destination storage location code cannot be a WA location.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			END

			--Check if location is not locked
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."StorLocCode" AS VARCHAR) + ' '
			FROM @documentLineTable AS Lines 
			LEFT JOIN PMX_OSSL AS OSE ON Lines."StorLocCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = OSE."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE OSE."LockedBy" IS NOT NULL
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50041;
				SET @error_message = 'Locations are locked, so no stock updates are allowed: ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			--Check if location has correct quality status code
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."DestinationStorLocCode" AS VARCHAR) + '-' + CAST(ISNULL(Lines."DestinationQualityStatusCode", Lines."QualityStatusCode") AS VARCHAR) + ' '
			FROM @documentLineTable AS Lines 
			LEFT JOIN PMX_OSSL AS OSE ON Lines."DestinationStorLocCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = OSE."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE ISNULL(OSE."QualityStatusCode", ISNULL(Lines."DestinationQualityStatusCode", Lines."QualityStatusCode")) <> ISNULL(Lines."DestinationQualityStatusCode", Lines."QualityStatusCode")
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50042;
				SET @error_message = 'Quality status does not match fixed quality status on location: ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END


			-- check if stock is free stock (only for stock exits)
			IF (@isInventoryExit=1 OR
				@isPurchaseReturnDocument=1 OR
				@isPurchaseCreditNote=1 OR
				@isSalesInvoice=1 OR
				@isStockTransfer = 1) BEGIN
				
				--Build ignore list
				--Declare memory table for storing the document lines of a purchase delivery note, sales delivery note, ...
				DECLARE @ignoreLockingTable TABLE (
					"DocEntry" int NOT NULL ,
					"ObjType" nvarchar(20) NOT NULL,
					"LineNum" int NOT NULL );
				
				INSERT INTO @ignoreLockingTable
				SELECT DISTINCT PMX_PLLI."DocEntry", 'PMX_PLHE', PMX_PLLI."LineNum"
				FROM PMX_PLLI
				INNER JOIN PMX_PLPL ON PMX_PLPL."LineNum" = PMX_PLLI."BaseLine"
				AND PMX_PLPL."DocEntry" = PMX_PLLI."BaseEntry"
				AND PMX_PLLI."BaseType" = 'PMX_PLPH'
				INNER JOIN @documentLineTable AS Lines
				ON Lines."BaseEntry" = PMX_PLPL."BaseEntry"
				AND Lines."BaseLine" = PMX_PLPL."BaseLine"
				AND Lines."BaseType" = PMX_PLPL."BaseType";

				--Check free stock for item - quality status
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."ItemCode" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
				INNER JOIN
				(
				SELECT "TOTAL"."Quantity" - ISNULL( "LOCK"."Quantity",0) AS "Quantity", "TOTAL"."ItemCode" AS "ItemCode", 
				"TOTAL"."QualityStatusCode" AS "QualityStatusCode", "TOTAL"."InvntryUom" AS "Uom" , "TOTAL"."SboWhsCode"
				FROM ( 

				SELECT SUM ("PMX_INVT"."Quantity") AS "Quantity", "PMX_INVT"."ItemCode", "OITM"."InvntryUom", 
				"PMX_INVT"."QualityStatusCode", "PMX_OSWH"."SboWhsCode" 
				FROM "PMX_INVT" 
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = "PMX_INVT"."StorLocCode" 
				INNER JOIN PMX_OSWH ON PMX_OSEL."PmxWhsCode" = PMX_OSWH."Code" 
				INNER JOIN "OITM" ON "OITM"."ItemCode" = "PMX_INVT"."ItemCode" 
				LEFT OUTER JOIN "PMX_LUID" ON "PMX_LUID"."InternalKey" = "PMX_INVT"."LogUnitIdentKey" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INVT"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey"
				GROUP BY "PMX_INVT"."ItemCode", "PMX_INVT"."QualityStatusCode", "OITM"."InvntryUom", PMX_OSWH."SboWhsCode" 

				) AS "TOTAL" 
				LEFT OUTER JOIN ( 

				SELECT SUM( "PMX_INLD"."Quantity") AS "Quantity", "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", PMX_OSWH."SboWhsCode"
				FROM "PMX_INLD" 
				INNER JOIN PMX_OSWH ON "PMX_INLD"."PmxWhsCode" = PMX_OSWH."Code" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INLD"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				LEFT JOIN @ignoreLockingTable AS IgnoreLocking
				ON IgnoreLocking."DocEntry" = PMX_INLD."BaseEntry"
				AND IgnoreLocking."LineNum" = PMX_INLD."BaseLine"
				AND IgnoreLocking."ObjType" = PMX_INLD."BaseType"
				WHERE IgnoreLocking."DocEntry" IS NULL
				GROUP BY "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode" , PMX_OSWH."SboWhsCode"

				) AS "LOCK" ON "TOTAL"."ItemCode" = "LOCK"."ItemCode" AND "TOTAL"."QualityStatusCode" = "LOCK"."QualityStatusCode" 
				AND "TOTAL"."SboWhsCode" = "LOCK"."SboWhsCode"
				) AS FreeStock ON lines."ItemCode" = FreeStock."ItemCode" 
				AND lines."QualityStatusCode" = FreeStock."QualityStatusCode"
				AND lines."WhsCode" = FreeStock."SboWhsCode"
				AND FreeStock."Quantity" < lines."Quantity";


				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50050;
					SET @error_message = 'Error for item(s): ' + @errorList + ' The stock for item and quality status is locked.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END


				--Check free stock for item - batch - best before date
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."ItemCode" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
				INNER JOIN
				(
				SELECT "TOTAL"."Quantity" - ISNULL( "LOCK"."Quantity",0) AS "Quantity", "TOTAL"."ItemCode" AS "ItemCode", 
				"TOTAL"."QualityStatusCode" AS "QualityStatusCode", "TOTAL"."InvntryUom" AS "Uom", 
				"TOTAL"."BatchNumber" AS "BatchNumber", 
				CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber_IsNull", 
				"TOTAL"."BatchNumber" AS "BatchNumber2", 
				CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber2_IsNull", 
				"TOTAL"."BestBeforeDate" AS "BestBeforeDate", 
				CASE WHEN ("TOTAL"."BestBeforeDate" IS NULL) THEN 1 ELSE 0 END AS "BestBeforeDate_IsNull", 
				"TOTAL"."ItemTransactionalInfoKey" AS "ItemTransactionalInfoKey", 
				CASE WHEN ("TOTAL"."ItemTransactionalInfoKey" IS NULL) THEN 1 ELSE 0 END AS "ItemTransactionalInfoKey_IsNull" ,
				"TOTAL"."SboWhsCode"
				FROM ( 

				SELECT SUM ("PMX_INVT"."Quantity") AS "Quantity", "PMX_INVT"."ItemCode", "OITM"."InvntryUom", 
				"PMX_INVT"."QualityStatusCode", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", 
				"PMX_INVT"."ItemTransactionalInfoKey", PMX_OSWH."SboWhsCode" 
				FROM "PMX_INVT" 
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = "PMX_INVT"."StorLocCode" 
				INNER JOIN PMX_OSWH ON PMX_OSEL."PmxWhsCode" = PMX_OSWH."Code" 
				INNER JOIN "OITM" ON "OITM"."ItemCode" = "PMX_INVT"."ItemCode" 
				LEFT OUTER JOIN "PMX_LUID" ON "PMX_LUID"."InternalKey" = "PMX_INVT"."LogUnitIdentKey" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INVT"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				GROUP BY "PMX_INVT"."ItemCode", "PMX_INVT"."QualityStatusCode", "OITM"."InvntryUom", 
				"PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INVT"."ItemTransactionalInfoKey",
				PMX_OSWH."SboWhsCode" 

				) AS "TOTAL" 
				LEFT OUTER JOIN ( 

				SELECT SUM ("PMX_INLD"."Quantity") AS "Quantity", "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", 
				"PMX_INLD"."ItemTransactionalInfoKey", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", PMX_OSWH."SboWhsCode" 
				FROM "PMX_INLD" INNER JOIN PMX_OSWH ON "PMX_INLD"."PmxWhsCode" = PMX_OSWH."Code" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INLD"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				LEFT JOIN @ignoreLockingTable AS IgnoreLocking
				ON IgnoreLocking."DocEntry" = PMX_INLD."BaseEntry"
				AND IgnoreLocking."LineNum" = PMX_INLD."BaseLine"
				AND IgnoreLocking."ObjType" = PMX_INLD."BaseType"
				WHERE IgnoreLocking."DocEntry" IS NULL
				GROUP BY "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", "PMX_ITRI"."BatchNumber", 
				"PMX_ITRI"."BestBeforeDate", "PMX_INLD"."ItemTransactionalInfoKey", PMX_OSWH."SboWhsCode"

				) AS "LOCK" 
				ON "TOTAL"."ItemCode" = "LOCK"."ItemCode" AND "TOTAL"."QualityStatusCode" = "LOCK"."QualityStatusCode" 
				AND "TOTAL"."SboWhsCode" = "LOCK"."SboWhsCode"
				AND ("TOTAL"."BatchNumber" = "LOCK"."BatchNumber" OR ("TOTAL"."BatchNumber"IS NULL AND "LOCK"."BatchNumber"IS NULL)) 
				AND ("TOTAL"."BestBeforeDate" = "LOCK"."BestBeforeDate" OR ("TOTAL"."BestBeforeDate"IS NULL AND "LOCK"."BestBeforeDate"IS NULL)) 
				AND ("TOTAL"."ItemTransactionalInfoKey" = "LOCK"."ItemTransactionalInfoKey" OR ("TOTAL"."ItemTransactionalInfoKey"IS NULL 
				AND "LOCK"."ItemTransactionalInfoKey"IS NULL)) 
				) AS FreeStock ON lines."ItemCode" = FreeStock."ItemCode" 
				AND lines."QualityStatusCode" = FreeStock."QualityStatusCode"
				AND lines."WhsCode" = FreeStock."SboWhsCode"
				AND FreeStock."Quantity" < lines."Quantity"
				AND (FreeStock."ItemTransactionalInfoKey" = lines."ItemTransactionalInfoKey" OR (FreeStock."ItemTransactionalInfoKey"IS NULL 
				AND lines."ItemTransactionalInfoKey"IS NULL));


				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50051;
					SET @error_message = 'Error for item(s): ' + @errorList + ' The stock for item, batch and best before date is locked.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

				--Check free stock for item - detail
				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."ItemCode" AS VARCHAR) + ' ' 
				FROM @documentLineTable AS Lines 
				INNER JOIN
				(
				SELECT "TOTAL"."Quantity" - ISNULL( "LOCK"."Quantity",0) AS "Quantity", "TOTAL"."ItemCode" AS "ItemCode", 
				"TOTAL"."QualityStatusCode" AS "QualityStatusCode", "TOTAL"."InvntryUom" AS "Uom", 
				"TOTAL"."StorLocCode" AS "StorLocCode", "TOTAL"."PmxWhsCode" AS "PmxWhsCode", PMX_OSSL."Sequence" AS "Sequence", 
				PMX_OSSL."IsPickLoc" AS "IsPickLoc", "TOTAL"."InternalKey" AS "InternalKey", "TOTAL"."LogUnitIdentKey" AS "LogUnitIdentKey", 
				CASE WHEN ("TOTAL"."LogUnitIdentKey" IS NULL) THEN 1 ELSE 0 END AS "LogUnitIdentKey_IsNull", "TOTAL"."SSCC" AS "SSCC", 
				CASE WHEN ("TOTAL"."SSCC" IS NULL) THEN 1 ELSE 0 END AS "SSCC_IsNull", "TOTAL"."IsFullPallet" AS "IsFullPallet", 
				"TOTAL"."BatchNumber" AS "BatchNumber", CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber_IsNull", 
				"TOTAL"."BatchNumber" AS "BatchNumber2", CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber2_IsNull", 
				"TOTAL"."BestBeforeDate" AS "BestBeforeDate", 
				CASE WHEN ("TOTAL"."BestBeforeDate" IS NULL) THEN 1 ELSE 0 END AS "BestBeforeDate_IsNull", 
				"TOTAL"."ItemTransactionalInfoKey" AS "ItemTransactionalInfoKey", 
				CASE WHEN ("TOTAL"."ItemTransactionalInfoKey" IS NULL) THEN 1 ELSE 0 END AS "ItemTransactionalInfoKey_IsNull" ,
				"TOTAL"."SboWhsCode"
				FROM ( 

				SELECT SUM ("PMX_INVT"."Quantity") AS "Quantity", "PMX_INVT"."ItemCode", "OITM"."InvntryUom", 
				"PMX_INVT"."QualityStatusCode", "PMX_INVT"."StorLocCode", PMX_OSEL."PmxWhsCode", "PMX_INVT"."InternalKey", 
				"PMX_INVT"."LogUnitIdentKey", ISNULL( "PMX_LUID"."IsFullPallet",'N') AS "IsFullPallet", 
				"PMX_INVT"."SSCC", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INVT"."ItemTransactionalInfoKey" ,
				PMX_OSWH."SboWhsCode" 
				FROM "PMX_INVT" 
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = "PMX_INVT"."StorLocCode" 
				INNER JOIN PMX_OSWH ON PMX_OSEL."PmxWhsCode" = PMX_OSWH."Code" 
				INNER JOIN "OITM" ON "OITM"."ItemCode" = "PMX_INVT"."ItemCode" 
				LEFT OUTER JOIN "PMX_LUID" ON "PMX_LUID"."InternalKey" = "PMX_INVT"."LogUnitIdentKey" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INVT"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				GROUP BY "PMX_INVT"."ItemCode", "PMX_INVT"."QualityStatusCode", "OITM"."InvntryUom", "PMX_INVT"."StorLocCode", 
				PMX_OSEL."PmxWhsCode", "PMX_INVT"."InternalKey", "PMX_INVT"."LogUnitIdentKey", "PMX_LUID"."IsFullPallet", 
				"PMX_INVT"."SSCC", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INVT"."ItemTransactionalInfoKey",
				PMX_OSWH."SboWhsCode" 

				) AS "TOTAL" 
				LEFT OUTER JOIN ( 

				SELECT SUM ("PMX_INLD"."Quantity") AS "Quantity", "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", 
				"PMX_INLD"."StorLocCode", "PMX_INLD"."LogUnitIdentKey", "PMX_INLD"."ItemTransactionalInfoKey", 
				"PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate" , PMX_OSWH."SboWhsCode" 
				FROM "PMX_INLD" INNER JOIN "PMX_OSWH" ON "PMX_INLD"."PmxWhsCode" = PMX_OSWH."Code"
				LEFT JOIN "PMX_ITRI" ON "PMX_INLD"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				LEFT JOIN @ignoreLockingTable AS IgnoreLocking
				ON IgnoreLocking."DocEntry" = PMX_INLD."BaseEntry"
				AND IgnoreLocking."LineNum" = PMX_INLD."BaseLine"
				AND IgnoreLocking."ObjType" = PMX_INLD."BaseType"
				WHERE IgnoreLocking."DocEntry" IS NULL
				GROUP BY "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", "PMX_INLD"."StorLocCode", "PMX_INLD"."LogUnitIdentKey", 
				"PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INLD"."ItemTransactionalInfoKey", PMX_OSWH."SboWhsCode"

				)AS "LOCK" ON "TOTAL"."ItemCode" = "LOCK"."ItemCode" AND "TOTAL"."QualityStatusCode" = "LOCK"."QualityStatusCode" 
				AND "TOTAL"."SboWhsCode" = "LOCK"."SboWhsCode"
				AND "TOTAL"."StorLocCode" = "LOCK"."StorLocCode" 
				AND ("TOTAL"."LogUnitIdentKey" = "LOCK"."LogUnitIdentKey" OR ("TOTAL"."LogUnitIdentKey"IS NULL AND "LOCK"."LogUnitIdentKey"IS NULL)) 
				AND ("TOTAL"."BatchNumber" = "LOCK"."BatchNumber" OR ("TOTAL"."BatchNumber"IS NULL AND "LOCK"."BatchNumber"IS NULL)) 
				AND ("TOTAL"."BestBeforeDate" = "LOCK"."BestBeforeDate" OR ("TOTAL"."BestBeforeDate"IS NULL AND "LOCK"."BestBeforeDate"IS NULL)) 
				AND ("TOTAL"."ItemTransactionalInfoKey" = "LOCK"."ItemTransactionalInfoKey" OR ("TOTAL"."ItemTransactionalInfoKey"IS NULL 
				AND "LOCK"."ItemTransactionalInfoKey"IS NULL)) 

				INNER JOIN PMX_OSSL ON "TOTAL"."StorLocCode" = PMX_OSSL."Code" 
				) AS FreeStock ON lines."ItemCode" = FreeStock."ItemCode" 
				AND lines."QualityStatusCode" = FreeStock."QualityStatusCode"
				AND lines."WhsCode" = FreeStock."SboWhsCode"
				AND FreeStock."Quantity" < lines."Quantity"
				AND (FreeStock."ItemTransactionalInfoKey" = lines."ItemTransactionalInfoKey" OR (FreeStock."ItemTransactionalInfoKey"IS NULL 
				AND lines."ItemTransactionalInfoKey"IS NULL)) 
				AND (FreeStock."LogUnitIdentKey" = lines."LogUnitIdentKey" OR (FreeStock."LogUnitIdentKey"IS NULL 
				AND lines."LogUnitIdentKey"IS NULL)) 
				AND FreeStock."StorLocCode" = lines."StorLocCode";

				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50052;
					SET @error_message = 'Error for item(s) ' + @errorList + ': The stock for item detail is locked.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			END


			--**********************************************************************************************************
			--If a document line is indicated that it is a logistic carrier then this logistic carrier is put on the
			--logistic unit.
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd = 1) BEGIN
				UPDATE PMX_LUID 
				SET "LogisticCarrierCode" = Lines."ItemCode" FROM @documentLineTable AS Lines INNER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
				WHERE Lines."IsLogisticCarrier" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks on goods receipt (purchase delivery note) and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isPurchaseDeliveryDocument = 1) BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			
			--It is not mandatory anymore that the LUID is filled in when doing a purchase delivery
	--		-- Check if LUID is filled in for the inventory items
	--		SET @errorList = ''
	--		SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
	--		FROM @documentLineTable AS Lines INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
	--		WHERE Items."InvntItem" = 'Y' AND Lines.U_PMX_LUID IS NULL
	--		IF LEN(@errorList) <> 0 BEGIN
	--			SET @error = 50500
	--			SET @error_message = 'Error on line(s): ' + @errorList + ' The LUID must be filled in for inventory items.'
	--			SELECT @error, @error_message
	--			EXEC xp_logevent @error, @error_message, ERROR
	--			RETURN
	--		END

			--Items can be stored on every storage location when doing reception
	--		-- Check if Item can be stored on the location
	--		-- Does the storage location exist, and is it active, and it is a reception dock
	--		SET @errorList = ''
	--		SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
	--		FROM @documentLineTable AS Lines 
	--		LEFT OUTER JOIN PMX_OSSL AS Locations ON Lines."StorLocCode" = Locations."Code"
	--		LEFT OUTER JOIN PMX_OSDO AS Docks ON Docks."Code" = Lines."StorLocCode"
	--		INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
	--		WHERE ((Locations."IsActive" = 'N' OR Locations."IsActive" IS NULL) OR (Docks.IsUnloadingDock <> 'Y' OR Docks.IsUnloadingDock IS NULL)) 
	--		AND Items."InvntItem" = 'Y'
	--		IF LEN(@errorList) <> 0 BEGIN
	--			SET @error = 50513
	--			SET @error_message = 'Error on line(s): ' + @errorList + ' The location is not a dock or is not active.'
	--			SELECT @error, @error_message
	--			EXEC xp_logevent @error, @error_message, ERROR
	--			RETURN
	--		END

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd = 1) BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT OPDN."ObjType", OPDN."DocEntry", OPDN."DocNum", OPDN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OPDN."CardCode", OPDN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OPDN."UserSign", OPDN."CreateDate", OPDN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  OPDN 
						  INNER JOIN @documentLineTable AS Lines ON OPDN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END

			--Check if we need to auto reserve the stock
			DECLARE @tempParamValue nvarchar(20);
			SELECT @tempParamValue = MAX(ISNULL( PMX_PAVA."ParameterValue", PMX_EXPA."DefaultValue"))
            FROM PMX_PROP 
			INNER JOIN PMX_EXPA ON PMX_EXPA."ExtensionCode" = PMX_PROP."ExtensionCode"
			LEFT JOIN PMX_PAVA ON PMX_PAVA."ParameterCode" = PMX_EXPA."Code"
			WHERE PMX_PROP."Code" = 'PDNGEN'
			AND PMX_EXPA."GeneralParameterCode" = 'PD-ARS';

			IF (@tempParamValue = 'true')
			BEGIN

				--insert data into locking table linked to sales order
				INSERT INTO PMX_INLD ( "BaseType","BaseEntry","BaseLine","CardCode","CardName",
				    "ItemCode","Quantity", "QuantityUom2","ItemTransactionalInfoKey","SerialNum",
				    "UserSign","CreateDate",
				    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
				    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
				SELECT RDR1."ObjType",RDR1."DocEntry",RDR1."LineNum",ORDR."CardCode",ORDR."CardName",
				    InvTable."ItemCode",InvTable."InventoryQuantity", InvTable."QuantityUom2",InvTable."ItemTransactionalInfoKey",InvTable."SerialNum",
				    InvTable."CreatedBy",InvTable."CreateDate",
				    InvTable."CreateTime",InvTable."LogUnitIdentKey", InvTable."QualityStatusCode",
                    'RESERVED', 'L' , PMX_OSEL."PmxWhsCode", 0
				FROM @inventoryDetailTable AS InvTable
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = InvTable."StorLocCode"
				LEFT  JOIN PCH1 ON PCH1."DocEntry" = InvTable."BaseEntry" AND PCH1."LineNum" = InvTable."BaseLine" AND PCH1."ObjType" = InvTable."BaseType"
				INNER JOIN POR1 ON ( POR1."DocEntry" = InvTable."BaseEntry" AND POR1."LineNum" = InvTable."BaseLine" AND POR1."ObjType" = InvTable."BaseType" )
				    OR             ( POR1."DocEntry" = PCH1."BaseEntry"     AND POR1."LineNum" = PCH1."BaseLine"     AND POR1."ObjType" = PCH1."BaseType" )
				INNER JOIN ORDR ON ORDR."DocEntry" = POR1."BaseEntry" 
				INNER JOIN RDR1 ON RDR1."DocEntry" = POR1."BaseEntry" AND RDR1."LineNum" = POR1."BaseLine" AND RDR1."ObjType" = POR1."BaseType"
				LEFT  JOIN PMX_DOLL ON PMX_DOLL."DocEntry" = POR1."BaseEntry" AND PMX_DOLL."LineNum" = POR1."BaseLine" AND PMX_DOLL."ObjType" = POR1."BaseType"
				WHERE InvTable."BaseEntry" IS NOT NULL AND InvTable."BaseLine" IS NOT NULL AND InvTable."BaseType" IS NOT NULL
				AND RDR1."LineStatus" = 'O'
				AND (PMX_DOLL."DocEntry" IS NULL OR PMX_DOLL."LockedQuantity" = 0);

				--insert data into locking table linked to reserve invoice
				INSERT INTO PMX_INLD ("BaseType","BaseEntry","BaseLine","CardCode","CardName",
				    "ItemCode","Quantity", "QuantityUom2", "ItemTransactionalInfoKey","SerialNum",
				    "UserSign","CreateDate",
				    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
				    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
				SELECT INV1."ObjType",INV1."DocEntry",INV1."LineNum",ORDR."CardCode",ORDR."CardName",
				    InvTable."ItemCode",InvTable."InventoryQuantity", InvTable."QuantityUom2",InvTable."ItemTransactionalInfoKey",InvTable."SerialNum",
				    InvTable."CreatedBy",InvTable."CreateDate",
				    InvTable."CreateTime",InvTable."LogUnitIdentKey", InvTable."QualityStatusCode",
                    'RESERVED', 'L' , PMX_OSEL."PmxWhsCode", 0
				FROM @inventoryDetailTable AS InvTable
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = InvTable."StorLocCode"
				LEFT  JOIN PCH1 ON PCH1."DocEntry" = InvTable."BaseEntry" AND PCH1."LineNum" = InvTable."BaseLine" AND PCH1."ObjType" = InvTable."BaseType"
				INNER JOIN POR1 ON ( POR1."DocEntry" = InvTable."BaseEntry" AND POR1."LineNum" = InvTable."BaseLine" AND POR1."ObjType" = InvTable."BaseType" )
				    OR             ( POR1."DocEntry" = PCH1."BaseEntry"     AND POR1."LineNum" = PCH1."BaseLine"     AND POR1."ObjType" = PCH1."BaseType" )
				INNER JOIN ORDR ON ORDR."DocEntry" = POR1."BaseEntry" 
				INNER JOIN RDR1 ON RDR1."DocEntry" = POR1."BaseEntry" AND RDR1."LineNum" = POR1."BaseLine" AND RDR1."ObjType" = POR1."BaseType"
				INNER JOIN INV1 ON RDR1."DocEntry" = INV1."BaseEntry" AND RDR1."LineNum" = INV1."BaseLine" AND RDR1."ObjType" = INV1."BaseType"
				INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry" 
				LEFT  JOIN PMX_DOLL ON PMX_DOLL."DocEntry" = INV1."DocEntry" AND PMX_DOLL."LineNum" = INV1."LineNum" AND PMX_DOLL."ObjType" = INV1."ObjType"
				WHERE InvTable."BaseEntry" IS NOT NULL AND InvTable."BaseLine" IS NOT NULL AND InvTable."BaseType" IS NOT NULL
				AND RDR1."LineStatus" = 'C'
				AND (PMX_DOLL."DocEntry" IS NULL OR PMX_DOLL."LockedQuantity" = 0)
				AND OINV."isIns" = 'Y';

			END

			


			--Close container (lines) if needed
			UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			FROM @documentLineTable AS Lines
			INNER JOIN PMX_COLI
			ON Lines."BaseEntry" = PMX_COLI."BaseEntry"
			AND Lines."BaseLine" = PMX_COLI."BaseLine"
			AND Lines."BaseType" = PMX_COLI."BaseType"
			INNER JOIN POR1
			ON Lines."BaseEntry" = POR1."DocEntry"
			AND Lines."BaseLine" = POR1."LineNum"
			AND Lines."BaseType" = POR1."ObjType"
			WHERE PMX_COLI."LineStatus" <> 'C'
			AND POR1."LineStatus" = 'C';


			--Close header is no open lines are found
			UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
			PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
			PMX_COHE."UserSign" = (SELECT MAX("CreatedBy") FROM @documentLineTable)
			WHERE PMX_COHE."DocStatus" = 'O'
			AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

		END

		--**********************************************************************************************************
		--Do checks on delivery (sales delivery note) and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isSalesDeliveryDocument = 1) BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			-- Check if the quality statuses on the sales delivery note are statuses that are allowed to be 
			-- to shipped (For sample orders no check is done)
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM @documentLineTable AS Lines 
			INNER JOIN DLN1 ON Lines."DocEntry" = DLN1."DocEntry"
			AND Lines."LineNum" = DLN1."LineNum"
			INNER JOIN PMX_QYST AS QualityStatus ON Lines."QualityStatusCode" = QualityStatus."Code"
			LEFT JOIN PMX_DLPL ON PMX_DLPL."SalesDeliveryNoteLineDocEntry" = Lines."DocEntry"
			AND PMX_DLPL."SalesDeliveryNoteLineLineNum" = Lines."LineNum"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."DocEntry" = PMX_DLPL."PickListLineDocEntry"
			AND PMX_PLLI."LineNum" = PMX_DLPL."PickListLineLineNum"
			WHERE ISNULL( PMX_PLLI."IsSampleOrder", 'N') = 'N'
			AND NOT ( (ISNULL( NULLIF(DLN1.U_PMX_SQOP,''), 'RELEASED') = 'RELEASED' AND QualityStatus."CanBeShipped" = 'Y' ) 
			OR (ISNULL( NULLIF(DLN1.U_PMX_SQOP, ''), 'RELEASED') = 'CAN_USE_SUQ' AND (QualityStatus."CanBeShipped" = 'Y' OR QualityStatus."CanBeShippedUnderQuarantine" = 'Y')  )
			OR (ISNULL( NULLIF(DLN1.U_PMX_SQOP, ''), 'RELEASED') = 'MUST_USE_SUQ' AND QualityStatus."CanBeShippedUnderQuarantine" = 'Y'  ));
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50700;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The quality status on these lines is not allowed to be shipped.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--The quantities must be made negative because the stock leaves the company
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd = 1) BEGIN

				INSERT INTO @inventoryDetailTable
					SELECT ODLN."ObjType", ODLN."DocEntry", ODLN."DocNum", ODLN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine", 
						  ODLN."CardCode", ODLN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection , Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ODLN."UserSign", ODLN."CreateDate", ODLN."DocTime", PMX_LUID.SSCC,
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  ODLN 
						  INNER JOIN @documentLineTable AS Lines ON ODLN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';

			END 

		END
			
		IF ((@isSalesDeliveryDocument = 1 OR @isStockTransfer = 1) AND @transaction_type = N'P') BEGIN
			BEGIN TRY
			
				-- This gets executed after the creation of a sales delivery note when loading a truck
				-- !! This routine does closing of : PickList(line)s, PickList Proposal(line)s, Route(line)s and Waves !!
				-- AND links serial numbers to the correct document.

				--Convert the list of key columns to an integer key
				SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

				--Adjust temp locking: remove delivered quantity
				DECLARE @tempLockItemCode nvarchar (20);
				DECLARE @tempLockDestLUID int;
				DECLARE @tempLockQualityStatusCode nvarchar (8);
				DECLARE @tempLockItri int;
				DECLARE @tempLockOriginalLUID int;
				DECLARE @tempLockDeliveredQuantity numeric(19,6);


				DECLARE deliveredStockCursor CURSOR LOCAL FOR 
				SELECT     dbo.PMX_MVLI."ItemCode", dbo.PMX_MVLI."DestLogUnitIdentKey", dbo.PMX_MVLI."DestQualityStatusCode", dbo.PMX_MVLI."ItemTransactionalInfoKey", 
									  dbo.PMX_PLPL."LogUnitIdentKey" AS OriginalLUID, ISNULL( dbo.PMX_DLPL."Quantity", PMX_MVLI."Quantity") * ISNULL( dbo.PMX_DLPL."QuantityPerUom", PMX_MVLI."QuantityPerUom") AS DeliveredQuantity
				FROM         dbo.PMX_DLPL INNER JOIN
									  dbo.PMX_MVLI ON dbo.PMX_DLPL."MoveLineDocEntry" = dbo.PMX_MVLI."DocEntry" AND 
									  dbo.PMX_DLPL."MoveLineLineNum" = dbo.PMX_MVLI."LineNum" INNER JOIN
									  dbo.PMX_PLLI ON dbo.PMX_DLPL."PickListLineDocEntry" = dbo.PMX_PLLI."DocEntry" AND 
									  dbo.PMX_DLPL."PickListLineLineNum" = dbo.PMX_PLLI."LineNum" INNER JOIN
									  dbo.PMX_PLPL ON dbo.PMX_PLLI."BaseEntry" = dbo.PMX_PLPL."DocEntry" AND dbo.PMX_PLLI."BaseLine" = dbo.PMX_PLPL."LineNum"
										AND (dbo.PMX_PLLI."BaseType" = N'PMX_PLPH')
				WHERE   (@isSalesDeliveryDocument=1 AND dbo.PMX_DLPL."SalesDeliveryNoteLineDocEntry" = @keyAsInteger) 
				      OR (@isStockTransfer=1 AND dbo.PMX_DLPL."StockTransferDocEntry" = @keyAsInteger);
				

				-- Reduce temp locking by delivered quantity
				OPEN deliveredStockCursor;
				FETCH NEXT FROM deliveredStockCursor INTO @tempLockItemCode,@tempLockDestLUID,@tempLockQualityStatusCode,@tempLockItri, @tempLockOriginalLUID, @tempLockDeliveredQuantity;
				WHILE @@FETCH_STATUS >= 0 BEGIN

					DECLARE @tempLockInternalKey int;
					DECLARE @tempLockQuantityLocked	 numeric (19,6);					
				
					--Get all temp lockings for stock
					DECLARE tempLockCursor CURSOR LOCAL FOR 
					SELECT     dbo.PMX_TINL."InternalKey", dbo.PMX_TINL."Quantity"
					FROM         dbo.PMX_TINL 
					WHERE     ISNULL( dbo.PMX_TINL."ItemTransactionalInfoKey", 0) = ISNULL( @tempLockItri,0)
					AND ISNULL( dbo.PMX_TINL."LogUnitIdentKey", 0) = ISNULL( @tempLockOriginalLUID,0)
					AND dbo.PMX_TINL."ItemCode" = @tempLockItemCode
					AND dbo.PMX_TINL."QualityStatusCode" = @tempLockQualityStatusCode;

					OPEN tempLockCursor;
					FETCH NEXT FROM tempLockCursor INTO @tempLockInternalKey,@tempLockQuantityLocked;
					WHILE @@FETCH_STATUS >= 0 BEGIN
						-- Reduce temp locking
						IF 	@tempLockDeliveredQuantity > 0
						BEGIN							
							IF (@tempLockQuantityLocked > @tempLockDeliveredQuantity)
							BEGIN
								--reduce quantity in temp lock
								UPDATE PMX_TINL SET "Quantity" = "Quantity" - @tempLockDeliveredQuantity, "QuantityUom2" = "QuantityUom2" - (("Quantity" - @tempLockDeliveredQuantity)* OITM."U_PMX_DQUM")
								FROM PMX_TINL
								INNER JOIN OITM ON OITM."ItemCode" = PMX_TINL."ItemCode"
								WHERE PMX_TINL."InternalKey" = @tempLockInternalKey;
								SET @tempLockDeliveredQuantity = 0;
							END
							ELSE
							BEGIN
								--we delivered more than what is on the temp lock line, so delete temp lock line and reduce quantity delivered
								SET @tempLockDeliveredQuantity = @tempLockDeliveredQuantity - @tempLockQuantityLocked
								DELETE FROM PMX_TINL
								WHERE PMX_TINL."InternalKey" = @tempLockInternalKey;
							END
						END

						FETCH NEXT FROM tempLockCursor INTO @tempLockInternalKey,@tempLockQuantityLocked;
					END
					CLOSE tempLockCursor ;
					DEALLOCATE tempLockCursor

					FETCH NEXT FROM deliveredStockCursor INTO @tempLockItemCode,@tempLockDestLUID,@tempLockQualityStatusCode,@tempLockItri, @tempLockOriginalLUID, @tempLockDeliveredQuantity;
				END
				CLOSE deliveredStockCursor ;
				DEALLOCATE deliveredStockCursor

				-- Set the IsInterfaced flag on delivery note if any of the BaseDoc ORDR's has the flag
				UPDATE ODLN SET U_PMX_INFC = (
					SELECT CASE WHEN COUNT(ORDR."DocEntry") + COUNT(OINV."DocEntry") > 0 THEN 'Y' ELSE 'N' END 
					FROM DLN1
					LEFT JOIN ORDR ON ORDR."ObjType" = DLN1."BaseType" AND ORDR."DocEntry" = DLN1."BaseEntry" AND ORDR.U_PMX_INFC = 'Y'
					LEFT JOIN OINV ON OINV."ObjType" = DLN1."BaseType" AND OINV."DocEntry" = DLN1."BaseEntry" AND OINV.U_PMX_INFC = 'Y'
					WHERE DLN1."DocEntry" = ODLN."DocEntry" )
				WHERE ODLN."DocEntry" = @keyAsInteger;
				
				-- Set the IsInterfaced flag on stock transfer if any of the BaseDoc ORDR's has the flag
				UPDATE OWTR SET U_PMX_INFC = (
					SELECT CASE WHEN COUNT(OWTQ."DocEntry") > 0 THEN 'Y' ELSE 'N' END 
					FROM WTR1
					LEFT JOIN OWTQ ON OWTQ."ObjType" = WTR1."BaseType" AND OWTQ."DocEntry" = WTR1."BaseEntry" AND OWTQ.U_PMX_INFC = 'Y'
					WHERE WTR1."DocEntry" = OWTR."DocEntry" )
				WHERE OWTR."DocEntry" = @keyAsInteger;


--				--Adjust serial number document link (Is now done in code instead of SP)
--				UPDATE PMX_SELD
--				SET PMX_SELD."BaseType" = DLN1."ObjType",
--					PMX_SELD."BaseEntry" = DLN1."DocEntry",
--					PMX_SELD."BaseLine" = DLN1."LineNum",
--					PMX_SELD."Version" = PMX_SELD."Version" + 1
--				FROM DLN1
--				INNER JOIN PMX_DLPL ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
--					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
--				INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
--					AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
--				INNER JOIN PMX_SELD ON PMX_SELD."BaseEntry" = PMX_PLLI."DocEntry"
--					AND PMX_SELD."BaseLine" = PMX_PLLI."LineNum"
--				AND PMX_SELD."BaseType" = 'PMX_PLHE'
--				WHERE DLN1."DocEntry" = @keyAsInteger
--				AND PMX_PLLI."LineStatus" <> 'C'

				--Adjust serial number in stock (Only for delivery, because stock transfer still has the serial number in stock
				IF @isSalesDeliveryDocument = 1 BEGIN
				UPDATE PMX_SENU
				SET PMX_SENU."InStock" = 'N',
					PMX_SENU."Version" = PMX_SENU."Version" + 1
				FROM PMX_SENU
				INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
				WHERE PMX_SELD."BaseEntry" = @keyAsInteger
				AND PMX_SELD."BaseType" = '15';  --Sales delivery
				END
				



				IF @isSalesDeliveryDocument = 1 BEGIN
					-- Decrease open quantity of PLLI's with MVLI quantity
					UPDATE PMX_PLLI
					SET PMX_PLLI."OpenQty" = PMX_PLLI."OpenQty" - DeliveredQuantity."Quantity",
						PMX_PLLI."OpenQtyUom2" = PMX_PLLI."OpenQtyUom2" - DeliveredQuantity."QuantityUom2",
						PMX_PLLI."Version" = PMX_PLLI."Version" + 1
					FROM PMX_PLLI
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( ISNULL( PMX_DLPL."QuantityUom2", PMX_MVLI."QuantityUom2")) AS "QuantityUom2", 
						SUM( ISNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity")) AS "Quantity" 
						FROM DLN1
						INNER JOIN PMX_DLPL ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
							AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						LEFT JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE DLN1."DocEntry" = @keyAsInteger
						AND DLN1."ObjType" = @object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_PLLI."DocEntry"
						AND DeliveredQuantity."LineNum" = PMX_PLLI."LineNum";
						
					--Decrease lockings
					UPDATE PMX_INLD  
					SET	PMX_INLD."Quantity" = PMX_INLD."Quantity" - DeliveredQuantity."Quantity",
						PMX_INLD."QuantityUom2" = PMX_INLD."QuantityUom2" - DeliveredQuantity."QuantityUom2",
						PMX_INLD."Version" = PMX_INLD."Version" + 1
					FROM PMX_INLD
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( ISNULL( PMX_DLPL."QuantityUom2" * PMX_DLPL."QuantityPerUom", PMX_MVLI."QuantityUom2"* PMX_MVLI."QuantityPerUom")) AS "QuantityUom2", 
						SUM( ISNULL( PMX_DLPL."Quantity" * PMX_DLPL."QuantityPerUom" , PMX_MVLI."Quantity" * PMX_MVLI."QuantityPerUom")) AS "Quantity" 
						FROM DLN1
						INNER JOIN PMX_DLPL ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
							AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						INNER JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE DLN1."DocEntry" = @keyAsInteger
						AND DLN1."ObjType" = @object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_INLD."BaseEntry"
						AND DeliveredQuantity."LineNum" = PMX_INLD."BaseLine"
						AND PMX_INLD."BaseType" = 'PMX_PLHE';
												
				END ELSE IF @isStockTransfer = 1 BEGIN
					-- Decrease open quantity of PLLI's with MVLI quantity
					UPDATE PMX_PLLI
					SET PMX_PLLI."OpenQty" = PMX_PLLI."OpenQty" - DeliveredQuantity."Quantity",
						PMX_PLLI."OpenQtyUom2" = PMX_PLLI."OpenQtyUom2" - DeliveredQuantity."QuantityUom2",
						PMX_PLLI."Version" = PMX_PLLI."Version" + 1
					FROM PMX_PLLI
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( ISNULL( PMX_DLPL."QuantityUom2", PMX_MVLI."QuantityUom2")) AS "QuantityUom2", 
						SUM( ISNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity")) AS "Quantity" 
						FROM WTR1
						INNER JOIN PMX_DLPL ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
							AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						LEFT JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE WTR1."DocEntry" = @keyAsInteger
						AND WTR1."ObjType" = @object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_PLLI."DocEntry"
						AND DeliveredQuantity."LineNum" = PMX_PLLI."LineNum";
						
					--Decrease lockings
					UPDATE PMX_INLD  
					SET	PMX_INLD."Quantity" = PMX_INLD."Quantity" - DeliveredQuantity."Quantity",
						PMX_INLD."QuantityUom2" = PMX_INLD."QuantityUom2" - DeliveredQuantity."QuantityUom2",
						PMX_INLD."Version" = PMX_INLD."Version" + 1
					FROM PMX_INLD
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( ISNULL( PMX_DLPL."QuantityUom2" * PMX_DLPL."QuantityPerUom", PMX_MVLI."QuantityUom2"* PMX_MVLI."QuantityPerUom")) AS "QuantityUom2", 
						SUM( ISNULL( PMX_DLPL."Quantity" * PMX_DLPL."QuantityPerUom" , PMX_MVLI."Quantity" * PMX_MVLI."QuantityPerUom")) AS "Quantity" 
						FROM WTR1
						INNER JOIN PMX_DLPL ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
							AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						INNER JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE WTR1."DocEntry" = @keyAsInteger
						AND WTR1."ObjType" = @object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_INLD."BaseEntry"
						AND DeliveredQuantity."LineNum" = PMX_INLD."BaseLine"
						AND PMX_INLD."BaseType" = 'PMX_PLHE';
					
				END
				
								
				-- Get the keys for the PLLI's to be closed
				DECLARE @SDLinesTable TABLE ( TmpDocEntry INT NOT NULL, TmpLineNum INT NOT NULL );

				
				INSERT INTO @SDLinesTable
				SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum"
				FROM PMX_DLPL
				INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
					AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
				WHERE (( @isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = @keyAsInteger ) 
				      OR ( @isStockTransfer = 1 AND WTR1."DocEntry" = @keyAsInteger ))
				AND PMX_PLLI."LineStatus" <> 'C'
				--AND ( PMX_PLLI."OpenQty" - DLN1."Quantity" ) < 0.000001
				AND ( PMX_PLLI."OpenQty") < 0.000001;
			

				
				-- Delete inventory locking for these picklist lines
				DELETE FROM PMX_INLD
				FROM @SDLinesTable
				INNER JOIN PMX_INLD ON 'PMX_PLHE' = PMX_INLD."BaseType"
					AND TmpDocEntry = PMX_INLD."BaseEntry"
					AND TmpLineNum = PMX_INLD."BaseLine"

				-- Set link status serial number links
				UPDATE PMX_SELD SET "LinkStatus" = 'C'
				FROM @SDLinesTable
				INNER JOIN PMX_SELD ON 'PMX_PLHE' = PMX_SELD."BaseType"
					AND TmpDocEntry = PMX_SELD."BaseEntry"
					AND TmpLineNum = PMX_SELD."BaseLine";



				--Update PMX_DOLL in case of partial delivery
				DECLARE @proposalQuantity2  numeric(19, 6);
				DECLARE @proposalQuantityUOM22  numeric(19, 6);
				DECLARE @proposalBaseEntry2  int;
				DECLARE @proposalBaseLineNum2  int;
				DECLARE @proposalBatchNum2  NVARCHAR(36);
				DECLARE @proposalBaseType2  NVARCHAR(20);
				DECLARE @proposalEntry2  int;
				DECLARE @proposalLineNum2  int;


				DECLARE lockCursor2 CURSOR LOCAL FOR 

--				SELECT PMX_PLLI."Quantity", PMX_PLPL."BaseEntry", PMX_PLPL."BaseLine", PMX_PLPL."BaseType", 
--				PMX_PLPL."QuantityUom2", "BatchNumber", PMX_PLPL."DocEntry", PMX_PLPL."LineNum"				
--				FROM PMX_PLPL 
--				INNER JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry"
--				AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum"
--				AND PMX_PLLI."LineStatus" = 'O'
--				INNER JOIN PMX_DLPL ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
--				AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
--				LEFT JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
--										
--				WHERE PMX_PLLI."DocEntry" IN (SELECT TmpDocEntry  FROM @SDLinesTable ) 
--				AND PMX_PLLI."LineStatus" = 'O'

				SELECT ISNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity"), PMX_PLPL."BaseEntry", PMX_PLPL."BaseLine", PMX_PLPL."BaseType", 
				ISNULL( PMX_DLPL."QuantityUom2", PMX_MVLI."QuantityUom2"), "BatchNumber", PMX_PLPL."DocEntry", PMX_PLPL."LineNum"	
				FROM PMX_DLPL
				INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
					AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
				INNER JOIN PMX_PLPL ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry"
				AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum"
				AND PMX_PLLI."LineStatus" = 'O'
				LEFT JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
					AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
				LEFT JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
				WHERE PMX_PLLI."LineStatus" <> 'C' AND
				 (( @isSalesDeliveryDocument=1 AND DLN1."DocEntry" = @keyAsInteger ) 
				      OR (@isStockTransfer=1 AND WTR1."DocEntry" = @keyAsInteger ));
				
				OPEN lockCursor2;
				FETCH NEXT FROM lockCursor2 INTO @proposalQuantity2,@proposalBaseEntry2,@proposalBaseLineNum2,
				@proposalBaseType2, @proposalQuantityUOM22, @proposalBatchNum2, @proposalEntry2, @proposalLineNum2;
				WHILE @@FETCH_STATUS >= 0 BEGIN
					-- Reduce document locking for all lines
					UPDATE PMX_DOLL 
					SET PMX_DOLL."LockedQuantity" = PMX_DOLL."LockedQuantity" - @proposalQuantity2,
					PMX_DOLL."LockedQuantityUom2" = PMX_DOLL."LockedQuantityUom2" - @proposalQuantityUOM22,
					PMX_DOLL."UpdateDate" = @updateDate,
					PMX_DOLL."UpdateTime" = @updateTime,
					PMX_DOLL."Version" = PMX_DOLL."Version" + 1
					WHERE "DocEntry" = @proposalBaseEntry2 AND "LineNum" = @proposalBaseLineNum2
					AND "ObjType" = @proposalBaseType2 
					AND ("BatchNum" = @proposalBatchNum2 OR "BatchNum" IS NULL);
					-- AND @proposalBaseType = '17' -- 2010-08-26  KrisM  Allow all types

					UPDATE PMX_PLPL
					SET PMX_PLPL."OpenQty" = PMX_PLPL."OpenQty" - @proposalQuantity2,
					PMX_PLPL."Version" = PMX_PLPL."Version" + 1
					WHERE "DocEntry" = @proposalEntry2 AND "LineNum" = @proposalLineNum2;

					FETCH NEXT FROM lockCursor2 INTO @proposalQuantity2,@proposalBaseEntry2,@proposalBaseLineNum2,
					@proposalBaseType2, @proposalQuantityUOM22, @proposalBatchNum2, @proposalEntry2, @proposalLineNum2;
				END
				CLOSE lockCursor2;
				DEALLOCATE lockCursor2


				-- Delete document locking for lines that have NO remaining "LockedQuantity"
				DELETE FROM PMX_DOLL
				WHERE PMX_DOLL."LockedQuantity" < 0.000001;



				-- Close picklist lines
				-- {PmxDocumentProvider.CloseLine(...)}
				-- {PmxPickListProvider.CloseLine(...)}
				UPDATE PMX_PLLI
				SET PMX_PLLI."OpenQty" = 0,
					PMX_PLLI."OpenQtyUom2" = PMX_PLLI."OpenQtyUom2" * 0,  -- Set to 0.0 except when NULL
					PMX_PLLI.PickListLineStatus = CASE PMX_PLLI."PickListLineStatus" WHEN 'F' THEN 'F' ELSE 'C' END,  -- Set to 'C' except when 'F'
					PMX_PLLI."LineStatus" = 'C',
					PMX_PLLI."Version" = PMX_PLLI."Version" + 1
				FROM @SDLinesTable
				INNER JOIN PMX_PLLI ON TmpDocEntry = PMX_PLLI."DocEntry"
					AND TmpLineNum = PMX_PLLI."LineNum";

				-- Empty @SDLinesTable
				DELETE FROM @SDLinesTable;
				
				-- Fill @SDLinesTable with all DocEntries for non-closed picklists with all lines closed
				-- Using TmpLineNum = NULL
				INSERT INTO @SDLinesTable
				SELECT DISTINCT PMX_DLPL."PickListLineDocEntry", -1
				FROM PMX_DLPL
				INNER JOIN PMX_PLHE ON PMX_DLPL."PickListLineDocEntry" = PMX_PLHE."DocEntry"
				LEFT JOIN PMX_PLLI ON PMX_PLHE."DocEntry" = PMX_PLLI."DocEntry"
					AND PMX_PLLI."LineStatus" <> 'C'
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"

				WHERE (( @isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = @keyAsInteger ) 
				      OR ( @isStockTransfer = 1 AND WTR1."DocEntry" = @keyAsInteger ))
				AND PMX_PLHE."DocStatus" <> 'C'
				AND PMX_PLLI."DocEntry" IS NULL;
					
				IF @@ROWCOUNT > 0  -- Found picklists to close ?
				BEGIN

					-- Close the picklists who have all lines closed
					UPDATE PMX_PLHE
					SET PMX_PLHE."PickListStatus" = CASE PMX_PLHE."PickListStatus" WHEN 'F' THEN 'F' ELSE 'C' END,  -- Set to 'C' except when 'F'
						PMX_PLHE."DocStatus" = 'C',
						PMX_PLHE."UpdateDate" = @updateDate,
						PMX_PLHE."UpdateTime" = @updateTime,
						PMX_PLHE."Version" = PMX_PLHE."Version" + 1
					WHERE PMX_PLHE."DocEntry" IN ( SELECT TmpDocEntry  FROM @SDLinesTable );

					-- Close the wave if the wave only has closed picklists
					UPDATE PMX_WAVE
					SET PMX_WAVE."DocStatus" = 'C',
						PMX_WAVE."UpdateDate" = @updateDate,
						PMX_WAVE."UpdateTime" = @updateTime,
						PMX_WAVE."Version" = PMX_WAVE."Version" + 1
					FROM PMX_PLHE ref
					INNER JOIN PMX_WAVE ON ref."WaveKey" = PMX_WAVE."InternalKey"
					LEFT JOIN PMX_PLHE tst ON PMX_WAVE."InternalKey" = tst."WaveKey"
						AND tst."DocStatus" <> 'C'
					WHERE ref."DocEntry" IN ( SELECT TmpDocEntry  FROM @SDLinesTable )
					AND PMX_WAVE."DocStatus" <> 'C'
					AND tst."DocEntry" IS NULL;
				
					-- Fill @SDLinesTable with all the lines of non-closed picklist proposals with all picklists closed
					-- (PickLists proposals to be closed)
					INSERT INTO @SDLinesTable
					SELECT DISTINCT PMX_PLPH."DocEntry", 1
					FROM PMX_PLHE ref
					INNER JOIN PMX_PLPH ON ref."PickListProposalEntry" = PMX_PLPH."DocEntry"
					LEFT JOIN PMX_PLHE tst ON PMX_PLPH."DocEntry" = tst."PickListProposalEntry"
						AND tst."DocStatus" <> 'C'
					WHERE ref."DocEntry" IN ( SELECT TmpDocEntry  FROM @SDLinesTable  WHERE TmpLineNum = -1 )
					AND PMX_PLPH."DocStatus" <> 'C'
					AND tst."DocEntry" IS NULL;

					IF @@ROWCOUNT > 0  -- Found picklist proposal(line)s to close ?
					BEGIN

						-- Remove PickList entries from @SDLinesTable so only PickList proposal lines remain
						DELETE FROM @SDLinesTable  WHERE TmpLineNum = -1;

						-- Delete inventory locking for the picklist proposal lines
						--  do not care about "LineNum", all lines need to be cleaned up
						DELETE FROM PMX_INLD
						FROM @SDLinesTable
						INNER JOIN PMX_INLD ON 'PMX_PLPH' = PMX_INLD."BaseType"
							AND TmpDocEntry = PMX_INLD."BaseEntry"
						
						DECLARE @proposalQuantity  numeric(19, 6);
						DECLARE @proposalQuantityUOM2  numeric(19, 6);
						DECLARE @proposalBaseEntry  int;
						DECLARE @proposalBaseLineNum  int;
						DECLARE @proposalBatchNum  NVARCHAR(36);
						DECLARE @proposalBaseType  NVARCHAR(20);

						DECLARE lockCursor CURSOR LOCAL FOR SELECT "OpenQty", "BaseEntry", "BaseLine", "BaseType", "QuantityUom2", "BatchNumber" 
						FROM PMX_PLPL LEFT JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
						WHERE "DocEntry" IN (SELECT TmpDocEntry  FROM @SDLinesTable) AND "LineStatus" = 'O';
						OPEN lockCursor;
						FETCH NEXT FROM lockCursor INTO @proposalQuantity,@proposalBaseEntry,@proposalBaseLineNum,@proposalBaseType, @proposalQuantityUOM2, @proposalBatchNum
						WHILE @@FETCH_STATUS >= 0 BEGIN
							-- Reduce document locking for all lines
							UPDATE PMX_DOLL 
							SET PMX_DOLL."LockedQuantity" = PMX_DOLL."LockedQuantity" - @proposalQuantity,
							PMX_DOLL."LockedQuantityUom2" = PMX_DOLL."LockedQuantityUom2" - @proposalQuantityUOM2,
							PMX_DOLL."UpdateDate" = @updateDate,
							PMX_DOLL."UpdateTime" = @updateTime,
							PMX_DOLL."Version" = PMX_DOLL."Version" + 1
							WHERE "DocEntry" = @proposalBaseEntry AND "LineNum" = @proposalBaseLineNum 
							AND "ObjType" = @proposalBaseType 
							AND ("BatchNum" = @proposalBatchNum OR "BatchNum" IS NULL);
							-- AND @proposalBaseType = '17' -- 2010-08-26  KrisM  Allow all types

							FETCH NEXT FROM lockCursor INTO @proposalQuantity,@proposalBaseEntry,@proposalBaseLineNum,@proposalBaseType, @proposalQuantityUOM2, @proposalBatchNum;
						END
						CLOSE lockCursor;
						DEALLOCATE lockCursor


						-- Delete document locking for lines that have NO remaining "LockedQuantity"
						DELETE FROM PMX_DOLL
						WHERE PMX_DOLL."LockedQuantity" < 0.000001;
						
						-- Close all proposal lines
						-- {PmxDocumentProvider.CloseLine(...)}
						UPDATE PMX_PLPL
						SET PMX_PLPL."OpenQty" = 0,
							PMX_PLPL."OpenQtyUom2" = PMX_PLPL."OpenQtyUom2" * 0,  -- Set to 0.0 except when NULL
							PMX_PLPL."LineStatus" = 'C',
							PMX_PLPL."Version" = PMX_PLPL."Version" + 1
						WHERE PMX_PLPL."LineStatus" <> 'C'
						AND PMX_PLPL."DocEntry" IN ( SELECT TmpDocEntry FROM @SDLinesTable );

						--Set back the temp lockings to the locking table
						--The LUID is set to NULL because it is impossible to retrieve the new luid
						INSERT INTO PMX_INLD ( "Canceled", "UserSign", "CreateDate", "CreateTime", "UpdateDate", "UpdateTime", "Version", "BaseType", "BaseEntry", "BaseLine", "ItemCode", "Quantity", 
											  "SerialNum", "LogUnitIdentKey", "InvLockStatusCode", "QualityStatusCode", "StorLocCode", "InvLockLevel", "Uom2", "QuantityUom2", "ItemTransactionalInfoKey", 
											  "PmxWhsCode", "CardCode", "CardName")

						SELECT    'N', dbo.PMX_TINL."InternalKey", dbo.PMX_TINL."CreateDate",
						dbo.PMX_TINL."CreateTime", @updateDate, @updateTime, 0, dbo.PMX_TINL."BaseType", dbo.PMX_TINL."BaseEntry", dbo.PMX_TINL."BaseLine", dbo.PMX_TINL."ItemCode", dbo.PMX_TINL."Quantity", 
						dbo.PMX_TINL."SerialNum", NULL, dbo.PMX_TINL."InvLockStatusCode", 
						dbo.PMX_TINL."QualityStatusCode", NULL, 'B', NULL, NULL, dbo.PMX_TINL."ItemTransactionalInfoKey", 
						dbo.PMX_TINL."PmxWhsCode", dbo.PMX_TINL."CardCode", dbo.PMX_TINL."CardName"

						FROM dbo.PMX_TINL 
						WHERE PMX_TINL."LockedByType" = 'PMX_PLPH'
						AND PMX_TINL."LockedByEntry" IN (SELECT TmpDocEntry FROM @SDLinesTable );


						--Remove temp lockings from table
						DELETE FROM PMX_TINL
						WHERE PMX_TINL."LockedByType" = 'PMX_PLPH'
						AND PMX_TINL."LockedByEntry" IN (SELECT TmpDocEntry FROM @SDLinesTable );


						
						-- Close the proposals
						UPDATE PMX_PLPH
						SET PMX_PLPH."DocStatus" = 'C',
							PMX_PLPH."UpdateDate" = @updateDate,
							PMX_PLPH."UpdateTime" = @updateTime,
							PMX_PLPH."Version" = PMX_PLPH."Version" + 1
						WHERE PMX_PLPH."DocStatus" <> 'C'
						AND PMX_PLPH."DocEntry" IN ( SELECT TmpDocEntry FROM @SDLinesTable );
						
						-- Close the route lines if all picklists are closed for that route
						UPDATE PMX_RTLI
						SET PMX_RTLI."LineStatus" = 'C',
							PMX_RTLI."Version" = PMX_RTLI."Version" + 1
						FROM @SDLinesTable
						INNER JOIN PMX_PLPH ref ON TmpDocEntry = ref."DocEntry"
						INNER JOIN PMX_RTHE ON ref."RouteLineDocEntry" = PMX_RTHE."DocEntry"
						INNER JOIN PMX_RTLI ON PMX_RTHE."DocEntry" = PMX_RTLI."DocEntry"
						LEFT JOIN PMX_PLPH tst ON PMX_RTLI."DocEntry" = tst."RouteLineDocEntry"
							AND tst."DocStatus" <> 'C'
						WHERE PMX_RTLI."LineStatus" <> 'C'
						AND tst."DocEntry" IS NULL;

						-- Close the routes if all route lines are closed
						UPDATE PMX_RTHE
						SET PMX_RTHE."RouteStatus" = 'C',
							PMX_RTHE."DocStatus" = 'C',
							PMX_RTHE."UpdateDate" = @updateDate,
							PMX_RTHE."UpdateTime" = @updateTime,
							PMX_RTHE."Version" = PMX_RTHE."Version" + 1
						FROM @SDLinesTable
						INNER JOIN PMX_PLPH ref ON TmpDocEntry = ref."DocEntry"
						INNER JOIN PMX_RTHE ON ref."RouteLineDocEntry" = PMX_RTHE."DocEntry"
						LEFT JOIN PMX_RTLI ON PMX_RTHE."DocEntry" = PMX_RTLI."DocEntry"
							AND PMX_RTLI."LineStatus" <> 'C'
						WHERE PMX_RTHE."DocStatus" <> 'C'
						AND PMX_RTLI."DocEntry" IS NULL

						--Close container (lines) if needed
						UPDATE PMX_COLI SET "LineStatus" = 'C', PMX_COLI."Version" = PMX_COLI."Version" + 1 
						FROM PMX_COLI
						INNER JOIN PMX_RTHE
						ON PMX_COLI."BaseEntry" = PMX_RTHE."DocEntry"
						AND PMX_COLI."BaseType" = 'PMX_RTHE'
						WHERE PMX_COLI."LineStatus" <> 'C'
						AND PMX_RTHE."DocStatus" = 'C';



					END  -- Found picklist proposals to close ?
					
				END  -- Found picklists to close ?
				

				--If pick list is not closed, change status to 'Partially delivered'
				UPDATE PMX_PLHE SET "PickListStatus" = 'E'
				FROM PMX_DLPL
				INNER JOIN PMX_PLHE ON PMX_DLPL."PickListLineDocEntry" = PMX_PLHE."DocEntry"								
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"

				WHERE (( @isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = @keyAsInteger ) 
				      OR ( @isStockTransfer = 1 AND WTR1."DocEntry" = @keyAsInteger ))
				AND PMX_PLHE."DocStatus" <> 'C'
				--Only set to partially delivered if there are no lines of status lower than packed
				AND PMX_PLHE."DocEntry" NOT IN (
				
					SELECT PMX_PLLI."DocEntry" 
					FROM PMX_PLLI WHERE "DocEntry" IN (
						SELECT PMX_PLHE."DocEntry"
						FROM PMX_DLPL
						INNER JOIN PMX_PLHE ON PMX_DLPL."PickListLineDocEntry" = PMX_PLHE."DocEntry"
						INNER JOIN PMX_PLLI ON PMX_PLLI."DocEntry" = PMX_PLHE."DocEntry"
						AND PMX_PLLI."LineNum" = PMX_DLPL."PickListLineLineNum"								
						LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
							AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
						LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
							AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"

						WHERE (( @isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = @keyAsInteger ) 
							  OR ( @isStockTransfer = 1 AND WTR1."DocEntry" = @keyAsInteger ))
						AND PMX_PLHE."DocStatus" <> 'C'
					)
					AND PMX_PLLI."PickListLineStatus" IN ('N', 'R', 'P', 'T', 'I')
				);
				
				
				
			END TRY
			BEGIN CATCH
				SET @error = 60004
				SET @error_message = 'Unhandled error update Produmex After shipping: Number:' + CAST(ERROR_NUMBER() AS NVARCHAR(MAX)) + ' Message: ' + ERROR_MESSAGE()
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END CATCH;


			--UPDATE LOCKING TABLE
			DELETE LOCKING FROM PMX_INLD LOCKING INNER JOIN ORDR ON ORDR."DocEntry" = LOCKING."BaseEntry"
			AND LOCKING."BaseType" = '17'
			WHERE ORDR."DocStatus" = 'C'

			DELETE LOCKING FROM PMX_INLD LOCKING INNER JOIN OINV ON OINV."DocEntry" = LOCKING."BaseEntry"
			AND LOCKING."BaseType" = '13'
			WHERE OINV."InvntSttus" = 'C' AND OINV."isIns" = 'Y'
			
			DELETE LOCKING FROM PMX_INLD LOCKING INNER JOIN OWTQ ON OWTQ."DocEntry" = LOCKING."BaseEntry"
			AND LOCKING."BaseType" = '1250000001'
			WHERE OWTQ."DocStatus" = 'C'

			--Close container (lines) if needed
			UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			FROM @documentLineTable AS Lines
			INNER JOIN PMX_COLI
			ON Lines."BaseEntry" = PMX_COLI."BaseEntry"
			AND Lines."BaseLine" = PMX_COLI."BaseLine"
			AND Lines."BaseType" = PMX_COLI."BaseType"
			INNER JOIN RDR1
			ON Lines."BaseEntry" = RDR1."DocEntry"
			AND Lines."BaseLine" = RDR1."LineNum"
			AND Lines."BaseType" = RDR1."ObjType"
			WHERE PMX_COLI."LineStatus" <> 'C'
			AND RDR1."LineStatus" = 'C';

			--Close header is no open lines are found
			UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
		 	PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
			PMX_COHE."UserSign" = (SELECT MAX("CreatedBy") FROM @documentLineTable )
			WHERE PMX_COHE."DocStatus" = 'O'
			AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');


		END
		

		--**********************************************************************************************************
		--Do checks on inventory entry and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isInventoryEntry = 1) BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd = 1) BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT OIGN."ObjType", OIGN."DocEntry", OIGN."DocNum", OIGN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OIGN."CardCode", OIGN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OIGN."UserSign", OIGN."CreateDate", OIGN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  OIGN 
						  INNER JOIN @documentLineTable AS Lines ON OIGN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks on inventory exit and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isInventoryExit = 1) BEGIN

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
					
			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--The quantities must be made negative because the stock leaves the company
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd = 1) BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT OIGE."ObjType", OIGE."DocEntry", OIGE."DocNum", OIGE."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OIGE."CardCode", OIGE."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OIGE."UserSign", OIGE."CreateDate", OIGE."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  OIGE 
						  INNER JOIN @documentLineTable AS Lines ON OIGE."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
					
					
					
			END
		END

		--**********************************************************************************************************
		--Do checks on sales return and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isSalesReturnDocument=1) BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--50800

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd=1) BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT ORDN."ObjType", ORDN."DocEntry", ORDN."DocNum", ORDN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORDN."CardCode", ORDN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORDN."UserSign", ORDN."CreateDate", ORDN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  ORDN 
						  INNER JOIN @documentLineTable AS Lines ON ORDN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks on purchase return and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isPurchaseReturnDocument = 1) BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************


			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--The quantities must be made negative because the stock leaves the company
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd = 1) BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT ORPD."ObjType", ORPD."DocEntry", ORPD."DocNum", ORPD."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORPD."CardCode", ORPD."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORPD."UserSign", ORPD."CreateDate", ORPD."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  ORPD 
						  INNER JOIN @documentLineTable AS Lines ON ORPD."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y'
			END
		END

		--**********************************************************************************************************
		--Do checks on sales invoice and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isSalesInvoice = 1) BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--The base document cannot have picklist(proposals)
			--We have to do the check here because the close of the sales order by sales invoice is not triggered!!
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM ORDR 
			INNER JOIN PMX_PLPL ON ORDR."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = CAST(ORDR."ObjType" AS nvarchar(20))
			INNER JOIN RDR1 ON ORDR."DocEntry" = RDR1."DocEntry" AND RDR1."LineNum" = PMX_PLPL."BaseLine" 
			INNER JOIN INV1 ON INV1."BaseEntry" = RDR1."DocEntry" AND INV1."BaseType" = RDR1."ObjType" AND INV1."BaseLine" = RDR1."LineNum"
			INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry"
			WHERE OINV."DocEntry" = @keyAsInteger AND (ORDR."DocStatus" = 'C' OR RDR1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0
			AND OINV."isIns" = 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51000;
				SET @error_message = 'The sales invoice is linked to a sales order with open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before adding the invoice.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			
			--Check if the sales order is closed that there are no open pick list proposals of this sales order
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM OINV 
			INNER JOIN PMX_PLPL ON OINV."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = CAST(OINV."ObjType" AS nvarchar(20))
			INNER JOIN INV1 ON OINV."DocEntry" = INV1."DocEntry" AND INV1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE OINV."DocEntry" = @keyAsInteger 
			AND ((OINV."InvntSttus" = 'C' AND OINV."isIns" = 'Y') OR INV1."LineStatus" = 'C' ) 
			AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51001;
				SET @error_message = 'The sales invoice ''' + @list_of_cols_val_tab_del + ''' has open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before closing the sales invoice (line).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END


			--Check if sales order lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the sales order line is lower than on the open pick list proposal line / open pick list line
			SET @errorList = ''
			SET @errorList2 = ''
			SELECT @errorList  = @errorList  + CASE WHEN        INV1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END,
				   @errorList2 = @errorList2 + CASE WHEN ISNULL( INV1."Quantity",0) < PMX_PLPL."Quantity" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN INV1 ON PMX_PLPL."BaseEntry" = INV1."DocEntry" AND PMX_PLPL."BaseLine" = INV1."LineNum" AND PMX_PLPL."BaseType" = CAST(INV1."ObjType" AS nvarchar(20))
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = '13' AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR ISNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
		
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51002;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' The item code on a sales invoice line cannot be changed if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
		
			IF LEN(@errorList2) <> 0 BEGIN
				SET @error = 51003;
				SET @error_message = 'Error on line(s):  ' + @errorList2 + ' The quantity on a sales invoice line cannot be lowered if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM INV1 AS Lines
			INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesInvoice."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"

			WHERE SalesInvoice."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O' AND Lines."U_PMX_AMBA" = 'N'
			GROUP BY Lines."LineNum"
			HAVING Count(*) > 1;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51004;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date minutes
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(ORDR.U_PMX_DDTM AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = @keyAsInteger
			AND (ISNULL( ORDR.U_PMX_DDTM, 0) > 59 OR
				ISNULL( ORDR.U_PMX_DDTM, 0) < 0);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51005;
				SET @error_message = 'The due date minutes: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date Hour
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(ORDR.U_PMX_DDTH AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = @keyAsInteger
			AND ( ISNULL( ORDR.U_PMX_DDTH, 0) < 0
					OR ( ISNULL( ORDR.U_PMX_DDTH, 0) > 23 AND ISNULL( ORDR.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR ( ISNULL( ORDR.U_PMX_DDTH, 0) > 12 AND ISNULL( ORDR.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51006;
				SET @error_message = 'The due date hours: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************


			IF (@isTransactionTypeAdd=1) BEGIN
				INSERT INTO PMX_INLD
				("BaseType","BaseEntry","BaseLine","CardCode","CardName",
				"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
				"UserSign","CreateDate",
				"CreateTime","LogUnitIdentKey","QualityStatusCode", 
				"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
				SELECT INV1."ObjType",INV1."DocEntry",INV1."LineNum",OINV."CardCode",OINV."CardName",	PMX_INLD."ItemCode", 
				CASE WHEN INV1."Quantity" > PMX_INLD."Quantity" THEN PMX_INLD."Quantity" ELSE INV1."Quantity" END,
				PMX_INLD."ItemTransactionalInfoKey",PMX_INLD."SerialNum",
				OINV."UserSign", @updateDate, @updateTime,
				PMX_INLD."LogUnitIdentKey", PMX_INLD."QualityStatusCode",
				 'RESERVED', PMX_INLD."InvLockLevel", PMX_INLD."PmxWhsCode", 0
				FROM PMX_INLD
				INNER JOIN INV1 ON CAST(INV1."BaseType" AS nvarchar(20)) = PMX_INLD."BaseType" AND INV1."BaseEntry" = PMX_INLD."BaseEntry" AND INV1."BaseLine" = PMX_INLD."BaseLine"
				INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry"
				WHERE OINV."DocEntry" = @keyAsInteger AND OINV."isIns" = 'Y'
				AND PMX_INLD."BaseEntry" = INV1."BaseEntry"
				AND PMX_INLD."BaseType" = CAST(INV1."BaseType" AS nvarchar(20))
				AND PMX_INLD."BaseLine" = INV1."BaseLine";


				--Update the locking of the sales order: adjust quantity
				UPDATE PMX_INLD SET PMX_INLD."Quantity" = PMX_INLD."Quantity" - INV1."Quantity"
				FROM PMX_INLD
				INNER JOIN INV1 ON CAST(INV1."BaseType" AS nvarchar(20)) = PMX_INLD."BaseType" AND INV1."BaseEntry" = PMX_INLD."BaseEntry" AND INV1."BaseLine" = PMX_INLD."BaseLine"
				INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry"
				WHERE OINV."DocEntry" = @keyAsInteger AND OINV."isIns" = 'Y'
				AND PMX_INLD."BaseEntry" = INV1."BaseEntry"
				AND PMX_INLD."BaseType" = CAST(INV1."BaseType" AS nvarchar(20))
				AND PMX_INLD."BaseLine" = INV1."BaseLine";


				DELETE FROM PMX_INLD WHERE "Quantity" <= 0;
			

				--**********************************************************************************************************
				--START: Add/update locking based on batch numbers
				--**********************************************************************************************************


				IF EXISTS(SELECT DocEntry FROM OINV WHERE OINV."DocEntry" = @keyAsInteger AND OINV."isIns" = 'Y')
				BEGIN


					--Start with deleting the lockings that are not correct
					DELETE FROM PMX_INLD
					WHERE PMX_INLD."InternalKey" NOT IN 
					(--Query to get the correct lockings
						SELECT 	PMX_INLD."InternalKey"
						FROM INV1 AS Lines
						INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
						INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						LEFT JOIN PMX_ITRI ON
						 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')
						--AND ISNULL( PMX_ITRI."BestBeforeDate", 1) = ISNULL( "ExpDate", 1)

						LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
						AND PMX_INLD."BaseEntry" = Lines."DocEntry"
						AND PMX_INLD."BaseLine" = Lines."LineNum"
						AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

						WHERE OITL."ApplyEntry" = @keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2

						AND PMX_INLD."InternalKey" IS NOT NULL


						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
						Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey", PMX_INLD."InvLockLevel"

						HAVING SUM("AllocQty") > 0



						--Exclude manual lockings on LUID level when no batch has been selected on sales invoice
						UNION SELECT PMX_INLD."InternalKey"
						FROM PMX_INLD 
						LEFT JOIN
						(
							SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
							FROM INV1 AS Lines
											INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
											LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
											LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
											LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
											AND ITL1."SysNumber" = OBTN."SysNumber"

							WHERE OITL."ApplyEntry"  = @keyAsInteger 
											AND Lines."LineStatus" = 'O'
											AND ISNULL( "StockEff", 2) = 2

								GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
											Lines."NumPerMsr", "DistNumber", "ExpDate"

							HAVING SUM(ISNULL( "AllocQty", 0)) > 0
						) AS OINV

						ON OINV."LineNum" = PMX_INLD."BaseLine"
						AND OINV."DocEntry" = PMX_INLD."BaseEntry"
						AND OINV."ObjType" = PMX_INLD."BaseType"


						WHERE PMX_INLD."InvLockLevel" = 'L'
						AND PMX_INLD."BaseEntry" = @keyAsInteger
						AND OINV."DocEntry" IS NULL

						

					) 

					AND PMX_INLD."BaseType" = '13'
					AND PMX_INLD."BaseEntry" = @keyAsInteger;



					BEGIN TRY

						DECLARE salesInvoiceBatchCursor CURSOR LOCAL FOR 
								SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", 
								Lines."ObjType", Lines."ItemCode", PMX_OSWH."Code", CASE WHEN OITM."U_PMX_ICAW" = 'Y' THEN OITM."U_PMX_DQUM" ELSE NULL END, Lines."U_PMX_SQOP"
								FROM INV1 AS Lines
								INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
								INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
								INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
								INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
								INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
								AND ITL1."SysNumber" = OBTN."SysNumber"

								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = Lines."WhsCode"

								LEFT JOIN PMX_PLPL
								ON PMX_PLPL."BaseEntry" = Lines."DocEntry"
								AND PMX_PLPL."BaseLine" = lines."LineNum"
								AND PMX_PLPL."BaseType" = Lines."ObjType"

								WHERE OITL."ApplyEntry" = @keyAsInteger 
								AND Lines."LineStatus" = 'O'
								AND "StockEff" = 2
								AND PMX_PLPL."DocEntry" IS NULL
								GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code", OITM."U_PMX_ICAW", OITM."U_PMX_DQUM", Lines."U_PMX_SQOP"

								HAVING SUM("AllocQty") > 0;


						OPEN salesInvoiceBatchCursor;
						FETCH NEXT FROM salesInvoiceBatchCursor 
						INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock,@pmxWarehouseToLock, @catchWeightRatioToLock, @shippingQualityOptionToLock
						WHILE @@FETCH_STATUS >= 0 BEGIN

							--Check for each batch if we have a corresponding locking line

							IF NOT EXISTS (SELECT TOP(1) PMX_INLD.InternalKey FROM PMX_INLD 
							INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
							WHERE PMX_INLD."BaseType" = @typeToLock
							AND PMX_INLD."BaseEntry" = @docEntryToLock
							AND PMX_INLD."BaseLine" = @lineNumToLock
							AND PMX_ITRI."BatchNumber" = @batchToLock
							--AND ISNULL( PMX_ITRI."BestBeforeDate", '') = ISNULL( @expDateToLock, '')
							)
							BEGIN

								--Entry does not exist, so we need to lock the stock
								--First try to get the itri key to lock
								SET @remainingQuantityToLock = @batchQuantityToLock;

								BEGIN TRY
									DECLARE itriSalesInvoiceCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
												INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
												AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
												WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
												AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
												AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0


									OPEN itriSalesInvoiceCursor;
									FETCH NEXT FROM itriSalesInvoiceCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

										IF @remainingQuantityToLock < @freeQuantityToLock
										BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END
										ELSE
										BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
										END

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ("BaseType","BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
										SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(OINV."CardCode"),MAX(OINV."CardName"),
											@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
											MAX(OINV."UserSign"),@updateDate,
											@updateTime,NULL, @qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, @currentQuantityToLock * @catchWeightRatioToLock
										FROM OINV 
										INNER JOIN INV1 ON INV1."DocEntry" = OINV."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = INV1."WhsCode"
										WHERE OINV."DocEntry" = @docEntryToLock
										AND INV1."LineNum" = @lineNumToLock

										--Reduce remaining
										SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


										FETCH NEXT FROM itriSalesInvoiceCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									END
									CLOSE itriSalesInvoiceCursor;
									DEALLOCATE itriSalesInvoiceCursor

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(@remainingQuantityToLock > 0)
									BEGIN
										SET @error = 51050;
										SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock + '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN;

									END

								END TRY
								BEGIN CATCH
									SET @error = 51090
									SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
									+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE itriSalesInvoiceCursor 
										DEALLOCATE itriSalesInvoiceCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE itriSalesInvoiceCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END CATCH
								END CATCH

							END
							ELSE 
							BEGIN

								--The line exists, so now we need to check if the quantity is the same/more/less
								SET @remainingQuantityToLock = (SELECT SUM(PMX_INLD."Quantity") -  @batchQuantityToLock AS "Quantity" FROM PMX_INLD 
									INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
									WHERE PMX_INLD."BaseType" = @typeToLock
									AND PMX_INLD."BaseEntry" = @docEntryToLock
									AND PMX_INLD."BaseLine" = @lineNumToLock
									AND PMX_ITRI."BatchNumber" = @batchToLock
									GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine")
									
												
								IF @remainingQuantityToLock <> 0 BEGIN
								--Only if deviation, then we try to update/insert
								--Loop tru all locking lines
									BEGIN TRY
										DECLARE inldCursor CURSOR LOCAL FOR SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey" FROM PMX_INLD 
										INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
										WHERE PMX_INLD."BaseType" = @typeToLock
										AND PMX_INLD."BaseEntry" = @docEntryToLock
										AND PMX_INLD."BaseLine" = @lineNumToLock
										AND PMX_ITRI."BatchNumber" = @batchToLock;

										OPEN inldCursor;
										FETCH NEXT FROM inldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
										WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock <> 0 BEGIN

											IF @remainingQuantityToLock > 0 BEGIN
												--Remaining is more than 0, so we can just remove the locking
												IF @remainingQuantityToLock >= @lockedQuantity BEGIN
													--Delete
													DELETE FROM PMX_INLD WHERE "InternalKey" = @currentLockInternalKey
													SET @remainingQuantityToLock = @remainingQuantityToLock - @lockedQuantity
												END ELSE BEGIN
													--update
													UPDATE PMX_INLD SET "Quantity" = "Quantity" - @remainingQuantityToLock WHERE "InternalKey" = @currentLockInternalKey
													SET @remainingQuantityToLock = 0
												END
											END	ELSE BEGIN
												--Remaining is less than 0, so we need to add locking
												--If there is still free stock for current itri, then take that one
												SELECT @freeQuantityToLock = ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) FROM PMX_ITRI 
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
												INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
												WHERE "QualityStatusCode" IN (
													SELECT "Code" FROM PMX_QYST 
													WHERE  (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
													) 
												GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
												WHERE "QualityStatusCode" IN (
													SELECT "Code" FROM PMX_QYST
													WHERE  (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
													OR (ISNULL(@shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
													) 
												GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
												AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
												WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
												AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
												AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0
												AND PMX_ITRI."InternalKey" = @itriToLock;

												IF @remainingQuantityToLock > @freeQuantityToLock
												BEGIN
													SET @currentQuantityToLock = @freeQuantityToLock;
												END
                                                ELSE
                                                BEGIN
													SET @currentQuantityToLock = @remainingQuantityToLock;
												END

												UPDATE PMX_INLD SET "Quantity" = "Quantity" + @currentQuantityToLock, "QuantityUom2" = "QuantityUom2" + (@currentQuantityToLock  * @catchWeightRatioToLock) WHERE PMX_INLD."InternalKey" = @currentLockInternalKey;

												-- add qty, because @remainingQuantityToLock is negative
												SET @remainingQuantityToLock = @remainingQuantityToLock + @currentQuantityToLock;
												
											END

											FETCH NEXT FROM inldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
										END
										CLOSE inldCursor;
										DEALLOCATE inldCursor

									END TRY
									BEGIN CATCH
										SET @error = 51091
										SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
										+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
										BEGIN TRY
											CLOSE inldCursor 
											DEALLOCATE inldCursor
											SELECT @error, @error_message
											EXEC xp_logevent @error, @error_message, ERROR
											RETURN
										END TRY
										BEGIN CATCH
											DEALLOCATE inldCursor
											SELECT @error, @error_message
											EXEC xp_logevent @error, @error_message, ERROR
											RETURN
										END CATCH
									END CATCH

									IF @remainingQuantityToLock <> 0 BEGIN
										-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
										--Set @remainingQuantityToLock as positive quanitty
										SET @remainingQuantityToLock = ABS( @remainingQuantityToLock );
										

										BEGIN TRY
											DECLARE itriSalesInvoiceCursor CURSOR LOCAL FOR
                                            SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
											LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
											INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
											WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
											LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
											WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
											AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
											WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
											AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
											AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0


											OPEN itriSalesInvoiceCursor;
											FETCH NEXT FROM itriSalesInvoiceCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
											WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

												IF(@remainingQuantityToLock < @freeQuantityToLock)
												BEGIN
													SET @currentQuantityToLock = @remainingQuantityToLock;
												END
												ELSE
												BEGIN
													SET @currentQuantityToLock = @freeQuantityToLock;
												END

												--Insert INTO the PMX_INLD table

												INSERT INTO PMX_INLD ("BaseType","BaseEntry","BaseLine","CardCode","CardName",
													"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
													"UserSign","CreateDate",
													"CreateTime","LogUnitIdentKey","QualityStatusCode", 
													"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
												SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(OINV."CardCode"),MAX(OINV."CardName"),
													@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
													MAX(OINV."UserSign"),@updateDate,
													@updateTime,NULL, @qualityStatusToLock,
													'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, @currentQuantityToLock * @catchWeightRatioToLock
												FROM OINV 
												INNER JOIN INV1 ON INV1."DocEntry" = OINV."DocEntry"
												INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = INV1."WhsCode"
												WHERE OINV."DocEntry" = @docEntryToLock
												AND INV1."LineNum" = @lineNumToLock

												--Reduce remaining
												SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


												FETCH NEXT FROM itriSalesInvoiceCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
											END;
											CLOSE itriSalesInvoiceCursor;
											DEALLOCATE itriSalesInvoiceCursor;

											--If there is remaining QTY, there is not enough free stock, so throw error
											IF(@remainingQuantityToLock > 0)
											BEGIN
												SET @error = 51052;
												SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
												SELECT @error, @error_message
												EXEC xp_logevent @error, @error_message, ERROR
												RETURN;
											END

										END TRY
										BEGIN CATCH
											SET @error = 51092
											SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
											+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
											BEGIN TRY
												CLOSE itriSalesInvoiceCursor 
												DEALLOCATE itriSalesInvoiceCursor
												SELECT @error, @error_message
												EXEC xp_logevent @error, @error_message, ERROR
												RETURN
											END TRY
											BEGIN CATCH
												DEALLOCATE itriSalesInvoiceCursor
												SELECT @error, @error_message
												EXEC xp_logevent @error, @error_message, ERROR
												RETURN
											END CATCH
										END CATCH

									END

								END

								IF @remainingQuantityToLock <> 0 BEGIN
								    -- if quantity is still different from 0, then there was not enough stock to reserve!
									CLOSE salesInvoiceBatchCursor;
									DEALLOCATE salesInvoiceBatchCursor;
									SET @error = 51053;
									SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN;
								END

							END

							FETCH NEXT FROM salesInvoiceBatchCursor
							INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock, @pmxWarehouseToLock,@catchWeightRatioToLock, @shippingQualityOptionToLock;
						END
						CLOSE salesInvoiceBatchCursor;
						DEALLOCATE salesInvoiceBatchCursor;
						
					END TRY
					BEGIN CATCH
						SET @error = 50699
						SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
						+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
						BEGIN TRY
							CLOSE salesInvoiceBatchCursor 
							DEALLOCATE salesInvoiceBatchCursor
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN
						END TRY
						BEGIN CATCH
							DEALLOCATE salesInvoiceBatchCursor
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN
						END CATCH
					END CATCH
				END
				--**********************************************************************************************************
				--END: Add/update locking based on batch numbers
				--**********************************************************************************************************
			END





			

			--UPDATE LOCKING TABLE
			DELETE FROM LOCKING
            FROM PMX_INLD LOCKING
            INNER JOIN OINV ON OINV."DocEntry" = LOCKING."BaseEntry"
			AND CAST(OINV."ObjType" AS nvarchar(20)) = LOCKING."BaseType"
			WHERE OINV."InvntSttus" = 'C' AND OINV."isIns" = 'Y'
			AND OINV."DocEntry" = @keyAsInteger;


			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF @isTransactionTypeAdd = 1 BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT OINV."ObjType", OINV."DocEntry", OINV."DocNum", OINV."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OINV."CardCode", OINV."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OINV."UserSign", OINV."CreateDate", OINV."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  OINV 
						  INNER JOIN @documentLineTable AS Lines ON OINV."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks on purchase invoice and add lines to detail inventory
		--**********************************************************************************************************
		IF @isPurchaseInvoice = 1 BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd=1) BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT OPCH."ObjType", OPCH."DocEntry", OPCH."DocNum", OPCH."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OPCH."CardCode", OPCH."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OPCH."UserSign", OPCH."CreateDate", OPCH."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  OPCH 
						  INNER JOIN @documentLineTable AS Lines ON OPCH."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks on purchase credit note and add lines to detail inventory
		--**********************************************************************************************************
		IF @isPurchaseCreditNote = 1 BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF @isTransactionTypeAdd = 1 BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT ORPC."ObjType", ORPC."DocEntry", ORPC."DocNum", ORPC."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORPC."CardCode", ORPC."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORPC."UserSign", ORPC."CreateDate", ORPC."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  ORPC 
						  INNER JOIN @documentLineTable AS Lines ON ORPC."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks on sales credit note and add lines to detail inventory
		--**********************************************************************************************************
		IF @isSalesCreditNote = 1 BEGIN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--The base document cannot have picklist(proposals)
			--We have to do the check here because the close of the sales invoice by sales credit note is not triggered!!
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM OINV 
			INNER JOIN PMX_PLPL ON OINV."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = OINV."ObjType"
			INNER JOIN INV1 ON OINV."DocEntry" = INV1."DocEntry" AND INV1."LineNum" = PMX_PLPL."BaseLine" 
			INNER JOIN RIN1 ON RIN1."BaseEntry" = INV1."DocEntry" AND RIN1."BaseType" = INV1."ObjType" AND RIN1."BaseLine" = INV1."LineNum"
			INNER JOIN ORIN ON ORIN."DocEntry" = RIN1."DocEntry"
			WHERE ORIN."DocEntry" = @keyAsInteger AND (ORIN."DocStatus" = 'C' OR RIN1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0
			AND OINV."isIns" = 'Y';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51100;
				SET @error_message = 'The sales credit note is linked to a sales invoice with open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before adding the invoice.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE: Delete locks for closed reserve invoices
			DELETE FROM LOCKING
            FROM PMX_INLD LOCKING
            INNER JOIN OINV ON OINV."DocEntry" = LOCKING."BaseEntry"
			AND OINV."ObjType" = LOCKING."BaseType"
			WHERE OINV."DocStatus" = 'C' AND OINV."isIns" = 'Y';


			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF @isTransactionTypeAdd = 1 BEGIN
				INSERT INTO @inventoryDetailTable
					SELECT ORIN."ObjType", ORIN."DocEntry", ORIN."DocNum", ORIN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORIN."CardCode", ORIN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * @stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORIN."UserSign", ORIN."CreateDate", ORIN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * @stockDirection, Lines."InventoryQuantity"
					FROM  ORIN 
						  INNER JOIN @documentLineTable AS Lines ON ORIN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END
		END

		--**********************************************************************************************************
		--Do checks for move and add lines to detail inventory
		--If a move line is indicated that it is a logistic carrier then this logistic carrier is put on the
		--logistic unit.
		--**********************************************************************************************************
		IF @isMoveDoc = 1 AND @isTransactionTypeAdd = 1 BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			--"ItemCode" filled in and correct
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines LEFT JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" AND Items."InvntItem" = 'Y'
			WHERE Items."ItemCode" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50100;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The item code does not exist, is not filled in or is not an inventory item.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

	--		--"BatchNumber" filled in if needed
	--		SET @errorList = ''
	--		SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
	--		FROM PMX_MVLI AS Lines INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
	--		LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
	--		WHERE PMX_ITRI."BatchNumber" IS NULL AND Items."ManBtchNum" = 'Y' AND Lines."DocEntry" = @keyAsInteger
	--		IF LEN(@errorList) <> 0 BEGIN
	--			SET @error = 50101
	--			SET @error_message = 'Error on line(s): ' + @errorList + ' The batchnumber must be filled in for item.'
	--			SELECT @error, @error_message
	--			EXEC xp_logevent @error, @error_message, ERROR
	--			RETURN
	--		END
	--
	--		-- Check if best before date is filled in for expirable goods
	--		SET @errorList = ''
	--		SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
	--		FROM PMX_MVLI AS Lines INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
	--		LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
	--		WHERE PMX_ITRI."BestBeforeDate" IS NULL AND Items.U_PMX_HBBD = 'Y' AND Lines."DocEntry" = @keyAsInteger
	--		IF LEN(@errorList) <> 0 BEGIN
	--			SET @error = 50102
	--			SET @error_message = 'Error on line(s): ' + @errorList + ' The best before date must be filled in for expirable item.'
	--			SELECT @error, @error_message
	--			EXEC xp_logevent @error, @error_message, ERROR
	--			RETURN
	--		END

			--SrcStorLocCode filled in and correct?
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSEL AS Locations ON Lines."SrcStorLocCode" = Locations."Code" AND Locations."IsStorageLocation" = 'Y'
			WHERE Locations."Code" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50103;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The source storage location does not exist or is not filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--"DestStorLocCode" filled in and correct?
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSEL AS Locations ON Lines."DestStorLocCode" = Locations."Code" AND Locations."IsStorageLocation" = 'Y'
			WHERE Locations."Code" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50104;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The destination storage location does not exist or is not filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--SrcLogUnitIdentKey correct
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_LUID AS LogicalUnitID ON Lines."SrcLogUnitIdentKey" = LogicalUnitID."InternalKey" 
			WHERE Lines."SrcLogUnitIdentKey" IS NOT NULL AND LogicalUnitID."InternalKey" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50105;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The source logistic unit identification does not exist.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--DestLogUnitIdentKey correct
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines
			LEFT JOIN PMX_LUID AS LogicalUnitID ON Lines."DestLogUnitIdentKey" = LogicalUnitID."InternalKey" 
			WHERE Lines."DestLogUnitIdentKey" IS NOT NULL AND LogicalUnitID."InternalKey" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50106;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The destination logistic unit identification does not exist.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--SrcQualityStatus filled in and correct
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_QYST AS QualityStatus ON Lines."SrcQualityStatusCode" = QualityStatus."Code" 
			WHERE QualityStatus."Code" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50107;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The entered source quality status does not exist or is not filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--DestQualityStatus filled in and correct
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines
			LEFT JOIN PMX_QYST AS QualityStatus ON Lines."DestQualityStatusCode" = QualityStatus."Code" 
			WHERE QualityStatus."Code" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50108;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The entered destination quality status does not exist or is not filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- If a line is set as logisic carrier then the LUID must be filled in
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Lines."DestLogUnitIdentKey" IS NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50109;
				SET @error_message = 'Error on line(s): ' + @errorList + ' If the flag is set that this line is the logistic carrier of the logistic unit then the logistic unit ID must be filled in.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if logistic carrier is filled in on the logistic unit and that there are no lines
			-- indicated as logistic carrier.
			-- Also check that not 2 logistic carriers are defined on the same logistic unit
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."DestLogUnitIdentKey" AS VARCHAR) + ' '
			FROM PMX_MVLI AS Lines 
				 INNER JOIN PMX_LUID AS LogisticUnitID ON Lines."DestLogUnitIdentKey" = LogisticUnitID."InternalKey"
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Lines."DocEntry" = @keyAsInteger
			GROUP BY lines."DocEntry",lines."DestLogUnitIdentKey",LogisticUnitID."LogisticCarrierCode"
			HAVING (count(lines."DocEntry") > 1) or (count(lines."DocEntry") = 1 AND  LogisticUnitID."LogisticCarrierCode" IS NOT NULL);
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50110;
				SET @error_message = 'Error for logistic unit(s): ' + @errorList + ' There is already a logistic carrier defined or there is more then 1 line is defined as logistic carrier for these logistic unit(s).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END;

			-- If a line is set as logisic carrier then the item must be a logistic carrier
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines
				INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Items.U_PMX_LOCA <> 'Y' AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50111;
				SET @error_message = 'Error on line(s): ' + @errorList + ' If the flag is set that this line is the logistic carrier of the logistic unit then the item must be a logistic carrier.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- If a line is set as logisic carrier then its quantity must be 1
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
			WHERE Lines."DocEntry" = @keyAsInteger AND Lines."IsLogisticCarrier" = 'Y' AND ((Lines."Quantity"*Lines."QuantityPerUom") <> 1);
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50112;
				SET @error_message = 'Error on line(s): ' + @errorList + ' If a line is a logistic carrier then its quantity must be 1.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Check if the warehouse of the source storage location is the same
			-- warehouse as the destination.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
				 INNER JOIN [PMX_OSEL] AS OSE ON Lines."DestStorLocCode" = OSE."Code"
				 INNER JOIN [PMX_OSEL] AS OSE2 ON Lines."SrcStorLocCode" = OSE2."Code"
			WHERE Lines."DocEntry" = @keyAsInteger AND OSE."PmxWhsCode" <> OSE2."PmxWhsCode"
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50113
				SET @error_message = 'Error on line(s): ' + @errorList + ' The source storage location code is of a different warehouse than the destination storage location.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END

			--SrcStorLocCode locked?
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines
			LEFT JOIN PMX_OSSL AS Locations ON Lines."SrcStorLocCode" = Locations."Code"
			WHERE Locations."LockedBy" IS NOT NULL AND Lines."DocEntry" = @keyAsInteger
			AND ISNULL( Lines."BaseType", '') <> 'PMX_CYCO';
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50114;
				SET @error_message = 'Error on line(s): ' + @errorList + ' The source storage location is locked.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--"DestStorLocCode" locked?
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSSL AS Locations ON Lines."DestStorLocCode" = Locations."Code"
			WHERE Locations."LockedBy" IS NOT NULL AND Lines."DocEntry" = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50115;
				SET @error_message = 'Error on line(s) ' + @errorList + ': The destination storage location is locked.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END;
			
			--"Quantity" uom2 filled in for catch weight?
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 
			FROM PMX_MVLI AS Lines LEFT JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" AND Items."InvntItem" = 'Y'
			WHERE Items.U_PMX_ICAW = 'Y' AND Lines."DocEntry" = @keyAsInteger
			AND Lines."QuantityUom2" IS NULL;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50116;
				SET @error_message = 'Error on line(s) ' + @errorList + ': The quantity for uom2 needs to be filled in for catch weight items.';
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

            -- Check if best before date is filled in for expirable goods
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' ' 			
			FROM PMX_MVLI AS Lines
			INNER JOIN WOR1 ON Lines.BaseEntry = WOR1.DocEntry 
			AND Lines.BaseLine = WOR1.LineNum 
			AND Lines.BaseType = N'202'
			INNER JOIN OWOR ON WOR1.DocENtry = OWOR.DocENtry
			INNER JOIN PMX_ITRI ON Lines.ItemTransactionalInfoKey = PMX_ITRI.InternalKey
			WHERE CASE WHEN "WOR1"."U_PMX_BBDO" = N'P' THEN CURRENT_TIMESTAMP WHEN "WOR1"."U_PMX_BBDO" = N'B' THEN ISNULL( "OWOR"."U_PMX_BBDT", "OWOR"."DueDate" ) ELSE "OWOR"."DueDate" END > PMX_ITRI.BestBeforeDate 
			AND WOR1.DocEntry = @keyAsInteger;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50118;
				SET @error_message = 'Error on line(s): ' + @errorList + ' Cannot move stock for production order. Invalid calculation for BestBeforeDate option.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			
			--Check if location has correct quality status code
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."DestStorLocCode" AS VARCHAR) + '-' + CAST(Lines."DestQualityStatusCode" AS VARCHAR) + ' '
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSSL AS OSE ON Lines."DestStorLocCode" COLLATE SQL_Latin1_General_Cp850_CS_AS = OSE."Code" COLLATE SQL_Latin1_General_Cp850_CS_AS
			WHERE ISNULL(OSE."QualityStatusCode", Lines."DestQualityStatusCode") <> Lines."DestQualityStatusCode"
			AND Lines."DocEntry" = @keyAsInteger
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 50119;
				SET @error_message = 'Quality status does not match fixed quality status on location: ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END


			--**********************************************************************************************************
			--If a move line is indicated that it is a logistic carrier then this logistic carrier is put on the
			--logistic unit.
			--**********************************************************************************************************
			UPDATE PMX_LUID 
			SET "LogisticCarrierCode" = Lines."ItemCode" FROM PMX_MVLI AS Lines INNER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."DestLogUnitIdentKey"
			WHERE Lines."IsLogisticCarrier" = 'Y'AND Lines."DocEntry" = @keyAsInteger;

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table
			--**********************************************************************************************************
			IF @isTransactionTypeAdd = 1 BEGIN
				--One move line represents 2 lines in the detail inventory table:
				-- *The destination as positive lines
				-- *The source as negative lines

				--Insert source lines -> negative
				INSERT INTO @inventoryDetailTable
					SELECT PMX_MVHE."ObjType", PMX_MVHE."DocEntry", PMX_MVHE."DocEntry", PMX_MVHE."CreateDate", PMX_MVLI."LineNum", PMX_MVLI."BaseType", PMX_MVLI."BaseEntry", PMX_MVLI."BaseLine",
						  NULL, NULL, PMX_MVLI."ItemCode", 
						  SUBSTRING(PMX_MVLI."Dscription", 0, 100), - PMX_MVLI."Quantity", PMX_OSSL."Code", PMX_MVLI."ItemTransactionalInfoKey", 
						  NULL AS Serial, PMX_MVLI."SrcLogUnitIdentKey", PMX_MVHE."UserSign", PMX_MVHE."CreateDate", PMX_MVHE."CreateTime", 
						  PMX_LUID.SSCC, PMX_MVLI."SrcQualityStatusCode", /*Use inventory "Uom"!*/ISNULL( OUOM."UomCode", OITM."InvntryUom"), ISNULL( OUOM."UomCode", OITM."InvntryUom"), PMX_MVLI."QuantityPerUom", PMX_MVLI."Uom2", -PMX_MVLI."QuantityUom2", -(PMX_MVLI."Quantity"*PMX_MVLI."QuantityPerUom")
					FROM  PMX_MVHE INNER JOIN
						  PMX_MVLI ON PMX_MVHE."DocEntry" = PMX_MVLI."DocEntry" INNER JOIN
						  PMX_OSSL ON PMX_MVLI."SrcStorLocCode" = PMX_OSSL."Code" LEFT OUTER JOIN
						  PMX_LUID ON PMX_LUID."InternalKey" = PMX_MVLI."SrcLogUnitIdentKey"
						  INNER JOIN OITM ON PMX_MVLI."ItemCode" = OITM."ItemCode"
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE PMX_MVHE."DocEntry" = @keyAsInteger;

				--Insert destination lines -> positive
				INSERT INTO @inventoryDetailTable
					SELECT PMX_MVHE."ObjType", PMX_MVHE."DocEntry", PMX_MVHE."DocEntry", PMX_MVHE."CreateDate", PMX_MVLI."LineNum", PMX_MVLI."BaseType", PMX_MVLI."BaseEntry", PMX_MVLI."BaseLine",
						  NULL, NULL, PMX_MVLI."ItemCode", 
						  SUBSTRING(PMX_MVLI."Dscription", 0, 100), PMX_MVLI."Quantity", PMX_OSSL."Code", PMX_MVLI."ItemTransactionalInfoKey", 
						  NULL AS Serial, PMX_MVLI."DestLogUnitIdentKey", PMX_MVHE."UserSign", PMX_MVHE."CreateDate", PMX_MVHE."CreateTime", 
						  PMX_LUID.SSCC, PMX_MVLI."DestQualityStatusCode", /*Use inventory "Uom"!*/ISNULL( OUOM."UomCode", OITM."InvntryUom"), ISNULL( OUOM."UomCode", OITM."InvntryUom"), PMX_MVLI."QuantityPerUom", PMX_MVLI."Uom2", PMX_MVLI."QuantityUom2", (PMX_MVLI."Quantity"*PMX_MVLI."QuantityPerUom")
					FROM  PMX_MVHE INNER JOIN
						  PMX_MVLI ON PMX_MVHE."DocEntry" = PMX_MVLI."DocEntry" INNER JOIN
						  PMX_OSSL ON PMX_MVLI."DestStorLocCode" = PMX_OSSL."Code" LEFT OUTER JOIN						  
						  PMX_LUID ON PMX_LUID."InternalKey" = PMX_MVLI."DestLogUnitIdentKey"
						  INNER JOIN OITM ON PMX_MVLI."ItemCode" = OITM."ItemCode"
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE PMX_MVHE."DocEntry" = @keyAsInteger;
			END
			
		
			--**********************************************************************************************************
			--add/update/delete if luids need to be linked to master luids
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd=1) BEGIN
			
			--Delete the previous links
				--DELETE FROM Links
    --            FROM PMX_LMLU Links
				--INNER JOIN PMX_MVLI Lines ON Links.LUID = Lines."DestLogUnitIdentKey"
				--WHERE Lines."DocEntry" = @keyAsInteger;
				
				
			--Delete the previous links
			DELETE FROM Links
			FROM PMX_LMLU Links
			INNER JOIN PMX_MVLI Lines ON Links."LUID" = Lines."SrcLogUnitIdentKey"
			WHERE Lines."DocEntry" = @keyAsInteger
			AND Lines."MasterLUID" <> Links."MasterLUID";
				
			--Delete links when all stock from luid is moved
				DELETE Links FROM PMX_LMLU Links
				WHERE Links.LUID IN 
				(
					SELECT Inventory.LogUnitIdentKey FROM PMX_MVLI Lines
					INNER JOIN PMX_LMLU Links ON Links.LUID = Lines.SrcLogUnitIdentKey
					INNER JOIN PMX_INVT Inventory ON Inventory.LogUnitIdentKey = Links.LUID 
					WHERE Lines.DocEntry = @keyAsInteger AND Lines.MasterLUID IS NULL 
					GROUP BY Inventory.LogUnitIdentKey
					HAVING SUM(Inventory.Quantity) = SUM(Lines.Quantity * Lines.QuantityPerUom)
				)
				
				
			--Add new links
				INSERT INTO PMX_LMLU   ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime",  "Version", "MasterLUID", "LUID")											
				SELECT DISTINCT 'N', Header.UserSign, @tsNow ,@tsNow, 0, Lines.MasterLUID, Lines.DestLogUnitIdentKey
				FROM	PMX_MVLI Lines INNER JOIN
						PMX_MVHE Header ON Lines."DocEntry" = Header."DocEntry" LEFT JOIN
						PMX_LMLU Links ON Lines.DestLogUnitIdentKey = Links.LUID AND Lines.MasterLUID = Links.MasterLUID
				WHERE	Lines.DocEntry = @keyAsInteger 
						AND Lines."MasterLUID" IS NOT NULL
						AND Links.InternalKey IS NULL									
				
			END
			
			--Check if all subluids are on the same location as master luid and master luid is on the same location as its subluids
			IF (@isTransactionTypeAdd=1) BEGIN

				SET @errorList = ''
				SELECT @errorList = @errorList + CAST(Lines."InternalKey" AS VARCHAR) + ' ' 
				FROM PMX_MVLI AS Lines
				INNER JOIN (
						SELECT PMX_INVT."StorLocCode", PMX_LMLU.LUID, PMX_LMLU."MasterLUID"
						FROM PMX_LMLU
						INNER JOIN PMX_INVT ON PMX_LMLU.LUID = PMX_INVT."LogUnitIdentKey"
					) t ON t."MasterLUID" = Lines."MasterLUID"
				WHERE t."StorLocCode" <> Lines."DestStorLocCode"
				AND t."MasterLUID" <> Lines."MasterLUID"
				AND Lines."DocEntry" = @keyAsInteger;
				IF LEN(@errorList) <> 0 BEGIN
					SET @error = 50119;
					SET @error_message = 'Error on line(s) ' + @errorList + ': SubLuids all need to be on the same location.'
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN;
				END

			END

		END


		IF @isWarehouseTransferRequestDocument = 1 BEGIN

			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );
			
			--Check if the warehouse inventory request is closed that there are no open pick list proposals of this warehouse inventory request
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_PLPL."DocEntry" AS VARCHAR) + ' ' 
			FROM OWTQ 
			INNER JOIN PMX_PLPL ON OWTQ."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = OWTQ."ObjType"
			INNER JOIN WTQ1 ON OWTQ."DocEntry" = WTQ1."DocEntry" AND WTQ1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE OWTQ."DocEntry" = @keyAsInteger AND (OWTQ."DocStatus" = 'C' OR WTQ1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51200;
				SET @error_message = 'The warehouse inventory request ''' + @list_of_cols_val_tab_del + ''' has open pick list proposal(s): ' + @errorList + '. The pick list proposal(s) must be closed first before closing the warehouse inventory request (line).'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if the full quantity is selected when entering batch numbers
--			SET @errorList = ''
--			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
--			FROM WTQ1 AS Lines
--			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
--			LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = WhsInvRequest."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
--			WHERE WhsInvRequest."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O'
--			GROUP BY Lines."LineNum", Lines."Quantity", Lines."NumPerMsr"
--			HAVING SUM(IBT1."Quantity") <> (Lines."Quantity" * Lines."NumPerMsr") AND SUM(IBT1."Quantity") > 0
--			IF LEN(@errorList) <> 0 BEGIN
--				SET @error = 51201
--				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
--				SELECT @error, @error_message
--				EXEC xp_logevent @error, @error_message, ERROR
--				RETURN
--			END

			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM WTQ1 AS Lines
            INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
            INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocatEnt" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
            INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
            INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
            AND ITL1."SysNumber" = OBTN."SysNumber"
            WHERE OITL."AllocatEnt" = @keyAsInteger 
			AND Lines."LineStatus" = 'O'
            AND "StockEff" = 2
            GROUP BY Lines."LineNum", Lines."OpenQty", Lines."NumPerMsr"
            HAVING SUM("AllocQty") <> (Lines."OpenQty" * Lines."NumPerMsr") AND SUM("AllocQty") > 0;
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51201;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' You need to fill the full quantity when entering a batch number'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			--Check if warehouse inventory request lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the warehouse inventory request line is lower than on the open pick list proposal line / open pick list line
			SET @errorList = ''
			SELECT @errorList  = @errorList  + CASE WHEN        WTQ1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN WTQ1 ON PMX_PLPL."BaseEntry" = WTQ1."DocEntry" AND PMX_PLPL."BaseLine" = WTQ1."LineNum"  AND PMX_PLPL."BaseType" = WTQ1."ObjType"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = 1250000001 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR ISNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51203;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' The item code on a warehouse inventory request line cannot be changed if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END
			
			SET @errorList2 = ''
			SELECT @errorList2 = @errorList2 + CASE WHEN ISNULL( WTQ1."OpenQty",0) < SUM(PMX_PLPL."Quantity") THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) + ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN WTQ1 ON PMX_PLPL."BaseEntry" = WTQ1."DocEntry" AND PMX_PLPL."BaseLine" = WTQ1."LineNum"  AND PMX_PLPL."BaseType" = WTQ1."ObjType"
			WHERE PMX_PLPL."BaseType" = 1250000001 AND PMX_PLPL."BaseEntry" = @keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C') 
			GROUP BY PMX_PLPL."BaseLine", WTQ1."OpenQty";
			IF LEN(@errorList2) <> 0 BEGIN
				SET @error = 51202;
				SET @error_message = 'Error on line(s):  ' + @errorList2 + ' The quantity on a warehouse inventory request line cannot be lowered if a picklist proposal is generated for it.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '
			FROM WTQ1 AS Lines
			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = WhsInvRequest."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"
						
			WHERE WhsInvRequest."DocEntry" = @keyAsInteger AND Lines."LineStatus" = 'O' AND Lines."U_PMX_AMBA" = 'N'
			GROUP BY Lines."LineNum"
			HAVING (Count(*)) > 1;
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51204;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END



			-- change of batch only allowed when not linked in PMX_PLPL and PMX_DOLL
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(Lines."LineNum" AS VARCHAR) + ' '		
			FROM WTQ1 AS Lines
			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"	
			
			INNER JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType" 
			AND PMX_PLPL."BaseEntry" = Lines."DocEntry" 
			AND PMX_PLPL."BaseLine" = Lines."LineNum" 
			
			INNER JOIN PMX_DOLL ON PMX_DOLL."ObjType" = Lines."ObjType" 
			AND PMX_DOLL."DocEntry" = Lines."DocEntry" 
			AND PMX_DOLL."LineNum" = Lines."LineNum" 
			
			INNER JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			
			LEFT JOIN 
			(	SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
								GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH

			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"		

			WHERE Lines."DocEntry" = @keyAsInteger
			AND Lines."LineStatus" = 'O'
			AND( ISNULL( BATCH."AllocQty", PMX_DOLL."LockedQuantity") < PMX_DOLL."LockedQuantity"
			OR BATCH."DistNumber" <> ISNULL( PMX_ITRI."BatchNumber", BATCH."DistNumber"));
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51205;
				SET @error_message = 'Error on line(s):  ' + @errorList + ' Proposal exists: Changing the batch number(s) not allowed'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Multiple batches on serial numbers are only allowed when multiple batches on line level are allowed
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(WTQ1."LineNum" AS VARCHAR) + ' '	
			FROM PMX_SELD	
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN WTQ1 on WTQ1."DocEntry" = PMX_SELD."BaseEntry"
			AND WTQ1."LineNum" = PMX_SELD."BaseLine"
			AND WTQ1."ObjType" = PMX_SELD."BaseType"
			WHERE WTQ1.U_PMX_AMBA = 'N'
			AND WTQ1."DocEntry" = @keyAsInteger	
			GROUP BY WTQ1."LineNum", PMX_SENU."ItriKey"
			HAVING COUNT(PMX_SENU."ItriKey") > 1;
            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51206;
				SET @error_message = 'Serial numbers linked with multiple batches, but multiple batches are not allowed. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			--Linked serial numbers cannot be more than open quantity
			SET @errorList = ''
			SELECT @errorList = @errorList + 'Line ' + CAST(WTQ1."LineNum" AS VARCHAR) + ', Link qty:' + CAST(COUNT(PMX_SELD."InternalKey") AS VARCHAR) + '; '	
			FROM PMX_SELD
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN WTQ1 on WTQ1."DocEntry" = PMX_SELD."BaseEntry"
			AND WTQ1."LineNum" = PMX_SELD."BaseLine"
			AND WTQ1."ObjType" = PMX_SELD."BaseType"
			WHERE WTQ1."DocEntry" = @keyAsInteger
			AND PMX_SENU."InStock" = 'Y'
			AND WTQ1."LineStatus" = 'O'
			GROUP BY WTQ1."LineNum", WTQ1."OpenQty", WTQ1."NumPerMsr"
			HAVING COUNT(PMX_SELD."InternalKey") > WTQ1."OpenQty" * WTQ1."NumPerMsr";

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51207;
				SET @error_message = 'Number of linked serial numbers cannot be more than open quantity. Error on line(s):  ' + @errorList 
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date minutes
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(OWTQ.U_PMX_DDTM AS VARCHAR) 
			FROM OWTQ
			WHERE OWTQ."DocEntry" = @keyAsInteger
			AND ( ISNULL( OWTQ.U_PMX_DDTM, 0) > 59 OR
				ISNULL( OWTQ.U_PMX_DDTM, 0) < 0 );

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51208;
				SET @error_message = 'The due date minutes: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

			-- Due date Hour
			SET @errorList = ''
			SELECT @errorList = @errorList + ' ' + CAST(OWTQ.U_PMX_DDTH AS VARCHAR) 
			FROM OWTQ
			WHERE OWTQ."DocEntry" = @keyAsInteger
			AND (ISNULL( OWTQ.U_PMX_DDTH, 0) < 0
					OR (ISNULL( OWTQ.U_PMX_DDTH, 0) > 23 AND ISNULL( OWTQ.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR (ISNULL( OWTQ.U_PMX_DDTH, 0) > 11 AND ISNULL( OWTQ.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);

            IF LEN(@errorList) <> 0 BEGIN
				SET @error = 51209;
				SET @error_message = 'The due date hours: ' + @errorList + ' are not valid for document ' + @list_of_cols_val_tab_del
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE
			DELETE FROM LOCKING
            FROM PMX_INLD LOCKING
            INNER JOIN OWTQ ON OWTQ."DocEntry" = LOCKING."BaseEntry"
			AND OWTQ."ObjType" = LOCKING."BaseType"
			WHERE OWTQ."DocStatus" = 'C' AND OWTQ."DocEntry" = @keyAsInteger;


			--IF(@isTransactionTypeAdd=0 AND
			--	EXISTS(SELECT * FROM WTQ1 WHERE WTQ1."LineStatus" = 'C' AND WTQ1."DocEntry" = @keyAsInteger)) 
			--BEGIN

			--	--Close container (lines) if needed
			--	UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			--	FROM PMX_COLI
			--	INNER JOIN WTQ1
			--	ON PMX_COLI."BaseEntry" = WTQ1."DocEntry"
			--	AND PMX_COLI."BaseLine" = WTQ1."LineNum"
			--	AND PMX_COLI."BaseType" = WTQ1."ObjType"
			--	WHERE PMX_COLI."LineStatus" <> 'C'
			--	AND WTQ1."DocEntry" = @keyAsInteger
			--	AND WTQ1."LineStatus" = 'C'

			--	--Close header if no open lines are found
			--	UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
			--	PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = @tsNow, 
			--	PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM OWTQ  WHERE OWTQ."DocEntry" = @keyAsInteger)
			--	WHERE PMX_COHE."DocStatus" = 'O'
			--	AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O')

			--END


			--**********************************************************************************************************
			--START: Add/update locking based on batch numbers
			--**********************************************************************************************************



			--Start with deleting the lockings that are not correct
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" NOT IN 
			(--Query to get the correct lockings
				SELECT 	PMX_INLD."InternalKey"
				FROM WTQ1 AS Lines
				INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
				INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')
				--AND ISNULL( PMX_ITRI."BestBeforeDate", 1) = ISNULL( "ExpDate", 1)

				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"
				AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

				WHERE OITL."ApplyEntry" = @keyAsInteger 
				AND Lines."LineStatus" = 'O'
				AND "StockEff" = 2

				AND PMX_INLD."InternalKey" IS NOT NULL


				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				HAVING SUM("AllocQty") > 0

				-- Exclude locked stock created on goods receipt.
				UNION SELECT PMX_INLD."InternalKey"
				FROM WTQ1 AS Lines 
				INNER JOIN POR1 ON 
				POR1."BaseType" = Lines."ObjType"
				AND POR1."BaseEntry" = Lines."DocEntry"
				AND POR1."BaseLine" = Lines."LineNum"

				INNER JOIN PDN1 ON 
				PDN1."BaseType" = POR1."ObjType"
				AND PDN1."BaseEntry" = POR1."DocEntry"
				AND PDN1."BaseLine" = POR1."LineNum"

				INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
				LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
				LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 ISNULL( PMX_ITRI."BatchNumber", '') = ISNULL( "DistNumber", '')


				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"

				WHERE Lines."DocEntry" = @keyAsInteger				

				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				--Only allow detailed locking from purchase order when no batch has been linked
				HAVING SUM( ISNULL( "AllocQty", 0) ) = 0 

				--Exclude manual lockings on LUID level when no batch has been selected on warehouse inventory request
				UNION SELECT PMX_INLD."InternalKey"
				FROM PMX_INLD 
				LEFT JOIN
				(
					SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
					FROM WTQ1 AS Lines
									INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
									LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
									LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
									LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
									AND ITL1."SysNumber" = OBTN."SysNumber"

					WHERE OITL."ApplyEntry"  = @keyAsInteger 
									AND Lines."LineStatus" = 'O'
									AND ISNULL( "StockEff", 2) = 2

						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
									Lines."NumPerMsr", "DistNumber", "ExpDate"

					HAVING SUM( ISNULL( "AllocQty", 0) ) > 0
				) AS OWTQ

				ON OWTQ."LineNum" = PMX_INLD."BaseLine"
				AND OWTQ."DocEntry" = PMX_INLD."BaseEntry"
				AND OWTQ."ObjType" = PMX_INLD."BaseType"


				WHERE PMX_INLD."InvLockLevel" = 'L'
				AND PMX_INLD."BaseEntry" = @keyAsInteger
				AND OWTQ."DocEntry" IS NULL

				

			) 

			AND PMX_INLD."BaseType" = '1250000001'
			AND PMX_INLD."BaseEntry" = @keyAsInteger;



			BEGIN TRY

				DECLARE WhsInvRequestBatchCursor CURSOR LOCAL FOR 
						SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."ItemCode", PMX_OSWH."Code"
						FROM WTQ1 AS Lines
						INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
						INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = WhsInvRequest."Filler"

						LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType"
						AND PMX_PLPL."BaseEntry" = Lines."DocEntry"
						AND PMX_PLPL."BaseLine" = Lines."LineNum"
						AND PMX_PLPL."LineStatus" = 'O'

						WHERE OITL."ApplyEntry" = @keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2
						--Only when no openpick list proposal is found
						AND PMX_PLPL."DocEntry" IS NULL
						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code"

						HAVING SUM("AllocQty") > 0;


				OPEN WhsInvRequestBatchCursor;
				FETCH NEXT FROM WhsInvRequestBatchCursor 
				INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock,@pmxWarehouseToLock;
				WHILE @@FETCH_STATUS >= 0 BEGIN

					--Check for each batch if we have a corresponding locking line
					IF NOT EXISTS (SELECT TOP(1) PMX_INLD.InternalKey FROM PMX_INLD 
    					INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
    					WHERE PMX_INLD."BaseType" = @typeToLock
    					AND PMX_INLD."BaseEntry" = @docEntryToLock
    					AND PMX_INLD."BaseLine" = @lineNumToLock
    					AND PMX_ITRI."BatchNumber" = @batchToLock
    					--AND ISNULL( PMX_ITRI."BestBeforeDate", '') = ISNULL( @expDateToLock, '')
					)
					BEGIN

						--Entry does not exist, so we need to lock the stock
						--First try to get the itri key to lock
						SET @remainingQuantityToLock = @batchQuantityToLock;

						BEGIN TRY
							DECLARE WhsInvRequestinldCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;


							OPEN WhsInvRequestinldCursor;
							FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
							WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

								IF(@remainingQuantityToLock < @freeQuantityToLock)
								BEGIN
									SET @currentQuantityToLock = @remainingQuantityToLock
								END
								ELSE
								BEGIN
									SET @currentQuantityToLock = @freeQuantityToLock
								END

								--Insert INTO the PMX_INLD table
								INSERT INTO PMX_INLD ( "BaseType","BaseEntry","BaseLine","CardCode","CardName",
									"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
									"UserSign","CreateDate",
									"CreateTime","LogUnitIdentKey","QualityStatusCode", 
									"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
								SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(OWTQ."CardCode"),MAX(OWTQ."CardName"),
									@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
									MAX(OWTQ."UserSign"),@updateDate,
									@updateTime,NULL, @qualityStatusToLock,
									'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0
								FROM OWTQ 
								INNER JOIN WTQ1 ON WTQ1."DocEntry" = OWTQ."DocEntry"
								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OWTQ."Filler"
								WHERE OWTQ."DocEntry" = @docEntryToLock
								AND WTQ1."LineNum" = @lineNumToLock

								--Reduce remaining
								SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


								FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
							END
							CLOSE WhsInvRequestinldCursor;
							DEALLOCATE WhsInvRequestinldCursor

							--If there is remaining QTY, there is not enough free stock, so throw error
							IF(@remainingQuantityToLock > 0)
							BEGIN
								SET @error = 51250;
								SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock + '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN;

							END

						END TRY
						BEGIN CATCH
							SET @error = 51290
							SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
							+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
							BEGIN TRY
								CLOSE WhsInvRequestinldCursor 
								DEALLOCATE WhsInvRequestinldCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN
							END TRY
							BEGIN CATCH
								DEALLOCATE WhsInvRequestinldCursor
								SELECT @error, @error_message
								EXEC xp_logevent @error, @error_message, ERROR
								RETURN
							END CATCH
						END CATCH

					END
					ELSE 
					BEGIN
						--The line exists, so now we need to check if the quantity is the same/more/less
						SET @remainingQuantityToLock = (
                            SELECT SUM(PMX_INLD."Quantity") -  @batchQuantityToLock AS "Quantity"
                            FROM PMX_INLD 
							INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
							WHERE PMX_INLD."BaseType" = @typeToLock
							AND PMX_INLD."BaseEntry" = @docEntryToLock
							AND PMX_INLD."BaseLine" = @lineNumToLock
							AND PMX_ITRI."BatchNumber" = @batchToLock
							GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine"
                        );
							
										
						IF @remainingQuantityToLock <> 0 BEGIN
						--Only if deviation, then we try to update/insert
						--Loop through all locking lines
							BEGIN TRY
								DECLARE WhsInvRequestinldCursor CURSOR LOCAL FOR SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey" FROM PMX_INLD 
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = @typeToLock
								AND PMX_INLD."BaseEntry" = @docEntryToLock
								AND PMX_INLD."BaseLine" = @lineNumToLock
								AND PMX_ITRI."BatchNumber" = @batchToLock;

								OPEN WhsInvRequestinldCursor;
								FETCH NEXT FROM WhsInvRequestinldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock <> 0 BEGIN

									IF @remainingQuantityToLock > 0 BEGIN

										--Remaining is more than 0, so we can just remove the locking
										IF @remainingQuantityToLock >= @lockedQuantity BEGIN
											--Delete
											DELETE FROM PMX_INLD WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = @remainingQuantityToLock - @lockedQuantity;
										END
                                        ELSE
                                        BEGIN
											--update
											UPDATE PMX_INLD SET "Quantity" = "Quantity" - @remainingQuantityToLock WHERE "InternalKey" = @currentLockInternalKey;
											SET @remainingQuantityToLock = 0;
										END
									END
                                    ELSE
                                    BEGIN
										--Remaining is less than 0, so we need to add locking
										--If there is still free stock for current itri, then take that one
										SELECT @freeQuantityToLock = ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
										AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
										AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0
										AND PMX_ITRI."InternalKey" = @itriToLock;

										IF @remainingQuantityToLock > @freeQuantityToLock BEGIN
											SET @currentQuantityToLock = @freeQuantityToLock;
    									END
                                        ELSE
                                        BEGIN
											SET @currentQuantityToLock = @remainingQuantityToLock;
										END

										UPDATE PMX_INLD SET "Quantity" = "Quantity" + @currentQuantityToLock WHERE PMX_INLD."InternalKey" = @currentLockInternalKey;

										-- add qty, because @remainingQuantityToLock is negative
										SET @remainingQuantityToLock = @remainingQuantityToLock + @currentQuantityToLock;
										
									END

									FETCH NEXT FROM WhsInvRequestinldCursor INTO @currentLockInternalKey, @lockedQuantity, @itriToLock;
								END
								CLOSE WhsInvRequestinldCursor;
								DEALLOCATE WhsInvRequestinldCursor

							END TRY
							BEGIN CATCH
								SET @error = 51291
								SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
								+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
								BEGIN TRY
									CLOSE WhsInvRequestinldCursor 
									DEALLOCATE WhsInvRequestinldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END TRY
								BEGIN CATCH
									DEALLOCATE WhsInvRequestinldCursor
									SELECT @error, @error_message
									EXEC xp_logevent @error, @error_message, ERROR
									RETURN
								END CATCH
							END CATCH

							IF @remainingQuantityToLock <> 0 BEGIN
								-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
								--Set @remainingQuantityToLock as positive quanitty
								SET @remainingQuantityToLock = ABS( @remainingQuantityToLock );
								

								BEGIN TRY
									DECLARE WhsInvRequestinldCursor CURSOR LOCAL FOR SELECT PMX_ITRI."InternalKey", ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
									LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
									INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
									WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
									LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
									WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
									AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
									WHERE PMX_ITRI."BatchNumber" = @batchToLock AND PMX_ITRI."ItemCode" = @itemCodeToLock
									AND Inventory."PmxWhsCode" = @pmxWarehouseToLock
									AND ISNULL( Inventory."Quantity", 0) - ISNULL( Locking."Quantity", 0) > 0;

									OPEN WhsInvRequestinldCursor;
									FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									WHILE @@FETCH_STATUS >= 0 AND @remainingQuantityToLock > 0 BEGIN

										IF @remainingQuantityToLock < @freeQuantityToLock
											SET @currentQuantityToLock = @remainingQuantityToLock;
										ELSE
											SET @currentQuantityToLock = @freeQuantityToLock;

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ( "BaseType","BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
										SELECT @typeToLock,@docEntryToLock,@lineNumToLock,MAX(OWTQ."CardCode"),MAX(OWTQ."CardName"),
											@itemCodeToLock,@currentQuantityToLock,@itriToLock,NULL,
											MAX(OWTQ."UserSign"),@updateDate,
											@updateTime,NULL, @qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0
										FROM OWTQ 
										INNER JOIN WTQ1 ON WTQ1."DocEntry" = OWTQ."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OWTQ."Filler"
										WHERE OWTQ."DocEntry" = @docEntryToLock
										AND WTQ1."LineNum" = @lineNumToLock

										--Reduce remaining
										SET @remainingQuantityToLock = @remainingQuantityToLock - @currentQuantityToLock;


										FETCH NEXT FROM WhsInvRequestinldCursor INTO @itriToLock, @freeQuantityToLock, @qualityStatusToLock;
									END
									CLOSE WhsInvRequestinldCursor;
									DEALLOCATE WhsInvRequestinldCursor

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(@remainingQuantityToLock > 0)
									BEGIN
										SET @error = 51252;
										SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN;
									END

								END TRY
								BEGIN CATCH
									SET @error = 51291
									SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
									+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE WhsInvRequestinldCursor 
										DEALLOCATE WhsInvRequestinldCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE WhsInvRequestinldCursor
										SELECT @error, @error_message
										EXEC xp_logevent @error, @error_message, ERROR
										RETURN
									END CATCH
								END CATCH

							END

						END

						IF @remainingQuantityToLock <> 0 BEGIN
						-- if quantity is still different from 0, then there was not enough stock to reserve!
							CLOSE WhsInvRequestBatchCursor;
							DEALLOCATE WhsInvRequestBatchCursor;
							SET @error = 51253;
							SET @error_message = 'Not enough free stock to allocate batch ' + @batchToLock + ' of item ' + @itemCodeToLock+ '. Missing qty: ' + CAST(@remainingQuantityToLock AS NVARCHAR(20))
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END

					END


					FETCH NEXT FROM WhsInvRequestBatchCursor
					INTO @batchQuantityToLock,@batchToLock,@expDateToLock,@lineNumToLock,@docEntryToLock,@typeToLock,@itemCodeToLock, @pmxWarehouseToLock;
				END
				CLOSE WhsInvRequestBatchCursor;
				DEALLOCATE WhsInvRequestBatchCursor;
			END TRY
			BEGIN CATCH
				SET @error = 51299
				SET @error_message = 'Error On batch allocation: (' + CAST(ERROR_NUMBER() AS VARCHAR) + ') ' + ERROR_MESSAGE()
				+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE WhsInvRequestBatchCursor 
					DEALLOCATE WhsInvRequestBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE WhsInvRequestBatchCursor
					SELECT @error, @error_message
					EXEC xp_logevent @error, @error_message, ERROR
					RETURN
				END CATCH
			END CATCH

			--**********************************************************************************************************
			--END: Add/update locking based on batch numbers
			--**********************************************************************************************************

			--**********************************************************************************************************
			--START: SAP Serial numbers
			--**********************************************************************************************************

			/*--Delete system serial links
			DELETE FROM PMX_SELD WHERE "LinkStatus" = 'S'
			AND PMX_SELD."BaseEntry" = @keyAsInteger
			AND PMX_SELD."BaseType" = '1250000001';

			--Add to PMX serial numbers table

			INSERT INTO PMX_SELD ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine", "LinkStatus") 
			SELECT 'N', SERIAL."UserSign", @tsNow, @tsNow, 1, PMX_SENU."InternalKey", "ApplyType", "ApplyEntry", "ApplyLine", 'S'	
			FROM 
				(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty", "StockEff", OWTQ."UserSign"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
				AND ITL1."SysNumber" = OSRN."SysNumber"
				INNER JOIN OWTQ ON OWTQ."DocEntry" = OITL."ApplyEntry"
				INNER JOIN WTQ1 ON WTQ1."DocEntry" = OITL."ApplyEntry"
				AND WTQ1."LineNum" = OITL."ApplyLine"
				AND WTQ1."ObjType" = OITL."ApplyType"

				WHERE "StockEff" = 2
				AND OITL."ApplyEntry" = @keyAsInteger
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", OWTQ."UserSign"
				HAVING SUM("AllocQty") > 0 AND "DistNumber" IS NOT NULL
				) AS SERIAL

			INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = SERIAL."ItemCode"
			AND PMX_SENU."SerialNumber" = SERIAL."DistNumber"
			LEFT JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
			AND PMX_SELD."BaseEntry" = SERIAL."ApplyEntry"
			AND PMX_SELD."BaseLine" = SERIAL."ApplyLine"
			AND PMX_SELD."BaseType" = SERIAL."ApplyType"
			WHERE PMX_SELD."InternalKey" IS NULL;*/



			--**********************************************************************************************************
			--END: SAP Serial numbers
			--**********************************************************************************************************


			RETURN;

		END

		--**********************************************************************************************************
		--Do checks for stock transfer and add lines to detail inventory
		--**********************************************************************************************************
		IF (@isStockTransfer=1) AND (@isTransactionTypeAdd=1 OR @isTransactionTypeCancel=1) BEGIN
			--Convert the list of key columns to an integer key
			SET @keyAsInteger = CAST( @list_of_cols_val_tab_del AS INT );

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table
			--**********************************************************************************************************
			IF (@isTransactionTypeAdd=1 OR @isTransactionTypeCancel=1) BEGIN
				--One stock transfer line represents 2 lines in the detail inventory table:
				-- *The destination as positive lines
				-- *The source as negative lines

				--Insert source lines -> negative
				INSERT INTO @inventoryDetailTable
					SELECT OWTR."ObjType", OWTR."DocEntry", OWTR."DocNum", OWTR."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OWTR."CardCode", OWTR."CardName", Lines."ItemCode", 
						  Lines."Dscription", -Lines."Quantity", Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", -Lines."QuantityUom2", -Lines."InventoryQuantity"
					FROM  OWTR 
						  INNER JOIN @documentLineTable AS Lines ON OWTR."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
		 				  INNER JOIN OWHS ON OWHS."WhsCode" = Lines."WhsCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y' AND OWHS.U_PMX_IMBP = 'Y';


				--Insert destination lines -> positive
				INSERT INTO @inventoryDetailTable
					SELECT OWTR."ObjType", OWTR."DocEntry", OWTR."DocNum", OWTR."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OWTR."CardCode", OWTR."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity", Lines."DestinationStorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."DestinationLUID", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", PMX_LUID.SSCC, 
						  Lines."DestinationQualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2", Lines."InventoryQuantity"
					FROM  OWTR 
						  INNER JOIN @documentLineTable AS Lines ON OWTR."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
		 				  INNER JOIN OWHS ON OWHS."WhsCode" = Lines."DestinationWhsCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."DestinationLUID"
					WHERE OITM."InvntItem" = 'Y' AND OWHS.U_PMX_IMBP = 'Y';
					
									
					
			--			--Delete the previous links
				--DELETE FROM Links
    --            FROM PMX_LMLU Links
				--INNER JOIN PMX_MVLI Lines ON Links.LUID = Lines."DestLogUnitIdentKey"
				--WHERE Lines."DocEntry" = @keyAsInteger;
				
			--Delete links when all stock from luid is moved
				DELETE Links FROM PMX_LMLU Links
				WHERE Links.LUID IN 
				(
					SELECT Inventory.LogUnitIdentKey FROM @documentLineTable Lines
					INNER JOIN PMX_LMLU Links ON Links.LUID = Lines."LogUnitIdentKey"
					INNER JOIN PMX_INVT Inventory ON Inventory.LogUnitIdentKey = Links.LUID 
					WHERE Lines.DocEntry = @keyAsInteger AND Lines."MasterLUID" IS NULL 
					GROUP BY Inventory.LogUnitIdentKey
					HAVING SUM(Inventory.Quantity) = SUM(Lines.Quantity * Lines."QuantityPerUom")
				)
				
				IF (@isTransactionTypeAdd=1 ) BEGIN	
				
			        ----Add new links
				    INSERT INTO PMX_LMLU   ( "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime",  "Version", "MasterLUID", "LUID")											
				    SELECT DISTINCT 'N', Header."UserSign", @tsNow ,@tsNow, 0, Lines."MasterLUID", Lines."LogUnitIdentKey"
				    FROM	@documentLineTable Lines INNER JOIN
						    OWTR Header ON Lines."DocEntry" = Header."DocEntry" LEFT JOIN
						    PMX_LMLU Links ON Lines."LogUnitIdentKey" = Links.LUID AND Lines."LogUnitIdentKey" = Links."MasterLUID"
				    WHERE	Lines."DocEntry" = @keyAsInteger 
					    AND Lines."MasterLUID" IS NOT NULL
					    AND Links."InternalKey" IS NULL	
								
				END
					
			END

		END

        -- *** Check if destination location can accept more LU's ***			
		SET @errorList = NULL
		SELECT @errorList = ISNULL( @errorList + ', ', '' ) + t."StorLocCode"
        FROM (
           SELECT "StorLocCode", "LogUnitIdentKey"
           FROM @inventoryDetailTable
           WHERE "Quantity" > 0
           UNION  -- 'UNION' will only get the distinct rows as needed
           SELECT PMX_INVT."StorLocCode", PMX_INVT."LogUnitIdentKey"
           FROM @inventoryDetailTable idt
           INNER JOIN PMX_INVT ON PMX_INVT."StorLocCode" = idt."StorLocCode"
           WHERE idt."Quantity" > 0
        ) t
        INNER JOIN PMX_OSSL ON PMX_OSSL."Code" = "StorLocCode"
        GROUP BY t."StorLocCode", PMX_OSSL."MaxLogUnits"
        HAVING COUNT(*) > PMX_OSSL."MaxLogUnits";
		IF @@ROWCOUNT > 0 BEGIN
			SET @error = 50116;
			SET @error_message = 'Error: The destination location ' + @errorList + 'has insufficient free space.'
			SELECT @error, @error_message
			EXEC xp_logevent @error, @error_message, ERROR
			RETURN;
		END


		--Check if an item with SAP serial number and track locations has an LUID (On a moveable location, no LUID is required)
		SET @errorList = ''
		SELECT @errorList = @errorList + Lines."ItemCode" + ' ' 
		FROM @inventoryDetailTable AS Lines
		INNER JOIN OITM ON OITM."ItemCode" = Lines."ItemCode"
		LEFT JOIN PMX_SENU ON PMX_SENU.LUID = Lines."LogUnitIdentKey"
		--LEFT JOIN PMX_OSML ON PMX_OSML."Code" = Lines."StorLocCode"
		WHERE OITM.U_PMX_TLSN = 'Y' AND Lines."LogUnitIdentKey" IS NULL
		AND Lines."StorLocCode" NOT IN ( SELECT "StorLocLostAndFound" FROM PMX_OSWH WHERE "StorLocLostAndFound" IS NOT NULL )
		--AND PMX_OSML."Code" IS NULL
		AND PMX_SENU."InternalKey" IS NOT NULL;
		IF LEN(@errorList) <> 0 BEGIN
			SET @error = 50117;
			SET @error_message = 'Serial nrs. locations are tracked, but no LUID is set for item items(s): ' + @errorList 
			SELECT @error, @error_message
			EXEC xp_logevent @error, @error_message, ERROR
			RETURN;
		END
		
		IF (@isTransactionTypeAdd = 1 AND ( @isPurchaseDeliveryDocument = 1 OR @isStockTransfer = 1 OR @isInventoryEntry = 1))
		BEGIN

			DECLARE @cardCodeAINL  NVARCHAR(15);
			DECLARE @cardNameAINL  NVARCHAR(100);
			DECLARE @internalKeyAINL  INT;
			DECLARE @itemCodeAINL  NVARCHAR(20);
			DECLARE @quantityAINL  NUMERIC(19,6);
			DECLARE @sboWhsCodeAINL  NVARCHAR(8);

			--*****************************************************************
			--* Lock stock if stock is available in Advance inventory locking *
			--*****************************************************************

			--Declare memory table for storing the details of the inventory (Temp table)
			DECLARE @tempDetailTable TABLE (
				"ItemCode" NVARCHAR (20) NOT NULL ,
				"ItemName" NVARCHAR (100)  NULL ,
				"Quantity" NUMERIC(19, 6) NOT NULL ,
				"StorLocCode" NVARCHAR (50) NOT NULL ,
				"ItemTransactionalInfoKey" INT  NULL ,
				"SerialNum" NVARCHAR (17)  NULL ,
				"LogUnitIdentKey" INT NULL ,
				"CreatedBy" INT NULL ,
				"CreateDate" DATETIME NULL ,
				"CreateTime" INT NULL ,
				"SSCC" NVARCHAR (18) NULL ,
				"QualityStatusCode" NVARCHAR (8) NULL,
				"Uom" NVARCHAR(20) NULL,
				"InventoryUom" NVARCHAR(20) NULL,
				"QuantityPerUom" NUMERIC(19,6) NULL,
				"Uom2" NVARCHAR(20) NULL,
				"QuantityUom2" NUMERIC(19,6) NULL,
				"BaseType" NVARCHAR(20) NULL ,
				"BaseEntry" INT NULL ,
				"BaseLine" SMALLINT NULL ,
                "SboWhsCode" NVARCHAR(8) NULL,
				"InternalKey" INT NOT NULL
			);

			--insert into temp table, because we will perform data actions on it
			INSERT INTO @tempDetailTable( "ItemCode", "ItemName", "Quantity", "StorLocCode", "ItemTransactionalInfoKey", 
				"SerialNum", "LogUnitIdentKey", "CreatedBy", "CreateDate", "CreateTime", "SSCC",
				"QualityStatusCode", "Uom", "InventoryUom", "QuantityPerUom", "Uom2", "QuantityUom2", "BaseType", 
				"BaseEntry", "BaseLine", "SboWhsCode", "InternalKey" ) 
			SELECT "ItemCode", "ItemName", "Quantity", "StorLocCode", "ItemTransactionalInfoKey", 
				"SerialNum", "LogUnitIdentKey", "CreatedBy", idt."CreateDate", idt."CreateTime", "SSCC",
				"QualityStatusCode", "Uom","InventoryUom", "QuantityPerUom", "Uom2", "QuantityUom2", "BaseType", 
				"BaseEntry", "BaseLine", PMX_OSWH."SboWhsCode", ROW_NUMBER() OVER ( ORDER BY "ItemCode", "BaseType", "BaseEntry", "BaseLine" )
			FROM @inventoryDetailTable idt
			LEFT JOIN ( -- inner join because "StorLocCode" is mandatory ?
			    PMX_OSEL
			    INNER JOIN PMX_OSWH ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
			) ON PMX_OSEL."Code" = idt."StorLocCode"
			INNER JOIN PMX_QYST ON PMX_QYST."Code" = idt."QualityStatusCode"
			WHERE idt."Quantity" > 0
			AND (PMX_QYST."CanBeShipped" = 'Y' OR PMX_QYST."CanBeShippedUnderQuarantine" = 'Y');


			DECLARE advanceLockingCursor CURSOR LOCAL FOR
			SELECT dbo.PMX_AINL."ItemCode", dbo.PMX_AINL."Quantity", dbo.PMX_AINL."InternalKey", dbo.PMX_AINL."CardCode", dbo.PMX_AINL."CardName", PMX_AINL."SboWhsCode"
			FROM PMX_AINL
			WHERE "ItemCode" IN ( SELECT "ItemCode" FROM @tempDetailTable )
			ORDER BY PMX_AINL."ItemCode", PMX_AINL."Sequence";

			OPEN advanceLockingCursor;
			FETCH NEXT FROM advanceLockingCursor INTO @itemCodeAINL,@quantityAINL,@internalKeyAINL,@cardCodeAINL, @cardNameAINL, @sboWhsCodeAINL
			WHILE @@FETCH_STATUS >= 0 BEGIN

				DECLARE @tempQuantityAINL  NUMERIC(19,6);
				DECLARE @currentInternalKey  INT;
				DECLARE @currentQuantityAINL  NUMERIC(19,6);

				SET @tempQuantityAINL = @quantityAINL;
				SET @currentInternalKey = -1;
				SET @currentQuantityAINL = 0;

				SELECT TOP 1 @currentQuantityAINL = "Quantity" * "QuantityPerUom", @currentInternalKey = "InternalKey" 
				FROM @tempDetailTable AS INV 
				WHERE INV."ItemCode" = @itemCodeAINL AND "Quantity" > 0 AND ( INV."SboWhsCode" = @sboWhsCodeAINL OR @sboWhsCodeAINL IS NULL OR @sboWhsCodeAINL = '' );

				WHILE @currentQuantityAINL > 0 AND @tempQuantityAINL > 0 BEGIN

					--check if we find a linked locking
					--IF this is the case, we do not need another locking 				

					IF NOT EXISTS ( SELECT InvTable.ItemCode
						FROM @tempDetailTable AS InvTable
						INNER JOIN POR1 ON 
						    POR1."DocEntry" = InvTable."BaseEntry" AND
						    POR1."LineNum" = InvTable."BaseLine" AND
						    POR1."ObjType" = InvTable."BaseType" 
						INNER JOIN PMX_INLD ON 
						    POR1."BaseEntry" = PMX_INLD."BaseEntry" AND
						    POR1."BaseLine" = PMX_INLD."BaseLine" AND
						    POR1."BaseType" = PMX_INLD."BaseType" 
						WHERE InvTable."InternalKey" = @currentInternalKey
    				) BEGIN

						DECLARE @quantityToReduce  NUMERIC(19,6);
						SET @quantityToReduce = @tempQuantityAINL;

						IF(@quantityToReduce> @currentQuantityAINL) BEGIN
							SET @quantityToReduce = @currentQuantityAINL;
						END

						IF EXISTS ( SELECT PMX_INLD.InternalKey FROM PMX_INLD 
									INNER JOIN @tempDetailTable AS InvTable ON
									    InvTable."ItemCode" = PMX_INLD."ItemCode"
									AND InvTable."ItemCode" = PMX_INLD."ItemCode"
									AND InvTable."ItemTransactionalInfoKey" = PMX_INLD."ItemTransactionalInfoKey"
									AND InvTable."SerialNum" = PMX_INLD."SerialNum"
									AND InvTable."LogUnitIdentKey" = PMX_INLD."LogUnitIdentKey"
									AND InvTable."QualityStatusCode" = PMX_INLD."QualityStatusCode"
									WHERE "CardCode" = @cardCodeAINL AND PMX_INLD."BaseType" IS NULL AND PMX_INLD."BaseEntry" IS NULL 
									AND PMX_INLD."BaseLine" IS NULL AND PMX_INLD."ItemCode" = @itemCodeAINL 
									AND InvTable."InternalKey" = @currentInternalKey
						) BEGIN

							--The entry already exists, so update the data
							UPDATE PMX_INLD SET "Version" = "Version" + 1, PMX_INLD."Quantity" = PMX_INLD."Quantity" + @quantityToReduce 
							FROM PMX_INLD 
							INNER JOIN @tempDetailTable AS InvTable ON
							    	InvTable."ItemCode" = PMX_INLD."ItemCode"
								AND InvTable."ItemCode" = PMX_INLD."ItemCode"
								AND InvTable."ItemTransactionalInfoKey" = PMX_INLD."ItemTransactionalInfoKey"
								AND InvTable."SerialNum" = PMX_INLD."SerialNum"
								AND InvTable."LogUnitIdentKey" = PMX_INLD."LogUnitIdentKey"
								AND InvTable."QualityStatusCode" = PMX_INLD."QualityStatusCode"
							WHERE "CardCode" = @cardCodeAINL AND PMX_INLD."BaseType" IS NULL AND PMX_INLD."BaseEntry" IS NULL 
								AND PMX_INLD."BaseLine" IS NULL AND PMX_INLD."ItemCode" = @itemCodeAINL 
								AND InvTable."InternalKey" = @currentInternalKey;
						END
						ELSE
                        BEGIN
							--Insert into locking
							INSERT INTO PMX_INLD ("BaseType","BaseEntry","BaseLine","CardCode","CardName",
							    "ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
							    "UserSign","CreateDate",
							    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
							    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
							SELECT  NULL, NULL,NULL,@cardCodeAINL,@cardNameAINL,
							    InvTable."ItemCode",@quantityToReduce,InvTable."ItemTransactionalInfoKey",InvTable."SerialNum",
							    InvTable."CreatedBy",InvTable."CreateDate",
							    InvTable."CreateTime",InvTable."LogUnitIdentKey", InvTable."QualityStatusCode",
							     'RESERVED', 'L' , PMX_OSEL."PmxWhsCode", 0
							FROM @tempDetailTable AS InvTable
							INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = InvTable."StorLocCode"
							WHERE InvTable."ItemCode" = @itemCodeAINL
							AND InvTable."InternalKey" = @currentInternalKey;
						END

						--Remove it from source temp table
						UPDATE @tempDetailTable SET "Quantity" = "Quantity" - @quantityToReduce WHERE "InternalKey" = @currentInternalKey;
						--Update the advance locking table
						UPDATE PMX_AINL SET "Quantity" = "Quantity" - @quantityToReduce WHERE "InternalKey" = @internalKeyAINL;

					END
					ELSE
                    BEGIN

						--Delete the current line from the temp table, so next time this is not taken in account
						DELETE FROM @tempDetailTable WHERE "InternalKey" = @currentInternalKey;

					END

					SET @tempQuantityAINL = @tempQuantityAINL - @quantityToReduce;

					IF (@tempQuantityAINL <= 0) BEGIN
						BREAK;
					END

					SET @currentInternalKey = -1;
					SET @currentQuantityAINL = 0;

                    SELECT TOP 1 @currentQuantityAINL = "Quantity", @currentInternalKey = "InternalKey" 
                    FROM @tempDetailTable AS INV 
                    WHERE INV."ItemCode" = @itemCodeAINL AND "Quantity" > 0 AND ( INV."SboWhsCode" = @sboWhsCodeAINL OR @sboWhsCodeAINL IS NULL OR @sboWhsCodeAINL = '' );
				
				END


				FETCH NEXT FROM advanceLockingCursor INTO @itemCodeAINL,@quantityAINL,@internalKeyAINL,@cardCodeAINL, @cardNameAINL, @sboWhsCodeAINL;
			END
			CLOSE advanceLockingCursor;
			DEALLOCATE advanceLockingCursor

			--Remove all not needed lockings
			DELETE FROM PMX_AINL WHERE "Quantity" <= 0;
			
		END
		
		
		
		--**********************************************************************************************************
		--Update Produmex inventory tables
		--**********************************************************************************************************

		--Insert inventory detail lines in real table
		INSERT INTO PMX_INVD ( "TransType","DocEntry","DocNum","DocDate","DocLineNum","BaseType","BaseEntry","BaseLine","CardCode","CardName","ItemCode","ItemName","Quantity","StorLocCode","ItemTransactionalInfoKey","SerialNum","LogUnitIdentKey","CreatedBy","CreateDate","CreateTime",SSCC,"QualityStatusCode","Uom","QuantityPerUom","Uom2","QuantityUom2") 
		SELECT "TransType","DocEntry","DocNum","DocDate","DocLineNum","BaseType","BaseEntry","BaseLine","CardCode","CardName","ItemCode","ItemName","Quantity","StorLocCode","ItemTransactionalInfoKey","SerialNum","LogUnitIdentKey","CreatedBy","CreateDate","CreateTime",SSCC,"QualityStatusCode","Uom","QuantityPerUom","Uom2","QuantityUom2"
        FROM @inventoryDetailTable;

		--Update total inventory table
		BEGIN TRY
    		--DECLARE @itemCode  nvarchar(20);
    		DECLARE @StorLocCode  nvarchar(50);
    		DECLARE @serialNum  nvarchar(17);
    		DECLARE @logUnitIdentKey  INT;
    		DECLARE @itemTransactionalInfoKey  INT;
    		--DECLARE @quantity  numeric(19, 6);
    		DECLARE @quantityInventoryTotal  numeric(19, 6);
    		DECLARE @quantityUom2InventoryTotal  numeric(19, 6);
    		DECLARE @inventoryTotalId  INT;
    		--DECLARE @sscc  nvarchar(18);
    		DECLARE @qualityStatusCode  NVARCHAR(8);
				
    		DECLARE @catchWeightBatchVarianceId  int;
    		DECLARE @quantityCatchWeightBatchVariance  numeric(19, 6);

				
			DECLARE invDetailCursor CURSOR LOCAL FOR 
            SELECT InvTable."ItemCode","StorLocCode","ItemTransactionalInfoKey",InvTable."SerialNum","LogUnitIdentKey", ROUND( "Quantity"*"QuantityPerUom", ISNULL( OITM.U_PMX_UOMD, 6) ) AS "Quantity", SSCC, "QualityStatusCode", "Uom2", "QuantityUom2", "InventoryUom" 
			FROM @inventoryDetailTable InvTable
			INNER JOIN OITM on OITM."ItemCode" = InvTable."ItemCode";

			OPEN invDetailCursor;
			FETCH NEXT FROM invDetailCursor INTO @itemCode,@StorLocCode,@itemTransactionalInfoKey,@serialNum,@logUnitIdentKey,@quantity,@sscc,@qualityStatusCode, @uom2, @quantityUom2, @uom;
			WHILE @@FETCH_STATUS >= 0 BEGIN

				--Because it is not possible to do column = @param when @param is NULL we have to do something special. We can use SET ANSI_NULLS OFF but this will not be supported in the future by Microsoft.
				--for every parameter that can be NULL we have to create a new variable that tells us if the parameter is NULL
				DECLARE @serialNumIsNull  SMALLINT;
				DECLARE @logUnitIdentKeyIsNull  SMALLINT;
				DECLARE @ssccIsNull  SMALLINT;
				DECLARE @itemTransactionalInfoKeyIsNull  SMALLINT ;

                SET @ssccIsNull = 0;
                SET @itemTransactionalInfoKeyIsNull = 0;

				IF @serialNum IS NULL SET @serialNumIsNull = 1; ELSE SET @serialNumIsNull = 0;
				IF @logUnitIdentKey IS NULL SET @logUnitIdentKeyIsNull = 1; ELSE SET @logUnitIdentKeyIsNull = 0;
				IF @sscc IS NULL SET @ssccIsNull = 1; ELSE SET @ssccIsNull = 0;
				IF @itemTransactionalInfoKey IS NULL SET @itemTransactionalInfoKeyIsNull = 1; ELSE SET @itemTransactionalInfoKeyIsNull = 0;

				--Get the summary line 
				SET @inventoryTotalId = NULL;
				SET @quantityInventoryTotal = 0;
				SET @quantityUom2InventoryTotal = 0;
				SELECT @inventoryTotalId = "InternalKey",@quantityInventoryTotal="Quantity",@quantityUom2InventoryTotal="QuantityUom2" 
                FROM PMX_INVT WITH(UPDLOCK)
                --FROM PMX_INVT WITH(XLOCK, ROWLOCK)
                WHERE "ItemCode" = @itemCode AND 
						"StorLocCode" = @StorLocCode AND 
						((@serialNumIsNull=1 AND "SerialNum" IS NULL) OR (@serialNumIsNull=0 AND "SerialNum"=@serialNum)) AND 
						((@logUnitIdentKeyIsNull=1 AND "LogUnitIdentKey" IS NULL) OR (@logUnitIdentKeyIsNull=0 AND "LogUnitIdentKey"=@logUnitIdentKey)) AND
						((@itemTransactionalInfoKeyIsNull=1 AND "ItemTransactionalInfoKey" IS NULL) OR (@itemTransactionalInfoKeyIsNull=0 AND "ItemTransactionalInfoKey"=@itemTransactionalInfoKey)) AND
						"QualityStatusCode"=@qualityStatusCode



				--Check if found an inventory total line		
				IF @inventoryTotalId IS NULL BEGIN

					IF (NOT EXISTS (SELECT Code FROM PMX_OSWH WHERE "StorLocLostAndFound" = @StorLocCode))
					BEGIN
						--Only check for quantity lower than 0 if it is not the lost and found location
						IF (@quantity < 0) BEGIN
							CLOSE invDetailCursor;
							DEALLOCATE invDetailCursor

							SET @error = 60000;
							SET @error_message = SUBSTRING('The quantity ' + ISNULL( CAST(@quantity AS VARCHAR),'') + ' goes below zero for item=' + ISNULL( @itemCode,'') + ' location=' + ISNULL( @StorLocCode,'') + ' batchID=' + ISNULL( CAST(@itemTransactionalInfoKey AS VARCHAR),'')  + ' "SerialNum"=' + ISNULL( @serialNum,'') + ' LUID=' + ISNULL( CAST(@logUnitIdentKey AS VARCHAR),'') + ' quality=' + ISNULL( @qualityStatusCode,'')  + ' . This is not allowed.',1,200);
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							
							RETURN;
						END
					END

					IF (@quantity = 0)
					BEGIN
						IF (@quantityUom2) <> 0 BEGIN
							--Update catch weight variance
							SET @catchWeightBatchVarianceId = NULL;
							SET @quantityCatchWeightBatchVariance = 0;
							
							SELECT @catchWeightBatchVarianceId = "InternalKey",@quantityCatchWeightBatchVariance="Quantity"
							FROM PMX_CWBV WITH(UPDLOCK)
							WHERE "ItemCode" = @itemCode AND 
									ISNULL( "ItriKey", 0) = ISNULL( @itemTransactionalInfoKey, 0)

							--Check if we find a variance line		
							IF @catchWeightBatchVarianceId IS NULL BEGIN
								INSERT INTO PMX_CWBV ("ItemCode","ItriKey","Quantity","Uom")
								VALUES (@itemCode, @itemTransactionalInfoKey, @quantityUom2, @uom2);
    						END
							ELSE
							BEGIN
								UPDATE PMX_CWBV SET "Quantity" = "Quantity" + (@quantityUom2)
								WHERE "InternalKey" = @catchWeightBatchVarianceId;
							END
						END 
					END
					ELSE BEGIN
					--Logging
	--				SET @MSG =  SUBSTRING('No inventory total table found for item='+ ISNULL( @itemCode,'') + ' storageLocation='  + ISNULL( @StorLocCode,'') + ' bestBeforeDate=' + ISNULL( CAST(@bestBeforeDate AS VARCHAR),'') + ' batchNumber=' + ISNULL( @batchNumber,'')  + ' "SerialNum"=' + ISNULL( @serialNum,'') + ' logisticUnitIdentificationKey=' + ISNULL( CAST(@logUnitIdentKey AS VARCHAR),'') + ' "QualityStatusCode"=' + ISNULL( @qualityStatusCode,''),1,255)
	--				EXEC xp_logevent 999999, @MSG, INFORMATIONAL
					INSERT INTO PMX_INVT ( "ItemCode","StorLocCode","ItemTransactionalInfoKey","SerialNum","LogUnitIdentKey","Quantity", SSCC, "QualityStatusCode", "Uom2", "QuantityUom2", "Uom")
                    VALUES (@itemCode,@StorLocCode,@itemTransactionalInfoKey,@serialNum,@logUnitIdentKey,@quantity, @sscc, @qualityStatusCode, @uom2, @quantityUom2, @uom);
					END
				END
                ELSE
                BEGIN

					IF (NOT EXISTS (SELECT Code FROM PMX_OSWH WHERE "StorLocLostAndFound" = @StorLocCode))
					BEGIN
						IF (@quantityInventoryTotal + @quantity) < 0 BEGIN
							CLOSE invDetailCursor;
							DEALLOCATE invDetailCursor

							SET @error = 60001
							SET @error_message = SUBSTRING('The quantity ' + ISNULL( CAST(@quantity AS VARCHAR),'') + ' goes below zero for item=' + ISNULL( @itemCode,'') + ' storageLocation=' + ISNULL( @StorLocCode,'') + ' "ItemTransactionalInfoKey"=' + ISNULL( CAST(@itemTransactionalInfoKey AS VARCHAR),'') + ' "SerialNum"=' + ISNULL( @serialNum,'') + ' logisticUnitIdentificationKey=' + ISNULL( CAST(@logUnitIdentKey AS VARCHAR),'') + ' "QualityStatusCode"=' + ISNULL( @qualityStatusCode,'')  + ' . This is not possible.',1,200);
							SELECT @error, @error_message
							EXEC xp_logevent @error, @error_message, ERROR
							RETURN;
						END
					END
					--Logging
	--				SET @MSG = SUBSTRING('Inventory total table with key=' + ISNULL( CAST(@inventoryTotalID AS VARCHAR),'') + ' found for item='+ ISNULL( @itemCode,'') + ' storageLocation='  + ISNULL( @StorLocCode,'') + ' bestBeforeDate=' + ISNULL( CAST(@bestBeforeDate AS VARCHAR),'') + ' batchNumber=' + ISNULL( @batchNumber,'')  + ' "SerialNum"=' + ISNULL( @serialNum,'') + ' logisticUnitIdentificationKey=' + ISNULL( CAST(@logUnitIdentKey AS VARCHAR),'') + ' "QualityStatusCode"=' + ISNULL( @qualityStatusCode,''),1,255)
	--				EXEC xp_logevent 999999, @MSG, INFORMATIONAL
	
					IF (@quantityInventoryTotal + @quantity) = 0 BEGIN

					
						--Quantity will be 0, so row can be deleted
						--But first we need to check the (catch) weight to see if we need to book the difference
						IF (@quantityUom2InventoryTotal + @quantityUom2) <> 0 BEGIN
							--Update catch weight variance
							SET @catchWeightBatchVarianceId = NULL;
							SET @quantityCatchWeightBatchVariance = 0;
							
							SELECT @catchWeightBatchVarianceId = "InternalKey",@quantityCatchWeightBatchVariance="Quantity"
							FROM PMX_CWBV WITH(UPDLOCK)
							WHERE "ItemCode" = @itemCode AND 
									ISNULL( "ItriKey", 0) = ISNULL( @itemTransactionalInfoKey, 0)

							--Check if we find a variance line		
							IF @catchWeightBatchVarianceId IS NULL BEGIN
								INSERT INTO PMX_CWBV ("ItemCode","ItriKey","Quantity","Uom")
								VALUES (@itemCode, @itemTransactionalInfoKey, @quantityUom2InventoryTotal + @quantityUom2, @uom2);
    						END
							ELSE
							BEGIN
								UPDATE PMX_CWBV SET "Quantity" = "Quantity" + (@quantityUom2InventoryTotal + @quantityUom2)
								WHERE "InternalKey" = @catchWeightBatchVarianceId;
							END
						END 
						DELETE FROM PMX_INVT WHERE "InternalKey" = @inventoryTotalId;
					END 
					ELSE BEGIN
						UPDATE PMX_INVT SET "Quantity" = "Quantity" + @quantity, "QuantityUom2" = "QuantityUom2" + @quantityUom2
						WHERE "InternalKey" = @inventoryTotalId;
                    END
				END

				FETCH NEXT FROM invDetailCursor INTO @itemCode, @StorLocCode,@itemTransactionalInfoKey,@serialNum,@logUnitIdentKey,@quantity,@sscc,@qualityStatusCode, @uom2, @quantityUom2, @uom;
			END
			CLOSE invDetailCursor;
			DEALLOCATE invDetailCursor
			
			
			----Update catch weight variances
			--BEGIN TRY

   -- 			DECLARE @catchWeightBatchVarianceId  int;
   -- 			DECLARE @quantityCatchWeightBatchVariance  numeric(19, 6);

			--	DECLARE catchWeightVarianceCursor CURSOR LOCAL FOR 
			--		SELECT PMX_INVT."ItemCode",PMX_INVT."ItemTransactionalInfoKey",PMX_INVT.QuantityUom2, PMX_INVT."Uom2"
			--		FROM PMX_INVT
			--		--INNER JOIN @inventoryDetailTable InvTable ON InvTable.ItemCode = PMX_INVT.ItemCode
			--		--AND ISNULL( PMX_INVT."ItemTransactionalInfoKey", 0) = ISNULL( InvTable."ItemTransactionalInfoKey", 0)
			--		WHERE PMX_INVT."Quantity" = 0 AND PMX_INVT."QuantityUom2" IS NOT NULL;

			--	OPEN catchWeightVarianceCursor;
			--	FETCH NEXT FROM catchWeightVarianceCursor INTO @itemCode,@itemTransactionalInfoKey,@quantity, @uom;
			--	WHILE @@FETCH_STATUS >= 0 BEGIN
			--		--Get the summary line 
			--		SET @catchWeightBatchVarianceId = NULL;
			--		SET @quantityCatchWeightBatchVariance = 0;
			--		SELECT @catchWeightBatchVarianceId = "InternalKey",@quantityCatchWeightBatchVariance="Quantity"
   --                 FROM PMX_CWBV WITH(UPDLOCK)
   --                 WHERE "ItemCode" = @itemCode AND 
			--				ISNULL( "ItriKey", 0) = ISNULL( @itemTransactionalInfoKey, 0)

			--		--Check if found an inventory total line		
			--		IF @catchWeightBatchVarianceId IS NULL BEGIN

			--			INSERT INTO PMX_CWBV ("ItemCode","ItriKey","Quantity","Uom")
   --                     VALUES (@itemCode, @itemTransactionalInfoKey, @quantity, @uom);
   -- 				END
   --                 ELSE
   --                 BEGIN
			--			UPDATE PMX_CWBV SET "Quantity" = "Quantity" + @quantity
   --                     WHERE "InternalKey" = @catchWeightBatchVarianceId;
			--		END

			--		FETCH NEXT FROM catchWeightVarianceCursor INTO @itemCode, @itemTransactionalInfoKey, @quantity, @uom
			--	END
			--	CLOSE catchWeightVarianceCursor;
			--	DEALLOCATE catchWeightVarianceCursor

			--END TRY
			--BEGIN CATCH
			--	SET @error = 60005
			--	SET @error_message = 'Unhandled error update Produmex catch weight variance: Number:' + CAST(ERROR_NUMBER() AS NVARCHAR(MAX)) + ' Message: ' + ERROR_MESSAGE()
			--	BEGIN TRY
			--		CLOSE catchWeightVarianceCursor 
			--		DEALLOCATE catchWeightVarianceCursor
			--		SELECT @error, @error_message
			--		EXEC xp_logevent @error, @error_message, ERROR
			--		RETURN
			--	END TRY
			--	BEGIN CATCH
			--		DEALLOCATE catchWeightVarianceCursor
			--		SELECT @error, @error_message
			--		EXEC xp_logevent @error, @error_message, ERROR
			--		RETURN
			--	END CATCH
			--END CATCH;

			--DELETE FROM PMX_INVT WHERE "Quantity" = 0; --AND ISNULL( "QuantityUom2", 0) = 0
			
			--Check if all Logistic Units are stored on the same storage location
			--It is not possible that a logistic unit is stored on more then 1 location.
			SET @errorList = ''
			SELECT @errorList = @errorList + CAST(PMX_INVENTORY_LUID_STORLOC."LogUnitIdentKey" AS VARCHAR) + ' ' 
			FROM PMX_INVENTORY_LUID_STORLOC
			WHERE ISNULL(PMX_INVENTORY_LUID_STORLOC."LogUnitIdentKey", -1) IN (
				SELECT DISTINCT ISNULL("LogUnitIdentKey", 0) FROM @inventoryDetailTable )
			GROUP BY PMX_INVENTORY_LUID_STORLOC."LogUnitIdentKey"
			HAVING (count(PMX_INVENTORY_LUID_STORLOC."LogUnitIdentKey") > 1);
			IF LEN(@errorList) <> 0 BEGIN
				SET @error = 70000;
				SET @error_message = 'The logistic unit(s) ' + @errorList + ' is/are stored on several storage locations. This is not possible. A logistic unit can only be stored on one storage location.'
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN;
			END

		END TRY
		BEGIN CATCH
			SET @error = 60002
			SET @error_message = 'Unhandled error update Produmex inventory tables: Number:' + CAST(ERROR_NUMBER() AS NVARCHAR(MAX)) + ' Message: ' + ERROR_MESSAGE()
			BEGIN TRY
				CLOSE invDetailCursor 
				DEALLOCATE invDetailCursor
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END TRY
			BEGIN CATCH
				DEALLOCATE invDetailCursor
				SELECT @error, @error_message
				EXEC xp_logevent @error, @error_message, ERROR
				RETURN
			END CATCH
		END CATCH;

		--**********************************************************************************************************
		--Do general checks
		--**********************************************************************************************************
		

	END TRY
	BEGIN CATCH
		SET @error = 60003
		SET @error_message = 'PMX_SP: sql error ' + CAST(ERROR_NUMBER() AS NVARCHAR) + ' : ' + ERROR_MESSAGE()
		   + ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' )
		SELECT @error, @error_message --> done at the end of SBO_SP_...
		EXEC xp_logevent @error, @error_message, ERROR
	END CATCH
	
END

----Following code must be added to the stored procedure SBO_SP_TransactionNotification
----to execute the PMX stored procedure:

----**********************************************************************************************************
----Start executing Produmex Logex Addon code
----**********************************************************************************************************
--IF @error = 0 BEGIN
--	BEGIN TRY
--		EXEC [dbo].[PMX_SP_TransactionNotification] 
--		@object_type,
--		@transaction_type,
--		@num_of_cols_in_key,
--		@list_of_key_cols_tab_del,
--		@list_of_cols_val_tab_del,
--		@error = @error OUTPUT,
--		@error_message = @error_message OUTPUT
--	END TRY
--	BEGIN CATCH
--		SET @error = ERROR_NUMBER()
--		SET @error_message = ERROR_MESSAGE()
--		DECLARE @msg as NVARCHAR(255)
--		SET @msg = SUBSTRING('PMX_SP: sql error ' + CAST(ERROR_NUMBER() AS NVARCHAR) + ' : ' + ERROR_MESSAGE()
--			+ ISNULL( ' line ' + CAST(ERROR_LINE() AS NVARCHAR), '' ) + ISNULL( ' in ' + ERROR_PROCEDURE(), '' ),1,255)
--		EXEC xp_logevent 999999, @msg, ERROR
--	END CATCH;
--END
----**********************************************************************************************************
----End executing Produmex Logex Addon code
----**********************************************************************************************************
