-- =============================================
-- Author:	PMX
-- Description:	Produmex extension on the SBO_SP_TransactionNotification
-- =============================================
CREATE PROCEDURE "PMX_SP_TransactionNotification" (
	-- Add the parameters for the stored procedure here
	p_object_type nvarchar(20), 			-- SBO Object Type
	p_transaction_type nchar(1),			-- [A]dd, [U]pdate, [D]elete, [C]ancel, C[L]ose
	p_num_of_cols_in_key int,
	p_list_of_key_cols_tab_del nvarchar(255),
	p_list_of_cols_val_tab_del nvarchar(255),
	OUT p_error int , -- Result (0 for no error)
	OUT p_error_message nvarchar (255) -- Error string to be displayed

) LANGUAGE SQLSCRIPT
AS
BEGIN
	--**********************************************************************************************************
	--Return and handling errors:
	--If a problem is found the function sets the parameters p_error and p_error_message
	--and logs the error in the windows event logging
	--**********************************************************************************************************
	
	--**********************************************************************************************************
	--Logging errors:
	--If there are errors and this procedure is called when using the DIAPI then the user does not see the error
	--Because of this problem the errors are also logged in the windows event logging
	--To log you have to use the function: --EXEC xp_logevent p_error, p_error_message, ERROR
	--**********************************************************************************************************

	--Logging
	DECLARE v_MSG NVARCHAR (255);
	DECLARE v_errorList NVARCHAR(255);
	DECLARE v_errorList2 VARCHAR(255);
	DECLARE v_count INT;

	DECLARE v_tsNow DATETIME;
	DECLARE v_updateDate DATETIME;
	DECLARE v_updateTime INT;
	
	--At this moment the function supports: purchase delivery note, sales delivery note, move, item and BOM
	DECLARE v_isPurchaseOrderDocument SMALLINT;
	DECLARE v_isPurchaseDeliveryDocument SMALLINT;
	DECLARE v_isPurchaseReturnDocument SMALLINT;
	DECLARE v_isPurchaseInvoice SMALLINT;
	DECLARE v_isPurchaseCreditNote SMALLINT;
	DECLARE v_isSalesDeliveryDocument SMALLINT;
	DECLARE v_isSalesReturnDocument SMALLINT;
	DECLARE v_isSalesOrderDocument SMALLINT;
	DECLARE v_isSalesInvoice SMALLINT;
	DECLARE v_isSalesCreditNote SMALLINT;
	DECLARE v_isInventoryEntry SMALLINT;
	DECLARE v_isInventoryExit SMALLINT;
	DECLARE v_isMoveDoc SMALLINT;
	DECLARE v_isItem SMALLINT;
	DECLARE v_isTransactionTypeAdd SMALLINT;
	DECLARE v_isTransactionTypeUpdate SMALLINT;
	DECLARE v_isTransactionTypeDelete SMALLINT;
	DECLARE v_isTransactionTypeCancel SMALLINT;
	DECLARE v_isBOM SMALLINT;
	DECLARE v_isUser SMALLINT;
	DECLARE v_isDocument SMALLINT;
	DECLARE v_isProductionOrder SMALLINT;
	DECLARE v_isItemBatchNumber SMALLINT;
	DECLARE v_isUomGroup SMALLINT;
	DECLARE v_isStockUpdate SMALLINT;
	DECLARE v_isBusinessPartner SMALLINT;
	DECLARE v_isStockTransfer SMALLINT;
	DECLARE v_isItemPackagingType SMALLINT;
	DECLARE v_isPickListType SMALLINT;
	DECLARE v_isPriority SMALLINT;
	DECLARE v_isFreightChargeDefinition SMALLINT;
	DECLARE v_isWarehouseTransferRequestDocument SMALLINT;
	DECLARE v_isWarehouse SMALLINT;
	DECLARE v_isBarcode SMALLINT;
	DECLARE v_isInventoryOpeningBalance SMALLINT;
	DECLARE v_isInventoryPosting SMALLINT;
	DECLARE v_isInventoryCounting SMALLINT;

	--General variables
	DECLARE v_keyAsInteger INT;
	DECLARE v_extraDB NVARCHAR(256);

	--Declaration of variables used for locking stock based on batches in sales order
	DECLARE v_batchToLock nvarchar (36);
	DECLARE v_batchQuantityToLock numeric (19,6);
	DECLARE v_expDateToLock DateTime;
	DECLARE v_shippingQualityOptionToLock nvarchar(30);
	DECLARE v_lineNumToLock int;
	DECLARE v_docEntryToLock int;
	DECLARE v_typeToLock nvarchar (20);
	DECLARE v_itemCodeToLock nvarchar (20);
	DECLARE v_itriToLock int;
	DECLARE v_qualityStatusToLock nvarchar (8);
	DECLARE v_pmxWarehouseToLock nvarchar (50);	
	DECLARE v_catchWeightRatioToLock numeric (19,6);

	DECLARE v_freeQuantityToLock numeric (19,6);
	DECLARE v_currentQuantityToLock numeric (19,6);
	DECLARE v_remainingQuantityToLock numeric (19,6);
	DECLARE v_lockedQuantity numeric (19,6);

	DECLARE v_currentLockInternalKey int;
	--END: Declaration of variables used for locking stock based on batches in sales order


	--Set default values
	p_error := 0;
	p_error_message := N'Ok';

	--Check if the object type is supported by Produmex
	-- DO NOT suppress this section, or the stored procedure might not compile anymore. HANA magic.
	IF p_object_type = N'xxxxxxxxxx' THEN 
		p_error := 50090;
		p_error_message := '"ObjType" ' || CAST(p_object_type AS NVARCHAR) || ' is not supported by the Produmex add-on!';
		SELECT p_error, p_error_message FROM dummy;
		-- --EXEC xp_logevent p_error, p_error_message, ERROR;
		RETURN;
	END IF;


	v_tsNow := CURRENT_TIMESTAMP;
	v_updateDate := TO_TIMESTAMP( TO_VARCHAR( v_tsNow, 'YYYY-MM-DD' ), 'YYYY-MM-DD' );
	v_updateTime := TO_INT( TO_VARCHAR( v_tsNow, 'HH24MI' ) );

	v_isPurchaseOrderDocument := 0;
	v_isPurchaseDeliveryDocument := 0;
	v_isPurchaseReturnDocument := 0;
	v_isPurchaseInvoice := 0;
	v_isPurchaseCreditNote := 0;
	v_isSalesDeliveryDocument := 0;
	v_isSalesReturnDocument := 0;
	v_isSalesOrderDocument := 0;
	v_isSalesInvoice := 0;
	v_isSalesCreditNote := 0;
	v_isInventoryEntry := 0;
	v_isInventoryExit := 0;
	v_isMoveDoc := 0;
	v_isTransactionTypeAdd := 0;
	v_isTransactionTypeUpdate := 0;
	v_isTransactionTypeDelete := 0;
	v_isTransactionTypeCancel := 0;
	v_isItem := 0;
	v_isBOM := 0;
	v_isUser := 0;
	v_isDocument := 0;
	v_isProductionOrder := 0;
	v_isItemBatchNumber := 0;
	v_isUomGroup := 0;
	v_isStockUpdate := 0;
	v_isBusinessPartner := 0;
	v_isStockTransfer := 0;
	v_isItemPackagingType := 0;
	v_isPickListType := 0;
	v_isPriority := 0;
	v_isFreightChargeDefinition := 0;
	v_isWarehouseTransferRequestDocument := 0;
	v_isWarehouse := 0;
	v_isBarcode := 0;
	v_isInventoryOpeningBalance := 0;
	v_isInventoryPosting := 0;
	v_isInventoryCounting := 0;


	--Test object type (this can be found in the SDK 'BoObjectTypes Enumeration'
	IF (p_object_type = N'20') THEN
		v_isPurchaseDeliveryDocument := 1;
	ELSEIF (p_object_type = N'PMX_MVHE') THEN
		v_isMoveDoc := 1;
	ELSEIF (p_object_type = N'22') THEN
		v_isPurchaseOrderDocument := 1;
	ELSEIF (p_object_type = N'4') THEN
		v_isItem := 1;
	ELSEIF (p_object_type = N'66') THEN
		v_isBOM := 1;
	ELSEIF (p_object_type = N'12') THEN
		v_isUser := 1;
	ELSEIF (p_object_type = N'15') THEN
		v_isSalesDeliveryDocument := 1;
	ELSEIF (p_object_type = N'59') THEN
		v_isInventoryEntry := 1;
	ELSEIF (p_object_type = N'60') THEN
		v_isInventoryExit := 1;
	ELSEIF (p_object_type = N'21') THEN
		v_isPurchaseReturnDocument := 1;
	ELSEIF (p_object_type = N'16') THEN
		v_isSalesReturnDocument := 1;
	ELSEIF (p_object_type = N'17') THEN
		v_isSalesOrderDocument := 1;
	ELSEIF (p_object_type = N'202') THEN
		v_isProductionOrder := 1;
	ELSEIF (p_object_type = N'106') THEN
		v_isItemBatchNumber := 1;
	ELSEIF (p_object_type = N'13') THEN
		v_isSalesInvoice := 1;
	ELSEIF (p_object_type = N'14') THEN
		v_isSalesCreditNote := 1;
	ELSEIF (p_object_type = N'18') THEN
		v_isPurchaseInvoice := 1;
	ELSEIF (p_object_type = N'19') THEN
		v_isPurchaseCreditNote := 1;
	ELSEIF (p_object_type = N'10000197') THEN
		v_isUomGroup := 1;
	ELSEIF (p_object_type = N'PMX_PRIO') THEN
		v_isPriority := 1;
	ELSEIF (p_object_type = N'58') THEN
		v_isStockUpdate := 1;
	ELSEIF (p_object_type = N'2') THEN
		v_isBusinessPartner := 1;
	ELSEIF (p_object_type = N'67') THEN
		v_isStockTransfer := 1;
	ELSEIF (p_object_type = N'64') THEN
		v_isWarehouse := 1;
	ELSEIF (p_object_type = N'PMX_IPTY') THEN
		v_isItemPackagingType := 1;
	ELSEIF (p_object_type = N'PMX_PLTY') THEN
		v_isPickListType := 1;
	ELSEIF (p_object_type = N'PMX_FCDE') THEN
		v_isFreightChargeDefinition := 1;
	ELSEIF (p_object_type = N'1250000001') THEN
		v_isWarehouseTransferRequestDocument := 1;
	ELSEIF (p_object_type = N'1470000062') THEN
		v_isBarcode := 1;
	ELSEIF (p_object_type = N'310000001') THEN
		v_isInventoryOpeningBalance := 1;
	ELSEIF (p_object_type = N'10000071') THEN
		v_isInventoryPosting := 1;
	ELSEIF (p_object_type = N'1470000065') THEN
		v_isInventoryCounting := 1;
	END IF;

	--Test transaction type
	IF (p_transaction_type = N'A') THEN
		v_isTransactionTypeAdd := 1;
	ELSEIF (p_transaction_type = N'U') THEN
		v_isTransactionTypeUpdate := 1;
	ELSEIF (p_transaction_type = N'D') THEN
		v_isTransactionTypeDelete := 1;
	ELSEIF (p_transaction_type = N'C') THEN
		v_isTransactionTypeCancel := 1;
	END IF;

	--Set if it is a document
	IF ((v_isPurchaseDeliveryDocument = 1) OR
		(v_isSalesDeliveryDocument = 1) OR
		(v_isInventoryEntry = 1) OR
		(v_isInventoryExit = 1) OR
		(v_isPurchaseReturnDocument = 1) OR
		(v_isSalesReturnDocument = 1) OR
		(v_isSalesInvoice = 1) OR
		(v_isSalesCreditNote = 1) OR
		(v_isPurchaseInvoice = 1) OR
		(v_isPurchaseCreditNote = 1) OR
		(v_isStockTransfer = 1) OR
		(v_isWarehouseTransferRequestDocument = 1) --OR
		--(v_isSalesOrderDocument = 1) --Sales order is a document but not a document that has changes the stock. With document we mean here documents that change stock
		) THEN
		v_isDocument := 1;
	END IF;

	--Check if we have to do something. If nothing to do then exit.	
	IF (NOT(
		(v_isPurchaseDeliveryDocument=1 AND v_isTransactionTypeAdd=1) OR				--Add of goods receipt(purchase delivery note) is supported
		(v_isMoveDoc=1 AND v_isTransactionTypeAdd=1) OR								--Add of move is supported
		(v_isItem=1)	OR																--All operations on an item are supported
		(v_isBOM=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR	--Add/update on a BOM
		(v_isUser=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1))	OR	--Add/update on a User
		(v_isSalesDeliveryDocument=1 AND LOCATE( N'APC',p_transaction_type ) > 0) OR		--Add of delivery (sales delivery note) is supported
		(v_isInventoryEntry=1 AND v_isTransactionTypeAdd=1) OR						--Add of inventory entry is supported
		(v_isInventoryExit=1 AND v_isTransactionTypeAdd=1) OR							--Add of inventory exit is supported
		(v_isPurchaseReturnDocument=1 AND v_isTransactionTypeAdd=1) OR				--Add of purchase return is supported
		(v_isSalesReturnDocument=1 AND v_isTransactionTypeAdd=1) OR					--Add of sales return is supported
		(v_isSalesOrderDocument=1) OR												--All operations on a sales order are supported
		(v_isPurchaseOrderDocument=1) OR												--All operations on a purchase order are supported
		(v_isProductionOrder=1) OR 	--Add/update of a production order is supported
		--(v_isItemBatchNumber=1 AND v_isTransactionTypeUpdate=1) OR					--Update of item batch numbers
		(v_isUomGroup=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR	--Add/update on UomGroup
		(v_isFreightChargeDefinition=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR	--Add/update on a Pmx "Uom"
		(v_isPriority=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR	--Add/update on a Pmx priority
		(v_isStockUpdate=1) OR														--Add/update of stock
		(v_isSalesInvoice=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR							--Add/update of sales invoice is supported
		(v_isSalesCreditNote=1 AND v_isTransactionTypeAdd=1) OR						--Add of sales credit note is supported
		(v_isPurchaseInvoice=1 AND v_isTransactionTypeAdd=1) OR						--Add of purchase invoice is supported
		(v_isPurchaseCreditNote=1 AND v_isTransactionTypeAdd=1) OR					--Add of purchase credit note is supported
		(v_isStockTransfer=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeCancel=1 OR p_transaction_type = N'P')) OR						--Add of stock transfer is supported
		(v_isItemPackagingType=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR 	--Add/update on a Pmx item packaging type
		(v_isWarehouse=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR 	--Add/update on SAP warehouse
		(v_isPickListType=1 AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1)) OR 	--Add/update on a Pmx pick list type
		(v_isBusinessPartner=1) OR	--Add/update on a business partner
		(v_isBarcode=1) OR	--Add/update on a barcode
		(v_isWarehouseTransferRequestDocument=1) OR
		(v_isInventoryCounting = 1) OR
		(v_isInventoryPosting = 1) OR
		(v_isInventoryOpeningBalance = 1)
		)) THEN
			RETURN;
	END IF;


	BEGIN -- TRY

		DECLARE v_cnt INT;
		DECLARE v_cnt2 INT;

		--Check if the object type is supported by Produmex
		IF (v_isInventoryPosting=1) THEN
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
		
			SELECT COUNT(1) INTO v_cnt FROM "IQR1" JOIN "OWHS" ON "OWHS"."WhsCode" = "IQR1"."WhsCode"
				WHERE "IQR1"."DocEntry" = v_keyAsInteger
				AND "OWHS"."U_PMX_IMBP" = 'Y';
			IF (v_cnt > 0) THEN
				p_error := 50097;
				p_error_message := 'Inventory posting documents (ObjType: ' || CAST(p_object_type AS NVARCHAR) || ') cannot reference warehouses that are managed by Produmex!';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END	IF;
			RETURN;
		END IF;
		
		IF (v_isInventoryCounting=1) THEN
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
		
			SELECT COUNT(1) INTO v_cnt FROM "INC1" JOIN "OWHS" ON "OWHS"."WhsCode" = "INC1"."WhsCode"
				WHERE "INC1"."DocEntry" = v_keyAsInteger
				AND "OWHS"."U_PMX_IMBP" = 'Y';
			IF (v_cnt > 0) THEN
				p_error := 50098;
				p_error_message := 'Inventory counting documents (ObjType: ' || CAST(p_object_type AS NVARCHAR) || ') cannot reference warehouses that are managed by Produmex!';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END	IF;
			RETURN;
		END IF;
		
		IF (v_isInventoryOpeningBalance=1) THEN
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
		
			SELECT COUNT(1) INTO v_cnt FROM "IQI1" JOIN "OWHS" ON "OWHS"."WhsCode" = "IQI1"."WhsCode"
				WHERE "IQI1"."DocEntry" = v_keyAsInteger
				AND "OWHS"."U_PMX_IMBP" = 'Y';
			IF (v_cnt > 0) THEN
				p_error := 50099;
				p_error_message := 'Opening balance documents (ObjType: ' || CAST(p_object_type AS NVARCHAR) || ') cannot reference warehouses that are managed by Produmex!';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END	IF;
			RETURN;
		END IF;
		
		--**********************************************************************************************************
		--Do that the stock update is allowed by the Produmex Add-On
		--Only stock changes that are supported by Produmex are allowed.
		--This line is put on top because this transaction notification contains 2 keys instead of one
		--**********************************************************************************************************
		IF (v_isStockUpdate=1) THEN
			DECLARE v_position INT;
			DECLARE v_transNum INT;
			DECLARE v_instance INT;
			DECLARE v_transType INT;
			DECLARE v_inQty NUMERIC(19, 6);
			DECLARE v_outQty NUMERIC(19, 6);

			--The number of columns must be always 2.
			IF (p_num_of_cols_in_key <> 2) THEN
				p_error := 50260;
				p_error_message := 'The number of columns for a stock update is always 2. In this case it is ' || CAST(p_num_of_cols_in_key AS VARCHAR) || ' AND the columns are ' || p_list_of_key_cols_tab_del || '. This is not supported!';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Stock updates are always adds and never updates.
			IF (v_isTransactionTypeAdd <> 1) THEN
				p_error := 50261;
				p_error_message := 'A stock update is always an ADD. In this case it is ''' || p_transaction_type || '''. This is not supported!';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Get the position of the second column value
			--If the returned position is zero then it can not find the second column. What is not possible.
			v_position := LOCATE( p_list_of_cols_val_tab_del, CHAR(9) );
			IF (v_position = 0) THEN
				p_error := 50262;
				p_error_message := 'No second column value could be found in ''' || p_list_of_cols_val_tab_del || '''';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Get the column values of "TransNum" and "Instance"
			v_transNum := CAST(SUBSTRING(p_list_of_cols_val_tab_del,1,v_position-1) AS INT );
			v_instance := CAST(SUBSTRING(p_list_of_cols_val_tab_del,v_position+1,LENGTH(p_list_of_cols_val_tab_del)-v_position-1) AS INT );

			--Get the stock update row
			SELECT "TransType", "InQty", "OutQty"  INTO v_transType, v_inQty, v_outQty  FROM OINM WHERE "TransNum" = v_transNum AND "Instance" = v_instance;
			
			--Check if "TransType" is filled in
			IF (v_transType IS NULL) THEN
				p_error := 50263;
				p_error_message := 'TransType IS NULL or could not be found for TransNum ' || CAST(v_transNum AS NVARCHAR) || ' and Instance ' || CAST(v_instance AS NVARCHAR);
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if there is quantity booked. If not then no stock update
			IF (v_inQty-v_outQty=0) THEN
				RETURN;
			END IF;

			--Check if the "TransType" is supported by Produmex
			SELECT COUNT(*) INTO v_cnt FROM dummy WHERE v_transType NOT IN (13,14,15,16,18,19,20,21,59,60,67);
			IF v_cnt = 1 THEN
				p_error := 50264;
				p_error_message := 'TransType ' || CAST(v_transType AS NVARCHAR) || ' is not supported by the Produmex add-on!';
				SELECT p_error, p_error_message FROM dummy;
				----EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--If we are here everything is OK for the stock update
			RETURN;
		END IF;


	--	--**********************************************************************************************************
	--	--Do checks on an item batchnumbers
	--	--This line is put on top because this transaction notification contains 2 keys instead of one
	--	--**********************************************************************************************************
	--	IF (v_isItemBatchNumber=1 AND v_isTransactionTypeUpdate=1) THEN
	--		p_error := 50270
	--		p_error_message := 'Batchnumber can not be updated when using the Produmex Add-On.'
	--		SELECT p_error, p_error_message
	--		--EXEC xp_logevent p_error, p_error_message, ERROR
	--		RETURN
	--	END

		--The number of columns of the primary key must always be one. 
		IF p_num_of_cols_in_key <> 1 THEN
			p_error := 50001;
			p_error_message := 'Number of keys must be 1! Contact Produmex Support.';
			SELECT p_error, p_error_message FROM dummy;
			----EXEC xp_logevent p_error, p_error_message, ERROR
			RETURN;
		END IF;

		--**********************************************************************************************************
		--Get "Name" of extra DB
		--**********************************************************************************************************
		SELECT TOP 1 '"' || "ExtraDb" || '"' INTO v_extraDB FROM PMX_EXDB;
		IF v_extraDB IS NULL THEN
			p_error := 50002;
			p_error_message := 'The extra database name is not filled in in the table PMX_EXDB. This must be filled in!';
			SELECT p_error, p_error_message FROM dummy;
			----EXEC xp_logevent p_error, p_error_message, ERROR
			RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on an item
		--**********************************************************************************************************
		IF (v_isItem=1) THEN

			--Check if this is a delete. 
			IF (v_isTransactionTypeDelete=1) THEN
				
				--Check if the item is not used for the bin items
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT PMX_SLIT."ParentCode"
				FROM PMX_SLIT
				WHERE PMX_SLIT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50250;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item is used for the BIN items ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the item has no zone types linked to it
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT PMX_ITZT."ZoneTypeCode"
				FROM PMX_ITZT
				WHERE PMX_ITZT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50251;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item has zone types linked to it for zone types: ' || v_errorList || '. Delete these zone types from the item.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the item has no shelflifes linked to it
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT PMX_ICSL."CardCode"
				FROM PMX_ICSL
				WHERE PMX_ICSL."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50252;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item has shelf lifes linked to it for customers: ' || v_errorList || '. Delete these customers from the item.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the item is not used in a steplist
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT PMX_PSLH."ParentItemCode"
				FROM PMX_PSLH
				WHERE PMX_PSLH."ParentItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50253;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item has production order step list linked to it.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the item is not used in a steplist lines
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT PMX_PSLH."ParentItemCode"
				FROM PMX_PSLH
				INNER JOIN PMX_PSLL ON PMX_PSLL."ParentKey" = PMX_PSLH."InternalKey"
				WHERE PMX_PSLL."ChildItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50254;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item is used in production order step list for items: ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the item is not used as default logistic carrier
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT OITM."ItemCode"
				FROM OITM
				WHERE OITM.U_PMX_DLCP = p_list_of_cols_val_tab_del OR OITM.U_PMX_DLCS = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50255;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item is used as default logistic carrier for items: ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

						--Check if the item does not have any batch-attributes linked to it
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT  PMX_IBAT."BatchAttributeCode" 
				FROM PMX_IBAT
				WHERE PMX_IBAT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50256;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the item has batch-attributes linked to it:' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the item does not have any packaging types linked to it
				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT PMX_IPTY."PackagingTypeCode"
				FROM PMX_IPTY
				WHERE PMX_IPTY."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50257;
					p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || '''can not be deleted because the item has packaging types linked to it:' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;


				--No other checks must be done for the item delete
				RETURN;
			END IF;
			
			BEGIN
			--Get data of the item
			DECLARE v_storageLocationCode NVARCHAR(50);
			DECLARE v_isReturnableItem NVARCHAR(1);
			DECLARE v_itemCodeReturnableItemNotInventory NVARCHAR(20);
			DECLARE v_isInventoryItem CHAR(1);
			DECLARE v_barcode NVARCHAR(254);
			DECLARE v_purchaseBarcode NVARCHAR(254);
			DECLARE v_salesBarcode NVARCHAR(254);
			DECLARE v_qualityStatusQuarantinedAfterReception NVARCHAR(8);
			DECLARE v_qualityStatusAfterSalesReturn NVARCHAR(8);
			DECLARE v_barcodeType NVARCHAR(1);
			DECLARE v_purchaseBarcodeType NVARCHAR(1);
			DECLARE v_salesBarcodeType NVARCHAR(1);
			DECLARE v_isSerialNumber NVARCHAR(1);
			DECLARE v_hasBestBefore NVARCHAR(1);
			DECLARE v_expiryDefinitionCodeProduction NVARCHAR(30);
			DECLARE v_numberOfDecimalsForUOM INT;
			DECLARE v_numberOfDecimalsForUOM2 INT;
			DECLARE v_gs1MeasureTypeForUOM NVARCHAR(30);
			DECLARE v_gs1MeasureTypeForUOM2 NVARCHAR(30);
			DECLARE v_itemUOM NVARCHAR(20);
			DECLARE v_itemUOM2 NVARCHAR(20);
			DECLARE v_defaultQuantityUOM NUMERIC(19, 6);
			DECLARE v_defaultQuantityUOM2 NUMERIC(19, 6);
			DECLARE v_UOMToUseForSales NVARCHAR(1);
			DECLARE v_purchaseQuantity NUMERIC(19, 6);
			DECLARE v_salesQuantity NUMERIC(19, 6);
			DECLARE v_purchaseUOM NVARCHAR(20);
			DECLARE v_salesUOM NVARCHAR(20);
			DECLARE v_correctStockForUOM NVARCHAR(1);
			DECLARE v_correctStockForUOM2 NVARCHAR(1);
			DECLARE v_UOMToUseForInventoryTransitions NVARCHAR(1);
			DECLARE v_UOMToUseForPurchase NVARCHAR(1);
			DECLARE v_isItemLogisticCarrier NVARCHAR(1);
			DECLARE v_sevesoClass  NVARCHAR(30);
			DECLARE v_qualityStatusReleasedAfterReception NVARCHAR(8);
			DECLARE v_qualityStatusReceptionController NVARCHAR(8);
			DECLARE v_hasSecondBatchNumber NVARCHAR(1);
			DECLARE v_defaultLogisticCarrierProduction NVARCHAR(20);
			DECLARE v_defaultLogisticCarrierPicking NVARCHAR(20);
			DECLARE v_hasPmxSerialNumber NVARCHAR(1);
			DECLARE v_qualityStatusProduction NVARCHAR(8);
			DECLARE v_pickType NVARCHAR(30);
			DECLARE v_trackLocationSerialNumber NVARCHAR(1);
			DECLARE v_formatSerialNumber NVARCHAR(50);
			DECLARE v_createSsccOnReception NVARCHAR(1);
			DECLARE v_isCatchWeight NVARCHAR(1);
			DECLARE v_addNonInventoryItemToPickList CHAR(1);

            SELECT  IFNULL( U_PMX_DSLC, ''), IFNULL( U_PMX_RETR, 'N'), IFNULL( U_PMX_ICRI, ''), 
                    IFNULL( "InvntItem", 'N'), IFNULL( "CodeBars", ''), IFNULL( U_PMX_BCTY, 'G'), 
                    U_PMX_QSCR, U_PMX_QSSR, IFNULL( "ManSerNum", 'N'), 
                    IFNULL( U_PMX_HBBD, 'N'), IFNULL( U_PMX_PBCO, ''), IFNULL( U_PMX_PBCT, 'G'), 
                    IFNULL( U_PMX_SBCO, ''), IFNULL( U_PMX_SBCT, 'G'), U_PMX_EXDP, 
                    U_PMX_UOMD, U_PMX_UM2D, IFNULL( U_PMX_UMMT, '') , IFNULL( U_PMX_U2MT, ''), 
                    IFNULL( UOM_INV."UomCode", IFNULL( "InvntryUom", '')) , IFNULL( U_PMX_UOM2, ''), U_PMX_DQUM, U_PMX_DQU2, 
                    U_PMX_UMSA, IFNULL( "NumInBuy", 0), IFNULL( "NumInSale", 0), 
                    IFNULL( UOM_PUR."UomCode", IFNULL( "BuyUnitMsr", '')), IFNULL( UOM_SAL."UomCode", IFNULL( "SalUnitMsr", '')), IFNULL( U_PMX_CSUM, 'N'), IFNULL( U_PMX_CSU2, 'N'), 
                    U_PMX_UMPU, U_PMX_UMIT, U_PMX_LOCA, U_PMX_SEVE, 
                    U_PMX_RQSR, U_PMX_QSRC, 
                    IFNULL( U_PMX_HBN2, 'N'), U_PMX_DLCP, U_PMX_DLCS, 
                    IFNULL( U_PMX_HSER, 'N'), U_PMX_QSPR, U_PMX_PITY, IFNULL( U_PMX_TLSN, 'N'), U_PMX_SNFO , IFNULL( U_PMX_CSOR, 'N'), 
                    U_PMX_ICAW, U_PMX_NIOP
			INTO v_storageLocationCode, v_isReturnableItem, v_itemCodeReturnableItemNotInventory,
				 v_isInventoryItem, v_barcode, v_barcodeType,
				 v_qualityStatusQuarantinedAfterReception, v_qualityStatusAfterSalesReturn, v_isSerialNumber,
				 v_hasBestBefore, v_purchaseBarcode, v_purchaseBarcodeType,
				 v_salesBarcode, v_salesBarcodeType, v_expiryDefinitionCodeProduction,
				 v_numberOfDecimalsForUOM, v_numberOfDecimalsForUOM2, v_gs1MeasureTypeForUOM, v_gs1MeasureTypeForUOM2,
				 v_itemUOM, v_itemUOM2, v_defaultQuantityUOM, v_defaultQuantityUOM2,
				 v_UOMToUseForSales, v_purchaseQuantity, v_salesQuantity,
				 v_purchaseUOM, v_salesUOM, v_correctStockForUOM, v_correctStockForUOM2,
				 v_UOMToUseForPurchase, v_UOMToUseForInventoryTransitions, v_isItemLogisticCarrier, v_sevesoClass,
				 v_qualityStatusReleasedAfterReception, v_qualityStatusReceptionController,
				 v_hasSecondBatchNumber, v_defaultLogisticCarrierProduction, v_defaultLogisticCarrierPicking,
				 v_hasPmxSerialNumber, v_qualityStatusProduction, v_pickType, v_trackLocationSerialNumber, v_formatSerialNumber, v_createSsccOnReception,
				 v_isCatchWeight, v_addNonInventoryItemToPickList
			FROM OITM
			LEFT JOIN OUOM AS UOM_INV ON UOM_INV."UomEntry" = OITM."IUoMEntry"
			LEFT JOIN OUOM AS UOM_PUR ON UOM_PUR."UomEntry" = OITM."PUoMEntry"
			LEFT JOIN OUOM AS UOM_SAL ON UOM_SAL."UomEntry" = OITM."SUoMEntry"
			WHERE "ItemCode" = p_list_of_cols_val_tab_del;

	--		print 'v_isReturnableItem='  + v_isReturnableItem
	--		print 'v_itemCodeReturnableItemNotInventory=' || v_itemCodeReturnableItemNotInventory
	--		print 'v_isInventoryItem=' || v_isInventoryItem
	--		print 'v_storageLocationCode=' || v_storageLocationCode
	--		print 'v_barcode=' || v_barcode

			--Check if the default storage location filled in is an existing storage location or a zone
			IF (LENGTH(v_storageLocationCode)>0) THEN
				SELECT COUNT(*) INTO v_cnt FROM PMX_OSSL WHERE "Code" = v_storageLocationCode AND "IsActive"='Y';
				IF v_cnt < 1 THEN
					SELECT COUNT(*) INTO v_cnt FROM PMX_OSZO WHERE "Code" = v_storageLocationCode;
					IF v_cnt < 1 THEN
						p_error := 50200;
						p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The storage location code or zone code ''' || v_storageLocationCode || ''' could not be found in the system or the storage location is not active!';
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;
				END IF;
			END IF;

			--The property ItemCodeReturnableItemNotInventory on an item must be filled in if the item is inventory and IsReturnableItem=True
			IF (v_isReturnableItem = 'Y') AND (v_isInventoryItem='Y') AND (LENGTH(v_itemCodeReturnableItemNotInventory)=0) THEN
				p_error := 50201;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item code returnable item not inventory is not filled. This must be filled in if the item is an inventory item and is returnable. Every inventory returnable item must have a not inventory returnable item linked to it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--The property ItemCodeReturnableItemNotInventory on an item can only be filled in if the item is an inventory returnable item.
			IF NOT((v_isReturnableItem = 'Y' OR v_isItemLogisticCarrier = 'Y') AND (v_isInventoryItem='Y')) AND (LENGTH(v_itemCodeReturnableItemNotInventory)>0) THEN
				p_error := 50202;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The field ItemCodeReturnableItemNotInventory may only be filled in if the item is an inventory returnable item or logistic carrier!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the item code that is filled in in v_itemCodeReturnableItemNotInventory is an existing not inventory returnable item.
			SELECT COUNT(*) INTO v_cnt FROM OITM WHERE "ItemCode" = v_itemCodeReturnableItemNotInventory AND "InvntItem"='N' AND (U_PMX_RETR='Y' OR U_PMX_LOCA='Y');
			IF (LENGTH(v_itemCodeReturnableItemNotInventory)>0) AND v_cnt<1 THEN
				p_error := 50203;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The not inventory returnable item with code ''' || v_itemCodeReturnableItemNotInventory || ''' could not be found!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check the GTINs of the item
			BEGIN

			DECLARE v_sqlStatementBegin  NCLOB;
			DECLARE v_sqlStatementEnd  NCLOB;
			DECLARE v_sqlStatement  NCLOB;
			DECLARE v_parameterList  NCLOB;

			v_sqlStatementBegin := 'INSERT INTO ' || v_extraDb || '."TMP_IntStr" SELECT Error, Message FROM ' || v_extraDb || '."PMX_FN_CheckGTIN"(''';
			v_sqlStatementEnd := ''')';
			v_parameterList := 'p_error INT output, p_error_message NVARCHAR(200) output';
			
			--Check the GTIN of the item
			IF ((LENGTH(v_barcode) <> 0) AND (v_barcodeType = 'G')) THEN

				DELETE FROM "WUK_4004_PMX"."TMP_IntStr";
				v_sqlStatement := v_sqlStatementBegin || v_barcode || v_sqlStatementEnd;
				EXEC v_sqlStatement; -- , v_parameterList, p_error output, p_error_message output
				SELECT TOP 1 "Key", "Value" INTO p_error, p_error_message FROM "WUK_4004_PMX"."TMP_IntStr";

				IF (p_error <> 0) THEN
					p_error_message := '"Barcode" ''' || v_barcode || ''' of Item ' || p_list_of_cols_val_tab_del || ': ' || p_error_message;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
			--Check the purchase GTIN of the item
			IF ((LENGTH(v_purchaseBarcode) <> 0) AND (v_purchaseBarcodeType = 'G')) THEN

				DELETE FROM "WUK_4004_PMX"."TMP_IntStr";
				v_sqlStatement := v_sqlStatementBegin || v_purchaseBarcode || v_sqlStatementEnd;
				EXEC v_sqlStatement; -- , v_parameterList, p_error output, p_error_message output
				SELECT TOP 1 "Key", "Value" INTO p_error, p_error_message FROM "WUK_4004_PMX"."TMP_IntStr";

				IF (p_error <> 0) THEN
					p_error_message := 'Purchase barcode ''' || v_purchaseBarcode || ''' of Item ' || p_list_of_cols_val_tab_del || ': ' || p_error_message;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
			--Check the sales GTIN of the item
			IF ((LENGTH(v_salesBarcode) <> 0) AND (v_salesBarcodeType = 'G')) THEN

				DELETE FROM "WUK_4004_PMX"."TMP_IntStr";
				v_sqlStatement := v_sqlStatementBegin || v_salesBarcode || v_sqlStatementEnd;
				EXEC v_sqlStatement; -- , v_parameterList, p_error output, p_error_message output
				SELECT TOP 1 "Key", "Value" INTO p_error, p_error_message FROM "WUK_4004_PMX"."TMP_IntStr";

				IF (p_error <> 0) THEN
					p_error_message := 'Sales barcode ''' || v_salesBarcode || ''' of Item ' || p_list_of_cols_val_tab_del || ': ' || p_error_message;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
			

			--Check the GTIN of the item packaging types
			BEGIN
				DECLARE v_barcodePackagingType NVARCHAR(16);
				DECLARE CURSOR barCodePackagingTypeCursor FOR SELECT "Barcode" FROM PMX_IPTY WHERE PMX_IPTY."ItemCode" = :p_list_of_cols_val_tab_del AND LENGTH("Barcode") <> 0 AND ("BarcodeType" = 'G');
				OPEN barCodePackagingTypeCursor;
				FETCH barCodePackagingTypeCursor INTO v_barcodePackagingType;
				WHILE v_barcodePackagingType IS NOT NULL DO

					DELETE FROM "WUK_4004_PMX"."TMP_IntStr";
					v_sqlStatement := v_sqlStatementBegin || v_barcodePackagingType || v_sqlStatementEnd;
					EXEC v_sqlStatement; -- , v_parameterList, p_error output, p_error_message output
					SELECT TOP 1 "Key", "Value" INTO p_error, p_error_message FROM "WUK_4004_PMX"."TMP_IntStr";

					IF (p_error <> 0) THEN
						CLOSE barCodePackagingTypeCursor;
						p_error_message := '"Barcode" ''' || v_barcodePackagingType || ''' of Item ' || p_list_of_cols_val_tab_del || ': ' || p_error_message;
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;

					FETCH barCodePackagingTypeCursor INTO v_barcodePackagingType;
				END WHILE;
				CLOSE barCodePackagingTypeCursor;
			END;


			--Check the GTIN of the "Uom" barcodes
			BEGIN
				DECLARE v_barcodeForUom NVARCHAR(254);
				DECLARE v_barcodeUom NVARCHAR(254);
				DECLARE CURSOR barCodeUomCursor FOR SELECT "BcdCode", OUOM."UomCode" 
													FROM OBCD 
													INNER JOIN OUOM ON OUOM."UomEntry" = OBCD."UomEntry" 
													WHERE OBCD."ItemCode" = :p_list_of_cols_val_tab_del AND LENGTH("BcdCode") <> 0 
													AND :v_barcodeType = 'G';
				OPEN barCodeUomCursor;
				FETCH barCodeUomCursor INTO v_barcodeForUom, v_barcodeUom;
				WHILE v_barcodeForUom IS NOT NULL DO

					DELETE FROM "WUK_4004_PMX"."TMP_IntStr";
					v_sqlStatement := v_sqlStatementBegin || v_barcodeForUom || v_sqlStatementEnd;
					EXEC v_sqlStatement; -- , v_parameterList, p_error output, p_error_message output
					SELECT TOP 1 "Key", "Value" INTO p_error, p_error_message FROM "WUK_4004_PMX"."TMP_IntStr";

					IF (p_error <> 0) THEN
						CLOSE barCodeUomCursor;
						p_error_message := '"Barcode" ''' || v_barcodeForUom || ''' for "Uom" ''' || v_barcodeUom || '''  of Item ' || p_list_of_cols_val_tab_del || ': ' || p_error_message;
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;
    				FETCH barCodeUomCursor INTO v_barcodeForUom, v_barcodeUom;
				END WHILE;
				CLOSE barCodeUomCursor;
			END;

			--Check if the quality status code that is filled in in v_qualityStatusQuarantinedAfterReception is an existing quality status code.
			SELECT COUNT("Code") INTO v_cnt FROM PMX_QYST WHERE "Code" = v_qualityStatusQuarantinedAfterReception;
			IF (v_qualityStatusQuarantinedAfterReception IS NOT NULL) AND (v_cnt=0) THEN
				p_error := 50207;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The quality status code after purchase return ''' || v_qualityStatusQuarantinedAfterReception || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the quality status code that is filled in in v_qualityStatusAfterSalesReturn is an existing quality status code.
			SELECT COUNT("Code") INTO v_cnt FROM PMX_QYST WHERE "Code" = v_qualityStatusAfterSalesReturn;
			IF (v_qualityStatusAfterSalesReturn IS NOT NULL) AND (v_cnt=0) THEN
				p_error := 50208;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The quality status code after sales return ''' || v_qualityStatusAfterSalesReturn || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if barcode does not occur more than once for inventory items
			IF (v_isInventoryItem = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT OITM."ItemCode"
				FROM OITM LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				WHERE OITM."ItemCode" <> p_list_of_cols_val_tab_del AND 
				((LENGTH(IFNULL( OITM."CodeBars",'')) <> 0 AND OITM."CodeBars" IN(v_barcode,v_purchaseBarcode,v_salesBarcode)) OR
				(LENGTH(IFNULL( OITM.U_PMX_PBCO,'')) <> 0 AND OITM.U_PMX_PBCO IN(v_barcode,v_purchaseBarcode,v_salesBarcode))OR
				(LENGTH(IFNULL( OITM.U_PMX_SBCO,'')) <> 0 AND OITM.U_PMX_SBCO IN(v_barcode,v_purchaseBarcode,v_salesBarcode))OR
				(LENGTH(IFNULL( PMX_IPTY."Barcode",'')) <> 0 AND PMX_IPTY."Barcode" IN(v_barcode,v_purchaseBarcode,v_salesBarcode))) AND
				OITM."InvntItem" = 'Y';
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50209;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item ' || v_errorList || 'has the same barcode.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--The property "ManSerNum" on an item cannot be true because the addon does not support it
--			IF (v_isSerialNumber = 'Y') THEN
--				p_error := 50210
--				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item cannot be managed by serial numbers.'
--				SELECT p_error, p_error_message
--				--EXEC xp_logevent p_error, p_error_message, ERROR
--				RETURN
--			END

			--When the item has bestBeforeDate, there cannot be an inventory item with bestBeforeDate is null
			IF (v_hasBestBefore = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST( PMX_INVT."InternalKey" AS NVARCHAR (11))
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."BestBeforeDate" IS NULL AND 
				PMX_INVT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50211;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has a best before date, but in the inventory the best before date is not set for the inventory total with keys: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--When the item has no bestBeforeDate, there cannot be an inventory item with a bestBeforeDate
			IF (v_hasBestBefore = 'N') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST( PMX_INVT."InternalKey" AS NVARCHAR (11))
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."BestBeforeDate" IS NOT NULL AND 
				PMX_INVT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50212;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has no best before date, but in the inventory the best before date is set for the inventory total with keys: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			-- Check if the barcode in purchase and sales are not the same as the base barcode
			IF (LENGTH(v_barcode)>0 AND v_barcode = v_purchasebarCode) OR (LENGTH(v_barcode)>0 AND v_barcode = v_salesBarcode) OR (LENGTH(v_salesBarcode)>0 AND v_salesBarcode=v_purchaseBarcode) THEN
				p_error := 50213;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The barcode or the purchase barcode or the sales barcode of  item ' || p_list_of_cols_val_tab_del || ' is the same. This is not allowed.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the expiry definition for production is filled in in v_expiryDefinitionCodeProduction is an existing expiry definition code.
			SELECT COUNT("Code") INTO v_cnt FROM "@PMX_EXDE" WHERE "Code"=v_expiryDefinitionCodeProduction;
			IF (v_expiryDefinitionCodeProduction IS NOT NULL) AND (v_cnt=0) THEN
				p_error := 50214;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The expiry defintion code for production ''' || v_expiryDefinitionCodeProduction || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the seveso class is filled in and if v_sevesoClass is an existing seveso class code.
			SELECT COUNT("Code") INTO v_cnt FROM "@PMX_SEVE" WHERE "Code" = v_sevesoClass;
			IF (v_sevesoClass IS NOT NULL) AND (v_cnt=0) THEN
				p_error := 50215;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The seveso class code ''' || v_sevesoClass || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the quality status code that is filled in in v_qualityStatusReleasedAfterReception is an existing quality status code.
			SELECT COUNT("Code") INTO v_cnt FROM PMX_QYST WHERE "Code" = v_qualityStatusReleasedAfterReception;
			IF (v_qualityStatusReleasedAfterReception IS NOT NULL) AND (v_cnt=0) THEN
				p_error := 50216;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The released quality status code after reception ''' || v_qualityStatusReleasedAfterReception || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the quality status controller exists
			SELECT COUNT("Code") INTO v_cnt FROM PMX_EXTE WHERE "Code" = v_qualityStatusReceptionController;
			IF (v_qualityStatusReceptionController IS NOT NULL) AND (v_cnt=0) THEN
				p_error := 50217;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The quality status controller ''' || v_qualityStatusReceptionController || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--When the item has second batchnumber, there cannot be an inventory item with second batchnumber is null
			IF (v_hasSecondBatchNumber = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST( PMX_INVT."InternalKey" AS NVARCHAR (11))
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."InternalBatchNumber" IS NULL AND 
				PMX_INVT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50218;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has a second batch number, but in the inventory the second batch number is not set for the inventory total with keys: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--When the item has no second batchnumber, there cannot be an inventory item with a second batchnumber
			IF (v_hasSecondBatchNumber = 'N') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST( PMX_INVT."InternalKey" AS NVARCHAR (11))
				FROM PMX_INVT
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INVT."ItemTransactionalInfoKey"
				WHERE PMX_ITRI."InternalBatchNumber" IS NOT NULL AND 
				PMX_INVT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50219;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has no second batch number, but in the inventory the second batch number is set for the inventory total with keys: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--If no batchnumber or best before date is set, user cannot ask for packaging type quantity during receipt
			IF (v_hasBestBefore = 'N') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT PMX_IPTY."PackagingTypeCode"
				FROM PMX_IPTY
				INNER JOIN OITM	ON OITM."ItemCode" = PMX_IPTY."ItemCode"
				WHERE PMX_IPTY."ItemCode" = p_list_of_cols_val_tab_del
				AND OITM."ManBtchNum" = 'N'
				AND PMX_IPTY."AskDuringReception" = 'Y';
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50220;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has no batch or BBD, so packaging quantity cannot be asked during receipt for packaging types: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--Check if the "Uom" does not exists in item packaging types
			SELECT COUNT("ItemCode") INTO v_cnt FROM PMX_IPTY WHERE "PackagingTypeCode"=v_itemUOM AND "ItemCode" = p_list_of_cols_val_tab_del;
			IF (LENGTH(v_itemUOM) > 0) AND v_cnt > 0 THEN
				p_error := 50221;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The "Uom" with code ''' || v_itemUOM || ''' is used in packaging types for the item';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Batchnumber cannot be set if item is logistic carrier
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT OITM."ItemCode"
			FROM OITM
			WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del
			AND OITM.U_PMX_LOCA = 'Y'
			AND (OITM."ManBtchNum" = 'Y' OR OITM.U_PMX_HBBD = 'Y' OR OITM.U_PMX_HBN2 = 'Y');
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50222;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': An item defined as logistic carrier cannot have a batch or BBD.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Default logistic carrier for production should be valid item and logistic carrier
			IF IFNULL( v_defaultLogisticCarrierProduction, '') <> '' THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT OITM.U_PMX_DLCP
				FROM OITM
				WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT * FROM OITM WHERE "ItemCode" = v_defaultLogisticCarrierProduction
				AND OITM.U_PMX_LOCA = 'Y');
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50223;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The default logistic carrier for Production ' || v_defaultLogisticCarrierProduction || ' is not a valid Logistic Carrier.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--Default logistic carrier for picking should be valid item and logistic carrier
			IF IFNULL( v_defaultLogisticCarrierProduction, '') <> '' THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT OITM.U_PMX_DLCS
				FROM OITM
				WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT * FROM OITM WHERE "ItemCode" = v_defaultLogisticCarrierPicking
				AND OITM.U_PMX_LOCA = 'Y');
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50224;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The default logistic carrier for Picking ' || v_defaultLogisticCarrierPicking || ' is not a valid Logistic Carrier.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--If item has Pmx serial number, the decimals for "Uom" should be 0
			IF ((v_hasPmxSerialNumber = 'Y' OR v_isSerialNumber = 'Y') AND v_numberOfDecimalsForUOM > 0) THEN
				p_error := 50225;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has serial numbers, so the number of decimals for "Uom" should be 0.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the quality status code that is filled in in v_qualityStatusProduction is an existing quality status code.
			SELECT COUNT(*) INTO v_cnt FROM PMX_QYST WHERE "Code" = v_qualityStatusProduction;
			IF (v_qualityStatusProduction IS NOT NULL) AND (v_cnt<1) THEN
				p_error := 50226;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The quality status code production ''' || v_qualityStatusProduction || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the pick type that is filled in in v_pickType is an existing pick type.
			SELECT COUNT(*) INTO v_cnt FROM "@PMX_IPIT" WHERE "Code"=v_picktype;
			IF (v_pickType IS NOT NULL) AND (v_cnt<1) THEN
				p_error := 50227;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The pick type ''' || v_pickType || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--When the item tracks location for serial numbers, there cannot be one without LUID
			IF (v_trackLocationSerialNumber = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT CAST( PMX_SENU."InternalKey" AS NVARCHAR (11))
				FROM PMX_SENU
				WHERE PMX_SENU."InStock" = 'Y' 
				AND PMX_SENU.LUID IS NULL
				AND PMX_SENU."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50228;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item needs to track locations of serial nr, but there are serial nrs in stock without LUID: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--When the item tracks location for serial numbers, there cannot be inventory totals without LUID (needed for "on release only" setting)
			IF (v_trackLocationSerialNumber = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT CAST( PMX_INVT."InternalKey" AS NVARCHAR (11))
				FROM PMX_INVT
				WHERE PMX_INVT."Quantity" > 0
				AND PMX_INVT."LogUnitIdentKey" IS NULL
				AND PMX_INVT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50238;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item needs to track locations of serial nr, but there are inventory totals without LUID: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--When the item does not tracks location for serial numbers, there cannot be one with LUID
			IF (v_trackLocationSerialNumber = 'N') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT CAST( PMX_SENU."InternalKey" AS NVARCHAR (11))
				FROM PMX_SENU
				WHERE PMX_SENU."InStock" = 'Y' 
				AND PMX_SENU.LUID IS NOT NULL
				AND PMX_SENU."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50229;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item does not need to track locations of ser. nr, but there are ser. nrs are in stock with LUID: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;


			--When the item does track location for serial numbers, either SAP or PMX serial numbers needs to be filled
			IF (v_trackLocationSerialNumber = 'Y' AND v_hasPmxSerialNumber = 'N' AND v_isSerialNumber = 'N') THEN
				p_error := 50230;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item tracks locations of ser. nr, but the item is not defined as serial number';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--When the item does track location for serial numbers, either SAP or PMX serial numbers needs to be filled
			IF (v_hasPmxSerialNumber = 'Y' AND v_isSerialNumber = 'Y') THEN
				p_error := 50231;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has SAP serial numbers and PMX serial numbers. Please choose only one of them.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the serial format is a correct one
			SELECT COUNT(*) INTO v_cnt FROM "@PMX_ISFT" WHERE "Code"=v_formatSerialNumber;
			IF (v_formatSerialNumber IS NOT NULL) AND (v_cnt<1) THEN
				p_error := 50232;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The serial number format ''' || v_formatSerialNumber || ''' could not be found in the system!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			IF (v_isInventoryItem = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT T0."ItemCode"
				FROM (
					SELECT DISTINCT Barcode2."ItemCode"				
					FROM OBCD AS Barcode1
					LEFT JOIN OBCD Barcode2 ON Barcode1."BcdCode" = Barcode2."BcdCode"
					WHERE Barcode1."ItemCode" = p_list_of_cols_val_tab_del
					AND Barcode1."BcdEntry" <> Barcode2."BcdEntry"
				) AS T0;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50233;
					p_error_message := 'Item ''' || v_errorList || ''' contains the same barcode ';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			IF (v_isInventoryItem = 'Y') THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT T0."ItemCode"
				FROM (
					SELECT OBCD."ItemCode"
					FROM OBCD 
					WHERE OBCD."ItemCode" <> p_list_of_cols_val_tab_del AND 
					(LENGTH(IFNULL( OBCD."BcdCode",'')) <> 0 AND OBCD."BcdCode" IN(v_barcode,v_purchaseBarcode,v_salesBarcode)) 
				) AS T0;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50234;
					p_error_message := 'Item ''' || p_list_of_cols_val_tab_del || ''': The item ''' || v_errorList || ''' has the same barcode.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			-- Check if item has serial that needs to be tracked -> need to have SSCC's at reception
			IF (v_hasPmxSerialNumber = 'Y' OR v_isSerialNumber = 'Y') AND v_trackLocationSerialNumber ='Y' AND v_createSsccOnReception = 'N' THEN
				p_error := 50235;
				p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || '''. Cannot add an item with SAP or PMX serial numbers and tracking location without create SSCC on reception.' || v_errorList || '.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			
			--When the item has no "SerialNumber", there cannot be an inventory item with "SerialNumber"
			IF (v_hasPmxSerialNumber = 'N' AND v_isSerialNumber = 'N' ) THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
			    INSERT INTO TMP_TN_CharListTableNoIndex
			    SELECT CAST( PMX_INVT."InternalKey" AS NVARCHAR (11))
				FROM PMX_INVT
				LEFT JOIN PMX_SENU ON PMX_SENU."ItemCode" = PMX_INVT."ItemCode"
				WHERE PMX_SENU."InStock" ='Y' AND 
				PMX_INVT."ItemCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50236;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The item has no serial number, but serial number(s) are set for the inventory total with keys: ' || v_errorList || '.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
			
			-- Check if item has serial that needs to be tracked -> need to have SSCC's at reception
			SELECT COUNT("ItemCode") INTO v_cnt FROM OITM WHERE "ItemCode" = p_list_of_cols_val_tab_del AND OITM."UgpEntry" <> -1;
			IF (v_isItemLogisticCarrier = 'Y' AND v_cnt > 0) THEN
			    p_error := 50237;
			    p_error_message := 'Item with code ''' || p_list_of_cols_val_tab_del || '''. A logistic carrier cannot have an "Uom" group.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			IF (v_isInventoryItem = 'Y' AND v_addNonInventoryItemToPickList = 'Y') THEN
			    p_error := 50239;
			    p_error_message := 'Add non-inventory item to pick list is not allowed for inventory item. Item with code ''' || p_list_of_cols_val_tab_del || '''';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--******************************
			--Catch weight
			--******************************

			--Check if the number of decimals for uom is between 0 and the number of decimals for quantities
			SELECT "QtyDec" INTO v_cnt FROM OADM;
			IF(v_numberOfDecimalsForUOM < 0 OR v_numberOfDecimalsForUOM > v_cnt) THEN
				p_error := 50350;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The number of decimals for "Uom" should be between 0 and ' || CAST( v_cnt AS NVARCHAR (10));
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the number of decimals for uom2 is between 0 and the number of decimals for quantities
			IF(v_numberOfDecimalsForUOM2 < 0 OR v_numberOfDecimalsForUOM2 > v_cnt) THEN
				p_error := 50351;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The number of decimals for "Uom2" should be between 0 and ' || CAST( v_cnt AS NVARCHAR (10));
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			END;

			--Check if the gs1 measure types are not the same
			IF( LENGTH(v_gs1MeasureTypeForUOM) > 0 AND LENGTH(v_gs1MeasureTypeForUOM2) > 0 AND v_gs1MeasureTypeForUOM = v_gs1MeasureTypeForUOM2) THEN
				p_error := 50352;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The GS1 Measure Type cannot be the same for "Uom" and "Uom2"';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the uom exists in UOM
			SELECT COUNT(*) INTO v_cnt FROM OUOM WHERE "UomCode"=v_itemUOM;
			IF(v_cnt<1 AND LENGTH(v_itemUOM) > 0) THEN
				p_error := 50355;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The "Uom" with code ''' || v_itemUOM || ''' does not exist in SAP "Uom"';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the purchase uom exists in UOM
			SELECT COUNT(*) INTO v_cnt FROM OUOM WHERE "UomCode"=v_purchaseUOM;
			IF(v_cnt<1 AND LENGTH(v_purchaseUOM) > 0) THEN
				p_error := 50356;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The purchase "Uom" with code ''' || v_purchaseUOM || ''' does not exist in SAP "Uom"';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the sales uom exists in UOM
			SELECT COUNT(*) INTO v_cnt FROM OUOM WHERE "UomCode"=v_salesUOM;
			IF(v_cnt<1 AND LENGTH(v_salesUOM) > 0) THEN
				p_error := 50357;
				p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The sales "Uom" with code ''' || v_salesUOM || ''' does not exist in SAP "Uom"';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--Checks for catch weight
			IF( v_isCatchWeight = 'Y' ) THEN

									
				--Check if uom decimals for UOM = 0 when it should be ** DISABLE FOR NOW **
				--IF( v_numberOfDecimalsForUOM <> 0 ) THEN
					--p_error := 50371;
					--p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The number of decimals for "Uom" should be 0 for inventory uom.';
					--SELECT p_error, p_error_message FROM dummy;
					----EXEC xp_logevent p_error, p_error_message, ERROR
					--RETURN;
				--END IF;
				
				--Check if uom decimals for UOM = 0 when it should be
				--SELECT COUNT(*) INTO v_cnt FROM OITM WHERE "ItemCode" = p_list_of_cols_val_tab_del AND IFNULL( "IUoMEntry", -1) <> -1;
				--IF v_cnt > 0 THEN
					--p_error := 50372;
					--p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': UOM groups are not allowed for catch weight items';
					--SELECT p_error, p_error_message FROM dummy;
					----EXEC xp_logevent p_error, p_error_message, ERROR
					--RETURN;
				--END IF;
				
				--IF (v_salesQuantity <> 1) THEN
					--p_error := 50373;
					--p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': Items per sales unit should be 1 for catch weight items';
					--SELECT p_error, p_error_message FROM dummy;
					----EXEC xp_logevent p_error, p_error_message, ERROR
					--RETURN;
				--END IF;
				
				--IF (v_purchaseQuantity <> 1) THEN
					--p_error := 50374;
					--p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': Items per purchase unit should be 1 for catch weight items';
					--SELECT p_error, p_error_message FROM dummy;
					----EXEC xp_logevent p_error, p_error_message, ERROR
					--RETURN;
				--END IF;
			
				IF IFNULL( v_defaultQuantityUOM, 0 ) = 0 THEN
					p_error := 50375;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': A default quantity for uom should be filled in for catch weight items';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				--Only allow catch weight items if no stock, or stock has quantity for Uom2
				SELECT COUNT(*) INTO v_cnt FROM PMX_INVT WHERE "ItemCode" = p_list_of_cols_val_tab_del AND "QuantityUom2" IS NULL;
				IF v_cnt > 0 THEN
					p_error := 50376;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ':  Cannot change item to catch weight item if stock exists';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				--Update the default quantity for uom2
				UPDATE OITM SET U_PMX_DQU2 = 1 / v_defaultQuantityUOM WHERE "ItemCode" = p_list_of_cols_val_tab_del;

			ELSE
			
				--Only allow non-catch weight items if no stock, or stock has no quantity for Uom2
				SELECT COUNT(*) INTO v_cnt FROM PMX_INVT WHERE "ItemCode" = p_list_of_cols_val_tab_del AND "QuantityUom2" IS NOT NULL;
				IF v_cnt > 0 THEN
					p_error := 50401;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ':  Cannot change item to NON-catch weight item if stock exists';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			

			END IF;


			--Do checks when UOM2 is filled in
			IF(LENGTH(v_itemUOM2) > 0) THEN
				--Check if the uom2 exists in UOM
				SELECT COUNT(*) INTO v_cnt FROM OUOM WHERE "UomCode"=v_itemUOM2;
				IF v_cnt<1 THEN
					p_error := 50358;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The "Uom2" with code ''' || v_itemUOM2 || ''' does not exist in SAP "Uom"';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

                /*				
				IF IFNULL( v_defaultQuantityUOM , 0 ) = 0 THEN
					UPDATE OITM SET U_PMX_DQUM = 1 WHERE "ItemCode" = p_list_of_cols_val_tab_del;
					v_defaultQuantityUOM := 1;
				END IF;
				IF IFNULL( v_defaultQuantityUOM2, 0 ) = 0 THEN
					UPDATE OITM SET U_PMX_DQU2 = 1 WHERE "ItemCode" = p_list_of_cols_val_tab_del;
					v_defaultQuantityUOM2 := 1;
				END IF;
				IF v_defaultQuantityUOM <> 1 AND v_defaultQuantityUOM2 <> 1 THEN
					p_error := 50359;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': At least one of the default quantities for "Uom" has to be 1';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the uom to use for orders is not 'A'
				IF( v_UOMToUseForOrders = 'A') THEN
					p_error := 50360;
					p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': "Uom" to use for orders cannot be ''UomAndUOM2''';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
                */
                
                IF( v_isCatchWeight = 'Y' ) THEN
                
					--Check if purchase quantity = 1
					IF(v_purchaseQuantity <> 1) THEN
						p_error := 50361;
						p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The items in purchase unit should be 1';
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;

					--Check if sales quantity = 1
					IF(v_purchaseQuantity <> 1) THEN
						p_error := 50362;
						p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The items in sales unit should be 1';
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;

					--Check if purchase, sales and inventory uom is the same
					IF ( NULLIF( v_purchaseUOM, '' ) <> v_itemUOM OR NULLIF( v_salesUOM, '' ) <> v_itemUOM ) THEN
						p_error := 50363;
						p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': The sales and purchase "Uom" should be the same as the inventory "Uom"';
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;

					/*
					IF(v_correctStockForUOM = 'Y' AND v_correctStockForUOM2 = 'Y') THEN
						--Check if the uom to use for inventory transitions is 'A'
						IF( v_UOMToUseForInventoryTransitions <> 'A') THEN
							p_error := 50364;
							p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': "Uom" to use for inventory transitions should be ''UomAndUOM2'' if correct stock for "Uom" and "Uom2" is set to true';
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;
						--Check if the uom to use for documents is 'A'
						IF( v_UOMToUseForDocuments <> 'A') THEN
							p_error := 50365;
							p_error_message := 'Item ' || p_list_of_cols_val_tab_del || ': "Uom" to use for documents should be ''UomAndUOM2'' if correct stock for "Uom" and "Uom2" is set to true';
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;
					END IF;
					*/
				END IF;
				
				-- Verify that UOM2 prices are possible (convert to UOM1 and back)

				BEGIN
					DECLARE CURSOR curPriceLists FOR
					SELECT U_PMX_UOM2PRICEUDF, CAST( "PriceDec" AS NVARCHAR ) FROM OPLN, OADM WHERE IFNULL( U_PMX_UOM2PRICEUDF, '' ) <> '';
	
    				DECLARE v_priceUDF NVARCHAR(20);
    				DECLARE v_priceDecimals NVARCHAR(20);
					DECLARE v_getPriceUDFs NVARCHAR(500);
					DECLARE v_badPrices NVARCHAR(500);
					DECLARE v_badPrices2 NVARCHAR(500);
	
					DELETE FROM TMP_TN_1;

                    OPEN curPriceLists;
				    FETCH curPriceLists INTO v_priceUDF, v_priceDecimals;
                    WHILE v_priceUDF IS NOT NULL DO
						v_getPriceUDFs := 'INSERT INTO TMP_TN_1 SELECT U_' || v_priceUDF || ',
							CEILING( FLOOR  ( U_' || v_priceUDF || ' / U_PMX_DQU2 * POWER(10,' || v_priceDecimals || ') ) * U_PMX_DQU2 ) / POWER(10,' || v_priceDecimals || '),
							CEILING( CEILING( U_' || v_priceUDF || ' / U_PMX_DQU2 * POWER(10,' || v_priceDecimals || ') ) * U_PMX_DQU2 ) / POWER(10,' || v_priceDecimals || ')
						FROM OITM WHERE "ItemCode" = ''' || p_list_of_cols_val_tab_del || '''';
						
						EXEC v_getPriceUDFs;

    				    FETCH curPriceLists INTO v_priceUDF, v_priceDecimals;
					END WHILE;
                    CLOSE curPriceLists;

					SELECT COUNT(1) INTO v_cnt FROM TMP_TN_1;
					IF v_cnt > 0 THEN

						DECLARE CURSOR curBadPrices FOR
						SELECT '"Uom2" price ' || CAST( priceFromUDF AS NVARCHAR ) || ' not possible, choose '
							|| CAST( flooredPrice AS NVARCHAR ) || ' or ' || CAST( ceiledPrice AS NVARCHAR )
						FROM TMP_TN_1
						WHERE priceFromUDF <> flooredPrice AND flooredPrice <> ceiledPrice;
						
						OPEN curBadPrices;
						FETCh curBadPrices INTO v_badPrices2;
											
						IF v_badPrices IS NOT NULL  THEN
							p_error := 50366;
							p_error_message := IFNULL( v_badPrices || '. ', '' ) || v_badPrices2;
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;

	                    CLOSE curBadPrices;

					END IF;

				END;
			END IF;


			END;




			--******************************
			--Updates
			--******************************

			-- * Set the custom description *

			UPDATE OITM SET U_PMX_CUDE = OITM."ItemCode" || IFNULL( ' - ' || OITM."ItemName", '') || IFNULL( ' - ' || OITM."CodeBars", '')
			WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del;

			-- * SBO UomGroup details to PMX_IPTY *

			DELETE FROM PMX_IPTY
			WHERE PMX_IPTY."ItemCode" IN (
				SELECT OITM."ItemCode"
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				LEFT JOIN (
					UGP1
					INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				) ON UGP1."UgpEntry" = OITM."UgpEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del
				  AND OITM."UgpEntry" <> -1
				  AND OUOM."UomCode" IS NULL
			);

			UPDATE PMX_IPTY SET "Quantity" = UGP1."BaseQty" / UGP1."AltQty", "UserSign" = IFNULL( OUGP."UserSign2", 1 ), "UpdateDateTime" = v_tsNow, "Version" = "Version" + 1
			FROM OITM
			INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
			INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
			INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
			INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
			WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del
			  AND OITM."UgpEntry" <> -1
			  AND "Quantity" <> CAST( UGP1."BaseQty" / UGP1."AltQty" AS DECIMAL(19,6) );

			INSERT INTO PMX_IPTY ( "InternalKey", "UserSign", "CreateDateTime", "Version", "ItemCode", "PackagingTypeCode", "Quantity", "BarcodeType" )
			SELECT PMX_IPTY_S.nextval, COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 ), v_tsNow, 1, OITM."ItemCode", OUOM."UomCode", UGP1."BaseQty" / UGP1."AltQty", 'G'
			FROM OITM
			INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
			INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
			INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
			LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode" AND PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
			WHERE OITM."ItemCode" = p_list_of_cols_val_tab_del
			  AND UGP1."UomEntry" <> OITM."IUoMEntry"
			  AND OITM."UgpEntry" <> -1
			  AND PMX_IPTY."PackagingTypeCode" IS NULL;


			--All checks are done so we can exit the checks.
			RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on BOM
		--**********************************************************************************************************
		IF (v_isBOM=1) THEN

			--The lines of BOM may not contain a returnable item that is not an inventory item
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(Lines."ChildNum" AS VARCHAR)
			FROM ITT1 AS Lines INNER JOIN OITM AS Item ON Lines."Code" = Item."ItemCode"
			WHERE Lines."Father" = p_list_of_cols_val_tab_del AND Item.U_PMX_RETR = 'Y' AND Item."InvntItem" = 'N';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50300;
				p_error_message := 'BOM ''' || p_list_of_cols_val_tab_del || ''': The line(s) ' || v_errorList || ' contain returnable items that are NOT inventory. All items can be added to a BOM except not inventory returnable items.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--The inventory items on the lines of a BOM may not have backflush selected.
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(Lines."ChildNum" AS VARCHAR)
			FROM ITT1 AS Lines INNER JOIN OITM AS Item ON Lines."Code" = Item."ItemCode"
			INNER JOIN OWHS ON Lines."Warehouse" = OWHS."WhsCode"
			WHERE Lines."Father" = p_list_of_cols_val_tab_del AND Item."InvntItem" = 'Y' AND Lines."IssueMthd" = 'B' AND OWHS."U_PMX_IMBP"='Y';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50301;
				p_error_message := 'BOM ''' || p_list_of_cols_val_tab_del || ''': The line(s) ' || v_errorList || ' have ''Backflush'' selected as issue method. If using the Produmex add-on this is not allowed. This must be set to ''Manual''.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--All checks are done so we can exit the checks.
			RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on ProductionOrder
		--**********************************************************************************************************
		IF (v_isProductionOrder=1) THEN
			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
			
			--The inventory items on the lines of a production order may not have backflush selected.
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM WOR1 AS Lines INNER JOIN OITM AS Item ON Lines."ItemCode" = Item."ItemCode"
			INNER JOIN OWOR AS ProductionOrder on ProductionOrder."DocEntry" = Lines."DocEntry"
			INNER JOIN OWHS ON Lines."wareHouse" = OWHS."WhsCode"
			WHERE Lines."DocEntry" = v_keyAsInteger AND Item."InvntItem" = 'Y' AND Lines."IssueType" = 'B' AND OWHS."U_PMX_IMBP"='Y' 
			AND ProductionOrder."Status" <> 'L';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50900;
				p_error_message := 'Production order ''' || p_list_of_cols_val_tab_del || ''': The line(s) ' || v_errorList || ' have ''Backflush'' selected as issue method. If using the Produmex add-on this is not allowed. This must be set to ''Manual''.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			BEGIN
				DECLARE v_parentDocEntry INT;
				DECLARE v_parentDocNum INT;
				DECLARE v_parentFound INT;
			
				SELECT this.U_PMX_PPRD, this.U_PMX_PPRN, parent."DocEntry"
				INTO v_parentDocEntry, v_parentDocNum, v_parentFound
				FROM OWOR this
				LEFT JOIN OWOR parent ON this.U_PMX_PPRD = parent."DocEntry" AND this.U_PMX_PPRN = parent."DocNum"
				WHERE this."DocEntry" = v_keyAsInteger;
				IF ( v_parentDocEntry IS NULL AND v_parentDocNum IS NOT NULL )
					OR ( v_parentDocEntry IS NOT NULL AND v_parentDocNum IS NULL )
					OR ( v_parentDocEntry IS NOT NULL AND v_parentDocNum IS NOT NULL AND v_parentFound IS NULL )
				THEN
					p_error := 50901;
					p_error_message := 'Production order ''' || p_list_of_cols_val_tab_del || ''': The "DocEntry" ''' || IFNULL( CAST( v_ParentDocEntry AS NVARCHAR ), '' ) || ''' and "DocNum" ''' || IFNULL( CAST( v_parentDocNum AS NVARCHAR ), '' ) || ''' referencing the parent production order do not match.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
	
				IF v_parentDocEntry = v_keyAsInteger THEN
					p_error := 50902;
					p_error_message := 'Production order ''' || p_list_of_cols_val_tab_del || ''': a production order cannot have itself as parent.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END;

			--Closed production orders cannot have the pmx status 'started'
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(OWOR."DocNum" AS VARCHAR)
			FROM OWOR										--Cancelled, Closed
			WHERE OWOR."DocEntry" = v_keyAsInteger AND OWOR."Status" IN ('C', 'L') AND OWOR.U_PMX_PRST = 'S';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50903;
				p_error_message := 'Production order ''' || v_errorList || ''': The production order cannot be closed when pmx status is started!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--Planned qty cannot be lowered if open proposals exist
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(OWOR."DocNum" AS VARCHAR) || '.' || CAST(WOR1."LineNum" AS VARCHAR)
			FROM OWOR
			INNER JOIN WOR1 ON WOR1."DocEntry" = OWOR."DocEntry"
			AND WOR1."PlannedQty" > 0
			LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseEntry" = WOR1."DocEntry"
			AND PMX_PLPL."BaseLine" = WOR1."LineNum"
			AND PMX_PLPL."BaseType" = '202'
			AND PMX_PLPL."LineStatus" = 'O'
			WHERE OWOR."DocEntry" = v_keyAsInteger
			GROUP BY OWOR."DocNum", WOR1."DocEntry", WOR1."LineNum", WOR1."PlannedQty"
			HAVING (SUM(IFNULL( PMX_PLPL."Quantity", 0)) > WOR1."PlannedQty");
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50904;
				p_error_message := 'Production order line ''' || v_errorList || ''': The production order planned quantity cannot be lowered because there are open proposal(s)!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--Productionline cannot be changed if open proposals exist
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(OWOR."DocNum" AS VARCHAR) 
			FROM OWOR
			INNER JOIN WOR1 ON WOR1."DocEntry" = OWOR."DocEntry"
			INNER JOIN PMX_PLPL ON PMX_PLPL."BaseEntry" = WOR1."DocEntry"
			AND PMX_PLPL."BaseLine" = WOR1."LineNum"
			AND PMX_PLPL."BaseType" = '202'
			INNER JOIN PMX_PLPH ON PMX_PLPH."DocEntry" = PMX_PLPL."DocEntry"
			INNER JOIN PMX_OSPL ON PMX_OSPL."Code" = OWOR.U_PMX_PLCD
			WHERE OWOR."DocEntry" = v_keyAsInteger 
			AND PMX_PLPH."DestStorLocCode" <> IFNULL(PMX_OSPL."PickToLocationCode", PMX_OSPL."InputLocationCode")
			AND PMX_PLPH."DocStatus" = 'O';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50905;
				p_error_message := 'Production order ''' || v_errorList || ''': The production order has a different production line than the proposal(s)!';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

		
			--Only one production order allowed for a production line with the pmx status 'started'
			SELECT COUNT(*) INTO v_cnt FROM OWOR WHERE OWOR."DocEntry" = v_keyAsInteger AND OWOR."Status" NOT IN ('C', 'L') 
				AND OWOR.U_PMX_PRST = 'S';
			IF v_cnt > 0 THEN

				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT CAST(OWOR."DocNum" AS VARCHAR)
				FROM OWOR	
				INNER JOIN OWOR AS currentOWOR
				ON OWOR.U_PMX_PLCD = currentOWOR.U_PMX_PLCD	
				INNER JOIN PMX_OSPL ON PMX_OSPL."Code" = currentOWOR.U_PMX_PLCD
										
				WHERE OWOR."DocEntry" <> v_keyAsInteger 
				AND currentOWOR."DocEntry" = v_keyAsInteger
									--Cancelled, Closed
				AND OWOR."Status" NOT IN ('C', 'L') 
				AND OWOR.U_PMX_PRST = 'S'
				AND PMX_OSPL."OnlyOneStartedProductionOrderAllowed" = 'Y';
    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50904;
					p_error_message := 'Production order(s) ''' || v_errorList || ''' started: There can only be 1 production order for a certain production line with the pmx status started!';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
			
			--All batchnumbers must be entered in upper case
			--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(OWOR."U_PMX_BTCH" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = v_keyAsInteger 
			AND OWOR."U_PMX_BTCH" <> UPPER(OWOR."U_PMX_BTCH");
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50906;
				p_error_message := 'Batch numbers must be upper case. This is not the case for the batchnumber: ' || v_errorList;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(OWOR."U_PMX_BTC2" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = v_keyAsInteger 
			AND OWOR."U_PMX_BTC2" <> UPPER(OWOR."U_PMX_BTC2");
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50907;
				p_error_message := 'Batch numbers must be upper case. This is not the case for the batchnumber 2: ' || v_errorList;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			DELETE FROM TMP_TN_CharListTableNoIndex;
		    INSERT INTO TMP_TN_CharListTableNoIndex
		    SELECT CAST(OWOR."U_PMX_NRPB" AS VARCHAR) 
			FROM OWOR
			WHERE OWOR."DocEntry" = v_keyAsInteger 
			AND OWOR."U_PMX_NRPB" = 0;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50908;
				p_error_message := 'The number of production batches cannot be 0 for production order ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--Add check for removing time registration items
			IF v_isTransactionTypeUpdate=1 THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST(PMX_TRHE."ItemCode" AS VARCHAR) 
				FROM PMX_TRHE
				LEFT JOIN WOR1 ON PMX_TRHE."BaseEntry" = WOR1."DocEntry" AND PMX_TRHE."ItemCode" = WOR1."ItemCode" 
				AND PMX_TRHE."BaseType" = '202'
				WHERE WOR1."ItemCode" IS NULL
				AND WOR1."DocEntry" = v_keyAsInteger;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50909;
					p_error_message := 'Cannot delete item codes because time has already been registered: ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
			
			--All checks are done so we can exit the checks.

			--Perform actions when adding production order

			--When a production order is duplicated, UDF's are also copied
			IF (v_isTransactionTypeAdd=1) THEN
				--The pmx status is also copied.
				--But the status should be (P)lanned when a production order is added.
				UPDATE OWOR SET OWOR.U_PMX_PRST = 'P' WHERE OWOR."DocEntry" = v_keyAsInteger AND OWOR.U_PMX_PRST <> 'P';
				--When production order is duplicated, the CurrentProdBatch should NOT be copied and set to default 1			
				UPDATE OWOR SET OWOR.U_PMX_CUPB = 1  WHERE OWOR."DocEntry" = v_keyAsInteger; --AND OWOR.U_PMX_PRST <> 'P';
				--Set picked quantity to 0
				UPDATE WOR1 SET WOR1.U_PMX_QTYP = 0 WHERE WOR1."DocEntry" = v_keyAsInteger AND U_PMX_QTYP <> 0;
			END IF;

            IF (v_isTransactionTypeUpdate=1) THEN
            
                DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT CAST(OWOR."DocNum" AS VARCHAR)
				FROM OWOR													
				WHERE OWOR."DocEntry" = v_keyAsInteger 				
				AND  OWOR.U_PMX_NRPB <= ( SELECT MAX(PMX_POBA."SeqNr") FROM PMX_POBA WHERE PMX_POBA."DocEntry" = 220 );
    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50906;
					p_error_message := 'Already components weighed for production order ''' || v_errorList || '''. Cannot alter number of production batches lower than current production batch.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			    --Remove locking when production order is closed
			    DELETE FROM PMX_INLD WHERE "BaseType" = '202' 
			    AND "BaseEntry" IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = v_keyAsInteger AND OWOR."Status" IN ('C', 'L'));
			    
			      --Remove dispended SSCC when production order is closed
			    DELETE FROM PMX_POPC WHERE "ProdOrderDocEntry"
			     IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = v_keyAsInteger AND 
						( OWOR."Status" IN ('C', 'L') OR OWOR.U_PMX_PRST IN ('C')));

                --Remove lined up locations when order is closed and production line is not used in another started production order
                DELETE FROM PMX_LIUP WHERE "ProductionLineCode" IN
                (SELECT U_PMX_PLCD FROM OWOR WHERE
                 OWOR."DocEntry" = v_keyAsInteger AND OWOR."Status" IN ('C', 'L') 
                 AND U_PMX_PLCD NOT IN 
                 (SELECT U_PMX_PLCD FROM OWOR WHERE
					OWOR."Status" = 'R' AND U_PMX_PRST = 'S'));

             END IF;
             
             IF(v_isTransactionTypeDelete=1 OR v_isTransactionTypeCancel=1) THEN
             
                --Remove dispended SSCC when production order is closed
			    DELETE FROM PMX_POPC WHERE "ProdOrderDocEntry"
			     IN (SELECT "DocEntry" FROM OWOR WHERE OWOR."DocEntry" = v_keyAsInteger );
             
             END IF;

			RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on a user
		--**********************************************************************************************************
		IF (v_isUser=1) THEN
			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

			--Check if the language key filled in at the user is an existing language key
			SELECT COUNT(OUSR.INTERNAL_K) INTO v_cnt
            FROM OUSR
			LEFT JOIN OLNG ON OUSR.U_PMX_LANG = OLNG."Code"
			WHERE OUSR.INTERNAL_K = v_keyAsInteger AND OLNG."Code" IS NULL AND OUSR.U_PMX_LANG IS NOT NULL;
			IF v_cnt > 0 THEN
				p_error := 50400;
				p_error_message := 'The language key filled in for user with key ' || p_list_of_cols_val_tab_del || ' does not exist.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
		
			--All checks are done so we can exit the checks.
			RETURN;	
		END IF;

		--**********************************************************************************************************
		--Do checks on a business partner
		--**********************************************************************************************************
		IF (v_isBusinessPartner=1) THEN

			--Check if this is a delete. 
			IF (v_isTransactionTypeDelete=1) THEN
				
				--Check if the customer is not used in a route template
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT PMX_RTTH."Name"
				FROM PMX_RTTH
				INNER JOIN PMX_RTTL ON PMX_RTTL."ParentCode" = PMX_RTTH."Code"
				WHERE PMX_RTTL."CardCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50500;
					p_error_message := 'Business partner with card code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the customer is used in route templates with name: ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the customer is not used in a picklist proposal
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT PMX_PLPH."DocEntry"
				FROM PMX_PLPH
				WHERE PMX_PLPH."CardCode" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50501;
					p_error_message := 'Business partner with card code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted because the customer is used in picklist proposals with doc number: ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--No other checks must be done for the item delete
				RETURN;
			END IF;

			IF (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate=1 ) THEN
			
				--Check if linked business partner exists
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT OCRD.U_PMX_LBPA
				FROM OCRD
				WHERE OCRD."CardCode" = p_list_of_cols_val_tab_del
				AND NOT EXISTS (SELECT * FROM OCRD AS InnerOCRD WHERE InnerOCRD."CardCode" = OCRD.U_PMX_LBPA)
				AND IFNULL( OCRD.U_PMX_LBPA, '') <> '';
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50502;
					p_error_message := 'Business partner with card code ''' || p_list_of_cols_val_tab_del || ''' has an incorrect linked business partner: ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--No other checks must be done for the item delete
				RETURN;
			END IF;

			RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on a sales order
		--**********************************************************************************************************
		IF (v_isSalesOrderDocument=1) THEN

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
			
			--Check if the sales order is closed that there are no open pick list proposals of this sales order
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(PMX_PLPL."DocEntry" AS VARCHAR)
			FROM ORDR 
			INNER JOIN PMX_PLPL ON ORDR."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = ORDR."ObjType"
			INNER JOIN RDR1 ON ORDR."DocEntry" = RDR1."DocEntry" AND RDR1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE ORDR."DocEntry" = v_keyAsInteger AND (ORDR."DocStatus" = 'C' OR RDR1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50600;
				p_error_message := 'The sales order ''' || p_list_of_cols_val_tab_del || ''' has open pick list proposal(s): ' || v_errorList || '. The pick list proposal(s) must be closed first before closing the sales order (line).';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the full quantity is selected when entering batch numbers
--			v_errorList := ''
--			SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' '
--			FROM RDR1 AS Lines
--			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
--			LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesOrder."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
--			WHERE SalesOrder."DocEntry" = v_keyAsInteger AND Lines."LineStatus" = 'O'
--			GROUP BY Lines."LineNum", Lines."Quantity", Lines."NumPerMsr"
--			HAVING SUM(IBT1."Quantity") <> (Lines."Quantity" * Lines."NumPerMsr") AND SUM(IBT1."Quantity") > 0
--			IF LENGTH(v_errorList) <> 0 THEN
--				p_error := 50601
--				p_error_message := 'Error on line(s):  ' || v_errorList || ' You need to fill the full quantity when entering a batch number'
--				SELECT p_error, p_error_message
--				--EXEC xp_logevent p_error, p_error_message, ERROR
--				RETURN
--			END

			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM RDR1 AS Lines
            INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
            INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
            INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
            INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
            AND ITL1."SysNumber" = OBTN."SysNumber"
            WHERE OITL."AllocatEnt" = v_keyAsInteger 
			AND Lines."LineStatus" = 'O'
            AND "StockEff" = 2
            GROUP BY Lines."LineNum", Lines."OpenQty", Lines."NumPerMsr"
            HAVING SUM("AllocQty") <> (Lines."OpenQty" * Lines."NumPerMsr") AND SUM("AllocQty") > 0;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50601;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' You need to fill the full quantity when entering a batch number';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;



			--Check if sales order lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the sales order line is lower than on the open pick list proposal line / open pick list line
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CASE WHEN        RDR1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) || ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN RDR1 ON PMX_PLPL."BaseEntry" = RDR1."DocEntry" AND PMX_PLPL."BaseLine" = RDR1."LineNum"  AND PMX_PLPL."BaseType" = RDR1."ObjType"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = 17 AND PMX_PLPL."BaseEntry" = v_keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR IFNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50603;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' The item code on a sales order line cannot be changed if a picklist proposal is generated for it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CASE WHEN IFNULL( RDR1."OpenInvQty",0) < SUM(ROUND( PMX_PLPL."OpenQty" * PMX_PLPL."QuantityPerUom", IFNULL( OITM.U_PMX_UOMD, 6)) ) THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) || ' '  ) ELSE '' END
			FROM PMX_PLPL
			INNER JOIN OITM ON OITM."ItemCode" = PMX_PLPL."ItemCode"
			LEFT JOIN RDR1 ON PMX_PLPL."BaseEntry" = RDR1."DocEntry" AND PMX_PLPL."BaseLine" = RDR1."LineNum"  AND PMX_PLPL."BaseType" = RDR1."ObjType"
			WHERE PMX_PLPL."BaseType" = 17 AND PMX_PLPL."BaseEntry" = v_keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C') 
			GROUP BY PMX_PLPL."BaseLine", RDR1."OpenQty", RDR1."OpenInvQty";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50602;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' The quantity on a sales order line cannot be lowered if a picklist proposal is generated for it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM RDR1 AS Lines
			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesOrder."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."AllocatEnt" = Lines."DocEntry" AND BATCH."AllocateLn" = Lines."LineNum" AND BATCH."AllocateTp" = Lines."ObjType"
			WHERE SalesOrder."DocEntry" = v_keyAsInteger AND Lines."LineStatus" = 'O' AND Lines.U_PMX_AMBA = 'N'
			GROUP BY Lines."LineNum"
			HAVING (Count(*)) > 1;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50604;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;



			-- change of batch only allowed when not linked in DOLL
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM RDR1 AS Lines
			INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
			INNER JOIN PMX_DOLL ON PMX_DOLL."ObjType" = Lines."ObjType" 
			AND PMX_DOLL."DocEntry" = Lines."DocEntry" 
			AND PMX_DOLL."LineNum" = Lines."LineNum" 
			LEFT JOIN 
			(SELECT "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."AllocatEnt", OITL."AllocateLn", OITL."AllocateTp"
				HAVING SUM("AllocQty") > 0) AS BATCH
			ON BATCH."AllocatEnt" = Lines."DocEntry" AND BATCH."AllocateLn" = Lines."LineNum" AND BATCH."AllocateTp" = Lines."ObjType"
			AND BATCH."DistNumber" = PMX_DOLL."BatchNum"
			WHERE Lines."DocEntry" = v_keyAsInteger 
			AND Lines."LineStatus" = 'O'
			AND (IFNULL( BATCH."AllocQty", PMX_DOLL."LockedQuantity") < PMX_DOLL."LockedQuantity"
				OR (PMX_DOLL."BatchNum" IS NOT NULL AND BATCH."DistNumber" IS NULL )
				);
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50605;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' Proposal exists: Changing the batch number(s) not allowed';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Multiple batches on serial numbers are only allowed when multiple batches on line level are allowed
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(RDR1."LineNum" AS VARCHAR)
			FROM PMX_SELD	
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN RDR1 on RDR1."DocEntry" = PMX_SELD."BaseEntry"
			AND RDR1."LineNum" = PMX_SELD."BaseLine"
			AND RDR1."ObjType" = PMX_SELD."BaseType"
			WHERE RDR1.U_PMX_AMBA = 'N'
			AND RDR1."DocEntry" = v_keyAsInteger	
			GROUP BY RDR1."LineNum", PMX_SENU."ItriKey"
			HAVING COUNT(PMX_SENU."ItriKey") > 1;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50606;
				p_error_message := 'Serial numbers linked with multiple batches, but multiple batches are not allowed. Error on line(s):  ' || v_errorList ;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Linked serial numbers cannot be more than open quantity
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT 'Line ' || CAST(RDR1."LineNum" AS VARCHAR) || ', Link qty:' || CAST(COUNT(PMX_SELD."InternalKey") AS VARCHAR) || '; '
			FROM PMX_SELD
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN RDR1 on RDR1."DocEntry" = PMX_SELD."BaseEntry"
			AND RDR1."LineNum" = PMX_SELD."BaseLine"
			AND RDR1."ObjType" = PMX_SELD."BaseType"
			WHERE RDR1."DocEntry" = v_keyAsInteger
			AND PMX_SENU."InStock" = 'Y'
			AND RDR1."LineStatus" = 'O'
			GROUP BY RDR1."LineNum", RDR1."OpenQty", RDR1."NumPerMsr"
			HAVING COUNT(PMX_SELD."InternalKey") > RDR1."OpenQty" * RDR1."NumPerMsr";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50607;
				p_error_message := 'Number of linked serial numbers cannot be more than open quantity. Error on line(s):  ' || v_errorList ;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Due date minutes
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(ORDR.U_PMX_DDTM AS VARCHAR)
			FROM ORDR
			WHERE ORDR."DocEntry" = v_keyAsInteger
			AND (IFNULL( ORDR.U_PMX_DDTM, 0) > 59 OR
				IFNULL( ORDR.U_PMX_DDTM, 0) < 0);
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50608;
				p_error_message := 'The due date minutes: ' || v_errorList || ' are not valid for document ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Due date Hour
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT CAST(ORDR.U_PMX_DDTH AS VARCHAR)
			FROM ORDR
			WHERE ORDR."DocEntry" = v_keyAsInteger
			AND ( IFNULL( ORDR.U_PMX_DDTH, 0) < 0
					OR ( IFNULL( ORDR.U_PMX_DDTH, 0) > 23 AND IFNULL( ORDR.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR ( IFNULL( ORDR.U_PMX_DDTH, 0) > 11 AND IFNULL( ORDR.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50609;
				p_error_message := 'The due date hours: ' || v_errorList || ' are not valid for document ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" IN (
				SELECT "InternalKey"
				FROM PMX_INLD
				INNER JOIN ORDR ON ORDR."DocEntry" = PMX_INLD."BaseEntry" AND ORDR."ObjType" = PMX_INLD."BaseType"
				WHERE ORDR."DocStatus" = 'C' AND ORDR."DocEntry" = v_keyAsInteger
			);

			SELECT COUNT(*) INTO v_cnt FROM RDR1 WHERE RDR1."LineStatus" = 'C' AND RDR1."DocEntry" = v_keyAsInteger;
			IF(v_isTransactionTypeAdd=0 AND v_cnt>0) 
			THEN

				--Close container (lines) if needed
				UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
				FROM PMX_COLI
				INNER JOIN RDR1
				ON PMX_COLI."BaseEntry" = RDR1."DocEntry"
				AND PMX_COLI."BaseLine" = RDR1."LineNum"
				AND PMX_COLI."BaseType" = RDR1."ObjType"
				WHERE PMX_COLI."LineStatus" <> 'C'
				AND RDR1."DocEntry" = v_keyAsInteger
				AND RDR1."LineStatus" = 'C';

				--Close header if no open lines are found
				UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
				PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = v_tsNow, 
				PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM ORDR  WHERE ORDR."DocEntry" = v_keyAsInteger)
				WHERE PMX_COHE."DocStatus" = 'O'
				AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

			END IF;


			--**********************************************************************************************************
			--START: Add/update locking based on batch numbers
			--**********************************************************************************************************



			--Start with deleting the lockings that are not correct
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" NOT IN 
			(--Query to get the correct lockings
				SELECT 	PMX_INLD."InternalKey"
				FROM RDR1 AS Lines
				INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
				INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND CAST(OITL."AllocateTp" AS NVARCHAR) = Lines."ObjType"
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( "DistNumber", '')
				--AND IFNULL( PMX_ITRI."BestBeforeDate", 1) = IFNULL( "ExpDate", 1)

				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"
				AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

				WHERE OITL."AllocatEnt" = v_keyAsInteger 
				AND Lines."LineStatus" = 'O'
				AND "StockEff" = 2

				AND PMX_INLD."InternalKey" IS NOT NULL


				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				HAVING SUM("AllocQty") > 0

				-- Exclude locked stock created on goods receipt.
				UNION SELECT PMX_INLD."InternalKey"
				FROM RDR1 Lines 
				INNER JOIN POR1 ON 
				CAST(POR1."BaseType" AS NVARCHAR) = Lines."ObjType"
				AND POR1."BaseEntry" = Lines."DocEntry"
				AND POR1."BaseLine" = Lines."LineNum"

				INNER JOIN PDN1 ON 
				CAST(PDN1."BaseType" AS NVARCHAR) = POR1."ObjType"
				AND PDN1."BaseEntry" = POR1."DocEntry"
				AND PDN1."BaseLine" = POR1."LineNum"

				INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
				LEFT JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND CAST(OITL."AllocateTp" AS NVARCHAR) = Lines."ObjType"
				LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 IFNULL(   PMX_ITRI."BatchNumber", '') = IFNULL( "DistNumber", '')


				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"

				WHERE Lines."DocEntry" = v_keyAsInteger				

				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				--Only allow detailed locking from purchase order when no batch has been linked
				HAVING SUM( IFNULL( "AllocQty", 0)) = 0 

				--Exclude manual lockings on LUID level when no batch has been selected on sales order
				UNION SELECT PMX_INLD."InternalKey"
				FROM PMX_INLD 
				LEFT JOIN
				(
					SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
					FROM RDR1 AS Lines
									INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
									LEFT JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND CAST(OITL."AllocateTp" AS NVARCHAR) = Lines."ObjType"
									LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
									LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
									AND ITL1."SysNumber" = OBTN."SysNumber"

					WHERE OITL."AllocatEnt"  = v_keyAsInteger 
									AND Lines."LineStatus" = 'O'
									AND IFNULL( "StockEff", 2) = 2

						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
									Lines."NumPerMsr", "DistNumber", "ExpDate"

					HAVING SUM( IFNULL( "AllocQty", 0)) > 0
				) AS ORDR

				ON ORDR."LineNum" = PMX_INLD."BaseLine"
				AND ORDR."DocEntry" = PMX_INLD."BaseEntry"
				AND ORDR."ObjType" = PMX_INLD."BaseType"


				WHERE PMX_INLD."InvLockLevel" = 'L'
				AND PMX_INLD."BaseEntry" = v_keyAsInteger
				AND ORDR."DocEntry" IS NULL

				

			) 

			AND PMX_INLD."BaseType" = '17'
			AND PMX_INLD."BaseEntry" = v_keyAsInteger;


			BEGIN --TRY
/*
				DECLARE EXIT HANDLER FOR SQLEXCEPTION
				BEGIN
					p_error := 50699;
					p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
						|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' );
				END;
	*/
				DECLARE CURSOR salesOrderBatchCursor FOR 
					SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."ItemCode", 
					PMX_OSWH."Code" AS "PmxWhsCode", CASE WHEN OITM."U_PMX_ICAW" = 'Y' THEN OITM."U_PMX_DQUM" ELSE NULL END AS "CatchWeightRatio", Lines."U_PMX_SQOP"
					FROM RDR1 AS Lines
					INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
					INNER JOIN ORDR AS SalesOrder ON Lines."DocEntry" = SalesOrder."DocEntry"
					INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocateLn" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
					INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
					INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
								   AND ITL1."SysNumber" = OBTN."SysNumber"
					INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = Lines."WhsCode"
					LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType"
									  AND PMX_PLPL."BaseEntry" = Lines."DocEntry"
									  AND PMX_PLPL."BaseLine" = Lines."LineNum"
									  AND PMX_PLPL."LineStatus" = 'O'
					WHERE OITL."AllocatEnt" = :v_keyAsInteger 
					  AND Lines."LineStatus" = 'O'
					  AND "StockEff" = 2
					  --Only when no openpick list proposal is found
					  AND PMX_PLPL."DocEntry" IS NULL
					GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code", OITM."U_PMX_ICAW", OITM."U_PMX_DQUM", Lines."U_PMX_SQOP"
					HAVING SUM("AllocQty") > 0;

				OPEN salesOrderBatchCursor;
				FETCH salesOrderBatchCursor 
				INTO v_batchQuantityToLock,v_batchToLock,v_expDateToLock,v_lineNumToLock,v_docEntryToLock,v_typeToLock,v_itemCodeToLock,v_pmxWarehouseToLock, v_catchWeightRatioToLock, v_shippingQualityOptionToLock;
				WHILE v_docEntryToLock IS NOT NULL DO

					--Check for each batch if we have a corresponding locking line

					SELECT COUNT(*) INTO v_cnt FROM PMX_INLD 
						INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
						WHERE PMX_INLD."BaseType" = v_typeToLock
						AND PMX_INLD."BaseEntry" = v_docEntryToLock
						AND PMX_INLD."BaseLine" = v_lineNumToLock
						AND PMX_ITRI."BatchNumber" = v_batchToLock;
						--AND IFNULL( PMX_ITRI."BestBeforeDate", '') = IFNULL( curRow."ExpDate", '');
					IF v_cnt < 1
					THEN

						--Entry does not exist, so we need to lock the stock
						--First try to get the itri key to lock
						v_remainingQuantityToLock := v_batchQuantityToLock;

						BEGIN --TRY
							/*
							DECLARE EXIT HANDLER FOR SQLEXCEPTION
							BEGIN
								p_error := 50650
								p_error_message := 'Not enough free stock to allocate batch ' || curRow."DistNumber" || ' of item ' || curRow."ItemCode" || '. Missing qty: ' || CAST(v_remainingQuantityToLock AS NVARCHAR(20))
							END;
							*/
							
							DECLARE CURSOR itriCursor FOR
							SELECT PMX_ITRI."InternalKey", IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0), Inventory."QualityStatusCode"
							FROM PMX_ITRI 
							LEFT JOIN (
								SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
								FROM PMX_INVT 
								INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
								WHERE "QualityStatusCode" IN (
									SELECT "Code" FROM PMX_QYST 
									WHERE  (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
											)
								GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
							) AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
							LEFT JOIN (
								SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
								FROM PMX_INLD 
								WHERE "QualityStatusCode" IN (
									SELECT "Code" FROM PMX_QYST 
									WHERE  (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
											)
								GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
							) AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
							 AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
							WHERE PMX_ITRI."BatchNumber" = :v_batchToLock AND PMX_ITRI."ItemCode" = :v_itemCodeToLock
							  AND Inventory."PmxWhsCode" = :v_pmxWarehouseToLock
							  AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0;

							OPEN itriCursor;
							FETCH itriCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
							WHILE v_itriToLock IS NOT NULL AND v_remainingQuantityToLock > 0 DO

								IF (v_remainingQuantityToLock < v_freeQuantityToLock) THEN
									v_currentQuantityToLock := v_remainingQuantityToLock;
								ELSE
									v_currentQuantityToLock := v_freeQuantityToLock;
								END IF;

								--Insert INTO the PMX_INLD table
								INSERT INTO PMX_INLD ("InternalKey","BaseType","BaseEntry","BaseLine","CardCode","CardName",
									"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
									"UserSign","CreateDate",
									"CreateTime","LogUnitIdentKey","QualityStatusCode", 
									"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2")
								SELECT PMX_INLD_S.nextval, * FROM (
									SELECT v_typeToLock,v_docEntryToLock,v_lineNumToLock,MAX(ORDR."CardCode"),MAX(ORDR."CardName"),
										v_itemCodeToLock,v_currentQuantityToLock,v_itriToLock,NULL,
										MAX(ORDR."UserSign"),v_updateDate,
										v_updateTime,NULL, v_qualityStatusToLock,
										'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, v_currentQuantityToLock * v_catchWeightRatioToLock
									FROM ORDR 
									INNER JOIN RDR1 ON RDR1."DocEntry" = ORDR."DocEntry"
									INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = RDR1."WhsCode"
									WHERE ORDR."DocEntry" = v_docEntryToLock
									AND RDR1."LineNum" = v_lineNumToLock
								);

								--Reduce remaining
								v_remainingQuantityToLock := v_remainingQuantityToLock - v_currentQuantityToLock;
									
    							FETCH itriCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
							END WHILE;
							CLOSE itriCursor;

							--If there is remaining QTY, there is not enough free stock, so throw error
							IF(v_remainingQuantityToLock > 0) THEN
								p_error := 50650;
								p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock || '. Missing qty: ' || v_remainingQuantityToLock;
								SELECT p_error, p_error_message FROM dummy;
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN;
							END IF;

						END; /* TRY
						BEGIN CATCH
							p_error := 50690
							p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
							|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
							BEGIN TRY
								CLOSE itriCursor 
								DEALLOCATE itriCursor
								SELECT p_error, p_error_message
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN
							END TRY
							BEGIN CATCH
								DEALLOCATE itriCursor
								SELECT p_error, p_error_message
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN
							END CATCH
						END CATCH
*/
					ELSE 
						--The line exists, so now we need to check if the quantity is the same/more/less
					 	SELECT SUM(PMX_INLD."Quantity") - v_batchQuantityToLock INTO v_remainingQuantityToLock
					 	FROM PMX_INLD 
						INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
						WHERE PMX_INLD."BaseType" = v_typeToLock
						AND PMX_INLD."BaseEntry" = v_docEntryToLock
						AND PMX_INLD."BaseLine" = v_lineNumToLock
						AND PMX_ITRI."BatchNumber" = v_batchToLock
						GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine";
							
										
						IF v_remainingQuantityToLock <> 0 THEN
						--Only if deviation, then we try to update/insert
						--Loop through all locking lines
							BEGIN --TRY
/*
								DECLARE EXIT HANDLER FOR SQLEXCEPTION
								BEGIN
									p_error := 50691;
									p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
										|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' );
								END;
*/
								DECLARE CURSOR inldCursor FOR 
								SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey" AS "ItriKey"
								FROM PMX_INLD 
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = :v_typeToLock
								AND PMX_INLD."BaseEntry" = :v_docEntryToLock
								AND PMX_INLD."BaseLine" = :v_lineNumToLock
								AND PMX_ITRI."BatchNumber" = :v_batchToLock;

								OPEN inldCursor;
								FETCH inldCursor INTO v_currentLockInternalKey, v_lockedQuantity, v_itriToLock;
								WHILE v_currentLockInternalKey IS NOT NULL AND v_remainingQuantityToLock <> 0 DO

									IF v_remainingQuantityToLock > 0 THEN
									
										--Remaining is more than 0, so we can just remove the locking
										IF v_remainingQuantityToLock >= v_lockedQuantity THEN
											--Delete
											DELETE FROM PMX_INLD WHERE "InternalKey" = v_currentLockInternalKey;
											v_remainingQuantityToLock := v_remainingQuantityToLock - v_lockedQuantity;
										ELSE
											--update
											UPDATE PMX_INLD SET "Quantity" = "Quantity" - v_remainingQuantityToLock, "QuantityUom2" = "QuantityUom2" - (v_remainingQuantityToLock * v_catchWeightRatioToLock) WHERE "InternalKey" = v_currentLockInternalKey;
											v_remainingQuantityToLock := 0;
										END IF;
										
									ELSE
									
										--Remaining is less than 0, so we need to add locking
										--If there is still free stock for current itri, then take that one
										SELECT IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) 
										INTO v_freeQuantityToLock
										FROM PMX_ITRI 
										LEFT JOIN (
											SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" 
											FROM PMX_INVT 
											INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
											WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
											GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
										) AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (
											SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
											FROM PMX_INLD 
											WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') 
											GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
										) AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" 
											AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										    AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = v_batchToLock AND PMX_ITRI."ItemCode" = v_itemCodeToLock
								  		  AND Inventory."PmxWhsCode" = v_pmxWarehouseToLock
										  AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0
										  AND PMX_ITRI."InternalKey" = v_itriToLock;

										IF(v_remainingQuantityToLock > v_freeQuantityToLock) THEN
											v_currentQuantityToLock := v_freeQuantityToLock;
										ELSE
											v_currentQuantityToLock := v_remainingQuantityToLock;
										END IF;

										UPDATE PMX_INLD SET "Quantity" = "Quantity" + v_currentQuantityToLock, "QuantityUom2" = "QuantityUom2" + (v_currentQuantityToLock * v_catchWeightRatioToLock) WHERE PMX_INLD."InternalKey" = v_currentLockInternalKey;

										-- add qty, because v_remainingQuantityToLock is negative
										v_remainingQuantityToLock := v_remainingQuantityToLock + v_currentQuantityToLock;
										
									END IF;

    								FETCH inldCursor INTO v_currentLockInternalKey, v_lockedQuantity, v_itriToLock;
								END WHILE;
								CLOSE inldCursor;

							END; /* TRY
							BEGIN CATCH
								p_error := 50691
								p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
								|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
								BEGIN TRY
									CLOSE inldCursor 
									DEALLOCATE inldCursor
									SELECT p_error, p_error_message
									--EXEC xp_logevent p_error, p_error_message, ERROR
									RETURN
								END TRY
								BEGIN CATCH
									DEALLOCATE inldCursor
									SELECT p_error, p_error_message
									--EXEC xp_logevent p_error, p_error_message, ERROR
									RETURN
								END CATCH
							END CATCH
*/
							
							IF v_remainingQuantityToLock <> 0 THEN
								-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
								--Set v_remainingQuantityToLock as positive quanitty
								v_remainingQuantityToLock := ABS(v_remainingQuantityToLock);
								

								BEGIN --TRY
/*
									DECLARE EXIT HANDLER FOR SQLEXCEPTION
									BEGIN
										p_error := 50691;
										p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
											|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' );
									END;
*/
									DECLARE CURSOR itriCursor FOR
									SELECT PMX_ITRI."InternalKey", IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0), Inventory."QualityStatusCode"
									FROM PMX_ITRI 
									LEFT JOIN (
										SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
										FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
										GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode"
									) AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
									LEFT JOIN (
										SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
										FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y')
										GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode"
									) AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
									  AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
									  AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
									WHERE PMX_ITRI."BatchNumber" = :v_batchToLock AND PMX_ITRI."ItemCode" = :v_itemCodeToLock
									  AND Inventory."PmxWhsCode" = :v_pmxWarehouseToLock
									  AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0;

									OPEN itriCursor;
									FETCH itriCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
									WHILE v_itriToLock IS NOT NULL AND v_remainingQuantityToLock > 0 DO

										IF (v_remainingQuantityToLock < v_freeQuantityToLock) THEN
											v_currentQuantityToLock := v_remainingQuantityToLock;
										ELSE
											v_currentQuantityToLock := v_freeQuantityToLock;
										END IF;
	
										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ( "InternalKey", "BaseType", "BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
										SELECT PMX_INLD_S.nextval, * FROM (
											SELECT v_typeToLock,v_docEntryToLock,v_lineNumToLock,MAX(ORDR."CardCode"),MAX(ORDR."CardName"),
												v_itemCodeToLock,v_currentQuantityToLock,v_itriToLock,NULL,
												MAX(ORDR."UserSign"),v_updateDate,
												v_updateTime,NULL, v_qualityStatusToLock,
												'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, v_currentQuantityToLock * v_catchWeightRatioToLock
											FROM ORDR 
											INNER JOIN RDR1 ON RDR1."DocEntry" = ORDR."DocEntry"
											INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = RDR1."WhsCode"
											WHERE ORDR."DocEntry" = v_docEntryToLock
											AND RDR1."LineNum" = v_lineNumToLock
										);

										--Reduce remaining
										v_remainingQuantityToLock := v_remainingQuantityToLock - v_currentQuantityToLock;
											
    									FETCH itriCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
									END WHILE;
									CLOSE itriCursor;

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(v_remainingQuantityToLock > 0) THEN
										p_error := 50652;
										p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock || '. Missing qty: ' || v_remainingQuantityToLock;
										SELECT p_error, p_error_message FROM dummy;
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN;
									END IF;

								END; /* TRY
								BEGIN CATCH
									p_error := 50691
									p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
									|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE itriCursor 
										DEALLOCATE itriCursor
										SELECT p_error, p_error_message
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE itriCursor
										SELECT p_error, p_error_message
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN
									END CATCH
								END CATCH
*/
							END IF;


						END IF;

						IF v_remainingQuantityToLock <> 0 THEN
						-- if quantity is still different from 0, then there was not enough stock to reserve!
							p_error := 50653;
							p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock || '. Missing qty: ' || v_remainingQuantityToLock;
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;

					END IF;

					FETCH salesOrderBatchCursor 
					INTO v_batchQuantityToLock,v_batchToLock,v_expDateToLock,v_lineNumToLock,v_docEntryToLock,v_typeToLock,v_itemCodeToLock,v_pmxWarehouseToLock,v_catchWeightRatioToLock, v_shippingQualityOptionToLock;
				END WHILE;
				CLOSE salesOrderBatchCursor;

			END; /* TRY
			BEGIN CATCH
				p_error := 50699
				p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
				|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE salesOrderBatchCursor 
					DEALLOCATE salesOrderBatchCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE salesOrderBatchCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END CATCH
			END CATCH
*/

			--**********************************************************************************************************
			--END: Add/update locking based on batch numbers
			--**********************************************************************************************************

			--**********************************************************************************************************
			--START: SAP Serial numbers
			--**********************************************************************************************************

			--Delete system serial links
			DELETE FROM PMX_SELD WHERE "LinkStatus" = 'S'
			AND PMX_SELD."BaseEntry" = v_keyAsInteger
			AND PMX_SELD."BaseType" = '17';

			--Add to PMX serial numbers table

			INSERT INTO PMX_SELD ( "InternalKey", "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine", "LinkStatus") 
			SELECT PMX_SELD_S.nextval, 'N', SERIAL."UserSign", v_tsNow, v_tsNow, 1, PMX_SENU."InternalKey", "ApplyType", "ApplyEntry", "ApplyLine", 'S'	
			FROM (
                SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty", "StockEff", ORDR."UserSign"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
				AND ITL1."SysNumber" = OSRN."SysNumber"
				INNER JOIN ORDR ON ORDR."DocEntry" = OITL."ApplyEntry"
				INNER JOIN RDR1 ON RDR1."DocEntry" = OITL."ApplyEntry"
				AND RDR1."LineNum" = OITL."ApplyLine"
				AND RDR1."ObjType" = OITL."ApplyType"

				WHERE "StockEff" = 2
				AND OITL."ApplyEntry" = v_keyAsInteger
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", ORDR."UserSign"
				HAVING SUM("AllocQty") > 0 AND "DistNumber" IS NOT NULL
			) AS SERIAL

			INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = SERIAL."ItemCode"
			AND PMX_SENU."SerialNumber" = SERIAL."DistNumber"
			LEFT JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
			AND PMX_SELD."BaseEntry" = SERIAL."ApplyEntry"
			AND PMX_SELD."BaseLine" = SERIAL."ApplyLine"
			AND PMX_SELD."BaseType" = '17'
			WHERE PMX_SELD."InternalKey" IS NULL;


			--**********************************************************************************************************
			--END: SAP Serial numbers
			--**********************************************************************************************************

			RETURN;

		END IF;


		--**********************************************************************************************************
		--Do checks on a purchase order - ERROR 51300 - 51399
		--**********************************************************************************************************
		IF (v_isPurchaseOrderDocument=1) THEN

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			SELECT COUNT(*) INTO v_cnt FROM POR1 WHERE POR1."LineStatus" = 'C' AND POR1."DocEntry" = v_keyAsInteger;
			IF(v_isTransactionTypeAdd=0 AND v_cnt>0)
			THEN

				--Close container (lines) if needed
				UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
				FROM PMX_COLI
				INNER JOIN POR1
				ON PMX_COLI."BaseEntry" = POR1."DocEntry"
				AND PMX_COLI."BaseLine" = POR1."LineNum"
				AND PMX_COLI."BaseType" = POR1."ObjType"
				WHERE PMX_COLI."LineStatus" <> 'C'
				AND POR1."DocEntry" = v_keyAsInteger
				AND POR1."LineStatus" = 'C';

				--Close header is no open lines are found
				UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
				PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = v_tsNow, 
				PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM OPOR  WHERE OPOR."DocEntry" = v_keyAsInteger)
				WHERE PMX_COHE."DocStatus" = 'O'
				AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

			END IF;
			
			IF ( v_isTransactionTypeAdd = 0 )
			THEN
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST(PMX_COLI."DocEntry" AS VARCHAR)
				FROM PMX_COLI 
				LEFT JOIN POR1 ON POR1."DocEntry" = PMX_COLI."BaseEntry" AND POR1."LineNum" = PMX_COLI."BaseLine" 
				 WHERE PMX_COLI."LineStatus" <> 'C' 
				 AND PMX_COLI."BaseEntry" = v_keyAsInteger 
				 AND PMX_COLI."BaseType" = N'22'
				 AND POR1."ItemCode" IS NULL;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 51300;
					p_error_message := 'Cannot delete purchase order line(s) because of container-document(s): ' || v_errorList || '. These container-document(s) have links to the purchase order line(s).';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;
	
			RETURN;
		END IF;
		
		--**********************************************************************************************************
		--Do checks on a warehouse: ERROR 50850 - 50900
		--**********************************************************************************************************
		IF (v_isWarehouse=1) THEN

			--Check if bin locations are not enabled
			DELETE FROM TMP_TN_CharListTableNoIndex;
			INSERT INTO TMP_TN_CharListTableNoIndex
			SELECT OWHS."WhsCode"
			FROM OWHS
			WHERE OWHS."WhsCode" = p_list_of_cols_val_tab_del
			AND OWHS."BinActivat" = 'Y'
			AND OWHS.U_PMX_IMBP = 'Y';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50850;
				p_error_message := 'Warehouse with code''' || p_list_of_cols_val_tab_del || ''' cannot have BIN locations enabled ';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			SELECT COUNT(*) INTO v_cnt FROM OWHS
				WHERE OWHS."WhsCode" = p_list_of_cols_val_tab_del
				AND OWHS.U_PMX_IMBP = 'Y';
			IF v_cnt>0 THEN

				--The warehouse needs to be managed by Produmex.
				--So now check if this is allowed
				SELECT count(*) INTO v_cnt FROM OITW 
					WHERE "WhsCode" = p_list_of_cols_val_tab_del 
					AND "OnHand" > 0;
				SELECT count(*) INTO v_cnt2 FROM PMX_OSWH
					LEFT JOIN PMX_OSEL ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
					LEFT JOIN PMX_INVT ON PMX_INVT."StorLocCode" = PMX_OSEL."Code"
					WHERE PMX_OSWH."SboWhsCode" = p_list_of_cols_val_tab_del 
					AND IFNULL(ABS(PMX_INVT."Quantity"), 0) > 0;
				IF v_cnt>0 AND v_cnt2=0 THEN
					p_error := 50851;
					p_error_message := 'Warehouse with code''' || p_list_of_cols_val_tab_del || ''' cannot be managed by Produmex, because there is stock in SAP, but not in PMX.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			END IF;
			
			SELECT COUNT(*) INTO v_cnt FROM OWHS
				WHERE OWHS."WhsCode" = p_list_of_cols_val_tab_del
				AND OWHS.U_PMX_IMBP = 'N';
			IF v_cnt>0 THEN
		
				--The warehouse does not need to be managed by Produmex.
				--So now check if this is allowed
				SELECT COUNT(*) INTO v_cnt FROM PMX_INVT 
						LEFT JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode"
						LEFT JOIN PMX_OSWH ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
						WHERE "SboWhsCode" = p_list_of_cols_val_tab_del;				
				IF v_cnt>0 THEN
					
					p_error := 50852;
					p_error_message := 'Warehouse with code''' || p_list_of_cols_val_tab_del || ''' needs to be managed by Produmex, because there is stock in PMX.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
			END IF;

            RETURN;
		END IF;


		--**********************************************************************************************************
		--Do checks on a barcode: ERROR 51900 - 51950
		--**********************************************************************************************************
		IF (v_isBarcode=1) THEN
		
			DECLARE v_sqlStatementBeginBarcode  NCLOB;
			DECLARE v_sqlStatementEndBarcode  NCLOB;
			DECLARE v_sqlStatementBarcode  NCLOB;
			DECLARE v_parameterListBarcode  NCLOB;

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
		
			--Check the GTINs of the item
			v_sqlStatementBeginBarcode := 'INSERT INTO ' || v_extraDb || '."TMP_IntStr" SELECT Error, Message FROM ' || v_extraDb || '."PMX_FN_CheckGTIN"(''';
			v_sqlStatementEndBarcode := ''')';
			v_parameterListBarcode := 'p_error INT output, p_error_message NVARCHAR(200) output';
			
			
			--Check the GTIN of the uom barcodes
			BEGIN
				DECLARE v_barcodeForBarcode NVARCHAR(254);
				DECLARE v_barcodeItemCode NVARCHAR(254);
				DECLARE CURSOR barCodeCursor FOR SELECT "BcdCode", OBCD."ItemCode" 
												 FROM OBCD 
												 INNER JOIN OITM ON OITM."ItemCode" = OBCD."ItemCode"
												 WHERE OBCD."BcdEntry" = :p_list_of_cols_val_tab_del AND LENGTH("BcdCode") <> 0 
												 AND (OITM.U_PMX_BCTY = 'G');

    			OPEN barCodeCursor;
    			FETCH barCodeCursor INTO v_barcodeForBarcode, v_barcodeItemCode;
				WHILE v_barcodeForBarcode IS NOT NULL DO
				
					DELETE FROM "WUK_4004_PMX"."TMP_IntStr";
					v_sqlStatementBarcode := v_sqlStatementBeginBarcode || v_barcodeForBarcode || v_sqlStatementEndBarcode;
					EXEC v_sqlStatementBarcode;  --, v_parameterListBarcode, p_error output, p_error_message output
					SELECT TOP 1 "Key", "Value" INTO p_error, p_error_message FROM "WUK_4004_PMX"."TMP_IntStr";

					IF (p_error <> 0) THEN
						p_error_message := '"Barcode" ''' || v_barcodeForBarcode ||  '''  of Item ' || v_barcodeItemCode || ': ' || p_error_message;
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
            			CLOSE barCodeCursor;
						RETURN;
					END IF;
				
        			FETCH barCodeCursor INTO v_barcodeForBarcode, v_barcodeItemCode;
				END WHILE;
    			CLOSE barCodeCursor;
			END;

            RETURN;
		END IF;

		--**********************************************************************************************************
		--Do actions on change of UomGroup
		--**********************************************************************************************************
		IF v_isUomGroup = 1 THEN

			IF (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate = 1) THEN

				DELETE FROM PMX_IPTY WHERE "InternalKey" IN ( SELECT "InternalKey"
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				LEFT JOIN (
					UGP1
					INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				) ON UGP1."UgpEntry" = OITM."UgpEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."UgpEntry" = p_list_of_cols_val_tab_del 
				AND OUOM."UomCode" IS NULL );

				UPDATE PMX_IPTY SET "Quantity" = UGP1."BaseQty" / UGP1."AltQty", "UserSign" = COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 )
                    , "UpdateDateTime" = v_tsNow, "Version" = "Version" + 1
				FROM OITM
				INNER JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode"
				INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
				INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry" AND OUOM."UomCode" = PMX_IPTY."PackagingTypeCode"
				WHERE OITM."UgpEntry" = p_list_of_cols_val_tab_del 
  				  AND "Quantity" <> CAST( UGP1."BaseQty" / UGP1."AltQty" AS DECIMAL(19,6) );

				INSERT INTO PMX_IPTY ( "InternalKey", "UserSign", "CreateDateTime", "Version", "ItemCode", "PackagingTypeCode", "Quantity", "BarcodeType", "Canceled", "HideDuringEnteringQuantity", "AskDuringReception" )
				SELECT PMX_IPTY_S.nextval, COALESCE( OUGP."UserSign2", OUGP."UserSign", 1 ), v_tsNow, 1, OITM."ItemCode", OUOM."UomCode", UGP1."BaseQty" / UGP1."AltQty", 'G', 'N', 'N', 'N'
				FROM OITM
				INNER JOIN OUGP ON OUGP."UgpEntry" = OITM."UgpEntry"
				INNER JOIN UGP1 ON UGP1."UgpEntry" = OUGP."UgpEntry"
				INNER JOIN OUOM ON OUOM."UomEntry" = UGP1."UomEntry"
				LEFT JOIN PMX_IPTY ON PMX_IPTY."ItemCode" = OITM."ItemCode" AND PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
				WHERE OITM."UgpEntry" = p_list_of_cols_val_tab_del 
				  AND UGP1."UomEntry" <> OITM."IUoMEntry"
				  AND PMX_IPTY."PackagingTypeCode" IS NULL;

			END IF;

	--This code is in comment because for some reason the delete is not captured!

	--		IF (v_isTransactionTypeDelete=1 OR v_isTransactionTypeUpdate = 1) THEN
	--			--Check if the "Uom" is used on item
	--			v_errorList := ''
	--			SELECT v_errorList = v_errorList || "OITM"."ItemCode" || ' ' 
	--			FROM "OITM"
	--			WHERE "BuyUnitMsr" = p_list_of_cols_val_tab_del OR
	--					"SalUnitMsr" = p_list_of_cols_val_tab_del OR
	--					"InvntryUom" = p_list_of_cols_val_tab_del OR
	--					U_PMX_UOM2 = p_list_of_cols_val_tab_del
	--			IF LENGTH(v_errorList) <> 0 THEN
	--				p_error := 80002
	--				p_error_message := 'PmxUom with code ''' || p_list_of_cols_val_tab_del || ''' can not be deleted or updated because it is used in items with code ''' || v_errorList || '''.'
	--				SELECT p_error, p_error_message
	--				--EXEC xp_logevent p_error, p_error_message, ERROR
	--				RETURN
	--			END 
	--		END

            RETURN;
		END IF;


		--**********************************************************************************************************
		--Do checks on an PmxPriority
		--**********************************************************************************************************
		IF (v_isPriority=1) THEN
			IF (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate = 1) THEN
                --Check if the Msequence is not null
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PRIO"."Code"
				FROM "@PMX_PRIO"
				WHERE "@PMX_PRIO"."Code" = p_list_of_cols_val_tab_del AND "@PMX_PRIO".U_PMX_SEQU IS NULL;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 80101;
					p_error_message := 'PmxPriority with code ''' || p_list_of_cols_val_tab_del || ''' can not be added or updated because the "Sequence" is not filled in.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if the Sequence is unique
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PRIO"."Code"
				FROM "@PMX_PRIO"
				WHERE "@PMX_PRIO"."Code" <> p_list_of_cols_val_tab_del 
				AND "@PMX_PRIO".U_PMX_SEQU = (SELECT TOP 1 "@PMX_PRIO".U_PMX_SEQU FROM "@PMX_PRIO" WHERE "@PMX_PRIO"."Code" = p_list_of_cols_val_tab_del);
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 80102;
					p_error_message := 'PmxPriority with code ''' || p_list_of_cols_val_tab_del || ''' can not be added or updated because the "Sequence" is not unique';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			END IF;

            RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on an v_isFreightChargeDefinition
		--**********************************************************************************************************
		IF (v_isFreightChargeDefinition = 1) THEN
			IF (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate = 1) THEN
				--Check if shipping type exists in table
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_FCDE".U_PMX_SHTY
				FROM "@PMX_FCDE" LEFT JOIN "OSHP"
				ON OSHP."TrnspCode" = "@PMX_FCDE".U_PMX_SHTY
				WHERE "@PMX_FCDE"."Code" = p_list_of_cols_val_tab_del AND "OSHP"."TrnspCode" IS NULL;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 80201;
					p_error_message := 'Shipping type with code ''' || v_errorList || ''' does not exist.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if freight exists in table
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_FCDE".U_PMX_FCDE
				FROM "@PMX_FCDE" LEFT JOIN "OEXD"
				ON OEXD."ExpnsCode" = "@PMX_FCDE".U_PMX_FCDE
				WHERE "@PMX_FCDE"."Code" = p_list_of_cols_val_tab_del AND "OEXD"."ExpnsCode" IS NULL;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 80202;
					p_error_message := 'Freight charge with code ''' || v_errorList || ''' does not exist.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
			END IF;

            RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on an ItemPackagingType
		--**********************************************************************************************************
		IF (v_isItemPackagingType = 1) THEN
			IF (v_isTransactionTypeAdd=1 OR v_isTransactionTypeUpdate = 1) THEN

				v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

				--Check if packaging type exists in table
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT PMX_IPTY."PackagingTypeCode"
				FROM PMX_IPTY 
                LEFT JOIN OUOM
				ON PMX_IPTY."PackagingTypeCode" = OUOM."UomCode"
				WHERE PMX_IPTY."InternalKey" = v_keyAsInteger AND OUOM."UomCode" IS NULL;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81001;
					p_error_message := 'ItemPackagingType with code ''' || v_errorList || ''' does not exist.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			END IF;

            RETURN;
		END IF;

		--**********************************************************************************************************
		--Do checks on an PickListType
		--**********************************************************************************************************
		IF (v_isPickListType = 1) THEN
			IF (v_isTransactionTypeAdd = 1 OR v_isTransactionTypeUpdate = 1) THEN

				--Check if split up pick list types are for the same type
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				FROM "@PMX_PLTY" 
				INNER JOIN  "@PMX_PLTY" PLTY_ITEM
				ON PLTY_ITEM."Code" = "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				WHERE PLTY_ITEM.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."Code" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81051;
					p_error_message := 'Split up for for items ''' || v_errorList || ''' is not of the same type as for pick list type with code ''' || p_list_of_cols_val_tab_del || '''';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check if split up pick list types are for the same type
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				FROM "@PMX_PLTY" 
				INNER JOIN  "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."Code" = p_list_of_cols_val_tab_del ;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81052;
					p_error_message := 'Split up type for full ''' || v_errorList || ''' is not of the same type as for pick list type with code ''' || p_list_of_cols_val_tab_del || '''';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				--Check if proposals exist with production that do not correspond to pick object type
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PLTY"."Code"
				FROM "@PMX_PLTY" 
				INNER JOIN  PMX_PLPH
				ON PMX_PLPH."PickListType" = "@PMX_PLTY"."Code"
				WHERE PMX_PLPH."PickObjType" = 'Production' 
				AND "@PMX_PLTY".U_PMX_PROD = 'N'
				AND "@PMX_PLTY"."Code" = p_list_of_cols_val_tab_del 
				AND PMX_PLPH."DocStatus" = 'O';
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81053;
					p_error_message := 'Proposals of pick object type ''production'' exist for pick list type with code ''' || p_list_of_cols_val_tab_del || '''';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				--Check if proposals exist without production that do not correspond to pick object type
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PLTY"."Code"
				FROM "@PMX_PLTY" 
				INNER JOIN  PMX_PLPH
				ON PMX_PLPH."PickListType" = "@PMX_PLTY"."Code"
				WHERE PMX_PLPH."PickObjType" <> 'Production' 
				AND "@PMX_PLTY".U_PMX_PROD = 'Y'
				AND "@PMX_PLTY"."Code" = p_list_of_cols_val_tab_del 
				AND PMX_PLPH."DocStatus" = 'O';
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81054;
					p_error_message := 'Proposals of pick object type different from ''production'' exist for pick list type with code ''' || p_list_of_cols_val_tab_del || '''';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				--Check if split up pick list types are for the same type
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PLTY"."Code"
				FROM "@PMX_PLTY" 
				INNER JOIN "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_FullPalletPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."U_PMX_FullPalletPLTY" = p_list_of_cols_val_tab_del ;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81055;
					p_error_message := 'The pick list type is used as split for full pallets, but the type does not correspond  on ''' || v_errorList ;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				--Check if split up pick list types are for the same type
				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT "@PMX_PLTY"."Code"
				FROM "@PMX_PLTY" 
				INNER JOIN "@PMX_PLTY" PLTY_FULL
				ON PLTY_FULL."Code" = "@PMX_PLTY"."U_PMX_ItemPickPLTY"
				WHERE PLTY_FULL.U_PMX_PROD <> "@PMX_PLTY".U_PMX_PROD
				AND "@PMX_PLTY"."U_PMX_ItemPickPLTY" = p_list_of_cols_val_tab_del;
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 81056;
					p_error_message := 'The pick list type is used as split for full pallets, but the type does not correspond  on ''' || v_errorList ;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			END IF;

            RETURN;
		END IF;


		--*******************************************************************
		--*																	*
		--*Code below is for maintaining inventory between SBO and Produmex	*
		--*Check on documents												*
		--*																	*
		--*******************************************************************

		BEGIN


		--Variables used for the document and document lines
		DECLARE v_lineNums  NVARCHAR(4000);
		DECLARE v_objType  NVARCHAR(20);
		DECLARE v_docEntry  INT;
		DECLARE v_docNum  INT;
		DECLARE v_docDate  DATETIME;
		DECLARE v_cardCode  NVARCHAR(15);
		DECLARE v_cardName  NVARCHAR(100);
		DECLARE v_userSign  INT;
		DECLARE v_createDate  DATETIME;
		DECLARE v_createTime  INT;
		DECLARE v_stockDirection  INT;
		DECLARE v_lineNum  INT;
		DECLARE v_baseType  NVARCHAR(20);
		DECLARE v_baseEntry  INT;
		DECLARE v_BaseLine  INT;
		DECLARE v_itemCode  NVARCHAR(20);
		DECLARE v_description  NVARCHAR(100);
		DECLARE v_quantity  numeric(19, 6);
		DECLARE v_inventoryQuantity  numeric(19, 6);
		DECLARE v_uom  NVARCHAR(20);
		DECLARE v_inventoryUom  NVARCHAR(20);
		DECLARE v_quantityPerUom  numeric(19, 6);
		DECLARE v_whsCode  nvarchar(8);
		DECLARE v_batchNumbers1  NVARCHAR(256000);
		DECLARE v_batchNumbers2  NVARCHAR(256000);
		DECLARE v_bestBeforeDates  NVARCHAR(256000);
		DECLARE v_quantities  NVARCHAR(256000);
		DECLARE v_storLocs  NVARCHAR(256000);
		DECLARE v_luids  NVARCHAR(256000);
		DECLARE v_ssccs  NVARCHAR(256000);
		DECLARE v_qualityStatuses  NVARCHAR(256000);
		DECLARE v_isLogisticCarrier  NVARCHAR(1);
		DECLARE v_reasonCode  NVARCHAR(256000);
		DECLARE v_quantityUom2  numeric(19, 6);
		DECLARE v_uom2  NVARCHAR(20);
		DECLARE v_quantitiesUom2  NVARCHAR(256000);
		DECLARE v_dropShip  NVARCHAR(1);
		--DECLARE	v_destinationStorLocCode  NVARCHAR(50);
		DECLARE	v_destinationStorLocCodes  NVARCHAR(256000);
		DECLARE	v_destinationWhsCode  NVARCHAR(8);
		DECLARE	v_destinationQualityStatusCodes  NVARCHAR(256000);
		--DECLARE	v_destinationLUID  INT;
		DECLARE v_destinationLuids  NVARCHAR(256000);
		DECLARE v_trackLocationForSerial  NVARCHAR(1);
		DECLARE v_isSAPSerial  NVARCHAR(1);
		DECLARE v_isCatchWeightForDocument  NVARCHAR(1);
		DECLARE v_masterLuids NVARCHAR(256000);


		--Declare memory tables for changing the raw document lines (with pipes) to normal document lines (without pipes)
		DELETE FROM TMP_TN_BatchNumbers1;
		DELETE FROM TMP_TN_BatchNumbers2;
		DELETE FROM TMP_TN_BestBeforeDates;
		DELETE FROM TMP_TN_Quantities;
		DELETE FROM TMP_TN_QuantitiesUom2;
		DELETE FROM TMP_TN_StorLocs;
		DELETE FROM TMP_TN_Luids;
		DELETE FROM TMP_TN_SSCCs;
		DELETE FROM TMP_TN_QualityStatuses;
		DELETE FROM TMP_TN_DestStorLocs;
		DELETE FROM TMP_TN_DestLuids;
		DELETE FROM TMP_TN_DestQualityStatuses;
		DELETE FROM TMP_TN_MasterLuids;

		--Declare memory table for storing the details of the inventory
		DELETE FROM TMP_TN_InventoryDetail;

		--Declare memory table for storing the document lines in raw format. We have to store them in a nicer format
		DELETE FROM TMP_TN_DocumentLineRaw;

		--Declare memory table for storing the document lines of a purchase delivery note, sales delivery note, ...
		DELETE FROM TMP_TN_DocumentLine;

		--**********************************************************************************************************
		--If the object that is changed is of the type document then
		--copy the document lines in a memory table
		--This memory table is used because it is easier and faster to do all the checks on this memory table
		--instead of getting the lines again from the database.
		--Checks are also done on the document lines
		--If a document line is indicated that it is a logistic carrier then this logistic carrier is put on the
		--logistic unit.
		--**********************************************************************************************************
		IF (v_isDocument=1) AND (v_isTransactionTypeAdd=1 OR (v_isTransactionTypeCancel=1 AND v_isStockTransfer=1)) THEN

			DECLARE v_docLineTable VARCHAR(20);

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

			--**********************************************************************************************************
			--Copy document lines in the Raw memory table
			--**********************************************************************************************************
			--Copy the good receipt lines (purchase delivery note lines) to the memory table.
			IF (v_isPurchaseDeliveryDocument = 1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OPDN."DocEntry", OPDN."ObjType", OPDN."DocNum", OPDN."DocDate",OPDN."CardCode", OPDN."CardName",OPDN."UserSign", OPDN."CreateDate", OPDN."DocTime", PDN1."LineNum", PDN1."BaseType", PDN1."BaseEntry", PDN1."BaseLine", PDN1."ItemCode", PDN1."Dscription", PDN1."Quantity", PDN1."InvQty", IFNULL( PDN1."UomCode", PDN1."unitMsr"), CASE WHEN NULLIF(PDN1."UomEntry2", -1) IS NULL THEN OITM."InvntryUom" ELSE IFNULL( PDN1."UomCode2", PDN1."unitMsr2") END, PDN1."NumPerMsr", PDN1."WhsCode", 
						  PDN1.U_PMX_BATC, PDN1.U_PMX_BAT2, PDN1.U_PMX_BBDT, PDN1.U_PMX_QUAN, PDN1.U_PMX_LOCO, PDN1.U_PMX_LUID, PDN1.U_PMX_SSCC, PDN1.U_PMX_QYSC,
						  PDN1.U_PMX_ISLC, PDN1.U_PMX_RSNC, PDN1.U_PMX_PQY2, PDN1.U_PMX_UOM2, PDN1.U_PMX_QTY2, PDN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  PDN1 INNER JOIN OPDN ON PDN1."DocEntry" = OPDN."DocEntry" 
								INNER JOIN OITM ON OITM."ItemCode" = PDN1."ItemCode"
					WHERE OPDN."DocEntry" = v_keyAsInteger;
					
				v_docLineTable := 'PDN1';
				v_stockDirection := +1;
				
				SELECT COUNT(*) INTO v_cnt FROM OPDN WHERE OPDN."DocEntry" = v_keyAsInteger AND OPDN."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := -1;
				END IF;

			END IF;

			--Copy the delivery lines (sales delivery note lines) to the memory table.
			IF (v_isSalesDeliveryDocument = 1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT ODLN."DocEntry", ODLN."ObjType", ODLN."DocNum", ODLN."DocDate",ODLN."CardCode", ODLN."CardName",ODLN."UserSign", ODLN."CreateDate", ODLN."DocTime", DLN1."LineNum", DLN1."BaseType", DLN1."BaseEntry", DLN1."BaseLine", DLN1."ItemCode", DLN1."Dscription", DLN1."Quantity", DLN1."InvQty", IFNULL( DLN1."UomCode", DLN1."unitMsr"), IFNULL( DLN1."UomCode2", DLN1."unitMsr2"), DLN1."NumPerMsr", DLN1."WhsCode", 
						  DLN1.U_PMX_BATC, DLN1.U_PMX_BAT2, DLN1.U_PMX_BBDT, DLN1.U_PMX_QUAN, DLN1.U_PMX_LOCO, DLN1.U_PMX_LUID, DLN1.U_PMX_SSCC, DLN1.U_PMX_QYSC,
						  DLN1.U_PMX_ISLC, DLN1.U_PMX_RSNC, DLN1.U_PMX_PQY2, DLN1.U_PMX_UOM2, DLN1.U_PMX_QTY2, DLN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  DLN1 INNER JOIN ODLN ON DLN1."DocEntry" = ODLN."DocEntry" 
								INNER JOIN OITM ON OITM."ItemCode" = DLN1."ItemCode"
					WHERE ODLN."DocEntry" = v_keyAsInteger AND DLN1."Quantity" > 0;

				v_stockDirection := -1;
				
				SELECT COUNT(*) INTO v_cnt FROM ODLN WHERE ODLN."DocEntry" = v_keyAsInteger AND ODLN."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := +1;
				END IF;

			END IF;

			--Copy the inventory entry lines to the memory table.
			IF (v_isInventoryEntry = 1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OIGN."DocEntry", OIGN."ObjType", OIGN."DocNum", OIGN."DocDate",OIGN."CardCode", OIGN."CardName",OIGN."UserSign", OIGN."CreateDate", OIGN."DocTime", IGN1."LineNum", IGN1."BaseType", IGN1."BaseEntry", IGN1."BaseLine", IGN1."ItemCode", IGN1."Dscription", IGN1."Quantity", IGN1."InvQty", IFNULL( OUOM."UomCode", OITM."InvntryUom"), IFNULL( OUOM."UomCode", OITM."InvntryUom") AS "unitMsr", 1 AS "NumPerMsr", IGN1."WhsCode", 
						  IGN1.U_PMX_BATC, IGN1.U_PMX_BAT2, IGN1.U_PMX_BBDT, IGN1.U_PMX_QUAN, IGN1.U_PMX_LOCO, IGN1.U_PMX_LUID, IGN1.U_PMX_SSCC, IGN1.U_PMX_QYSC,
						  IGN1.U_PMX_ISLC, IGN1.U_PMX_RSNC, IGN1.U_PMX_PQY2, IGN1.U_PMX_UOM2, IGN1.U_PMX_QTY2, IGN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  IGN1 INNER JOIN OIGN ON IGN1."DocEntry" = OIGN."DocEntry"
						  INNER JOIN OITM ON IGN1."ItemCode" = OITM."ItemCode"
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE OIGN."DocEntry" = v_keyAsInteger;
	
				v_docLineTable := 'IGN1';
				v_stockDirection := +1;
				
				SELECT COUNT(*) INTO v_cnt FROM OIGN WHERE OIGN."DocEntry" = v_keyAsInteger AND OIGN."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := -1;
				END IF;
				
			END IF;

			--Copy the inventory exit lines to the memory table.
			IF (v_isInventoryExit=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OIGE."DocEntry", OIGE."ObjType", OIGE."DocNum", OIGE."DocDate", OIGE."CardCode", OIGE."CardName", OIGE."UserSign", OIGE."CreateDate", OIGE."DocTime", IGE1."LineNum", IGE1."BaseType", IGE1."BaseEntry", IGE1."BaseLine", IGE1."ItemCode", IGE1."Dscription", IGE1."Quantity", IGE1."InvQty", IFNULL( OUOM."UomCode", OITM."InvntryUom"), IFNULL( OUOM."UomCode", OITM."InvntryUom") AS "unitMsr", 1 AS "NumPerMsr", IGE1."WhsCode", 
						  IGE1.U_PMX_BATC, IGE1.U_PMX_BAT2, IGE1.U_PMX_BBDT, IGE1.U_PMX_QUAN, IGE1.U_PMX_LOCO, IGE1.U_PMX_LUID, IGE1.U_PMX_SSCC, IGE1.U_PMX_QYSC,
						  IGE1.U_PMX_ISLC, IGE1.U_PMX_RSNC, IGE1.U_PMX_PQY2, IGE1.U_PMX_UOM2,IGE1.U_PMX_QTY2, IGE1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  IGE1 INNER JOIN OIGE ON IGE1."DocEntry" = OIGE."DocEntry"
						  INNER JOIN OITM ON IGE1."ItemCode" = OITM."ItemCode"
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE OIGE."DocEntry" = v_keyAsInteger;
	
				v_stockDirection := -1;
				
				SELECT COUNT(*) INTO v_cnt FROM OIGE WHERE OIGE."DocEntry" = v_keyAsInteger AND OIGE."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := +1;
				END IF;
				
			END IF;

			--Copy the sales return lines to the memory table
			IF (v_isSalesReturnDocument=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT ORDN."DocEntry", ORDN."ObjType", ORDN."DocNum", ORDN."DocDate", ORDN."CardCode", ORDN."CardName", ORDN."UserSign", ORDN."CreateDate", ORDN."DocTime", RDN1."LineNum", RDN1."BaseType", RDN1."BaseEntry", RDN1."BaseLine", RDN1."ItemCode", RDN1."Dscription", RDN1."Quantity", RDN1."InvQty", IFNULL( RDN1."UomCode", RDN1."unitMsr"), IFNULL( RDN1."UomCode2", RDN1."unitMsr2"), RDN1."NumPerMsr", RDN1."WhsCode", 
						  RDN1.U_PMX_BATC, RDN1.U_PMX_BAT2, RDN1.U_PMX_BBDT, RDN1.U_PMX_QUAN, RDN1.U_PMX_LOCO, RDN1.U_PMX_LUID, RDN1.U_PMX_SSCC, RDN1.U_PMX_QYSC,
						  RDN1.U_PMX_ISLC, RDN1.U_PMX_RSNC, RDN1.U_PMX_PQY2, RDN1.U_PMX_UOM2, RDN1.U_PMX_QTY2, RDN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RDN1 INNER JOIN ORDN ON RDN1."DocEntry" = ORDN."DocEntry"
						INNER JOIN OITM ON RDN1."ItemCode" = OITM."ItemCode"
					WHERE ORDN."DocEntry" = v_keyAsInteger;
		
				v_docLineTable := 'RDN1';
				v_stockDirection := +1;
				
				SELECT COUNT(*) INTO v_cnt FROM ORDN WHERE ORDN."DocEntry" = v_keyAsInteger AND ORDN."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := -1;
				END IF;
				
			END IF;

			--Copy the purchase return lines to the memory table
			IF (v_isPurchaseReturnDocument=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT ORPD."DocEntry", ORPD."ObjType", ORPD."DocNum", ORPD."DocDate", ORPD."CardCode", ORPD."CardName", ORPD."UserSign", ORPD."CreateDate", ORPD."DocTime", RPD1."LineNum", RPD1."BaseType", RPD1."BaseEntry", RPD1."BaseLine", RPD1."ItemCode", RPD1."Dscription", RPD1."Quantity", RPD1."InvQty", IFNULL( RPD1."UomCode", RPD1."unitMsr"), IFNULL( RPD1."UomCode2", RPD1."unitMsr2"), RPD1."NumPerMsr", RPD1."WhsCode", 
						  RPD1.U_PMX_BATC, RPD1.U_PMX_BAT2, RPD1.U_PMX_BBDT, RPD1.U_PMX_QUAN, RPD1.U_PMX_LOCO, RPD1.U_PMX_LUID, RPD1.U_PMX_SSCC, RPD1.U_PMX_QYSC,
						  RPD1.U_PMX_ISLC, RPD1.U_PMX_RSNC, RPD1.U_PMX_PQY2, RPD1.U_PMX_UOM2, RPD1.U_PMX_QTY2, RPD1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RPD1 INNER JOIN ORPD ON RPD1."DocEntry" = ORPD."DocEntry"
						  INNER JOIN OITM ON RPD1."ItemCode" = OITM."ItemCode"
					WHERE ORPD."DocEntry" = v_keyAsInteger;

				v_stockDirection := -1;
				
				SELECT COUNT(*) INTO v_cnt FROM ORPD WHERE ORPD."DocEntry" = v_keyAsInteger AND ORPD."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := +1;
				END IF;
				
			END IF;

			--Copy the purchase invoice lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a good receipt
			IF (v_isPurchaseInvoice=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OPCH."DocEntry", OPCH."ObjType", OPCH."DocNum", OPCH."DocDate", OPCH."CardCode", OPCH."CardName", OPCH."UserSign", OPCH."CreateDate", OPCH."DocTime", PCH1."LineNum", PCH1."BaseType", PCH1."BaseEntry", PCH1."BaseLine", PCH1."ItemCode", PCH1."Dscription", PCH1."Quantity", PCH1."InvQty", IFNULL( PCH1."UomCode", PCH1."unitMsr"), CASE WHEN NULLIF(PCH1."UomEntry2", -1) IS NULL THEN OITM."InvntryUom" ELSE IFNULL( PCH1."UomCode2", PCH1."unitMsr2") END, PCH1."NumPerMsr", PCH1."WhsCode", 
						  PCH1.U_PMX_BATC, PCH1.U_PMX_BAT2, PCH1.U_PMX_BBDT, PCH1.U_PMX_QUAN, PCH1.U_PMX_LOCO, PCH1.U_PMX_LUID, PCH1.U_PMX_SSCC, PCH1.U_PMX_QYSC,
						  PCH1.U_PMX_ISLC, PCH1.U_PMX_RSNC, PCH1.U_PMX_PQY2, PCH1.U_PMX_UOM2, PCH1.U_PMX_QTY2, PCH1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  PCH1 
						INNER JOIN OPCH ON PCH1."DocEntry" = OPCH."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = v_keyAsInteger AND "TransType"=18 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON OPCH."DocEntry" = OINM."CreatedBy" AND PCH1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = OPCH."ObjType"
						  INNER JOIN OITM ON PCH1."ItemCode" = OITM."ItemCode"
					WHERE OPCH."DocEntry" = v_keyAsInteger AND OINM."InQty" <> OINM."OutQty";

				v_stockDirection := +1;
				
				SELECT COUNT(*) INTO v_cnt FROM OPCH WHERE OPCH."DocEntry" = v_keyAsInteger AND OPCH."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := -1;
				END IF;
				
			END IF;

			--Copy the purchase credit note lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a goods return
			IF (v_isPurchaseCreditNote=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT ORPC."DocEntry", ORPC."ObjType", ORPC."DocNum", ORPC."DocDate", ORPC."CardCode", ORPC."CardName", ORPC."UserSign", ORPC."CreateDate", ORPC."DocTime", RPC1."LineNum", RPC1."BaseType", RPC1."BaseEntry", RPC1."BaseLine", RPC1."ItemCode", RPC1."Dscription", RPC1."Quantity", RPC1."InvQty", IFNULL( RPC1."UomCode", RPC1."unitMsr"), IFNULL( RPC1."UomCode2", RPC1."unitMsr2"), RPC1."NumPerMsr", RPC1."WhsCode", 
						  RPC1.U_PMX_BATC, RPC1.U_PMX_BAT2, RPC1.U_PMX_BBDT, RPC1.U_PMX_QUAN, RPC1.U_PMX_LOCO, RPC1.U_PMX_LUID, RPC1.U_PMX_SSCC, RPC1.U_PMX_QYSC,
						  RPC1.U_PMX_ISLC, RPC1.U_PMX_RSNC, RPC1.U_PMX_PQY2, RPC1.U_PMX_UOM2, RPC1.U_PMX_QTY2, RPC1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RPC1 
						INNER JOIN ORPC ON RPC1."DocEntry" = ORPC."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = v_keyAsInteger AND "TransType"=19 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON ORPC."DocEntry" = OINM."CreatedBy" AND RPC1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = ORPC."ObjType"
						  INNER JOIN OITM ON RPC1."ItemCode" = OITM."ItemCode"
					WHERE ORPC."DocEntry" = v_keyAsInteger AND OINM."InQty" <> OINM."OutQty";

				v_stockDirection := -1;
				
				SELECT COUNT(*) INTO v_cnt FROM ORPC WHERE ORPC."DocEntry" = v_keyAsInteger AND ORPC."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := +1;
				END IF;
				
			END IF;

			--Copy the sales invoice lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a delivery
			IF (v_isSalesInvoice=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OINV."DocEntry", OINV."ObjType", OINV."DocNum", OINV."DocDate", OINV."CardCode", OINV."CardName", OINV."UserSign", OINV."CreateDate", OINV."DocTime", INV1."LineNum", INV1."BaseType", INV1."BaseEntry", INV1."BaseLine", INV1."ItemCode", INV1."Dscription", INV1."Quantity", INV1."InvQty", IFNULL( INV1."UomCode", INV1."unitMsr"), IFNULL( INV1."UomCode2", INV1."unitMsr2"), INV1."NumPerMsr", INV1."WhsCode", 
						  INV1.U_PMX_BATC, INV1.U_PMX_BAT2, INV1.U_PMX_BBDT, INV1.U_PMX_QUAN, INV1.U_PMX_LOCO, INV1.U_PMX_LUID, INV1.U_PMX_SSCC, INV1.U_PMX_QYSC,
						  INV1.U_PMX_ISLC, INV1.U_PMX_RSNC, INV1.U_PMX_PQY2, INV1.U_PMX_UOM2, INV1.U_PMX_QTY2, INV1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  INV1 
						INNER JOIN OINV ON INV1."DocEntry" = OINV."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = v_keyAsInteger AND "TransType"=13 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON OINV."DocEntry" = OINM."CreatedBy" AND INV1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = OINV."ObjType"
						  INNER JOIN OITM ON INV1."ItemCode" = OITM."ItemCode"
					WHERE OINV."DocEntry" = v_keyAsInteger AND OINM."InQty" <> OINM."OutQty";

				v_stockDirection := -1;
				
				SELECT COUNT(*) INTO v_cnt FROM OINV WHERE OINV."DocEntry" = v_keyAsInteger AND OINV."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := +1;
				END IF;
				
			END IF;

			--Copy the sales credit note lines that have new inventory lines to the memory table
			--We do an inner join with OINM to get only the inventory items and a where where "InQty" is different
			--from out quantity to know if this inventory line is not already added using a goods return
			IF (v_isSalesCreditNote=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT ORIN."DocEntry", ORIN."ObjType", ORIN."DocNum", ORIN."DocDate", ORIN."CardCode", ORIN."CardName", ORIN."UserSign", ORIN."CreateDate", ORIN."DocTime", RIN1."LineNum", RIN1."BaseType", RIN1."BaseEntry", RIN1."BaseLine", RIN1."ItemCode", RIN1."Dscription", RIN1."Quantity", RIN1."InvQty", IFNULL( RIN1."UomCode", RIN1."unitMsr"), IFNULL( RIN1."UomCode2", RIN1."unitMsr2"), RIN1."NumPerMsr", RIN1."WhsCode", 
						  RIN1.U_PMX_BATC, RIN1.U_PMX_BAT2, RIN1.U_PMX_BBDT, RIN1.U_PMX_QUAN, RIN1.U_PMX_LOCO, RIN1.U_PMX_LUID, RIN1.U_PMX_SSCC, RIN1.U_PMX_QYSC,
						  RIN1.U_PMX_ISLC, RIN1.U_PMX_RSNC, RIN1.U_PMX_PQY2, RIN1.U_PMX_UOM2, RIN1.U_PMX_QTY2, RIN1."DropShip", NULL, NULL, NULL, NULL,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, NULL
					FROM  RIN1 
						INNER JOIN ORIN ON RIN1."DocEntry" = ORIN."DocEntry"
						INNER JOIN 
							(SELECT "TransType","CreatedBy","DocLineNum", SUM("InQty") AS "InQty", SUM("OutQty") AS "OutQty" FROM OINM WHERE "CreatedBy" = v_keyAsInteger AND "TransType"=14 GROUP BY "TransType","CreatedBy","DocLineNum")
							AS OINM ON ORIN."DocEntry" = OINM."CreatedBy" AND RIN1."LineNum" = OINM."DocLineNum" AND OINM."TransType" = ORIN."ObjType"
						  INNER JOIN OITM ON RIN1."ItemCode" = OITM."ItemCode"
					WHERE ORIN."DocEntry" = v_keyAsInteger AND OINM."InQty" <> OINM."OutQty";

				v_stockDirection := +1;
				
				SELECT COUNT(*) INTO v_cnt FROM ORIN WHERE ORIN."DocEntry" = v_keyAsInteger AND ORIN."CANCELED" = 'C';
				IF v_cnt>0 THEN
					--Cancellation of document
					v_stockDirection := -1;
				END IF;
				
			END IF;

			--Copy the stock transfer lines 
			IF (v_isStockTransfer=1 AND v_isTransactionTypeCancel=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OWTR."DocEntry", OWTR."ObjType", OWTR."DocNum", OWTR."DocDate", OWTR."CardCode", OWTR."CardName", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", WTR1."LineNum", IFNULL( WTR1."BaseType", WTR1.U_PMX_BATY), IFNULL( WTR1."BaseEntry", WTR1.U_PMX_BAEN), IFNULL( WTR1."BaseLine", WTR1.U_PMX_BALI), WTR1."ItemCode", WTR1."Dscription", WTR1."Quantity", WTR1."InvQty", IFNULL( WTR1."UomCode", WTR1."unitMsr"), IFNULL( WTR1."UomCode2", WTR1."unitMsr2"), WTR1."NumPerMsr", WTR1."WhsCode", 
						  WTR1.U_PMX_BATC, WTR1.U_PMX_BAT2, WTR1.U_PMX_BBDT, WTR1.U_PMX_QUAN, WTR1.U_PMX_LOCO, WTR1.U_PMX_LUID, NULL, WTR1.U_PMX_QYSC,
						  WTR1.U_PMX_ISLC, WTR1.U_PMX_RSNC, WTR1.U_PMX_PQY2, WTR1.U_PMX_UOM2, WTR1.U_PMX_QTY2, WTR1."DropShip", WTR1.U_PMX_SLOC, WTR1."FromWhsCod", WTR1.U_PMX_SQSC, WTR1.U_PMX_SLUI,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, REPLACE(CAST(WTR1.U_PMX_MLUI AS NVARCHAR),'0123456789','')
					FROM  WTR1 
						INNER JOIN OWTR ON WTR1."DocEntry" = OWTR."DocEntry"
						  INNER JOIN OITM ON WTR1."ItemCode" = OITM."ItemCode"
					WHERE OWTR."DocEntry" = v_keyAsInteger;

				v_stockDirection := +1;
			
			END IF;

			--Copy the stock transfer lines 
			IF (v_isStockTransfer=1 AND v_isTransactionTypeAdd=1) THEN
			
				INSERT INTO TMP_TN_DocumentLineRaw
					SELECT OWTR."DocEntry", OWTR."ObjType", OWTR."DocNum", OWTR."DocDate", OWTR."CardCode", OWTR."CardName", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", WTR1."LineNum", IFNULL( WTR1."BaseType", WTR1.U_PMX_BATY), IFNULL( WTR1."BaseEntry", WTR1.U_PMX_BAEN), IFNULL( WTR1."BaseLine", WTR1.U_PMX_BALI), WTR1."ItemCode", WTR1."Dscription", WTR1."Quantity", WTR1."InvQty", IFNULL( WTR1."UomCode", WTR1."unitMsr"), IFNULL( WTR1."UomCode2", WTR1."unitMsr2"), WTR1."NumPerMsr", WTR1."FromWhsCod", 
						  WTR1.U_PMX_BATC, WTR1.U_PMX_BAT2, WTR1.U_PMX_BBDT, WTR1.U_PMX_QUAN, WTR1.U_PMX_SLOC, WTR1.U_PMX_SLUI, NULL, WTR1.U_PMX_SQSC,
						  WTR1.U_PMX_ISLC, WTR1.U_PMX_RSNC, WTR1.U_PMX_PQY2, WTR1.U_PMX_UOM2, WTR1.U_PMX_QTY2, WTR1."DropShip", WTR1.U_PMX_LOCO, WTR1."WhsCode", WTR1.U_PMX_QYSC, WTR1.U_PMX_LUID,
							OITM.U_PMX_TLSN, OITM."ManSerNum", OITM.U_PMX_ICAW, WTR1.U_PMX_MLUI
					FROM  WTR1 
						INNER JOIN OWTR ON WTR1."DocEntry" = OWTR."DocEntry"
						  INNER JOIN OITM ON WTR1."ItemCode" = OITM."ItemCode"
					WHERE OWTR."DocEntry" = v_keyAsInteger;

				v_stockDirection := -1;
				
			END IF;


			--We have to remove lines that are drop ships because this inventory is not added to SAP
			-- OR warehouses not managed by Produmex
			DELETE FROM TMP_TN_DocumentLineRaw
			WHERE "DropShip" = 'Y'
			OR ( "WhsCode" IN ( SELECT "WhsCode" FROM OWHS WHERE U_PMX_IMBP = 'N' )
			AND ( "DestinationWhsCode" IS NULL OR "DestinationWhsCode" IN ( SELECT "WhsCode" FROM OWHS WHERE U_PMX_IMBP = 'N' ) ) );

			--Only inventory items must be added to TMP_TN_DocumentLineRaw
			--This means we have to remove not inventory items
			DELETE FROM TMP_TN_DocumentLineRaw WHERE "ItemCode" IN (SELECT "ItemCode" FROM OITM WHERE "InvntItem" <> 'Y');
			
			--**********************************************************************************************************
			--Convert Raw document lines to normal document lines
			--Remove the pipes
			--**********************************************************************************************************
			BEGIN --TRY
/*
				DECLARE EXIT HANDLER FOR SQLEXCEPTION
				BEGIN
					p_error := 50016;
					p_error_message := 'Error raw lines->normal lines: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
						|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' );
				END;
	*/
				DECLARE v_mustAutoSelectBatches SMALLINT;

				DECLARE CURSOR documentLineRawCursor FOR
					SELECT "DocEntry", "ObjType", "DocNum", "DocDate", "CardCode", "CardName", "CreatedBy", "CreateDate", "CreateTime", "LineNum", "BaseType",
					"BaseEntry", "BaseLine", "ItemCode", "Dscription", "Quantity", "InventoryQuantity", "Uom", "InventoryUom", "QuantityPerUom",
					"WhsCode", CAST("U_PMX_BATC" AS NVARCHAR(256000)), CAST("U_PMX_BAT2" AS NVARCHAR(256000)), CAST("U_PMX_BBDT" AS NVARCHAR(256000)), CAST("U_PMX_QUTY" AS NVARCHAR(256000)),
					CAST("U_PMX_LOCO" AS NVARCHAR(256000)), CAST("U_PMX_LUID" AS NVARCHAR(256000)), CAST("U_PMX_SSCC" AS NVARCHAR(256000)), CAST("U_PMX_QYSC" AS NVARCHAR(256000)),
					"U_PMX_ISLC", "U_PMX_RSNC", CAST("U_PMX_PQY2" AS NVARCHAR(256000)), "Uom2", "QuantityUom2", "DropShip",
					CAST("DestinationStorLocCode" AS NVARCHAR(256000)), "DestinationWhsCode", "DestinationQualityStatusCode", CAST("DestinationLUID" AS NVARCHAR(256000)),
					"TrackLocationForSerial", "IsSAPSerial", "IsCatchWeight", CAST("MasterLUID" AS NVARCHAR(256000))
					FROM TMP_TN_DocumentLineRaw;
				OPEN documentLineRawCursor;
				FETCH documentLineRawCursor INTO v_docEntry, v_objType, v_docNum, v_docDate, v_cardCode, v_cardName, v_userSign, v_createDate, v_createTime, v_lineNum, v_baseType,
					v_baseEntry, v_BaseLine, v_itemCode, v_description, v_quantity, v_inventoryQuantity, v_uom, v_inventoryUom, v_quantityPerUom,
					v_whsCode, v_batchNumbers1, v_batchNumbers2, v_bestBeforeDates, v_quantities,
					v_storLocs, v_luids, v_ssccs, v_qualityStatuses,
					v_isLogisticCarrier, v_reasonCode, v_quantitiesUom2, v_uom2, v_quantityUom2, v_dropShip, 
					v_destinationStorLocCodes, v_destinationWhsCode, v_destinationQualityStatusCodes, v_destinationLuids,
					v_trackLocationForSerial, v_isSAPSerial, v_isCatchWeightForDocument, v_masterLuids;
				WHILE v_docEntry IS NOT NULL DO
					v_mustAutoSelectBatches := 0;

					IF v_destinationStorLocCodes       IS NULL THEN v_destinationStorLocCodes       := ''; END IF;
					IF v_destinationLuids              IS NULL THEN v_destinationLuids              := ''; END IF;
					IF v_destinationQualityStatusCodes IS NULL THEN v_destinationQualityStatusCodes := ''; END IF;
					IF v_masterLuids                   IS NULL THEN v_masterLuids                   := ''; END IF;
				
					--Delete all temp tables
					DELETE FROM TMP_TN_BatchNumbers1;
					DELETE FROM TMP_TN_BatchNumbers2;
					DELETE FROM TMP_TN_BestBeforeDates;
					DELETE FROM TMP_TN_Quantities;
					DELETE FROM TMP_TN_QuantitiesUom2;
					DELETE FROM TMP_TN_StorLocs;
					DELETE FROM TMP_TN_Luids;
					DELETE FROM TMP_TN_QualityStatuses;
					DELETE FROM TMP_TN_SSCCs;

					DELETE FROM TMP_TN_DestStorLocs;
					DELETE FROM TMP_TN_DestLuids;
					DELETE FROM TMP_TN_DestQualityStatuses;
					DELETE FROM TMP_TN_MasterLuids;

					--Parse all the concatenated columns
					--The function that is used is stored in the extra database
					IF v_batchNumbers1 IS NULL THEN
						v_batchNumbers1 := '';
						v_mustAutoSelectBatches := 1;
					END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_batchNumbers1                 || ''', ''|'',1,0, TMP_TN_BatchNumbers1       ) WITH OVERVIEW';
					IF v_batchNumbers2 IS NULL THEN v_batchNumbers2 := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_batchNumbers2                 || ''', ''|'',1,0, TMP_TN_BatchNumbers2       ) WITH OVERVIEW';
					IF v_bestBeforeDates IS NULL THEN
						v_bestBeforeDates := '';
						v_mustAutoSelectBatches := 1;
					END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_bestBeforeDates               || ''', ''|'',1,0, TMP_TN_BestBeforeDates     ) WITH OVERVIEW';
					IF v_quantities IS NULL THEN
						v_quantities := '';
						v_mustAutoSelectBatches := 1;
					END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_quantities                    || ''', ''|'',1,0, TMP_TN_SSCCs               ) WITH OVERVIEW';

					INSERT INTO TMP_TN_Quantities
						SELECT "listPos", CASE LOCATE("nstr", 'e') + LOCATE("nstr", 'E') WHEN 0 THEN CAST("nstr" AS numeric(19,6)) ELSE CAST(CAST("nstr" as real) AS numeric(19,6)) END FROM TMP_TN_SSCCs;
					DELETE FROM TMP_TN_SSCCs;

					IF v_quantitiesUom2 IS NULL THEN v_quantitiesUom2 := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_quantitiesUom2                || ''', ''|'',1,0, TMP_TN_SSCCs               ) WITH OVERVIEW';

					INSERT INTO TMP_TN_QuantitiesUom2
						SELECT "listPos", CAST("nstr" AS numeric(19,6)) FROM TMP_TN_SSCCs;
					DELETE FROM TMP_TN_SSCCs;

					IF v_storLocs IS NULL THEN v_storLocs := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_storLocs                      || ''', ''|'',1,0, TMP_TN_StorLocs            ) WITH OVERVIEW';
					IF v_luids IS NULL THEN v_luids := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_luids                         || ''', ''|'',1,0, TMP_TN_Luids               ) WITH OVERVIEW';
					IF v_qualityStatuses IS NULL THEN v_qualityStatuses := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_qualityStatuses               || ''', ''|'',1,0, TMP_TN_QualityStatuses     ) WITH OVERVIEW';

					IF v_destinationStorLocCodes IS NULL THEN v_destinationStorLocCodes := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_destinationStorLocCodes       || ''', ''|'',1,0, TMP_TN_DestStorLocs        ) WITH OVERVIEW';
					IF v_destinationLuids IS NULL THEN v_destinationLuids := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_destinationLuids              || ''', ''|'',1,0, TMP_TN_DestLuids           ) WITH OVERVIEW';
					IF v_destinationQualityStatusCodes IS NULL THEN v_destinationQualityStatusCodes := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_destinationQualityStatusCodes || ''', ''|'',1,0, TMP_TN_DestQualityStatuses ) WITH OVERVIEW';

					IF v_ssccs IS NULL THEN v_ssccs := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_ssccs                         || ''', ''|'',1,0, TMP_TN_SSCCs               ) WITH OVERVIEW';
					IF v_masterLuids IS NULL THEN v_masterLuids := ''; END IF;
					EXEC 'CALL ' || v_extraDb || '."PMX_SP_IterateCharListToTable"(''' || v_masterLuids                   || ''', ''|'',1,0, TMP_TN_MasterLuids           ) WITH OVERVIEW';

					--Sometimes SBO set "BaseType" to -1 if there is no base document
					--We don't want this we want that the baseType is NULL then
					IF v_baseType = '-1' THEN
						v_baseType := NULL;
					END IF;

               -- Auto-fill PMX BatchNumber1, "BestBeforeDate" and "Quantity"
               IF v_mustAutoSelectBatches = 1 THEN
					DECLARE v_tmpITLRows INT;

					DELETE FROM TMP_TN_ITL;

					INSERT INTO TMP_TN_ITL
					SELECT ROW_NUMBER() OVER ( ORDER BY OBTN."DistNumber" ), OBTN."DistNumber", OBTN."ExpDate", SUM( ITL1."Quantity" ) * v_stockDirection * CASE WHEN v_isStockTransfer = 1 THEN -1 ELSE +1 END
					FROM OITL 
					INNER JOIN (   ITL1
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"
					) ON ITL1."LogEntry" = OITL."LogEntry"
					WHERE OITL."DocType" = v_objType AND OITL."DocEntry" = v_docEntry AND OITL."DocLine" = v_lineNum AND OITL."StockEff" = 1  -- Affects "OnHand"
					AND ( OITL."LocCode" = v_destinationWhsCode OR v_destinationWhsCode IS NULL )
					GROUP BY OBTN."DistNumber", OBTN."ExpDate"
					HAVING SUM( ITL1."Quantity" ) <> 0;
 
					SELECT COUNT(1) INTO v_tmpITLRows FROM TMP_TN_ITL;

					IF v_tmpITLRows = 0 THEN  -- No batch numbers
						IF LENGTH(v_quantities) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_Quantities;

							INSERT INTO TMP_TN_Quantities ( "listPos", "nstr" )
							--VALUES ( 1, v_quantity * v_quantityPerUom )
							VALUES ( 1, v_inventoryQuantity);
						END IF;
					ELSEIF v_tmpITLRows = 1 OR ( v_tmpITLRows > 1 AND LENGTH(v_batchNumbers1) = 0 AND LENGTH(v_bestBeforeDates) = 0 AND LENGTH(v_quantities) = 0 ) THEN
						IF LENGTH(v_batchNumbers1) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_BatchNumbers1;

							INSERT INTO TMP_TN_BatchNumbers1 ( "listPos", "nstr" )
							SELECT "RowNum", "BatchNum1" FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;

						IF LENGTH(v_bestBeforeDates) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_BestBeforeDates;

							INSERT INTO TMP_TN_BestBeforeDates ( "listPos", "nstr" )
							SELECT "RowNum", "BestBeforeDate" FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;

						IF LENGTH(v_quantities) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_Quantities;

							INSERT INTO TMP_TN_Quantities ( "listPos", "nstr" )
							SELECT "RowNum", "Quantity" FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;
					END IF;

					IF v_tmpITLRows > 1 THEN
						IF LENGTH(v_batchNumbers2) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_BatchNumbers2;

							INSERT INTO TMP_TN_BatchNumbers2 ( "listPos", "nstr" )
							SELECT "RowNum", NULL FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;

						IF LENGTH(v_storLocs) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_StorLocs;

							INSERT INTO TMP_TN_StorLocs ( "listPos", "nstr" )
							SELECT "RowNum", NULL FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;

						IF LENGTH(v_luids) = 0 AND LENGTH(v_ssccs) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_LUIDs;

							INSERT INTO TMP_TN_LUIDs ( "listPos", "nstr" )
							SELECT "RowNum", NULL FROM TMP_TN_ITL ORDER BY "RowNum";

							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_SSCCs;

							INSERT INTO TMP_TN_SSCCs ( "listPos", "nstr" )
							SELECT "RowNum", NULL FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;

						IF LENGTH(v_qualityStatuses) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_QualityStatuses;

							INSERT INTO TMP_TN_QualityStatuses ( "listPos", "nstr" )
							SELECT "RowNum", NULL FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;

						IF LENGTH(v_quantitiesUom2) = 0 THEN
							--Delete the NULL row inserted before
							DELETE FROM TMP_TN_QuantitiesUom2;

							INSERT INTO TMP_TN_QuantitiesUom2 ( "listPos", "nstr" )
							SELECT "RowNum", NULL FROM TMP_TN_ITL ORDER BY "RowNum";
						END IF;
					END IF;
				END IF;

					IF v_isStockTransfer = 0 THEN
					
						IF v_docLineTable IS NOT NULL THEN

						    DECLARE v_pos  INT;
						    DECLARE v_sscc NVARCHAR(4000);
						    DECLARE v_len  INT;
						    DECLARE v_luid INT;

   						    DECLARE v_query NVARCHAR(4000);
						    DECLARE v_pos_1 INT;
						    DECLARE v_pos_2 INT;
						    SELECT pos INTO v_pos_1 FROM "WUK_4004_PMX"."PMX_FN_PositionOfFirstDigit"( :v_luids );
						    SELECT pos INTO v_pos_2 FROM "WUK_4004_PMX"."PMX_FN_PositionOfFirstDigit"( :v_ssccs );
    					
						    IF ( LENGTH(v_luids) = 0 OR v_pos_1 = 0 ) AND ( v_pos_2 > 0 ) THEN

							    -- Fill in LUID's from SSCC's
							    DECLARE CURSOR curSSCC FOR
							    SELECT T0."listPos", T0."nstr", LENGTH(T0."nstr"), PMX_INVT."LogUnitIdentKey"
							    FROM TMP_TN_SSCCs T0
							    LEFT JOIN PMX_INVT ON PMX_INVT.SSCC = T0."nstr"
							    GROUP BY T0."listPos", T0."nstr", LENGTH(T0."nstr"), PMX_INVT."LogUnitIdentKey";

							    DELETE FROM TMP_TN_Luids;
    							
							    OPEN curSSCC;
							    FETCH curSSCC INTO v_pos, v_sscc, v_len, v_luid;
							    WHILE v_pos IS NOT NULL
                                DO

								    IF v_luid IS NULL THEN
									    IF v_len <> 18 THEN
										    p_error := 50031;
										    p_error_message := 'Error on line ' || v_lineNum || ' : SSCC must have 18 digits.';

										    SELECT p_error, p_error_message FROM dummy;
										    --EXEC xp_logevent p_error, p_error_message, ERROR
										    CLOSE curSSCC;
										    CLOSE documentLineRawCursor;
										    RETURN;
									    END IF;

									    BEGIN
										    DECLARE v_multiplier  INT;
										    DECLARE v_checkSum  INT;
										    v_multiplier := 1;
										    v_checkSum := 0;
										    WHILE v_len > 0 DO
											    v_checkSum := v_checkSum + v_multiplier * CAST( SUBSTRING( v_sscc, v_len, 1 ) AS INT );
											    v_multiplier := 4 - v_multiplier;
											    v_len := v_len - 1;
										    END WHILE;
										    IF MOD( v_checkSum, 10 ) <> 0 THEN
											    p_error := 50032;
											    p_error_message := 'Error on line ' || v_lineNum || ' SSCC ' || v_sscc || ' : check digit invalid.';
    	
											    SELECT p_error, p_error_message FROM dummy;
											    --EXEC xp_logevent p_error, p_error_message, ERROR
											    CLOSE curSSCC;
											    CLOSE documentLineRawCursor;
											    RETURN;
										    END IF;
									    END;
    									
									    INSERT INTO PMX_LUID ( "InternalKey", "CreateDate", "CreateTime", "Version", SSCC, "IsGenerated", "IsFullPallet" ) 
									    VALUES ( PMX_LUID_S.nextval, v_updateDate, v_updateTime, 1, v_sscc, 'N', 'Y' );
									    SELECT PMX_LUID_S.currval INTO v_luid FROM dummy;
								    END IF;
    								
								    INSERT INTO TMP_TN_Luids ( "listPos", "nstr" ) VALUES ( v_pos, v_luid );
    								
								    FETCH curSSCC INTO v_pos, v_sscc, v_len, v_luid;
							    END WHILE;
							    CLOSE curSSCC;
    							
							    DELETE FROM "WUK_4004".TMP_TN_CharListTable;
							    INSERT INTO "WUK_4004".TMP_TN_CharListTable
							    SELECT * FROM TMP_TN_Luids;

							    SELECT str INTO v_luids FROM "WUK_4004_PMX"."PMX_FN_TableToCharList"( "WUK_4004".TMP_TN_CharListTable, '|' );
							
							    v_query := 'UPDATE ' || v_docLineTable || ' SET U_PMX_LUID = ' || '''' || v_luids || ''''
							    	    || ' WHERE ' || v_docLineTable || '."DocEntry" = ' || CAST( v_docEntry AS NVARCHAR )
								          || ' AND ' || v_docLineTable || '."LineNum"  = ' || CAST( v_lineNum  AS NVARCHAR );
							    EXEC v_query;

						    ELSEIF ( LENGTH(v_ssccs) = 0 OR v_pos_2 = 0 ) AND ( v_pos_1 > 0 ) THEN
							    -- Fill in SSCC's from LUID's
							    DECLARE v_currentSSCC BIGINT;
							    DECLARE v_newLuidCreated INT;

							    DECLARE CURSOR curSSCC FOR
								    SELECT "listPos", "nstr" FROM TMP_TN_SSCCs;

							    DELETE FROM TMP_TN_SSCCs;
							    INSERT INTO TMP_TN_SSCCs ( "listPos", "nstr" )
							    SELECT TMP_TN_LUIDs."listPos", PMX_LUID.SSCC
							    FROM TMP_TN_LUIDs
							    LEFT JOIN PMX_LUID ON PMX_LUID."InternalKey" = TMP_TN_LUIDs."nstr";
							    
								
								-- Generate new SSCC if luid = -1
							    v_newLuidCreated := 0;

							    OPEN curSSCC;
							    FETCH curSSCC INTO v_pos, v_sscc;
							    WHILE v_pos IS NOT NULL
							    DO
									-- if no SSCC was found for the current LUID, check if the LUID was -1
									IF v_sscc IS NULL THEN
										SELECT "nstr" INTO v_luid FROM TMP_TN_LUIDs WHERE "listPos" = v_pos;
									END IF;
									
									-- if the LUID was -1, generate a new SSCC
									IF v_sscc IS NULL AND v_luid = -1 THEN
										-- generate new SSCC
										SELECT TOP 1 CAST("CurrentSSCC" AS BIGINT) + 1 INTO v_currentSSCC FROM PMX_SSCC;
										UPDATE PMX_SSCC SET "CurrentSSCC" = CAST(CAST("CurrentSSCC" AS BIGINT) + 1 AS NVARCHAR(20)), "Version" = "Version" + 1;

										-- format new SSCC with checksum into variable
										SELECT sscc INTO v_sscc FROM "WUK_4004_PMX"."PMX_FN_GenerateSSCC"( :v_currentSSCC );

										-- insert new LUID
									    INSERT INTO PMX_LUID ( "InternalKey", "CreateDate", "CreateTime", "Version", SSCC, "IsGenerated", "IsFullPallet" ) 
									    VALUES ( PMX_LUID_S.nextval, v_updateDate, v_updateTime, 1, v_sscc, 'N', 'Y' );
									    SELECT PMX_LUID_S.currval INTO v_luid FROM dummy;

										-- update temp tables with newly generated LUID and SSCC
										UPDATE TMP_TN_SSCCs SET "nstr" = v_sscc WHERE "listPos" = v_pos;
										UPDATE TMP_TN_LUIDs SET "nstr" = v_luid WHERE "listPos" = v_pos;

										v_newLuidCreated := 1;
									END IF;
									FETCH curSSCC INTO v_pos, v_sscc;
							    END WHILE;
							    CLOSE curSSCC;
							    

							    DELETE FROM "WUK_4004".TMP_TN_CharListTable;
							    INSERT INTO "WUK_4004".TMP_TN_CharListTable
							    SELECT * FROM TMP_TN_SSCCs;

							    SELECT str INTO v_ssccs FROM "WUK_4004_PMX"."PMX_FN_TableToCharList"( "WUK_4004".TMP_TN_CharListTable, '|' );

							    
								-- if a new LUID was generated, rebuild the list of LUID to replace -1 with new values
							    IF v_newLuidCreated=1 THEN
								    DELETE FROM "WUK_4004".TMP_TN_CharListTable;
									INSERT INTO "WUK_4004".TMP_TN_CharListTable
									SELECT * FROM TMP_TN_Luids;

									SELECT str INTO v_luids FROM "WUK_4004_PMX"."PMX_FN_TableToCharList"( "WUK_4004".TMP_TN_CharListTable, '|' );
							    END IF;
							
							    v_query := 'UPDATE ' || v_docLineTable || ' SET U_PMX_SSCC = ' || '''' || v_ssccs || '''';
							    IF v_newLuidCreated=1 THEN
									v_query := v_query || ' , U_PMX_LUID = ' || '''' || v_luids || '''';
							    END IF;
								v_query := v_query
											|| ' WHERE ' || v_docLineTable || '."DocEntry" = ' || CAST( v_docEntry AS NVARCHAR )
											|| ' AND ' || v_docLineTable || '."LineNum"  = ' || CAST( v_lineNum  AS NVARCHAR );
							    EXEC v_query;

					        END IF;
					        
					    END IF;
					    
					END IF;

					--All the tables must have the same number of records
					BEGIN
						DECLARE v_numOfBN1  INT;
						DECLARE v_numOfBN2  INT;
						DECLARE v_numOfBBD  INT;
						DECLARE v_numOfQTY  INT;
						DECLARE v_numOfSL   INT;
						DECLARE v_numOfLUID INT;
						DECLARE v_numOfSSCC INT;
						DECLARE v_numOfQS   INT;
						DECLARE v_numOfQU2  INT;
						DECLARE v_numOfDstSL   INT;
						DECLARE v_numOfDstLUID INT;
						DECLARE v_numOfDstQS   INT;
						DECLARE v_numOfMstrLUID INT;
						
						SELECT COUNT("listPos") INTO v_numOfBN1  FROM TMP_TN_BatchNumbers1;
						SELECT COUNT("listPos") INTO v_numOfBN2  FROM TMP_TN_BatchNumbers2;
						SELECT COUNT("listPos") INTO v_numOfBBD  FROM TMP_TN_BestBeforeDates;
						SELECT COUNT("listPos") INTO v_numOfQTY  FROM TMP_TN_Quantities;
						SELECT COUNT("listPos") INTO v_numOfSL   FROM TMP_TN_StorLocs;
						SELECT COUNT("listPos") INTO v_numOfLUID FROM TMP_TN_LUIDs;
						SELECT COUNT("listPos") INTO v_numOfSSCC FROM TMP_TN_SSCCs;
						SELECT COUNT("listPos") INTO v_numOfQS   FROM TMP_TN_QualityStatuses;
						SELECT COUNT("listPos") INTO v_numOfQU2  FROM TMP_TN_QuantitiesUom2;
						IF NOT(
							v_numOfBN2 = v_numOfBN1 AND
							v_numOfBBD = v_numOfBN1 AND
							v_numOfQTY = v_numOfBN1 AND
							v_numOfSL = v_numOfBN1 AND
							v_numOfLUID = v_numOfBN1 AND
							(v_isStockTransfer = 1 OR v_numOfSSCC = v_numOfBN1) AND
							v_numOfQS = v_numOfBN1 AND
							v_numOfQU2 = v_numOfBN1
						) THEN
							p_error := 50014;

							SELECT 'Line ' || CAST(v_lineNum AS NCLOB) || ': Values for BatchNumbers1 = '|| CAST((SELECT COUNT("listPos") FROM TMP_TN_BatchNumbers1) AS NCLOB) ||
							',BatchNumbers2 = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_BatchNumbers2) AS NCLOB)||
							',BestBeforeDates = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_BestBeforeDates) AS NCLOB) ||
							',Quantities = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_Quantities) AS NCLOB) ||
							',StorLocs = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_StorLocs) AS NCLOB) ||
							',LUIDs = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_LUIDs) AS NCLOB) ||
							',QualityStatus = ' ||CAST((SELECT COUNT("listPos") FROM TMP_TN_QualityStatuses) AS NCLOB)|| 
							',Quantities "Uom2" = ' ||CAST((SELECT COUNT("listPos") FROM TMP_TN_QuantitiesUom2) AS NCLOB)|| 
							',SSCCs = ' ||CAST((SELECT COUNT("listPos") FROM TMP_TN_SSCCs) AS NCLOB)
							INTO p_error_message FROM dummy;
	
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							CLOSE documentLineRawCursor ;
							RETURN;
						END IF;
					
						SELECT COUNT("listPos") INTO v_numOfDstSL   FROM TMP_TN_DestStorLocs;
						SELECT COUNT("listPos") INTO v_numOfDstLUID FROM TMP_TN_DestLuids;
						SELECT COUNT("listPos") INTO v_numOfDstQS   FROM TMP_TN_DestQualityStatuses;
						SELECT COUNT("listPos") INTO v_numOfMstrLUID FROM TMP_TN_MasterLuids;
						IF v_isStockTransfer = 1 AND
							NOT(
							v_numOfDstSL = v_numOfBN1 AND
							v_numOfDstLUID = v_numOfBN1 AND
							v_numOfDstQS = v_numOfBN1 AND
							v_numOfMstrLUID = v_numOfBN1
						) THEN
							p_error := 50015;
							--p_error_message := 'Error on line ' || CAST(v_lineNum AS NCLOB) || ': One or more columns (BatchNumbers1,BatchNumbers2,BestBeforeDates,Quantities,StorLocs,Luids or QualityStatus) have more or less pipes(|) than the other columns.'
							
							SELECT 'Line ' || CAST(v_lineNum AS NCLOB) || ': Values for DestStorLocs = '|| CAST((SELECT COUNT("listPos") FROM TMP_TN_DestStorLocs) AS NCLOB) ||
							',DestLUIDs = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_DestLuids) AS NCLOB)||
							',DestQualityStatus = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_DestQualityStatuses) AS NCLOB) ||
							',Quantities = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_Quantities) AS NCLOB) ||
							',MasterLUIDs = '||CAST((SELECT COUNT("listPos") FROM TMP_TN_MasterLuids) AS NCLOB)
							INTO p_error_message FROM dummy;

							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							CLOSE documentLineRawCursor;
							RETURN;
						END IF;

					END;

					--Check if luid corresponds to SSCC
					IF v_isStockTransfer = 0 THEN
						DELETE FROM TMP_TN_CharListTableNoIndex;
				        INSERT INTO TMP_TN_CharListTableNoIndex
				        SELECT CAST(LUIDs."nstr" AS NVARCHAR(255)) || ' <> ' || SSCCs."nstr"
						FROM TMP_TN_LUIDs AS LUIDs
						INNER JOIN TMP_TN_SSCCs AS SSCCs ON LUIDs."listPos" = SSCCs."listPos"
						WHERE LUIDs."nstr" NOT IN (SELECT "InternalKey" FROM PMX_LUID WHERE "InternalKey" = LUIDs."nstr" AND SSCC = SSCCs."nstr");
        				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
						IF LENGTH(v_errorList) <> 0 THEN
							p_error := 50033;
							p_error_message := 'Error, LUID''s do not match with SSCC''s: ' || v_errorList;
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							CLOSE documentLineRawCursor;
							RETURN;
						END IF;
					END IF;

					BEGIN
						DECLARE v_uomDecimals INT;
						DECLARE v_SUM_qty DECIMAL(19,6);
						SELECT IFNULL( U_PMX_UOMD,6) INTO v_uomDecimals FROM OITM WHERE "ItemCode" = v_itemCode;

						--The sum of the quantities must be the same as the quantity on the line
						--IF (SELECT SUM("nstr") FROM TMP_TN_Quantities) <> ROUND ( ROUND((v_quantity * v_quantityPerUom),6), v_uomDecimals) THEN
						SELECT SUM("nstr") INTO v_SUM_qty FROM TMP_TN_Quantities;
						IF v_SUM_qty <> ROUND ( ROUND(v_inventoryQuantity,6), v_uomDecimals) THEN
							p_error := 50034;
							SELECT 'Error on line: ' || CAST(v_lineNum AS NCLOB) || ' The total of the quantities in the column  (Pmx Quantities) ' || CAST(v_SUM_qty AS NCLOB)
													 || ' is not the same as the SAP quantity ' || CAST(ROUND(v_inventoryQuantity, v_uomDecimals) AS NCLOB) || ' fields on the line.'
							INTO p_error_message FROM dummy;
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							CLOSE documentLineRawCursor;
							RETURN;
						END IF;

					END;

					--All batchnumbers1 must be entered in upper case
					--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
					v_batchNumbers1 := '';
					v_lineNums := '';
					DELETE FROM TMP_TN_CharListTableNoIndex;
			        INSERT INTO TMP_TN_CharListTableNoIndex
			        SELECT CAST(v_lineNum AS NVARCHAR(255)) || ' -> ' || CAST(BatchNumbers."nstr" AS NVARCHAR(255))
					FROM TMP_TN_BatchNumbers1 AS BatchNumbers
					WHERE BatchNumbers."nstr" <> UPPER(BatchNumbers."nstr");
       				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
					IF LENGTH(v_errorList) <> 0 THEN
						p_error := 50018;
						p_error_message := 'BatchNumbers1 must be upper case. This is not the case for lines->batchnumbers: ' || v_errorList;
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;

					--All batchnumbers2 must be entered in upper case
					--By doing a where that is case sensitive we can find all batchnumbers that are not in upper case
					v_batchNumbers2 := '';
					v_lineNums := '';
					DELETE FROM TMP_TN_CharListTableNoIndex;
			        INSERT INTO TMP_TN_CharListTableNoIndex
			        SELECT CAST(v_lineNum AS NVARCHAR(255)) || ' -> ' || CAST(BatchNumbers."nstr" AS NVARCHAR(255))
					FROM TMP_TN_BatchNumbers2 AS BatchNumbers
					WHERE BatchNumbers."nstr" <> UPPER(BatchNumbers."nstr");
       				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
					IF LENGTH(v_errorList) <> 0 THEN
						p_error := 50024;
						p_error_message := 'BatchNumbers2 must be upper case. This is not the case for lines->batchnumbers: ' || v_errorList;
						SELECT p_error, p_error_message FROM dummy;
						--EXEC xp_logevent p_error, p_error_message, ERROR
						RETURN;
					END IF;
					
					--When this is a catch weight item, the quantity for UOM2 should be filled in
					IF(v_isCatchWeightForDocument = 'Y' ) THEN
					
						DELETE FROM TMP_TN_CharListTableNoIndex;
				        INSERT INTO TMP_TN_CharListTableNoIndex
						SELECT CAST(v_lineNum AS NVARCHAR(255)) || ' '
						FROM TMP_TN_QuantitiesUom2 AS QuantitiesUom2 
						WHERE QuantitiesUom2."nstr" IS NULL;
	       				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
							
						IF LENGTH(v_errorList) <> 0 THEN
							p_error := 50039;
							p_error_message := 'Error on line(s):  ' || v_errorList || ' "Quantity" for "Uom" needs to be filled for catch weight items.';
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;
						
						--Check if quantity is within tolerance
						--For goods issue/receipt, the check is not done, to make sure adjustments can be made.
						DELETE FROM TMP_TN_CharListTableNoIndex;
				        INSERT INTO TMP_TN_CharListTableNoIndex
						SELECT CAST(v_lineNum AS NVARCHAR(255)) || ' '
						FROM TMP_TN_QuantitiesUom2 AS QuantitiesUom2 
						INNER JOIN TMP_TN_Quantities AS Quantities ON Quantities."listPos" = QuantitiesUom2."listPos"
						INNER JOIN OITM ON OITM."ItemCode" = v_itemCode
						WHERE OITM."U_PMX_DQUM" < 100
						AND v_objType NOT IN ('59', '60')
						AND ( QuantitiesUom2."nstr" > ( Quantities."nstr" * OITM."U_PMX_DQUM" * (1 + (OITM."U_PMX_CWTO"/100)) ) OR
				   		      QuantitiesUom2."nstr" < ( Quantities."nstr" * OITM."U_PMX_DQUM" * (1 - (OITM."U_PMX_CWTO"/100)) ) );
	       				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
						
						IF LENGTH(v_errorList) <> 0 THEN
							p_error := 50040;
							p_error_message := 'Error on line(s):  ' || v_errorList || ' The weight for UOM 2 is not within the tolerance.';
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;
					 
					END IF;

					--Insert new item transactional info where this line does not exist
					--The line is only inserted if a batchnumber or best before date is filled in
					--TODO: Add function to external database to get the date and time
					INSERT INTO PMX_ITRI( "InternalKey", "Canceled", "UserSign", "CreateDate", "CreateTime", "UpdateDate", "UpdateTime", "Version", "ItemCode", "BatchNumber","InternalBatchNumber","BestBeforeDate")
					SELECT PMX_ITRI_S.nextval, * FROM (
						SELECT DISTINCT 'N', v_userSign, v_createDate,v_createTime,v_createDate,v_createTime, 1, v_itemCode, BatchNumbers1."nstr", BatchNumbers2."nstr", BestBeforeDates."nstr"
						FROM TMP_TN_BatchNumbers1 AS BatchNumbers1 
						INNER JOIN TMP_TN_BestBeforeDates AS BestBeforeDates ON BatchNumbers1."listPos" = BestBeforeDates."listPos"
						INNER JOIN TMP_TN_BatchNumbers2 AS BatchNumbers2 ON BatchNumbers1."listPos" = BatchNumbers2."listPos"
						LEFT JOIN PMX_ITRI ON 
							IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( BatchNumbers1."nstr", '') AND 
							IFNULL( PMX_ITRI."InternalBatchNumber", '') = IFNULL( BatchNumbers2."nstr", '') AND 
							(PMX_ITRI."BestBeforeDate" = BestBeforeDates."nstr" OR (PMX_ITRI."BestBeforeDate" IS NULL AND BestBeforeDates."nstr" IS NULL)) AND
							(PMX_ITRI."ItemCode" = v_itemCode)
						WHERE PMX_ITRI."InternalKey" IS NULL AND (NOT(BatchNumbers1."nstr" IS NULL AND BatchNumbers2."nstr" IS NULL AND BestBeforeDates."nstr" IS NULL))
					);


					--Check if we already have a link between serial number and document.
					--If not, only 1 itri key can be used, because otherwise, we cannot decide what the serial numbers are linked to the different itri keys
					SELECT COUNT(*) INTO v_cnt FROM PMX_SELD WHERE "BaseEntry" = v_docEntry AND "BaseLine" = v_lineNum AND "BaseType" = v_objType;
					IF( v_isSAPSerial = 'Y' AND v_cnt < 1 )
					THEN

						SELECT COUNT(*) INTO v_cnt
						FROM 
						(
							SELECT "DistNumber"
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							WHERE OITL."ApplyEntry" = v_docEntry
							AND OITL."ApplyLine" = v_lineNum
							AND OITL."ApplyType" = v_objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = v_whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0
							AND IFNULL("DistNumber", '')  <> ''
						) AS TEMP;
						
						--Only do checks if actual serial numbers are added
						IF( v_cnt > 0 )
						THEN

						--Check for only 1 itri key
						SELECT (
							(SELECT COUNT(*) FROM TMP_TN_BatchNumbers1) +
							(SELECT COUNT(*) FROM TMP_TN_BatchNumbers2) +
							(SELECT COUNT(*) FROM TMP_TN_BestBeforeDates) ) INTO v_cnt FROM dummy;
						IF (v_cnt > 3)
						THEN
							p_error := 50060;
							p_error_message := 'Only 1 ITRI allowed by line. Error on line ' || v_lineNum || ' and "DocEntry" ' || v_docEntry;
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;

						--Get the itri key
						BEGIN
							DECLARE v_itriForSerial int;
							SELECT TOP 1 PMX_ITRI."InternalKey"
							INTO v_itriForSerial
							FROM TMP_TN_BatchNumbers1 AS BatchNumbers1 
							 INNER JOIN TMP_TN_BestBeforeDates AS BestBeforeDates ON BatchNumbers1."listPos" = BestBeforeDates."listPos"
							 INNER JOIN TMP_TN_BatchNumbers2 AS BatchNumbers2 ON BatchNumbers1."listPos" = BatchNumbers2."listPos"
							 LEFT JOIN PMX_ITRI ON 
									IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( BatchNumbers1."nstr", '') AND 
									IFNULL( PMX_ITRI."InternalBatchNumber", '') = IFNULL( BatchNumbers2."nstr", '') AND 
								(PMX_ITRI."BestBeforeDate" = BestBeforeDates."nstr" OR (PMX_ITRI."BestBeforeDate" IS NULL AND BestBeforeDates."nstr" IS NULL)) AND
								(PMX_ITRI."ItemCode" = v_itemCode);

						--If we need to keep track of locations of the serial number, the LUID needs to be filled, and there can only be 1 LUID!
						BEGIN
							DECLARE v_luidForSerial  INT;
							DECLARE v_destLuidForSerial  INT;
	
							SELECT COUNT(*) INTO v_cnt FROM TMP_TN_LUIDs;
                            SELECT TOP 1 "nstr" INTO v_luidForSerial FROM TMP_TN_LUIDs;
							IF (v_trackLocationForSerial = 'Y'
								AND ( v_cnt > 1 OR v_luidForSerial IS NULL)) THEN
								p_error := 50061;
								p_error_message := 'Serial location tracked for item ' || v_itemCode || ' and "DocEntry" ' || v_docEntry || ' requires 1 LUID';
								SELECT p_error, p_error_message FROM dummy;
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN;
							END IF;
	
							IF (v_trackLocationForSerial = 'Y') THEN
								 SELECT TOP 1 "nstr" INTO v_luidForSerial FROM TMP_TN_LUIDs;
								 SELECT TOP 1 "nstr" INTO v_destLuidForSerial FROM TMP_TN_DestLuids;
							ELSE
								v_luidForSerial := NULL;
								v_destLuidForSerial := NULL;
							END IF;

						--Add to PMX serial numbers table
						--DECLARE v_outputSerialTable TABLE (internalkey int, "SerialNumber" nvarchar(36));
						BEGIN
							DECLARE v_firstKey INT;
							DECLARE v_lastKey  INT;
						
							SELECT PMX_SENU_S.nextval INTO v_firstKey FROM dummy; -- cannot use currval here, currval is only valid after a nextval call. We waste 1 seq number here.

						INSERT INTO PMX_SENU ("InternalKey", "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialNumber", "InStock", "ItriKey", "ItemCode", "LUID", "PmxWhsCode") 
						SELECT PMX_SENU_S.nextval, 'N', v_userSign, v_tsNow, v_tsNow, 1, "DistNumber", CASE WHEN v_stockDirection > 0 OR v_isStockTransfer = 1 THEN 'Y' ELSE 'N' END, v_itriForSerial, OSRN."ItemCode", IFNULL( v_destLuidForSerial, v_luidForSerial), PMX_OSWH."Code"
						FROM OITL 
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
						AND ITL1."SysNumber" = OSRN."SysNumber"
						INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OITL."LocCode"
						LEFT JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
						AND PMX_SENU."SerialNumber" = "DistNumber"
						WHERE OITL."ApplyEntry" = v_docEntry
						AND OITL."ApplyLine" = v_lineNum
						AND OITL."ApplyType" = v_objType
						AND OITM."ManSerNum" = 'Y'
						AND OITL."LocCode" = v_whsCode
						AND PMX_SENU."InternalKey" IS NULL
						GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", PMX_OSWH."Code"
						HAVING SUM(ITL1."Quantity") <> 0
						AND "DistNumber" IS NOT NULL;

						SELECT PMX_SENU_S.currval INTO v_lastKey FROM dummy;

						--Update the 'in stock' values
						UPDATE PMX_SENU SET "InStock" = CASE WHEN v_stockDirection > 0  OR v_isStockTransfer = 1  THEN 'Y' ELSE 'N' END, "PmxWhsCode" = CASE WHEN v_isStockTransfer = 1 THEN PMX_OSWH."Code" ELSE "PmxWhsCode" END
						FROM PMX_SENU 
						INNER JOIN OITM ON OITM."ItemCode" = PMX_SENU."ItemCode"
						INNER JOIN
							(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM(ITL1."Quantity") * v_StockDirection AS "Quantity", "StockEff", "LocCode"

							FROM OSRN
							INNER JOIN ITL1 ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITL ON ITL1."LogEntry" = OITL."LogEntry"
							WHERE OITL."ApplyEntry" = v_docEntry
							AND OITL."ApplyLine" = v_lineNum
							AND OITL."ApplyType" = v_objType
							AND OITL."LocCode" = CASE WHEN v_isStockTransfer = 1 THEN v_destinationWhsCode ELSE v_whsCode END 
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", "LocCode"
							HAVING SUM(ITL1."Quantity") <> 0
							) AS SerialTable ON SerialTable."ItemCode" = PMX_SENU."ItemCode"
							AND SerialTable."DistNumber" = PMX_SENU."SerialNumber"
						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = SerialTable."LocCode"
						WHERE OITM."ManSerNum" = 'Y'
						AND IFNULL(PMX_SENU."ItriKey", -1) = IFNULL(v_itriForSerial, -1);

						IF (v_trackLocationForSerial = 'Y'
							AND IFNULL( v_luidForSerial, 0) <> IFNULL( v_destLuidForSerial, 0)) THEN

							--Check if move for track location serial number does not split up LUID when linked to document
							DELETE FROM TMP_TN_CharListTableNoIndex;
                            INSERT INTO TMP_TN_CharListTableNoIndex
                            SELECT CAST(OSRN."ItemCode" AS VARCHAR)
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
							AND PMX_SENU."SerialNumber" = "DistNumber"
							AND PMX_SENU.LUID = v_luidForSerial
							INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
							INNER JOIN RDR1 ON RDR1."ObjType" = PMX_SELD."BaseType"
							AND RDR1."DocEntry" = PMX_SELD."BaseEntry"
							AND RDR1."LineNum" = PMX_SELD."BaseLine"
							AND RDR1."LineStatus" = 'O'
							WHERE OITL."ApplyEntry" = v_docEntry
							AND OITL."ApplyLine" = v_lineNum
							AND OITL."ApplyType" = v_objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = v_whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0;
							SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
							IF LENGTH(v_errorList) <> 0 THEN
								p_error := 50062;
								p_error_message := 'LUID cannot be split up when serial numbers are linked to sales order for items: ' || v_errorList ;
								SELECT p_error, p_error_message FROM dummy;
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN;
							END IF;

							--Check if move for track location serial number does not split up LUID when linked to document
							DELETE FROM TMP_TN_CharListTableNoIndex;
                            INSERT INTO TMP_TN_CharListTableNoIndex
                            SELECT CAST(OSRN."ItemCode" AS VARCHAR)
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
							AND PMX_SENU."SerialNumber" = "DistNumber"
							AND PMX_SENU.LUID = v_luidForSerial
							INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
							INNER JOIN PMX_PLPL ON PMX_PLPL."DocEntry" = PMX_SELD."BaseEntry"
							AND PMX_PLPL."LineNum" = PMX_SELD."BaseLine"
							AND PMX_PLPL."LineStatus" = 'O'
							AND PMX_SELD."BaseType" = 'PMX_PLPH'
							WHERE OITL."ApplyEntry" = v_docEntry
							AND OITL."ApplyLine" = v_lineNum
							AND OITL."ApplyType" = v_objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = v_whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0;
							SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
							IF LENGTH(v_errorList) <> 0 THEN
								p_error := 50063;
								p_error_message := 'LUID cannot be split up when serial numbers are linked to pick list proposal for items: ' || v_errorList ;
								SELECT p_error, p_error_message FROM dummy;
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN;
							END IF;

							--Check if move for track location serial number does not split up LUID when linked to document
							DELETE FROM TMP_TN_CharListTableNoIndex;
                            INSERT INTO TMP_TN_CharListTableNoIndex
                            SELECT CAST(OSRN."ItemCode" AS VARCHAR)
							FROM OITL 
							INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
							INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITM ON OITM."ItemCode" = OSRN."ItemCode"
							INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = OITM."ItemCode"
							AND PMX_SENU."SerialNumber" = "DistNumber"
							AND PMX_SENU.LUID = v_luidForSerial
							INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
							INNER JOIN PMX_PLLI ON PMX_PLLI."DocEntry" = PMX_SELD."BaseEntry"
							AND PMX_PLLI."LineNum" = PMX_SELD."BaseLine"
							AND PMX_PLLI."LineStatus" = 'O'
							AND PMX_SELD."BaseType" = 'PMX_PLHE'
							WHERE OITL."ApplyEntry" = v_docEntry
							AND OITL."ApplyLine" = v_lineNum
							AND OITL."ApplyType" = v_objType
							AND OITM."ManSerNum" = 'Y'
							AND OITL."LocCode" = v_whsCode
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff"
							HAVING SUM(ITL1."Quantity") <> 0;
							SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
							IF LENGTH(v_errorList) <> 0 THEN
								p_error := 50064;
								p_error_message := 'LUID cannot be split up when serial numbers are linked to pick list for items: ' || v_errorList;
								SELECT p_error, p_error_message FROM dummy;
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN;
							END IF;

						END IF;

						--Add link to serial number document link table
						INSERT INTO PMX_SELD ( "InternalKey", "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine") 
						SELECT PMX_SELD_S.nextval, 'N', v_userSign, v_tsNow, v_tsNow, 1, "InternalKey", v_objType, v_docEntry, v_lineNum
						FROM PMX_SENU
						WHERE "InternalKey" >= v_firstKey AND "InternalKey" <= v_lastKey;

						END;
						END;
						END;	


					ELSE IF (v_isSAPSerial = 'Y' AND v_isStockTransfer = 1) THEN
						--Adjust serial number whs code
						UPDATE PMX_SENU
						SET PMX_SENU."PmxWhsCode" = PMX_OSWH."Code",
							PMX_SENU."Version" = PMX_SENU."Version" + 1
						FROM PMX_SENU
						INNER JOIN OITM ON OITM."ItemCode" = PMX_SENU."ItemCode"
						INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
						INNER JOIN
							(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM(ITL1."Quantity") * v_StockDirection AS "Quantity", "StockEff", "LocCode"

							FROM OSRN
							INNER JOIN ITL1 ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
							AND ITL1."SysNumber" = OSRN."SysNumber"
							INNER JOIN OITL ON ITL1."LogEntry" = OITL."LogEntry"
							WHERE OITL."ApplyEntry" = v_keyAsInteger
							AND OITL."ApplyType" = '67'
							AND OITL."LocCode" = v_destinationWhsCode 
							GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", "LocCode"
							HAVING SUM(ITL1."Quantity") <> 0
							) AS SerialTable ON SerialTable."ItemCode" = PMX_SENU."ItemCode"
							AND SerialTable."DistNumber" = PMX_SENU."SerialNumber"
							AND SerialTable."ApplyLine" = PMX_SELD."BaseLine"
							AND SerialTable."ApplyEntry" = PMX_SELD."BaseEntry"
							AND SerialTable."ApplyType" = PMX_SELD."BaseType"
						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = SerialTable."LocCode"
						WHERE OITM."ManSerNum" = 'Y'
						AND PMX_SELD."BaseEntry" = v_keyAsInteger
						AND PMX_SELD."BaseType" = '67';  --Warehouse transfer
						
						END IF;
					END IF;
					END IF;

					
					--Insert the document lines in a nice format in the table
					--Quantities."nstr" is the number of items in inventory and not in sales/purchase "Uom". Therefore we devide the quantity again by the quantityPerUom
					INSERT INTO TMP_TN_documentLine 
						SELECT v_docEntry,v_objType,v_docNum,v_docDate,v_cardCode,v_cardName,v_userSign,v_createDate,v_createTime,v_lineNum, v_baseType ,v_baseEntry,v_BaseLine,v_itemCode,v_description, Quantities."nstr"/v_quantityPerUom, Quantities."nstr", v_uom,v_inventoryUom, v_quantityPerUom, v_whsCode, StorLocs."nstr", PMX_ITRI."InternalKey", Luids."nstr", PMX_LUID.SSCC, Qualities."nstr", v_isLogisticCarrier, v_reasonCode, v_uom2, QuantitiesUom2."nstr", 
						DestStorLocs."nstr", v_destinationWhsCode, DestQualities."nstr", DestLuids."nstr", MasterLuids."nstr"
						FROM TMP_TN_BatchNumbers1 AS BatchNumbers1 
							 INNER JOIN TMP_TN_BatchNumbers2 AS BatchNumbers2 ON BatchNumbers1."listPos" = BatchNumbers2."listPos"
							 INNER JOIN TMP_TN_BestBeforeDates AS BestBeforeDates ON BatchNumbers1."listPos" = BestBeforeDates."listPos"
							 INNER JOIN TMP_TN_Quantities AS Quantities ON BatchNumbers1."listPos" = Quantities."listPos"
							 INNER JOIN TMP_TN_QuantitiesUom2 AS QuantitiesUom2 ON BatchNumbers1."listPos" = QuantitiesUom2."listPos"
							 INNER JOIN TMP_TN_StorLocs AS StorLocs ON BatchNumbers1."listPos" = StorLocs."listPos"
							 INNER JOIN TMP_TN_LUIDs AS Luids ON BatchNumbers1."listPos" = Luids."listPos"
							 INNER JOIN TMP_TN_QualityStatuses AS Qualities ON BatchNumbers1."listPos" = Qualities."listPos"
		 					 INNER JOIN OITM ON OITM."ItemCode" = v_itemCode
							 LEFT JOIN TMP_TN_DestStorLocs AS DestStorLocs ON BatchNumbers1."listPos" = DestStorLocs."listPos"
							 LEFT JOIN TMP_TN_DestLuids AS DestLuids ON BatchNumbers1."listPos" = DestLuids."listPos"
							 LEFT JOIN TMP_TN_DestQualityStatuses AS DestQualities ON BatchNumbers1."listPos" = DestQualities."listPos"
							 LEFT JOIN TMP_TN_MasterLuids as MasterLuids ON BatchNumbers1."listPos" = MasterLuids."listPos"
							 LEFT JOIN PMX_ITRI ON 
								IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( BatchNumbers1."nstr", '') AND 
								IFNULL( PMX_ITRI."InternalBatchNumber", '') = IFNULL( BatchNumbers2."nstr", '') AND 
								(PMX_ITRI."BestBeforeDate" = BestBeforeDates."nstr" OR (PMX_ITRI."BestBeforeDate" IS NULL AND BestBeforeDates."nstr" IS NULL)) AND
								(PMX_ITRI."ItemCode" = v_itemCode)
							 LEFT JOIN PMX_LUID ON PMX_LUID."InternalKey" = Luids."nstr"
						WHERE OITM."InvntItem" = 'Y';
						
					FETCH documentLineRawCursor INTO v_docEntry,v_objType,v_docNum,v_docDate,v_cardCode,v_cardName,v_userSign,v_createDate,v_createTime,v_lineNum,v_baseType,v_baseEntry,v_BaseLine,v_itemCode,v_description,v_quantity, v_inventoryQuantity,v_uom,v_inventoryUom,v_quantityPerUom,v_whsCode,v_batchNumbers1,v_batchNumbers2,v_bestBeforeDates,v_quantities,v_storLocs,v_luids,v_ssccs,v_qualityStatuses,v_isLogisticCarrier,v_reasonCode, v_quantitiesUom2, v_uom2, v_quantityUom2, v_dropShip, v_destinationStorLocCodes, v_destinationWhsCode, v_destinationQualityStatusCodes, v_destinationLuids, v_trackLocationForSerial, v_isSAPSerial, v_isCatchWeightForDocument, v_masterLuids;
				END WHILE;
				CLOSE documentLineRawCursor;
				
			END; /* TRY
			BEGIN CATCH
				p_error := 50016
				p_error_message := 'Error raw lines->normal lines: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
				|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE documentLineRawCursor 
					DEALLOCATE documentLineRawCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE documentLineRawCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END CATCH
			END CATCH */


			--**********************************************************************************************************
			--Check if the converted lines are the same as the SAP lines
			--**********************************************************************************************************

			--The batchnumbers and the quantities must be the same as what is saved in SAP
			--The batchnumbers are checked using case sensitive and not like default case insensitive
			--This means if in SAP batch is saved as b1 and in extra fields as B1 that this is not the same
			v_lineNums := '';
			v_batchNumbers1 := '';
			v_quantities := '';
			
			SELECT COUNT(1) INTO v_cnt FROM TMP_TN_documentLine;
			IF v_cnt > 0 THEN
				SELECT TOP 1 "ObjType" INTO v_objType FROM TMP_TN_documentLine;  --Get the object type of the document


				DELETE FROM TMP_TN_CharListTableNoIndex;
				INSERT INTO TMP_TN_CharListTableNoIndex
				SELECT CAST(IFNULL( Lines."LineNum",TranLog."DocLine") AS NVARCHAR(255))
					|| ',' || IFNULL( IFNULL( Lines."BatchNumber", TranLog."DistNumber"),'<NULL>')
					|| ',' || CAST(CAST(IFNULL( Lines."Quantity", 0) AS DECIMAL(19,6) ) AS NVARCHAR) || '<>' || CAST(CAST(IFNULL( TranLog."Quantity", 0) AS DECIMAL(19,6) ) AS NVARCHAR)
				FROM (
					SELECT LinesTable."ObjType", LinesTable."DocEntry", LinesTable."LineNum", PMX_ITRI."BatchNumber", SUM(LinesTable."Quantity"*LinesTable."QuantityPerUom") AS "Quantity" , OITM.U_PMX_UOMD AS UOMDecimals, LinesTable."WhsCode"
   					FROM TMP_TN_documentLine AS LinesTable 
					INNER JOIN OITM ON LinesTable."ItemCode" = OITM."ItemCode" 
					LEFT JOIN  PMX_ITRI ON LinesTable."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
					WHERE OITM."ManBtchNum" = 'Y' 
					GROUP BY LinesTable."ObjType", LinesTable."DocEntry", LinesTable."LineNum", PMX_ITRI."BatchNumber", OITM.U_PMX_UOMD, LinesTable."WhsCode"
   				) AS Lines 
				LEFT JOIN (
					SELECT OITL."DocType", OITL."DocEntry", OITL."DocLine", OBTN."DistNumber", SUM( ITL1."Quantity" ) * v_stockDirection AS "Quantity", OITL."LocCode"
					FROM OITL 
					LEFT JOIN (   ITL1  -- or INNER JOIN?
					   INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
					   AND ITL1."SysNumber" = OBTN."SysNumber"
					) ON ITL1."LogEntry" = OITL."LogEntry"
					WHERE OITL."DocType" = v_objType AND OITL."DocEntry" = v_keyAsInteger AND "StockEff" = 1  -- Affects "OnHand"
					GROUP BY OITL."DocType", OITL."DocEntry", OITL."DocLine", OBTN."DistNumber", OITL."LocCode"
					HAVING SUM( ITL1."Quantity" ) <> 0
				) AS TranLog ON TranLog."DocType" = Lines."ObjType" 
				AND TranLog."DocEntry" = Lines."DocEntry" 
				AND TranLog."DocLine" = Lines."LineNum"
				AND TranLog."LocCode" = Lines."WhsCode"
				AND TranLog."DistNumber" = Lines."BatchNumber"
				--WHERE IFNULL( ROUND(TranLog."Quantity" * (CASE WHEN v_isTransactionTypeCancel=0 THEN 1 ELSE -1 END), IFNULL( Lines.UOMDecimals, 0)), 0) <> IFNULL( ROUND(Lines."Quantity" , IFNULL( Lines.UOMDecimals, 0)), 0)
				WHERE IFNULL( ROUND(TranLog."Quantity", IFNULL( Lines.UOMDecimals, 0)), 0) <> IFNULL( ROUND(Lines."Quantity" , IFNULL( Lines.UOMDecimals, 0)), 0)
				-- OR (TranLog."Quantity" IS NULL AND Lines."Quantity" > 0) 
				-- OR Lines."Quantity" IS NULL
				;
				
				SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' | ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50017;
					p_error_message := 'Batch quantities do not match between PMX and SAP (line,batch,pmx<>sap): ' || v_errorList;
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			END IF;

			--**********************************************************************************************************
			--Do checks on document lines
			--**********************************************************************************************************
			-- Check if storage location code exists and is filled in in the organizational structure
			-- This check is only done for the inventory lines.
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
            INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
            INNER JOIN OWHS ON OWHS."WhsCode" = Lines."WhsCode"
            LEFT JOIN PMX_OSEL AS OSE ON Lines."StorLocCode" = OSE."Code" AND OSE."IsStorageLocation" = 'Y'
			WHERE Items."InvntItem" = 'Y' AND OSE."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y'
			GROUP BY Lines."LineNum";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50003;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The storage location code is not filled in or does not exist in the organizational structure.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if batch number 2 is filled in for items with a second batch number
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
				INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
				LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."InternalBatchNumber" IS NULL AND Items.U_PMX_HBN2 = 'Y'
			GROUP BY Lines."LineNum";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50022;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The second batch number must be filled in for items that have a second batch number.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if batch number 2 is NOT filled in for items that have no second batch number
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
			LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."InternalBatchNumber" IS NOT NULL AND Items.U_PMX_HBN2 = 'N'
			GROUP BY Lines."LineNum";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50023;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The second batch number cannot be filled in for items that have NO second batch number.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if best before date is filled in for expirable goods
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
			LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."BestBeforeDate" IS NULL AND Items.U_PMX_HBBD = 'Y'
			GROUP BY Lines."LineNum";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50004;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The best before date must be filled in for expirable item.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if best before date is NOT filled in for NON expirable goods
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines
			INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
			LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
			WHERE PMX_ITRI."BestBeforeDate" IS NOT NULL AND Items.U_PMX_HBBD = 'N'
			GROUP BY Lines."LineNum";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50005;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The best before date cannot be filled in for a non-expirable item.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if the entered LUID exists for the inventory items
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			LEFT JOIN PMX_LUID ON Lines."LogUnitIdentKey" = PMX_LUID."InternalKey" 
			WHERE Lines."LogUnitIdentKey" IS NOT NULL AND OITM."InvntItem" = 'Y' AND PMX_LUID."InternalKey" IS NULL
			GROUP BY Lines."LineNum";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50006;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The entered LUID does not exist.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

	--		-- Check if the entered SSCC matches the stored SSCC for the specified LUID
	--		v_errorList := ''
	--		SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' ' 
	--		FROM TMP_TN_documentLine AS Lines LEFT JOIN PMX_LUID AS LogicalUnitID ON Lines.U_PMX_LUID = LogicalUnitID."InternalKey" 
	--		WHERE Lines.U_PMX_SSCC <> LogicalUnitID.SSCC
	--		IF LENGTH(v_errorList) <> 0 THEN
	--			p_error := 50007
	--			p_error_message := 'Error on line(s): ' || v_errorList || ' The entered SSCC does not match the stored SSCC for the specified LUID.'
	--			SELECT p_error, p_error_message
	--			--EXEC xp_logevent p_error, p_error_message, ERROR
	--			RETURN
	--		END

			-- Check if the entered quality statuses exists and is filled in for inventory items
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			INNER JOIN OWHS ON OWHS."WhsCode" = Lines."WhsCode"
			LEFT JOIN PMX_QYST ON Lines."QualityStatusCode" = PMX_QYST."Code"
			WHERE OITM."InvntItem" = 'Y' AND PMX_QYST."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50008;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The entered quality status does not exist or is not filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			-- Check if the entered quality statuses exists and is filled in for inventory items
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			INNER JOIN OWHS ON OWHS."WhsCode" = Lines."DestinationWhsCode"
			LEFT JOIN PMX_QYST ON Lines."DestinationQualityStatusCode" = PMX_QYST."Code"
			WHERE OITM."InvntItem" = 'Y' AND PMX_QYST."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50008;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The entered destination quality status does not exist or is not filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if the warehouse of the storage location on the document line is the same
			-- warehouse as the one that is stored on the document line itself.
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN PMX_OSEL AS OSE ON Lines."StorLocCode" = OSE."Code"
			INNER JOIN PMX_OSWH AS WHS ON OSE."PmxWhsCode" = WHS."Code"
			WHERE Lines."WhsCode" <> WHS."SboWhsCode";
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50009;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The storage location code is of a different warehouse than the warehouse defined on the line.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- If a line is set as logisic carrier then the LUID must be filled in
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Lines."LogUnitIdentKey" IS NULL;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50010;
				p_error_message := 'Error on line(s): ' || v_errorList || ' If the flag is set that this line is the logistic carrier of the logistic unit then the logistic unit ID must be filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if logistic carrier is filled in on the logistic unit and that there are no lines
			-- indicated as logistic carrier.
			-- Also check that not 2 logistic carriers are defined on the same logistic unit
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LogUnitIdentKey" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN PMX_LUID ON Lines."LogUnitIdentKey" = PMX_LUID."InternalKey"
			WHERE Lines."IsLogisticCarrier" = 'Y'
			GROUP BY lines."DocEntry", lines."LogUnitIdentKey", PMX_LUID."LogisticCarrierCode"
			HAVING (count(lines."DocEntry") > 1) or (count(lines."DocEntry") = 1 AND PMX_LUID."LogisticCarrierCode" IS NOT NULL);
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50011;
				p_error_message := 'Error for logistic unit(s): ' || v_errorList || ' There is already a logistic carrier defined or there is more then 1 line is defined as logistic carrier for these logistic unit(s).';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- If a line is set as logisic carrier then the item must be a logistic carrier
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			WHERE Lines."IsLogisticCarrier" = 'Y' AND OITM.U_PMX_LOCA <> 'Y';
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50012;
				p_error_message := 'Error on line(s): ' || v_errorList || ' If the flag is set that this line is the logistic carrier of the logistic unit then the item must be a logistic carrier.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- If a line is set as logisic carrier then its quantity must be 1
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			WHERE Lines."IsLogisticCarrier" = 'Y' AND ( Lines."Quantity" * Lines."QuantityPerUom" ) <> 1;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50013;
				p_error_message := 'Error on line(s): ' || v_errorList || ' If a line is a logistic carrier then its quantity must be 1.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Does the item have zone types and is it allowed by the location
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode" 
			INNER JOIN PMX_OSEL AS Ose ON Lines."StorLocCode" = Ose."Code"
			LEFT OUTER JOIN PMX_ITZT AS ItemZoneTypes ON OITM."ItemCode" = ItemZoneTypes."ItemCode"
			LEFT OUTER JOIN PMX_LZZT AS LinkedZoneTypes ON ItemZoneTypes."ZoneTypeCode" = LinkedZoneTypes."ZoneTypeCode"
				AND Ose."PmxZoneCode" = LinkedZoneTypes."ParentCode"
			WHERE ItemZoneTypes."InternalKey" IS NOT NULL AND LinkedZoneTypes."ZoneTypeCode" IS NULL;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50019;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The location does not have a zone type that corresponds to a zone type on the item.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if the filled in reason code exist
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			LEFT JOIN PMX_REAS ON Lines."ReasonCode" = PMX_REAS."Code"
			WHERE NULLIF( Lines."ReasonCode", '' ) IS NOT NULL AND PMX_REAS."Code" IS NULL;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50020;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The filled in reason code does not exist.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if the filled in "Uom2" is the same as the "Uom2" on an item
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			LEFT JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
			WHERE Lines."Uom2" <> OITM.U_PMX_UOM2;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50021;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The filled in "Uom2" is not the same as the one defined on the item.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if batch number is filled in for items with a batch number
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode" 
			LEFT JOIN PMX_ITRI ON Lines."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			WHERE OITM."ManBtchNum" = 'Y' AND NULLIF( PMX_ITRI."BatchNumber", '' ) IS NULL;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50025;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The batch number must be filled in for items that have a batch number.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if batch number is NOT filled in for items that have batch number
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode" 
			LEFT JOIN PMX_ITRI ON Lines."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			WHERE OITM."ManBtchNum" = 'N' AND NULLIF( PMX_ITRI."BatchNumber", '' ) IS NOT NULL;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50026;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The batch number cannot be filled in for items that have NO batch number.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- check if stock is stored on logistic carrier location for non logistic carriers (only for stock receipts)
			SELECT COUNT(*) INTO v_cnt FROM PMX_OSCO WHERE "LogCarOn1Loc" = 'Y';
			IF ((v_isInventoryEntry=1 OR 
				v_isPurchaseDeliveryDocument=1 OR		
				v_isPurchaseInvoice=1 OR 
				v_isSalesReturnDocument=1 OR		
				v_isSalesCreditNote=1)
				AND v_cnt = 1) THEN

			    DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."LineNum" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
					 INNER JOIN PMX_OSEL AS OSE ON Lines."StorLocCode" = OSE."Code"
					 INNER JOIN PMX_OSWH AS WHS ON OSE."PmxWhsCode" = WHS."Code"
					 INNER JOIN OITM on OITM."ItemCode" = Lines."ItemCode"
				WHERE Lines."StorLocCode" = WHS."StorLocLogCar" AND 
						OITM.U_PMX_LOCA = 'N';
    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50027;
					p_error_message := 'Error on line(s): ' || v_errorList || ' Items that are no logistic carrier cannot be stored on the location for logistic carriers.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			END IF;

			IF (v_isStockTransfer = 1) THEN

				-- Check if 'destination' storage location code exists and is filled in in the organizational structure
				-- This check is only done for the inventory lines.
			    DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."LineNum" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
					 INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
					 INNER JOIN OWHS ON OWHS."WhsCode" = Lines."DestinationWhsCode"
					 LEFT JOIN PMX_OSEL AS OSE ON Lines."DestinationStorLocCode" = OSE."Code" AND OSE."IsStorageLocation" = 'Y'
				WHERE Items."InvntItem" = 'Y' AND OSE."Code" IS NULL AND OWHS.U_PMX_IMBP = 'Y';
    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50028;
					p_error_message := 'Error on line(s): ' || v_errorList || ' The destination storage location code is not filled in or does not exist in the organizational structure.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				-- Check if the warehouse of the 'destination' storage location on the document line is the same
				-- 'Destiniation' warehouse as the one that is stored on the document line itself.
			    DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."LineNum" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
				INNER JOIN PMX_OSEL AS OSE ON Lines."DestinationStorLocCode" = OSE."Code"
				INNER JOIN PMX_OSWH AS WHS ON OSE."PmxWhsCode" = WHS."Code"
				WHERE Lines."DestinationWhsCode" <> WHS."SboWhsCode";
    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50029;
					p_error_message := 'Error on line(s): ' || v_errorList || ' The storage location code is of a different warehouse than the warehouse defined on the line.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
				
				-- Check if 'destination' storage location code exists and is filled in in the organizational structure
				-- This check is only done for the inventory lines.
				DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."DestinationStorLocCode" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
                INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
				WHERE Items."InvntItem" = 'Y'
				AND Lines."DestinationStorLocCode" IN (
									 SELECT "Code" FROM PMX_OSWA
									 UNION SELECT "InputLocationCode" FROM PMX_OSWA WHERE "InputLocationCode" IS NOT NULL
									 UNION SELECT "InputWithBoxLocationCode" FROM PMX_OSWA WHERE "InputWithBoxLocationCode" IS NOT NULL);
    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50030;
					p_error_message := 'Error for location(s): ' || v_errorList || ' The destination storage location code cannot be a WA location.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

			END IF;

			--Check if location is not locked
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."StorLocCode" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
		    LEFT JOIN PMX_OSSL AS OSE ON Lines."StorLocCode" = OSE."Code"
			WHERE OSE."LockedBy" IS NOT NULL;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50041;
				p_error_message := 'Locations are locked, so no stock updates are allowed: ' || v_errorList;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--Check if location matches quality status
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."DestinationStorLocCode" AS VARCHAR) || '-' || CAST(IFNULL(Lines."DestinationQualityStatusCode", Lines."QualityStatusCode") AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
		    LEFT JOIN PMX_OSSL AS OSE ON Lines."DestinationStorLocCode" = OSE."Code"
			WHERE IFNULL(OSE."QualityStatusCode", IFNULL(Lines."DestinationQualityStatusCode", Lines."QualityStatusCode")) <> IFNULL(Lines."DestinationQualityStatusCode", Lines."QualityStatusCode") ;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50042;
				p_error_message := 'Quality status does not match fixed quality status on location: ' || v_errorList;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			

			-- check if stock is free stock (only for stock exits)
			IF (v_isInventoryExit=1 OR
				v_isPurchaseReturnDocument=1 OR
				v_isPurchaseCreditNote=1 OR
				v_isSalesInvoice=1 OR
				v_isStockTransfer = 1) THEN
				
				--Build ignore list
				DELETE FROM TMP_TN_IgnoreLocking;
				INSERT INTO TMP_TN_IgnoreLocking
				SELECT DISTINCT PMX_PLLI."DocEntry", 'PMX_PLHE', PMX_PLLI."LineNum"
				FROM PMX_PLLI
				INNER JOIN PMX_PLPL ON PMX_PLPL."LineNum" = PMX_PLLI."BaseLine"
				AND PMX_PLPL."DocEntry" = PMX_PLLI."BaseEntry"
				AND PMX_PLLI."BaseType" = 'PMX_PLPH'
				INNER JOIN TMP_TN_documentLine AS Lines
				ON Lines."BaseEntry" = PMX_PLPL."BaseEntry"
				AND Lines."BaseLine" = PMX_PLPL."BaseLine"
				AND Lines."BaseType" = PMX_PLPL."BaseType";

				--Check free stock for item - quality status
				DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."ItemCode" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
				INNER JOIN
				(
				SELECT "TOTAL"."Quantity" - IFNULL( "LOCK"."Quantity",0) AS "Quantity", "TOTAL"."ItemCode" AS "ItemCode", 
				"TOTAL"."QualityStatusCode" AS "QualityStatusCode", "TOTAL"."InvntryUom" AS "Uom" , "TOTAL"."SboWhsCode"
				FROM ( 

				SELECT SUM ("PMX_INVT"."Quantity") AS "Quantity", "PMX_INVT"."ItemCode", "OITM"."InvntryUom", 
				"PMX_INVT"."QualityStatusCode", PMX_OSWH."SboWhsCode" 
				FROM "PMX_INVT" 
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = "PMX_INVT"."StorLocCode" 
				INNER JOIN PMX_OSWH ON PMX_OSEL."PmxWhsCode" = PMX_OSWH."Code" 
				INNER JOIN "OITM" ON "OITM"."ItemCode" = "PMX_INVT"."ItemCode" 
				LEFT OUTER JOIN "PMX_LUID" ON "PMX_LUID"."InternalKey" = "PMX_INVT"."LogUnitIdentKey" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INVT"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey"
				GROUP BY "PMX_INVT"."ItemCode", "PMX_INVT"."QualityStatusCode", "OITM"."InvntryUom", PMX_OSWH."SboWhsCode" 

				) AS "TOTAL" 
				LEFT OUTER JOIN ( 

				SELECT SUM( "PMX_INLD"."Quantity") AS "Quantity", "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", PMX_OSWH."SboWhsCode"
				FROM "PMX_INLD" 
				INNER JOIN PMX_OSWH ON "PMX_INLD"."PmxWhsCode" = PMX_OSWH."Code" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INLD"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				LEFT JOIN TMP_TN_IgnoreLocking AS IgnoreLocking
				ON IgnoreLocking."DocEntry" = PMX_INLD."BaseEntry"
				AND IgnoreLocking."LineNum" = PMX_INLD."BaseLine"
				AND IgnoreLocking."ObjType" = PMX_INLD."BaseType"
				WHERE IgnoreLocking."DocEntry" IS NULL
				GROUP BY "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode" , PMX_OSWH."SboWhsCode"

				) AS "LOCK" ON "TOTAL"."ItemCode" = "LOCK"."ItemCode" AND "TOTAL"."QualityStatusCode" = "LOCK"."QualityStatusCode" 
				AND "TOTAL"."SboWhsCode" = "LOCK"."SboWhsCode"
				) AS FreeStock ON lines."ItemCode" = FreeStock."ItemCode" 
				AND lines."QualityStatusCode" = FreeStock."QualityStatusCode"
				AND lines."WhsCode" = FreeStock."SboWhsCode"
				AND FreeStock."Quantity" < lines."Quantity";

    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50050;
					p_error_message := 'Error for item(s): ' || v_errorList || ' The stock for item and quality status is locked.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;


				--Check free stock for item - batch - best before date
				DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."ItemCode" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
				INNER JOIN
				(
				SELECT "TOTAL"."Quantity" - IFNULL( "LOCK"."Quantity",0) AS "Quantity", "TOTAL"."ItemCode" AS "ItemCode", 
				"TOTAL"."QualityStatusCode" AS "QualityStatusCode", "TOTAL"."InvntryUom" AS "Uom", 
				"TOTAL"."BatchNumber" AS "BatchNumber", 
				CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber_IsNull", 
				"TOTAL"."BatchNumber" AS "BatchNumber2", 
				CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber2_IsNull", 
				"TOTAL"."BestBeforeDate" AS "BestBeforeDate", 
				CASE WHEN ("TOTAL"."BestBeforeDate" IS NULL) THEN 1 ELSE 0 END AS "BestBeforeDate_IsNull", 
				"TOTAL"."ItemTransactionalInfoKey" AS "ItemTransactionalInfoKey", 
				CASE WHEN ("TOTAL"."ItemTransactionalInfoKey" IS NULL) THEN 1 ELSE 0 END AS "ItemTransactionalInfoKey_IsNull" ,
				"TOTAL"."SboWhsCode"
				FROM ( 

				SELECT SUM ("PMX_INVT"."Quantity") AS "Quantity", "PMX_INVT"."ItemCode", "OITM"."InvntryUom", 
				"PMX_INVT"."QualityStatusCode", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", 
				"PMX_INVT"."ItemTransactionalInfoKey", PMX_OSWH."SboWhsCode" 
				FROM "PMX_INVT" 
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = "PMX_INVT"."StorLocCode" 
				INNER JOIN PMX_OSWH ON PMX_OSEL."PmxWhsCode" = PMX_OSWH."Code" 
				INNER JOIN "OITM" ON "OITM"."ItemCode" = "PMX_INVT"."ItemCode" 
				LEFT OUTER JOIN "PMX_LUID" ON "PMX_LUID"."InternalKey" = "PMX_INVT"."LogUnitIdentKey" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INVT"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				GROUP BY "PMX_INVT"."ItemCode", "PMX_INVT"."QualityStatusCode", "OITM"."InvntryUom", 
				"PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INVT"."ItemTransactionalInfoKey",
				PMX_OSWH."SboWhsCode" 

				) AS "TOTAL" 
				LEFT OUTER JOIN ( 

				SELECT SUM ("PMX_INLD"."Quantity") AS "Quantity", "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", 
				"PMX_INLD"."ItemTransactionalInfoKey", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", PMX_OSWH."SboWhsCode" 
				FROM "PMX_INLD" INNER JOIN PMX_OSWH ON "PMX_INLD"."PmxWhsCode" = PMX_OSWH."Code" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INLD"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				LEFT JOIN TMP_TN_IgnoreLocking AS IgnoreLocking
				ON IgnoreLocking."DocEntry" = PMX_INLD."BaseEntry"
				AND IgnoreLocking."LineNum" = PMX_INLD."BaseLine"
				AND IgnoreLocking."ObjType" = PMX_INLD."BaseType"
				WHERE IgnoreLocking."DocEntry" IS NULL
				GROUP BY "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", "PMX_ITRI"."BatchNumber", 
				"PMX_ITRI"."BestBeforeDate", "PMX_INLD"."ItemTransactionalInfoKey", PMX_OSWH."SboWhsCode"

				) AS "LOCK" 
				ON "TOTAL"."ItemCode" = "LOCK"."ItemCode" AND "TOTAL"."QualityStatusCode" = "LOCK"."QualityStatusCode" 
				AND "TOTAL"."SboWhsCode" = "LOCK"."SboWhsCode"
				AND ("TOTAL"."BatchNumber" = "LOCK"."BatchNumber" OR ("TOTAL"."BatchNumber"IS NULL AND "LOCK"."BatchNumber"IS NULL)) 
				AND ("TOTAL"."BestBeforeDate" = "LOCK"."BestBeforeDate" OR ("TOTAL"."BestBeforeDate"IS NULL AND "LOCK"."BestBeforeDate"IS NULL)) 
				AND ("TOTAL"."ItemTransactionalInfoKey" = "LOCK"."ItemTransactionalInfoKey" OR ("TOTAL"."ItemTransactionalInfoKey"IS NULL 
				AND "LOCK"."ItemTransactionalInfoKey"IS NULL)) 
				) AS FreeStock ON lines."ItemCode" = FreeStock."ItemCode" 
				AND lines."QualityStatusCode" = FreeStock."QualityStatusCode"
				AND lines."WhsCode" = FreeStock."SboWhsCode"
				AND FreeStock."Quantity" < lines."Quantity"
				AND (FreeStock."ItemTransactionalInfoKey" = lines."ItemTransactionalInfoKey" OR (FreeStock."ItemTransactionalInfoKey"IS NULL 
				AND lines."ItemTransactionalInfoKey"IS NULL));

    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50051;
					p_error_message := 'Error for item(s): ' || v_errorList || ' The stock for item, batch and best before date is locked.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;

				--Check free stock for item - detail
				DELETE FROM TMP_TN_CharListTableNoIndex;
	            INSERT INTO TMP_TN_CharListTableNoIndex
	            SELECT CAST(Lines."ItemCode" AS VARCHAR)
				FROM TMP_TN_documentLine AS Lines 
				INNER JOIN
				(
				SELECT "TOTAL"."Quantity" - IFNULL( "LOCK"."Quantity",0) AS "Quantity", "TOTAL"."ItemCode" AS "ItemCode", 
				"TOTAL"."QualityStatusCode" AS "QualityStatusCode", "TOTAL"."InvntryUom" AS "Uom", 
				"TOTAL"."StorLocCode" AS "StorLocCode", "TOTAL"."PmxWhsCode" AS "PmxWhsCode", PMX_OSSL."Sequence" AS "Sequence", 
				PMX_OSSL."IsPickLoc" AS "IsPickLoc", "TOTAL"."InternalKey" AS "InternalKey", "TOTAL"."LogUnitIdentKey" AS "LogUnitIdentKey", 
				CASE WHEN ("TOTAL"."LogUnitIdentKey" IS NULL) THEN 1 ELSE 0 END AS "LogUnitIdentKey_IsNull", "TOTAL"."SSCC" AS "SSCC", 
				CASE WHEN ("TOTAL"."SSCC" IS NULL) THEN 1 ELSE 0 END AS "SSCC_IsNull", "TOTAL"."IsFullPallet" AS "IsFullPallet", 
				"TOTAL"."BatchNumber" AS "BatchNumber", CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber_IsNull", 
				"TOTAL"."BatchNumber" AS "BatchNumber2", CASE WHEN ("TOTAL"."BatchNumber" IS NULL) THEN 1 ELSE 0 END AS "BatchNumber2_IsNull", 
				"TOTAL"."BestBeforeDate" AS "BestBeforeDate", 
				CASE WHEN ("TOTAL"."BestBeforeDate" IS NULL) THEN 1 ELSE 0 END AS "BestBeforeDate_IsNull", 
				"TOTAL"."ItemTransactionalInfoKey" AS "ItemTransactionalInfoKey", 
				CASE WHEN ("TOTAL"."ItemTransactionalInfoKey" IS NULL) THEN 1 ELSE 0 END AS "ItemTransactionalInfoKey_IsNull" ,
				"TOTAL"."SboWhsCode"
				FROM ( 

				SELECT SUM ("PMX_INVT"."Quantity") AS "Quantity", "PMX_INVT"."ItemCode", "OITM"."InvntryUom", 
				"PMX_INVT"."QualityStatusCode", "PMX_INVT"."StorLocCode", PMX_OSEL."PmxWhsCode", "PMX_INVT"."InternalKey", 
				"PMX_INVT"."LogUnitIdentKey", IFNULL( "PMX_LUID"."IsFullPallet",'N') AS "IsFullPallet", 
				"PMX_INVT"."SSCC", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INVT"."ItemTransactionalInfoKey" ,
				PMX_OSWH."SboWhsCode" 
				FROM "PMX_INVT" 
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = "PMX_INVT"."StorLocCode" 
				INNER JOIN PMX_OSWH ON PMX_OSEL."PmxWhsCode" = PMX_OSWH."Code" 
				INNER JOIN "OITM" ON "OITM"."ItemCode" = "PMX_INVT"."ItemCode" 
				LEFT OUTER JOIN "PMX_LUID" ON "PMX_LUID"."InternalKey" = "PMX_INVT"."LogUnitIdentKey" 
				LEFT JOIN "PMX_ITRI" ON "PMX_INVT"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				GROUP BY "PMX_INVT"."ItemCode", "PMX_INVT"."QualityStatusCode", "OITM"."InvntryUom", "PMX_INVT"."StorLocCode", 
				PMX_OSEL."PmxWhsCode", "PMX_INVT"."InternalKey", "PMX_INVT"."LogUnitIdentKey", "PMX_LUID"."IsFullPallet", 
				"PMX_INVT"."SSCC", "PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INVT"."ItemTransactionalInfoKey",
				PMX_OSWH."SboWhsCode" 

				) AS "TOTAL" 
				LEFT OUTER JOIN ( 

				SELECT SUM ("PMX_INLD"."Quantity") AS "Quantity", "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", 
				"PMX_INLD"."StorLocCode", "PMX_INLD"."LogUnitIdentKey", "PMX_INLD"."ItemTransactionalInfoKey", 
				"PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate" , PMX_OSWH."SboWhsCode" 
				FROM "PMX_INLD" INNER JOIN PMX_OSWH ON "PMX_INLD"."PmxWhsCode" = PMX_OSWH."Code"
				LEFT JOIN "PMX_ITRI" ON "PMX_INLD"."ItemTransactionalInfoKey" = "PMX_ITRI"."InternalKey" 
				LEFT JOIN TMP_TN_IgnoreLocking AS IgnoreLocking
				ON IgnoreLocking."DocEntry" = PMX_INLD."BaseEntry"
				AND IgnoreLocking."LineNum" = PMX_INLD."BaseLine"
				AND IgnoreLocking."ObjType" = PMX_INLD."BaseType"
				WHERE IgnoreLocking."DocEntry" IS NULL
				GROUP BY "PMX_INLD"."ItemCode", "PMX_INLD"."QualityStatusCode", "PMX_INLD"."StorLocCode", "PMX_INLD"."LogUnitIdentKey", 
				"PMX_ITRI"."BatchNumber", "PMX_ITRI"."BestBeforeDate", "PMX_INLD"."ItemTransactionalInfoKey", PMX_OSWH."SboWhsCode"

				)AS "LOCK" ON "TOTAL"."ItemCode" = "LOCK"."ItemCode" AND "TOTAL"."QualityStatusCode" = "LOCK"."QualityStatusCode" 
				AND "TOTAL"."SboWhsCode" = "LOCK"."SboWhsCode"
				AND "TOTAL"."StorLocCode" = "LOCK"."StorLocCode" 
				AND ("TOTAL"."LogUnitIdentKey" = "LOCK"."LogUnitIdentKey" OR ("TOTAL"."LogUnitIdentKey"IS NULL AND "LOCK"."LogUnitIdentKey"IS NULL)) 
				AND ("TOTAL"."BatchNumber" = "LOCK"."BatchNumber" OR ("TOTAL"."BatchNumber"IS NULL AND "LOCK"."BatchNumber"IS NULL)) 
				AND ("TOTAL"."BestBeforeDate" = "LOCK"."BestBeforeDate" OR ("TOTAL"."BestBeforeDate"IS NULL AND "LOCK"."BestBeforeDate"IS NULL)) 
				AND ("TOTAL"."ItemTransactionalInfoKey" = "LOCK"."ItemTransactionalInfoKey" OR ("TOTAL"."ItemTransactionalInfoKey"IS NULL 
				AND "LOCK"."ItemTransactionalInfoKey"IS NULL)) 

				INNER JOIN PMX_OSSL ON "TOTAL"."StorLocCode" = PMX_OSSL."Code" 
				) AS FreeStock ON lines."ItemCode" = FreeStock."ItemCode" 
				AND lines."QualityStatusCode" = FreeStock."QualityStatusCode"
				AND lines."WhsCode" = FreeStock."SboWhsCode"
				AND FreeStock."Quantity" < lines."Quantity"
				AND (FreeStock."ItemTransactionalInfoKey" = lines."ItemTransactionalInfoKey" OR (FreeStock."ItemTransactionalInfoKey"IS NULL 
				AND lines."ItemTransactionalInfoKey"IS NULL)) 
				AND (FreeStock."LogUnitIdentKey" = lines."LogUnitIdentKey" OR (FreeStock."LogUnitIdentKey"IS NULL 
				AND lines."LogUnitIdentKey"IS NULL)) 
				AND FreeStock."StorLocCode" = lines."StorLocCode";

    			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
					p_error := 50052;
					p_error_message := 'Error for item(s) ' || v_errorList || ': The stock for item detail is locked.';
					SELECT p_error, p_error_message FROM dummy;
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN;
				END IF;
			
			END IF;


			--**********************************************************************************************************
			--If a document line is indicated that it is a logistic carrier then this logistic carrier is put on the
			--logistic unit.
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd = 1) THEN
				UPDATE PMX_LUID 
				SET "LogisticCarrierCode" = Lines."ItemCode" FROM TMP_TN_documentLine AS Lines INNER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
				WHERE Lines."IsLogisticCarrier" = 'Y';
			END IF;
		END IF;
		

		--**********************************************************************************************************
		--Do checks on goods receipt (purchase delivery note) and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isPurchaseDeliveryDocument = 1) THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			
			--It is not mandatory anymore that the LUID is filled in when doing a purchase delivery
	--		-- Check if LUID is filled in for the inventory items
	--		v_errorList := ''
	--		SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' ' 
	--		FROM TMP_TN_documentLine AS Lines INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
	--		WHERE Items."InvntItem" = 'Y' AND Lines.U_PMX_LUID IS NULL
	--		IF LENGTH(v_errorList) <> 0 THEN
	--			p_error := 50500
	--			p_error_message := 'Error on line(s): ' || v_errorList || ' The LUID must be filled in for inventory items.'
	--			SELECT p_error, p_error_message
	--			--EXEC xp_logevent p_error, p_error_message, ERROR
	--			RETURN
	--		END

			--Items can be stored on every storage location when doing reception
	--		-- Check if Item can be stored on the location
	--		-- Does the storage location exist, and is it active, and it is a reception dock
	--		v_errorList := ''
	--		SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' ' 
	--		FROM TMP_TN_documentLine AS Lines 
	--		LEFT OUTER JOIN PMX_OSSL AS Locations ON Lines."StorLocCode" = Locations."Code"
	--		LEFT OUTER JOIN PMX_OSDO AS Docks ON Docks."Code" = Lines."StorLocCode"
	--		INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
	--		WHERE ((Locations."IsActive" = 'N' OR Locations."IsActive" IS NULL) OR (Docks.IsUnloadingDock <> 'Y' OR Docks.IsUnloadingDock IS NULL)) 
	--		AND Items."InvntItem" = 'Y'
	--		IF LENGTH(v_errorList) <> 0 THEN
	--			p_error := 50513
	--			p_error_message := 'Error on line(s): ' || v_errorList || ' The location is not a dock or is not active.'
	--			SELECT p_error, p_error_message
	--			--EXEC xp_logevent p_error, p_error_message, ERROR
	--			RETURN
	--		END

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd = 1) THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OPDN."ObjType", OPDN."DocEntry", OPDN."DocNum", OPDN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OPDN."CardCode", OPDN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OPDN."UserSign", OPDN."CreateDate", OPDN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  OPDN 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OPDN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;

			--Check if we need to auto reserve the stock
			BEGIN
				DECLARE v_tempParamValue nvarchar(20);
				SELECT MAX(IFNULL( PMX_PAVA."ParameterValue", PMX_EXPA."DefaultValue"))
				INTO v_tempParamValue 
				FROM PMX_PROP 
				INNER JOIN PMX_EXPA ON PMX_EXPA."ExtensionCode" = PMX_PROP."ExtensionCode"
				LEFT JOIN PMX_PAVA ON PMX_PAVA."ParameterCode" = PMX_EXPA."Code"
				WHERE PMX_PROP."Code" = 'PDNGEN'
				AND PMX_EXPA."GeneralParameterCode" = 'PD-ARS';

			IF (v_tempParamValue = 'true' OR v_tempParamValue = 'True')
			THEN

				--insert data into locking table linked to sales order
				INSERT INTO PMX_INLD ( "InternalKey", "BaseType","BaseEntry","BaseLine","CardCode","CardName",
				    "ItemCode","Quantity", "QuantityUom2", "ItemTransactionalInfoKey","SerialNum",
				    "UserSign","CreateDate",
				    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
				    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
				SELECT PMX_INLD_S.nextval, RDR1."ObjType",RDR1."DocEntry",RDR1."LineNum",ORDR."CardCode",ORDR."CardName",
				    InvTable."ItemCode",InvTable."InventoryQuantity",InvTable."QuantityUom2",InvTable."ItemTransactionalInfoKey",InvTable."SerialNum",
				    InvTable."CreatedBy",InvTable."CreateDate",
				    InvTable."CreateTime",InvTable."LogUnitIdentKey", InvTable."QualityStatusCode",
                    'RESERVED', 'L' , PMX_OSEL."PmxWhsCode", 0
				FROM TMP_TN_InventoryDetail AS InvTable
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = InvTable."StorLocCode"
				LEFT  JOIN PCH1 ON PCH1."DocEntry" = InvTable."BaseEntry" AND PCH1."LineNum" = InvTable."BaseLine" AND PCH1."ObjType" = InvTable."BaseType"
				INNER JOIN POR1 ON ( POR1."DocEntry" = InvTable."BaseEntry" AND POR1."LineNum" = InvTable."BaseLine" AND POR1."ObjType" = InvTable."BaseType" )
				    OR             ( POR1."DocEntry" = PCH1."BaseEntry"     AND POR1."LineNum" = PCH1."BaseLine"     AND POR1."ObjType" = PCH1."BaseType" )
				INNER JOIN ORDR ON ORDR."DocEntry" = POR1."BaseEntry" 
				INNER JOIN RDR1 ON RDR1."DocEntry" = POR1."BaseEntry" AND RDR1."LineNum" = POR1."BaseLine" AND RDR1."ObjType" = POR1."BaseType"
				LEFT  JOIN PMX_DOLL ON PMX_DOLL."DocEntry" = POR1."BaseEntry" AND PMX_DOLL."LineNum" = POR1."BaseLine" AND PMX_DOLL."ObjType" = POR1."BaseType"
				WHERE InvTable."BaseEntry" IS NOT NULL AND InvTable."BaseLine" IS NOT NULL AND InvTable."BaseType" IS NOT NULL
				AND RDR1."LineStatus" = 'O'
				AND (PMX_DOLL."DocEntry" IS NULL OR PMX_DOLL."LockedQuantity" = 0);

				--insert data into locking table linked to reserve invoice
				INSERT INTO PMX_INLD ("InternalKey","BaseType","BaseEntry","BaseLine","CardCode","CardName",
				    "ItemCode","Quantity", "QuantityUom2","ItemTransactionalInfoKey","SerialNum",
				    "UserSign","CreateDate",
				    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
				    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
				SELECT PMX_INLD_S.nextval, INV1."ObjType",INV1."DocEntry",INV1."LineNum",ORDR."CardCode",ORDR."CardName",
				    InvTable."ItemCode",InvTable."InventoryQuantity",InvTable."QuantityUom2",InvTable."ItemTransactionalInfoKey",InvTable."SerialNum",
				    InvTable."CreatedBy",InvTable."CreateDate",
				    InvTable."CreateTime",InvTable."LogUnitIdentKey", InvTable."QualityStatusCode",
                    'RESERVED', 'L' , PMX_OSEL."PmxWhsCode", 0
				FROM TMP_TN_InventoryDetail AS InvTable
				INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = InvTable."StorLocCode"
				LEFT  JOIN PCH1 ON PCH1."DocEntry" = InvTable."BaseEntry" AND PCH1."LineNum" = InvTable."BaseLine" AND PCH1."ObjType" = InvTable."BaseType"
				INNER JOIN POR1 ON ( POR1."DocEntry" = InvTable."BaseEntry" AND POR1."LineNum" = InvTable."BaseLine" AND POR1."ObjType" = InvTable."BaseType" )
				    OR             ( POR1."DocEntry" = PCH1."BaseEntry"     AND POR1."LineNum" = PCH1."BaseLine"     AND POR1."ObjType" = PCH1."BaseType" )
				INNER JOIN ORDR ON ORDR."DocEntry" = POR1."BaseEntry" 
				INNER JOIN RDR1 ON RDR1."DocEntry" = POR1."BaseEntry" AND RDR1."LineNum" = POR1."BaseLine" AND RDR1."ObjType" = POR1."BaseType"
				INNER JOIN INV1 ON RDR1."DocEntry" = INV1."BaseEntry" AND RDR1."LineNum" = INV1."BaseLine" AND RDR1."ObjType" = INV1."BaseType"
				INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry" 
				LEFT  JOIN PMX_DOLL ON PMX_DOLL."DocEntry" = INV1."DocEntry" AND PMX_DOLL."LineNum" = INV1."LineNum" AND PMX_DOLL."ObjType" = INV1."ObjType"
				WHERE InvTable."BaseEntry" IS NOT NULL AND InvTable."BaseLine" IS NOT NULL AND InvTable."BaseType" IS NOT NULL
				AND RDR1."LineStatus" = 'C'
				AND (PMX_DOLL."DocEntry" IS NULL OR PMX_DOLL."LockedQuantity" = 0)
				AND OINV."isIns" = 'Y';

			END IF;

			END;




			--Close container (lines) if needed
			UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			FROM TMP_TN_documentLine AS Lines
			INNER JOIN PMX_COLI
			ON Lines."BaseEntry" = PMX_COLI."BaseEntry"
			AND Lines."BaseLine" = PMX_COLI."BaseLine"
			AND Lines."BaseType" = PMX_COLI."BaseType"
			INNER JOIN POR1
			ON Lines."BaseEntry" = POR1."DocEntry"
			AND Lines."BaseLine" = POR1."LineNum"
			AND Lines."BaseType" = POR1."ObjType"
			WHERE PMX_COLI."LineStatus" <> 'C'
			AND POR1."LineStatus" = 'C';


			--Close header is no open lines are found
			UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
			PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = v_tsNow, 
			PMX_COHE."UserSign" = (SELECT MAX("CreatedBy") FROM TMP_TN_documentLine)
			WHERE PMX_COHE."DocStatus" = 'O'
			AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');

		END IF;

		--**********************************************************************************************************
		--Do checks on delivery (sales delivery note) and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isSalesDeliveryDocument = 1) THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			-- Check if the quality statuses on the sales delivery note are statuses that are allowed to be 
			-- to shipped (For sample orders no check is done)
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM TMP_TN_documentLine AS Lines 
			INNER JOIN DLN1 ON Lines."DocEntry" = DLN1."DocEntry"
			AND Lines."LineNum" = DLN1."LineNum"
			INNER JOIN PMX_QYST AS QualityStatus ON Lines."QualityStatusCode" = QualityStatus."Code"
			LEFT JOIN PMX_DLPL ON PMX_DLPL."SalesDeliveryNoteLineDocEntry" = Lines."DocEntry"
			AND PMX_DLPL."SalesDeliveryNoteLineLineNum" = Lines."LineNum"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."DocEntry" = PMX_DLPL."PickListLineDocEntry"
			AND PMX_PLLI."LineNum" = PMX_DLPL."PickListLineLineNum"
			WHERE IFNULL( PMX_PLLI."IsSampleOrder", 'N') = 'N'
			AND NOT ( (IFNULL( DLN1.U_PMX_SQOP, 'RELEASED') = 'RELEASED' AND QualityStatus."CanBeShipped" = 'Y' ) 
			OR (IFNULL( DLN1.U_PMX_SQOP, 'RELEASED') = 'CAN_USE_SUQ' AND (QualityStatus."CanBeShipped" = 'Y' OR QualityStatus."CanBeShippedUnderQuarantine" = 'Y')  )
			OR (IFNULL( DLN1.U_PMX_SQOP, 'RELEASED') = 'MUST_USE_SUQ' AND QualityStatus."CanBeShippedUnderQuarantine" = 'Y'  ));
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50700;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The quality status on these lines is not allowed to be shipped.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--The quantities must be made negative because the stock leaves the company
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd = 1) THEN

				INSERT INTO TMP_TN_InventoryDetail
					SELECT ODLN."ObjType", ODLN."DocEntry", ODLN."DocNum", ODLN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine", 
						  ODLN."CardCode", ODLN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection , Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ODLN."UserSign", ODLN."CreateDate", ODLN."DocTime", PMX_LUID.SSCC,
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  ODLN 
						  INNER JOIN TMP_TN_documentLine AS Lines ON ODLN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';

			END IF;

		END IF;
			
		IF ((v_isSalesDeliveryDocument = 1 OR v_isStockTransfer = 1) AND p_transaction_type = N'P') THEN
			BEGIN --TRY
			
				-- This gets executed after the creation of a sales delivery note when loading a truck
				-- !! This routine does closing of : PickList(line)s, PickList Proposal(line)s, Route(line)s and Waves !!
				-- AND links serial numbers to the correct document.

				--Convert the list of key columns to an integer key
				v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

BEGIN
				--Adjust temp locking: remove delivered quantity
				DECLARE v_tempLockItemCode nvarchar (20);
				DECLARE v_tempLockDestLUID int;
				DECLARE v_tempLockQualityStatusCode nvarchar (8);
				DECLARE v_tempLockItri int;
				DECLARE v_tempLockOriginalLUID int;
				DECLARE v_tempLockDeliveredQuantity numeric(19,6);


				DECLARE CURSOR deliveredStockCursor FOR 
				SELECT     PMX_MVLI."ItemCode", PMX_MVLI."DestLogUnitIdentKey", PMX_MVLI."DestQualityStatusCode", PMX_MVLI."ItemTransactionalInfoKey", 
									  PMX_PLPL."LogUnitIdentKey" AS OriginalLUID, IFNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity") * IFNULL( PMX_DLPL."QuantityPerUom", PMX_MVLI."QuantityPerUom") AS DeliveredQuantity
				FROM         PMX_DLPL INNER JOIN
									  PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry" AND 
									  PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum" INNER JOIN
									  PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry" AND 
									  PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum" INNER JOIN
									  PMX_PLPL ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum"
										AND (PMX_PLLI."BaseType" = N'PMX_PLPH')
				WHERE   (:v_isSalesDeliveryDocument=1 AND PMX_DLPL."SalesDeliveryNoteLineDocEntry" = :v_keyAsInteger) 
				      OR (:v_isStockTransfer=1 AND PMX_DLPL."StockTransferDocEntry" = :v_keyAsInteger);

				-- Reduce temp locking by delivered quantity
				OPEN deliveredStockCursor;
				FETCH deliveredStockCursor INTO v_tempLockItemCode,v_tempLockDestLUID,v_tempLockQualityStatusCode,v_tempLockItri, v_tempLockOriginalLUID, v_tempLockDeliveredQuantity;
				WHILE v_tempLockItemCode IS NOT NULL DO

					DECLARE v_tempLockInternalKey int;
					DECLARE v_tempLockQuantityLocked	 numeric (19,6);
				
					--Get all temp lockings for stock
					DECLARE CURSOR tempLockCursor FOR 
					SELECT     PMX_TINL."InternalKey", PMX_TINL."Quantity"
					FROM         PMX_TINL 
					WHERE     IFNULL( PMX_TINL."ItemTransactionalInfoKey", 0) = IFNULL( :v_tempLockItri,0)
					AND IFNULL( PMX_TINL."LogUnitIdentKey", 0) = IFNULL( :v_tempLockOriginalLUID,0)
					AND PMX_TINL."ItemCode" = :v_tempLockItemCode
					AND PMX_TINL."QualityStatusCode" = :v_tempLockQualityStatusCode;

					OPEN tempLockCursor;
					FETCH tempLockCursor INTO v_tempLockInternalKey, v_tempLockQuantityLocked;
					WHILE v_tempLockInternalKey IS NOT NULL DO
						-- Reduce temp locking
						IF 	v_tempLockDeliveredQuantity > 0 THEN							
							IF (v_tempLockQuantityLocked > v_tempLockDeliveredQuantity) THEN
								--reduce quantity in temp lock
								UPDATE PMX_TINL SET "Quantity" = "Quantity" - v_tempLockDeliveredQuantity, "QuantityUom2" = "QuantityUom2" - (("Quantity" - v_tempLockDeliveredQuantity)* OITM."U_PMX_DQUM")
								FROM PMX_TINL
								INNER JOIN OITM ON OITM."ItemCode" = PMX_TINL."ItemCode"
								WHERE PMX_TINL."InternalKey" = v_tempLockInternalKey;
								v_tempLockDeliveredQuantity := 0;
							
							ELSE
								--we delivered more than what is on the temp lock line, so delete temp lock line and reduce quantity delivered
								v_tempLockDeliveredQuantity := v_tempLockDeliveredQuantity - v_tempLockQuantityLocked;
								DELETE FROM PMX_TINL
								WHERE PMX_TINL."InternalKey" = v_tempLockInternalKey;
							END IF;
						END IF;

						FETCH tempLockCursor INTO v_tempLockInternalKey,v_tempLockQuantityLocked;
					END WHILE;
					CLOSE tempLockCursor;

					FETCH deliveredStockCursor INTO v_tempLockItemCode,v_tempLockDestLUID,v_tempLockQualityStatusCode,v_tempLockItri, v_tempLockOriginalLUID, v_tempLockDeliveredQuantity;
				END WHILE;
				CLOSE deliveredStockCursor;

END;			

				-- Set the IsInterfaced flag on delivery note if any of the BaseDoc ORDR's has the flag
				UPDATE ODLN SET U_PMX_INFC = (
					SELECT CASE WHEN COUNT(ORDR."DocEntry") + COUNT(OINV."DocEntry") > 0 THEN 'Y' ELSE 'N' END 
					FROM DLN1
					LEFT JOIN ORDR ON ORDR."ObjType" = DLN1."BaseType" AND ORDR."DocEntry" = DLN1."BaseEntry" AND ORDR.U_PMX_INFC = 'Y'
					LEFT JOIN OINV ON OINV."ObjType" = DLN1."BaseType" AND OINV."DocEntry" = DLN1."BaseEntry" AND OINV.U_PMX_INFC = 'Y'
					WHERE DLN1."DocEntry" = ODLN."DocEntry" )
				WHERE ODLN."DocEntry" = v_keyAsInteger;
				
				-- Set the IsInterfaced flag on stock transfer if any of the BaseDoc ORDR's has the flag
				UPDATE OWTR SET U_PMX_INFC = (
					SELECT CASE WHEN COUNT(OWTQ."DocEntry") > 0 THEN 'Y' ELSE 'N' END 
					FROM WTR1
					LEFT JOIN OWTQ ON OWTQ."ObjType" = WTR1."BaseType" AND OWTQ."DocEntry" = WTR1."BaseEntry" AND OWTQ.U_PMX_INFC = 'Y'
					WHERE WTR1."DocEntry" = OWTR."DocEntry" )
				WHERE OWTR."DocEntry" = v_keyAsInteger;


--				--Adjust serial number document link (Is now done in code instead of SP)
--				UPDATE PMX_SELD
--				SET PMX_SELD."BaseType" = DLN1."ObjType",
--					PMX_SELD."BaseEntry" = DLN1."DocEntry",
--					PMX_SELD."BaseLine" = DLN1."LineNum",
--					PMX_SELD."Version" = PMX_SELD."Version" + 1
--				FROM DLN1
--				INNER JOIN PMX_DLPL ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
--					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
--				INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
--					AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
--				INNER JOIN PMX_SELD ON PMX_SELD."BaseEntry" = PMX_PLLI."DocEntry"
--					AND PMX_SELD."BaseLine" = PMX_PLLI."LineNum"
--				AND PMX_SELD."BaseType" = 'PMX_PLHE'
--				WHERE DLN1."DocEntry" = v_keyAsInteger
--				AND PMX_PLLI."LineStatus" <> 'C'

				--Adjust serial number in stock (Only for delivery, because stock transfer still has the serial number in stock
				UPDATE PMX_SENU
				SET PMX_SENU."InStock" = 'N',
					PMX_SENU."Version" = PMX_SENU."Version" + 1
				FROM PMX_SENU
				INNER JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
				WHERE PMX_SELD."BaseEntry" = v_keyAsInteger
				AND PMX_SELD."BaseType" = '15';  --Sales delivery



				IF v_isSalesDeliveryDocument = 1 THEN
				
					-- Decrease open quantity of PLLI's with MVLI quantity
					UPDATE PMX_PLLI
					SET PMX_PLLI."OpenQty" = PMX_PLLI."OpenQty" - DeliveredQuantity."Quantity",
						PMX_PLLI."OpenQtyUom2" = PMX_PLLI."OpenQtyUom2" - DeliveredQuantity."QuantityUom2",
						PMX_PLLI."Version" = PMX_PLLI."Version" + 1
					FROM PMX_PLLI
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( IFNULL( PMX_DLPL."QuantityUom2", PMX_MVLI."QuantityUom2")) AS "QuantityUom2", 
						SUM( IFNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity")) AS "Quantity" 
						FROM DLN1
						INNER JOIN PMX_DLPL ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
							AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						LEFT JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE DLN1."DocEntry" = v_keyAsInteger
						AND DLN1."ObjType" = p_object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_PLLI."DocEntry"
						AND DeliveredQuantity."LineNum" = PMX_PLLI."LineNum";
						
					--Decrease lockings
					UPDATE PMX_INLD  
					SET	PMX_INLD."Quantity" = PMX_INLD."Quantity" - DeliveredQuantity."Quantity",
						PMX_INLD."QuantityUom2" = PMX_INLD."QuantityUom2" - DeliveredQuantity."QuantityUom2",
						PMX_INLD."Version" = PMX_INLD."Version" + 1
					FROM PMX_INLD
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( IFNULL( PMX_DLPL."QuantityUom2" * PMX_DLPL."QuantityPerUom", PMX_MVLI."QuantityUom2"* PMX_MVLI."QuantityPerUom")) AS "QuantityUom2", 
						SUM( IFNULL( PMX_DLPL."Quantity" * PMX_DLPL."QuantityPerUom" , PMX_MVLI."Quantity" * PMX_MVLI."QuantityPerUom")) AS "Quantity" 
						FROM DLN1
						INNER JOIN PMX_DLPL ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
							AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						INNER JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE DLN1."DocEntry" = v_keyAsInteger
						AND DLN1."ObjType" = p_object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_INLD."BaseEntry"
						AND DeliveredQuantity."LineNum" = PMX_INLD."BaseLine"
						AND PMX_INLD."BaseType" = 'PMX_PLHE';
												
				ELSEIF v_isStockTransfer = 1 THEN
				
					-- Decrease open quantity of PLLI's with MVLI quantity
					UPDATE PMX_PLLI
					SET PMX_PLLI."OpenQty" = PMX_PLLI."OpenQty" - DeliveredQuantity."Quantity",
						PMX_PLLI."OpenQtyUom2" = PMX_PLLI."OpenQtyUom2" - DeliveredQuantity."QuantityUom2",
						PMX_PLLI."Version" = PMX_PLLI."Version" + 1
					FROM PMX_PLLI
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( IFNULL( PMX_DLPL."QuantityUom2", PMX_MVLI."QuantityUom2")) AS "QuantityUom2", 
						SUM( IFNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity")) AS "Quantity" 
						FROM WTR1
						INNER JOIN PMX_DLPL ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
							AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						LEFT JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE WTR1."DocEntry" = v_keyAsInteger
						AND WTR1."ObjType" = p_object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_PLLI."DocEntry"
						AND DeliveredQuantity."LineNum" = PMX_PLLI."LineNum";
						
					--Decrease lockings
					UPDATE PMX_INLD  
					SET	PMX_INLD."Quantity" = PMX_INLD."Quantity" - DeliveredQuantity."Quantity",
						PMX_INLD."QuantityUom2" = PMX_INLD."QuantityUom2" - DeliveredQuantity."QuantityUom2",
						PMX_INLD."Version" = PMX_INLD."Version" + 1
					FROM PMX_INLD
					INNER JOIN 
						(SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum", 
						SUM( IFNULL( PMX_DLPL."QuantityUom2" * PMX_DLPL."QuantityPerUom", PMX_MVLI."QuantityUom2"* PMX_MVLI."QuantityPerUom")) AS "QuantityUom2", 
						SUM( IFNULL( PMX_DLPL."Quantity" * PMX_DLPL."QuantityPerUom" , PMX_MVLI."Quantity" * PMX_MVLI."QuantityPerUom")) AS "Quantity" 
						FROM WTR1
						INNER JOIN PMX_DLPL ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
							AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
						INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
							AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
						INNER JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
							AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
						WHERE WTR1."DocEntry" = v_keyAsInteger
						AND WTR1."ObjType" = p_object_type
						AND PMX_PLLI."LineStatus" <> 'C'
						GROUP BY PMX_PLLI."DocEntry", PMX_PLLI."LineNum") AS DeliveredQuantity
					ON DeliveredQuantity."DocEntry" = PMX_INLD."BaseEntry"
						AND DeliveredQuantity."LineNum" = PMX_INLD."BaseLine"
						AND PMX_INLD."BaseType" = 'PMX_PLHE';
					
				END IF;
								
				-- Get the keys for the PLLI's to be closed
				DELETE FROM TMP_TN_SDLines;
				INSERT INTO TMP_TN_SDLines
				SELECT PMX_PLLI."DocEntry", PMX_PLLI."LineNum"
				FROM PMX_DLPL
				INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
					AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
				WHERE (( v_isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = v_keyAsInteger ) 
				      OR ( v_isStockTransfer = 1 AND WTR1."DocEntry" = v_keyAsInteger ))
				AND PMX_PLLI."LineStatus" <> 'C'
				--AND ( PMX_PLLI."OpenQty" - DLN1."Quantity" ) < 0.000001
				AND ( PMX_PLLI."OpenQty") < 0.000001;
			

				
				-- Delete inventory locking for these picklist lines
				DELETE FROM PMX_INLD
				WHERE "InternalKey" IN (
					SELECT PMX_INLD."InternalKey"
					FROM TMP_TN_SDLines
					INNER JOIN PMX_INLD ON 'PMX_PLHE' = PMX_INLD."BaseType"
					AND TmpDocEntry = PMX_INLD."BaseEntry"
					AND TmpLineNum = PMX_INLD."BaseLine"
				);

				-- Set link status serial number links
				UPDATE PMX_SELD SET "LinkStatus" = 'C'
				FROM TMP_TN_SDLines
				INNER JOIN PMX_SELD ON 'PMX_PLHE' = PMX_SELD."BaseType"
					AND TmpDocEntry = PMX_SELD."BaseEntry"
					AND TmpLineNum = PMX_SELD."BaseLine";



				--Update PMX_DOLL in case of partial delivery
begin
				DECLARE v_proposalQuantity2  numeric(19, 6);
				DECLARE v_proposalQuantityUOM22  numeric(19, 6);
				DECLARE v_proposalBaseEntry2  int;
				DECLARE v_proposalBaseLineNum2  int;
				DECLARE v_proposalBatchNum2  NVARCHAR(36);
				DECLARE v_proposalBaseType2  NVARCHAR(20);
				DECLARE v_proposalEntry2  int;
				DECLARE v_proposalLineNum2  int;


				DECLARE CURSOR lockCursor2 FOR 

--				SELECT PMX_PLLI."Quantity", PMX_PLPL."BaseEntry", PMX_PLPL."BaseLine", PMX_PLPL."BaseType", 
--				PMX_PLPL."QuantityUom2", "BatchNumber", PMX_PLPL."DocEntry", PMX_PLPL."LineNum"				
--				FROM PMX_PLPL 
--				INNER JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry"
--				AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum"
--				AND PMX_PLLI."LineStatus" = 'O'
--				INNER JOIN PMX_DLPL ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
--				AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
--				LEFT JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
--										
--				WHERE PMX_PLLI."DocEntry" IN (SELECT TmpDocEntry  FROM TMP_TN_SDLines ) 
--				AND PMX_PLLI."LineStatus" = 'O'

				SELECT IFNULL( PMX_DLPL."Quantity", PMX_MVLI."Quantity"), PMX_PLPL."BaseEntry", PMX_PLPL."BaseLine", PMX_PLPL."BaseType", 
					IFNULL( PMX_DLPL."QuantityUom2", PMX_MVLI."QuantityUom2"), "BatchNumber", PMX_PLPL."DocEntry", PMX_PLPL."LineNum"	
				FROM PMX_DLPL
				INNER JOIN PMX_PLLI ON PMX_DLPL."PickListLineDocEntry" = PMX_PLLI."DocEntry"
					AND PMX_DLPL."PickListLineLineNum" = PMX_PLLI."LineNum"
				INNER JOIN PMX_MVLI ON PMX_DLPL."MoveLineDocEntry" = PMX_MVLI."DocEntry"
					AND PMX_DLPL."MoveLineLineNum" = PMX_MVLI."LineNum"
				INNER JOIN PMX_PLpl ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry"
				AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum"
				AND PMX_PLLI."LineStatus" = 'O'
				LEFT JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"
				WHERE PMX_PLLI."LineStatus" <> 'C' AND
				 (( :v_isSalesDeliveryDocument=1 AND DLN1."DocEntry" = :v_keyAsInteger ) 
				      OR ( :v_isStockTransfer=1 AND WTR1."DocEntry" = :v_keyAsInteger ));

				OPEN lockCursor2;
				FETCH lockCursor2 INTO v_proposalQuantity2,v_proposalBaseEntry2,v_proposalBaseLineNum2,
					v_proposalBaseType2, v_proposalQuantityUOM22, v_proposalBatchNum2, v_proposalEntry2, v_proposalLineNum2;
				WHILE v_proposalEntry2 IS NOT NULL DO
					-- Reduce document locking for all lines
					UPDATE PMX_DOLL 
					SET PMX_DOLL."LockedQuantity" = PMX_DOLL."LockedQuantity" - v_proposalQuantity2,
					PMX_DOLL."LockedQuantityUom2" = PMX_DOLL."LockedQuantityUom2" - v_proposalQuantityUOM22,
					PMX_DOLL."UpdateDate" = v_updateDate,
					PMX_DOLL."UpdateTime" = v_updateTime,
					PMX_DOLL."Version" = PMX_DOLL."Version" + 1
					WHERE "DocEntry" = v_proposalBaseEntry2 AND "LineNum" = v_proposalBaseLineNum2
					AND "ObjType" = v_proposalBaseType2 
					AND ("BatchNum" = v_proposalBatchNum2 OR "BatchNum" IS NULL);
					-- AND v_proposalBaseType = '17' -- 2010-08-26  KrisM  Allow all types

					UPDATE PMX_PLPL
					SET PMX_PLPL."OpenQty" = PMX_PLPL."OpenQty" - v_proposalQuantity2,
					PMX_PLPL."Version" = PMX_PLPL."Version" + 1
					WHERE "DocEntry" = v_proposalEntry2 AND "LineNum" = v_proposalLineNum2;

					FETCH lockCursor2 INTO v_proposalQuantity2,v_proposalBaseEntry2,v_proposalBaseLineNum2,
						v_proposalBaseType2, v_proposalQuantityUOM22, v_proposalBatchNum2, v_proposalEntry2, v_proposalLineNum2;
				END WHILE;
				CLOSE lockCursor2;

END;				

				-- Delete document locking for lines that have NO remaining "LockedQuantity"
				DELETE FROM PMX_DOLL
				WHERE PMX_DOLL."LockedQuantity" < 0.000001;



				-- Close picklist lines
				-- {PmxDocumentProvider.CloseLine(...)}
				-- {PmxPickListProvider.CloseLine(...)}
				UPDATE PMX_PLLI
				SET PMX_PLLI."OpenQty" = 0,
					PMX_PLLI."OpenQtyUom2" = PMX_PLLI."OpenQtyUom2" * 0,  -- Set to 0.0 except when NULL
					PMX_PLLI."PickListLineStatus" = CASE PMX_PLLI."PickListLineStatus" WHEN 'F' THEN 'F' ELSE 'C' END,  -- Set to 'C' except when 'F'
					PMX_PLLI."LineStatus" = 'C',
					PMX_PLLI."Version" = PMX_PLLI."Version" + 1
				FROM TMP_TN_SDLines
				INNER JOIN PMX_PLLI ON TmpDocEntry = PMX_PLLI."DocEntry"
					AND TmpLineNum = PMX_PLLI."LineNum";

				-- Empty TMP_TN_SDLines
				DELETE FROM TMP_TN_SDLines;
				
				-- Fill TMP_TN_SDLines with all DocEntries for non-closed picklists with all lines closed
				-- Using TmpLineNum = NULL
				INSERT INTO TMP_TN_SDLines
				SELECT DISTINCT PMX_DLPL."PickListLineDocEntry", -1
				FROM PMX_DLPL
				INNER JOIN PMX_PLHE ON PMX_DLPL."PickListLineDocEntry" = PMX_PLHE."DocEntry"
				LEFT JOIN PMX_PLLI ON PMX_PLHE."DocEntry" = PMX_PLLI."DocEntry"
					AND PMX_PLLI."LineStatus" <> 'C'
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"

				WHERE (( v_isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = v_keyAsInteger ) 
				      OR ( v_isStockTransfer = 1 AND WTR1."DocEntry" = v_keyAsInteger ))
				AND PMX_PLHE."DocStatus" <> 'C'
				AND PMX_PLLI."DocEntry" IS NULL;

				SELECT COUNT(*) INTO v_cnt FROM TMP_TN_SDLines;					
				IF v_cnt > 0  -- Found picklists to close ?
				THEN

					-- Close the picklists who have all lines closed
					UPDATE PMX_PLHE
					SET PMX_PLHE."PickListStatus" = CASE PMX_PLHE."PickListStatus" WHEN 'F' THEN 'F' ELSE 'C' END,  -- Set to 'C' except when 'F'
						PMX_PLHE."DocStatus" = 'C',
						PMX_PLHE."UpdateDate" = v_updateDate,
						PMX_PLHE."UpdateTime" = v_updateTime,
						PMX_PLHE."Version" = PMX_PLHE."Version" + 1
					WHERE PMX_PLHE."DocEntry" IN ( SELECT TmpDocEntry  FROM TMP_TN_SDLines );

					-- Close the wave if the wave only has closed picklists
					UPDATE PMX_WAVE
					SET PMX_WAVE."DocStatus" = 'C',
						PMX_WAVE."UpdateDate" = v_updateDate,
						PMX_WAVE."UpdateTime" = v_updateTime,
						PMX_WAVE."Version" = PMX_WAVE."Version" + 1
					FROM PMX_PLHE ref
					INNER JOIN PMX_WAVE ON ref."WaveKey" = PMX_WAVE."InternalKey"
					LEFT JOIN PMX_PLHE tst ON PMX_WAVE."InternalKey" = tst."WaveKey"
						AND tst."DocStatus" <> 'C'
					WHERE ref."DocEntry" IN ( SELECT TmpDocEntry  FROM TMP_TN_SDLines )
					AND PMX_WAVE."DocStatus" <> 'C'
					AND tst."DocEntry" IS NULL;
				
					-- Fill TMP_TN_SDLines with all the lines of non-closed picklist proposals with all picklists closed
					-- (PickLists proposals to be closed)
					INSERT INTO TMP_TN_SDLines
					SELECT DISTINCT PMX_PLPH."DocEntry", 1
					FROM PMX_PLHE ref
					INNER JOIN PMX_PLPH ON ref."PickListProposalEntry" = PMX_PLPH."DocEntry"
					LEFT JOIN PMX_PLHE tst ON PMX_PLPH."DocEntry" = tst."PickListProposalEntry"
						AND tst."DocStatus" <> 'C'
					WHERE ref."DocEntry" IN ( SELECT TmpDocEntry  FROM TMP_TN_SDLines  WHERE TmpLineNum = -1 )
					AND PMX_PLPH."DocStatus" <> 'C'
					AND tst."DocEntry" IS NULL;

					SELECT COUNT(*) INTO v_cnt FROM TMP_TN_SDLines WHERE TmpLineNum = 1;
					IF v_cnt > 0  -- Found picklist proposal(line)s to close ?
					THEN

						-- Remove PickList entries from TMP_TN_SDLines so only PickList proposal lines remain
						DELETE FROM TMP_TN_SDLines  WHERE TmpLineNum = -1;

						-- Delete inventory locking for the picklist proposal lines
						--  do not care about "LineNum", all lines need to be cleaned up
						DELETE FROM PMX_INLD WHERE "InternalKey" IN (
							SELECT PMX_INLD."InternalKey"
							FROM TMP_TN_SDLines
							INNER JOIN PMX_INLD ON 'PMX_PLPH' = PMX_INLD."BaseType"
								AND TmpDocEntry = PMX_INLD."BaseEntry"
						);
						
begin
						DECLARE v_proposalQuantity  numeric(19, 6);
						DECLARE v_proposalQuantityUOM2  numeric(19, 6);
						DECLARE v_proposalBaseEntry  INT;
						DECLARE v_proposalBaseLineNum  INT;
						DECLARE v_proposalBatchNum  NVARCHAR(36);
						DECLARE v_proposalBaseType  NVARCHAR(20);

						DECLARE CURSOR lockCursor FOR SELECT "OpenQty", "BaseEntry", "BaseLine", "BaseType", "QuantityUom2", "BatchNumber" 
						FROM PMX_PLPL LEFT JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
						WHERE "DocEntry" IN (SELECT TmpDocEntry  FROM TMP_TN_SDLines) AND "LineStatus" = 'O';

						OPEN lockCursor;
						FETCH lockCursor INTO v_proposalQuantity,v_proposalBaseEntry,v_proposalBaseLineNum,v_proposalBaseType, v_proposalQuantityUOM2, v_proposalBatchNum;
						WHILE v_proposalQuantity IS NOT NULL DO
							-- Reduce document locking for all lines
							UPDATE PMX_DOLL 
							SET PMX_DOLL."LockedQuantity" = PMX_DOLL."LockedQuantity" - v_proposalQuantity,
							PMX_DOLL."LockedQuantityUom2" = PMX_DOLL."LockedQuantityUom2" - v_proposalQuantityUOM2,
							PMX_DOLL."UpdateDate" = v_updateDate,
							PMX_DOLL."UpdateTime" = v_updateTime,
							PMX_DOLL."Version" = PMX_DOLL."Version" + 1
							WHERE "DocEntry" = v_proposalBaseEntry AND "LineNum" = v_proposalBaseLineNum 
							AND "ObjType" = v_proposalBaseType 
							AND ("BatchNum" = v_proposalBatchNum OR "BatchNum" IS NULL);
							-- AND v_proposalBaseType = '17' -- 2010-08-26  KrisM  Allow all types

							FETCH lockCursor INTO v_proposalQuantity,v_proposalBaseEntry,v_proposalBaseLineNum,v_proposalBaseType, v_proposalQuantityUOM2, v_proposalBatchNum;
						END WHILE;
						CLOSE lockCursor;

END;


						-- Delete document locking for lines that have NO remaining "LockedQuantity"
						DELETE FROM PMX_DOLL
						WHERE PMX_DOLL."LockedQuantity" < 0.000001;
						
						-- Close all proposal lines
						-- {PmxDocumentProvider.CloseLine(...)}
						UPDATE PMX_PLPL
						SET PMX_PLPL."OpenQty" = 0,
							PMX_PLPL."OpenQtyUom2" = PMX_PLPL."OpenQtyUom2" * 0,  -- Set to 0.0 except when NULL
							PMX_PLPL."LineStatus" = 'C',
							PMX_PLPL."Version" = PMX_PLPL."Version" + 1
						WHERE PMX_PLPL."LineStatus" <> 'C'
						AND PMX_PLPL."DocEntry" IN ( SELECT TmpDocEntry FROM TMP_TN_SDLines );

						--Set back the temp lockings to the locking table
						--The LUID is set to NULL because it is impossible to retrieve the new luid
						INSERT INTO PMX_INLD ( "InternalKey","Canceled", "UserSign", "CreateDate", "CreateTime", "UpdateDate", "UpdateTime", "Version", "BaseType", "BaseEntry", "BaseLine", "ItemCode", "Quantity", 
											  "SerialNum", "LogUnitIdentKey", "InvLockStatusCode", "QualityStatusCode", "StorLocCode", "InvLockLevel", "Uom2", "QuantityUom2", "ItemTransactionalInfoKey", 
											  "PmxWhsCode", "CardCode", "CardName")

						SELECT PMX_INLD_S.nextval, 'N', PMX_TINL."InternalKey", PMX_TINL."CreateDate",
						PMX_TINL."CreateTime", v_updateDate, v_updateTime, 0, PMX_TINL."BaseType", PMX_TINL."BaseEntry", PMX_TINL."BaseLine", PMX_TINL."ItemCode", PMX_TINL."Quantity", 
						PMX_TINL."SerialNum", NULL, PMX_TINL."InvLockStatusCode", 
						PMX_TINL."QualityStatusCode", NULL, 'B', NULL, NULL, PMX_TINL."ItemTransactionalInfoKey", 
						PMX_TINL."PmxWhsCode", PMX_TINL."CardCode", PMX_TINL."CardName"

						FROM PMX_TINL 
						WHERE PMX_TINL."LockedByType" = 'PMX_PLPH'
						AND PMX_TINL."LockedByEntry" IN (SELECT TmpDocEntry FROM TMP_TN_SDLines );


						--Remove temp lockings from table
						DELETE FROM PMX_TINL
						WHERE PMX_TINL."LockedByType" = 'PMX_PLPH'
						AND PMX_TINL."LockedByEntry" IN (SELECT TmpDocEntry FROM TMP_TN_SDLines );


						
						-- Close the proposals
						UPDATE PMX_PLPH
						SET PMX_PLPH."DocStatus" = 'C',
							PMX_PLPH."UpdateDate" = v_updateDate,
							PMX_PLPH."UpdateTime" = v_updateTime,
							PMX_PLPH."Version" = PMX_PLPH."Version" + 1
						WHERE PMX_PLPH."DocStatus" <> 'C'
						AND PMX_PLPH."DocEntry" IN ( SELECT TmpDocEntry FROM TMP_TN_SDLines );
						
						-- Close the route lines if all picklists are closed for that route
						UPDATE PMX_RTLI
						SET PMX_RTLI."LineStatus" = 'C',
							PMX_RTLI."Version" = PMX_RTLI."Version" + 1
						FROM TMP_TN_SDLines
						INNER JOIN PMX_PLPH ref ON TmpDocEntry = ref."DocEntry"
						INNER JOIN PMX_RTHE ON ref."RouteLineDocEntry" = PMX_RTHE."DocEntry"
						INNER JOIN PMX_RTLI ON PMX_RTHE."DocEntry" = PMX_RTLI."DocEntry"
						LEFT JOIN PMX_PLPH tst ON PMX_RTLI."DocEntry" = tst."RouteLineDocEntry"
							AND tst."DocStatus" <> 'C'
						WHERE PMX_RTLI."LineStatus" <> 'C'
						AND tst."DocEntry" IS NULL;

						-- Close the routes if all route lines are closed
						UPDATE PMX_RTHE
						SET PMX_RTHE."RouteStatus" = 'C',
							PMX_RTHE."DocStatus" = 'C',
							PMX_RTHE."UpdateDate" = v_updateDate,
							PMX_RTHE."UpdateTime" = v_updateTime,
							PMX_RTHE."Version" = PMX_RTHE."Version" + 1
						FROM TMP_TN_SDLines
						INNER JOIN PMX_PLPH ref ON TmpDocEntry = ref."DocEntry"
						INNER JOIN PMX_RTHE ON ref."RouteLineDocEntry" = PMX_RTHE."DocEntry"
						LEFT JOIN PMX_RTLI ON PMX_RTHE."DocEntry" = PMX_RTLI."DocEntry"
							AND PMX_RTLI."LineStatus" <> 'C'
						WHERE PMX_RTHE."DocStatus" <> 'C'
						AND PMX_RTLI."DocEntry" IS NULL;

						--Close container (lines) if needed
						UPDATE PMX_COLI SET "LineStatus" = 'C', PMX_COLI."Version" = PMX_COLI."Version" + 1 
						FROM PMX_COLI
						INNER JOIN PMX_RTHE
						ON PMX_COLI."BaseEntry" = PMX_RTHE."DocEntry"
						AND PMX_COLI."BaseType" = 'PMX_RTHE'
						WHERE PMX_COLI."LineStatus" <> 'C'
						AND PMX_RTHE."DocStatus" = 'C';



					END IF;  -- Found picklist proposals to close ?
					
				END IF;  -- Found picklists to close ?
				

				--If pick list is not closed, change status to 'Partially delivered'
				UPDATE PMX_PLHE SET "PickListStatus" = 'E'
				FROM PMX_DLPL
				INNER JOIN PMX_PLHE ON PMX_DLPL."PickListLineDocEntry" = PMX_PLHE."DocEntry"								
				LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
					AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
				LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
					AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"

				WHERE (( v_isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = v_keyAsInteger ) 
				      OR ( v_isStockTransfer = 1 AND WTR1."DocEntry" = v_keyAsInteger ))
				AND PMX_PLHE."DocStatus" <> 'C'
				--Only set to partially delivered if there are no lines of status lower than packed
				AND PMX_PLHE."DocEntry" NOT IN (
				
					SELECT PMX_PLLI."DocEntry" 
					FROM PMX_PLLI WHERE "DocEntry" IN (
						SELECT PMX_PLHE."DocEntry"
						FROM PMX_DLPL
						INNER JOIN PMX_PLHE ON PMX_DLPL."PickListLineDocEntry" = PMX_PLHE."DocEntry"
						INNER JOIN PMX_PLLI ON PMX_PLLI."DocEntry" = PMX_PLHE."DocEntry"
						AND PMX_PLLI."LineNum" = PMX_DLPL."PickListLineLineNum"								
						LEFT JOIN DLN1 ON DLN1."DocEntry" = PMX_DLPL."SalesDeliveryNoteLineDocEntry"
							AND DLN1."LineNum" = PMX_DLPL."SalesDeliveryNoteLineLineNum"
						LEFT JOIN WTR1 ON WTR1."DocEntry" = PMX_DLPL."StockTransferDocEntry"
							AND WTR1."LineNum" = PMX_DLPL."StockTransferLineNum"

						WHERE (( v_isSalesDeliveryDocument = 1 AND DLN1."DocEntry" = v_keyAsInteger )
							  OR ( v_isStockTransfer = 1 AND WTR1."DocEntry" = v_keyAsInteger ))
						AND PMX_PLHE."DocStatus" <> 'C'
					)
					AND PMX_PLLI."PickListLineStatus" IN ('N', 'R', 'P', 'T', 'I')
				);
			
				
			END; /* TRY
			BEGIN CATCH
				p_error := 60004
				p_error_message := 'Unhandled error update Produmex After shipping: Number:' || CAST(ERROR_NUMBER() AS NCLOB) || ' Message: ' || ERROR_MESSAGE()
				SELECT p_error, p_error_message
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN
			END CATCH;
*/
			--END IF;

			--UPDATE LOCKING TABLE
			DELETE FROM PMX_INLD WHERE "InternalKey" IN ( SELECT "InternalKey" FROM PMX_INLD LOCKING INNER JOIN ORDR ON ORDR."DocEntry" = LOCKING."BaseEntry"
			AND ORDR."ObjType" = LOCKING."BaseType"
			WHERE ORDR."DocStatus" = 'C' );

			DELETE FROM PMX_INLD WHERE "InternalKey" IN ( SELECT "InternalKey" FROM PMX_INLD LOCKING INNER JOIN OINV ON OINV."DocEntry" = LOCKING."BaseEntry"
			AND OINV."ObjType" = LOCKING."BaseType"
			WHERE OINV."InvntSttus" = 'C' AND OINV."isIns" = 'Y');
			
			DELETE FROM PMX_INLD WHERE "InternalKey" IN ( SELECT "InternalKey" FROM PMX_INLD LOCKING INNER JOIN OWTQ ON OWTQ."DocEntry" = LOCKING."BaseEntry"
			AND OWTQ."ObjType" = LOCKING."BaseType"
			WHERE OWTQ."DocStatus" = 'C');

			--Close container (lines) if needed
			UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			FROM TMP_TN_documentLine AS Lines
			INNER JOIN PMX_COLI
			ON Lines."BaseEntry" = PMX_COLI."BaseEntry"
			AND Lines."BaseLine" = PMX_COLI."BaseLine"
			AND Lines."BaseType" = PMX_COLI."BaseType"
			INNER JOIN RDR1
			ON Lines."BaseEntry" = RDR1."DocEntry"
			AND Lines."BaseLine" = RDR1."LineNum"
			AND Lines."BaseType" = RDR1."ObjType"
			WHERE PMX_COLI."LineStatus" <> 'C'
			AND RDR1."LineStatus" = 'C';

			--Close header is no open lines are found
			UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
			PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = v_tsNow, 
			PMX_COHE."UserSign" = (SELECT MAX("CreatedBy") FROM TMP_TN_documentLine )
			WHERE PMX_COHE."DocStatus" = 'O'
			AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O');


		END IF;
		

		--**********************************************************************************************************
		--Do checks on inventory entry and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isInventoryEntry = 1) THEN

		--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
			
			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			
			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd = 1) THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OIGN."ObjType", OIGN."DocEntry", OIGN."DocNum", OIGN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OIGN."CardCode", OIGN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OIGN."UserSign", OIGN."CreateDate", OIGN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  OIGN 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OIGN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on inventory exit and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isInventoryExit = 1) THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--The quantities must be made negative because the stock leaves the company
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd = 1) THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OIGE."ObjType", OIGE."DocEntry", OIGE."DocNum", OIGE."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OIGE."CardCode", OIGE."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OIGE."UserSign", OIGE."CreateDate", OIGE."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  OIGE 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OIGE."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on sales return and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isSalesReturnDocument=1) THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--50800

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd=1) THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT ORDN."ObjType", ORDN."DocEntry", ORDN."DocNum", ORDN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORDN."CardCode", ORDN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORDN."UserSign", ORDN."CreateDate", ORDN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  ORDN 
						  INNER JOIN TMP_TN_documentLine AS Lines ON ORDN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on purchase return and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isPurchaseReturnDocument = 1) THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************


			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--The quantities must be made negative because the stock leaves the company
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd = 1) THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT ORPD."ObjType", ORPD."DocEntry", ORPD."DocNum", ORPD."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORPD."CardCode", ORPD."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORPD."UserSign", ORPD."CreateDate", ORPD."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  ORPD 
						  INNER JOIN TMP_TN_documentLine AS Lines ON ORPD."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on sales invoice and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isSalesInvoice = 1) THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

			--The base document cannot have picklist(proposals)
			--We have to do the check here because the close of the sales order by sales invoice is not triggered!!
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(PMX_PLPL."DocEntry" AS VARCHAR)
			FROM ORDR 
			INNER JOIN PMX_PLPL ON ORDR."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = CAST(ORDR."ObjType" AS nvarchar(20))
			INNER JOIN RDR1 ON ORDR."DocEntry" = RDR1."DocEntry" AND RDR1."LineNum" = PMX_PLPL."BaseLine" 
			INNER JOIN INV1 ON INV1."BaseEntry" = RDR1."DocEntry" AND INV1."BaseType" = RDR1."ObjType" AND INV1."BaseLine" = RDR1."LineNum"
			INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry"
			WHERE OINV."DocEntry" = v_keyAsInteger AND (ORDR."DocStatus" = 'C' OR RDR1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0
			AND OINV."isIns" = 'Y';
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51000;
				p_error_message := 'The sales invoice is linked to a sales order with open pick list proposal(s): ' || v_errorList || '. The pick list proposal(s) must be closed first before adding the invoice.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			
			--Check if the sales order is closed that there are no open pick list proposals of this sales order
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(PMX_PLPL."DocEntry" AS VARCHAR)
			FROM OINV 
			INNER JOIN PMX_PLPL ON OINV."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = CAST(OINV."ObjType" AS nvarchar(20))
			INNER JOIN INV1 ON OINV."DocEntry" = INV1."DocEntry" AND INV1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE OINV."DocEntry" = v_keyAsInteger 
			AND ((OINV."InvntSttus" = 'C' AND OINV."isIns" = 'Y') OR INV1."LineStatus" = 'C' ) 
			AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0;
		    SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51001;
				p_error_message := 'The sales invoice ''' || p_list_of_cols_val_tab_del || ''' has open pick list proposal(s): ' || v_errorList || '. The pick list proposal(s) must be closed first before closing the sales invoice (line).';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;


			--Check if sales order lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the sales order line is lower than on the open pick list proposal line / open pick list line
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CASE WHEN        INV1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) || ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN INV1 ON PMX_PLPL."BaseEntry" = INV1."DocEntry" AND PMX_PLPL."BaseLine" = INV1."LineNum" AND PMX_PLPL."BaseType" = CAST(INV1."ObjType" AS nvarchar(20))
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = '13' AND PMX_PLPL."BaseEntry" = v_keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR IFNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
		    SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51002;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' The item code on a sales invoice line cannot be changed if a picklist proposal is generated for it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
		
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CASE WHEN IFNULL( INV1."Quantity",0) < PMX_PLPL."Quantity" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) || ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN INV1 ON PMX_PLPL."BaseEntry" = INV1."DocEntry" AND PMX_PLPL."BaseLine" = INV1."LineNum" AND PMX_PLPL."BaseType" = CAST(INV1."ObjType" AS nvarchar(20))
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = '13' AND PMX_PLPL."BaseEntry" = v_keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR IFNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51003;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' The quantity on a sales invoice line cannot be lowered if a picklist proposal is generated for it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM INV1 AS Lines
			INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = SalesInvoice."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"

			WHERE SalesInvoice."DocEntry" = v_keyAsInteger AND Lines."LineStatus" = 'O' AND Lines."U_PMX_AMBA" = 'N'
			GROUP BY Lines."LineNum"
			HAVING Count(*) > 1;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51004;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Due date minutes
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(ORDR.U_PMX_DDTM AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = v_keyAsInteger
			AND (IFNULL( ORDR.U_PMX_DDTM, 0) > 59 OR
				IFNULL( ORDR.U_PMX_DDTM, 0) < 0);
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51005;
				p_error_message := 'The due date minutes: ' || v_errorList || ' are not valid for document ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Due date Hour
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(ORDR.U_PMX_DDTH AS VARCHAR) 
			FROM ORDR
			WHERE ORDR."DocEntry" = v_keyAsInteger
			AND ( IFNULL( ORDR.U_PMX_DDTH, 0) < 0
					OR ( IFNULL( ORDR.U_PMX_DDTH, 0) > 23 AND IFNULL( ORDR.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR ( IFNULL( ORDR.U_PMX_DDTH, 0) > 12 AND IFNULL( ORDR.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51006;
				p_error_message := 'The due date hours: ' || v_errorList || ' are not valid for document ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			IF (v_isTransactionTypeAdd=1) THEN
				INSERT INTO PMX_INLD ("InternalKey","BaseType","BaseEntry","BaseLine","CardCode","CardName",
				    "ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
				    "UserSign","CreateDate",
				    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
				    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
				SELECT PMX_INLD_S.nextval, INV1."ObjType",INV1."DocEntry",INV1."LineNum",OINV."CardCode",OINV."CardName",	PMX_INLD."ItemCode", 
				CASE WHEN INV1."Quantity" > PMX_INLD."Quantity" THEN PMX_INLD."Quantity" ELSE INV1."Quantity" END,
				PMX_INLD."ItemTransactionalInfoKey",PMX_INLD."SerialNum",
				OINV."UserSign", v_updateDate, v_updateTime,
				PMX_INLD."LogUnitIdentKey", PMX_INLD."QualityStatusCode",
				 'RESERVED', PMX_INLD."InvLockLevel", PMX_INLD."PmxWhsCode", 0
				FROM PMX_INLD
				INNER JOIN INV1 ON CAST(INV1."BaseType" AS nvarchar(20)) = PMX_INLD."BaseType" AND INV1."BaseEntry" = PMX_INLD."BaseEntry" AND INV1."BaseLine" = PMX_INLD."BaseLine"
				INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry"
				WHERE OINV."DocEntry" = v_keyAsInteger AND OINV."isIns" = 'Y'
				AND PMX_INLD."BaseEntry" = INV1."BaseEntry"
				AND PMX_INLD."BaseType" = CAST(INV1."BaseType" AS nvarchar(20))
				AND PMX_INLD."BaseLine" = INV1."BaseLine";


				--Update the locking of the sales order: adjust quantity
				UPDATE PMX_INLD SET PMX_INLD."Quantity" = PMX_INLD."Quantity" - INV1."Quantity"
				FROM PMX_INLD
				INNER JOIN INV1 ON CAST(INV1."BaseType" AS nvarchar(20)) = PMX_INLD."BaseType" AND INV1."BaseEntry" = PMX_INLD."BaseEntry" AND INV1."BaseLine" = PMX_INLD."BaseLine"
				INNER JOIN OINV ON OINV."DocEntry" = INV1."DocEntry"
				WHERE OINV."DocEntry" = v_keyAsInteger AND OINV."isIns" = 'Y'
				AND PMX_INLD."BaseEntry" = INV1."BaseEntry"
				AND PMX_INLD."BaseType" = CAST(INV1."BaseType" AS nvarchar(20))
				AND PMX_INLD."BaseLine" = INV1."BaseLine";


				DELETE FROM PMX_INLD WHERE "Quantity" <= 0;
			

				--**********************************************************************************************************
				--START: Add/update locking based on batch numbers
				--**********************************************************************************************************

				SELECT COUNT(*) INTO v_cnt FROM OINV WHERE OINV."DocEntry" = v_keyAsInteger AND OINV."isIns" = 'Y';
				IF v_cnt > 0
				THEN

					--Start with deleting the lockings that are not correct
					DELETE FROM PMX_INLD
					WHERE PMX_INLD."InternalKey" NOT IN 
					(--Query to get the correct lockings
						SELECT 	PMX_INLD."InternalKey"
						FROM INV1 AS Lines
						INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
						INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						LEFT JOIN PMX_ITRI ON
						 IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( "DistNumber", '')
						--AND IFNULL( PMX_ITRI."BestBeforeDate", 1) = IFNULL( "ExpDate", 1)

						LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
						AND PMX_INLD."BaseEntry" = Lines."DocEntry"
						AND PMX_INLD."BaseLine" = Lines."LineNum"
						AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

						WHERE OITL."ApplyEntry" = v_keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2

						AND PMX_INLD."InternalKey" IS NOT NULL


						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
						Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey", PMX_INLD."InvLockLevel"

						HAVING SUM("AllocQty") > 0



						--Exclude manual lockings on LUID level when no batch has been selected on sales invoice
						UNION SELECT PMX_INLD."InternalKey"
						FROM PMX_INLD 
						LEFT JOIN
						(
							SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
							FROM INV1 AS Lines
											INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
											LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
											LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
											LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
											AND ITL1."SysNumber" = OBTN."SysNumber"

							WHERE OITL."ApplyEntry"  = v_keyAsInteger 
											AND Lines."LineStatus" = 'O'
											AND IFNULL( "StockEff", 2) = 2

								GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
											Lines."NumPerMsr", "DistNumber", "ExpDate"

							HAVING SUM(IFNULL( "AllocQty", 0)) > 0
						) AS OINV

						ON OINV."LineNum" = PMX_INLD."BaseLine"
						AND OINV."DocEntry" = PMX_INLD."BaseEntry"
						AND OINV."ObjType" = PMX_INLD."BaseType"


						WHERE PMX_INLD."InvLockLevel" = 'L'
						AND PMX_INLD."BaseEntry" = v_keyAsInteger
						AND OINV."DocEntry" IS NULL

						

					) 

					AND PMX_INLD."BaseType" = '13'
					AND PMX_INLD."BaseEntry" = v_keyAsInteger;



					BEGIN --TRY

						DECLARE CURSOR salesInvoiceBatchCursor FOR 
								SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", 
								Lines."ObjType", Lines."ItemCode", PMX_OSWH."Code", CASE WHEN OITM."U_PMX_ICAW" = 'Y' THEN OITM."U_PMX_DQUM" ELSE NULL END AS "CatchWeightRatio", Lines."U_PMX_SQOP"
								FROM INV1 AS Lines
								INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
								INNER JOIN OINV AS SalesInvoice ON Lines."DocEntry" = SalesInvoice."DocEntry"
								INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
								INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
								INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
								AND ITL1."SysNumber" = OBTN."SysNumber"

								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = Lines."WhsCode"

								LEFT JOIN PMX_PLPL
								ON PMX_PLPL."BaseEntry" = Lines."DocEntry"
								AND PMX_PLPL."BaseLine" = lines."LineNum"
								AND PMX_PLPL."BaseType" = Lines."ObjType"

								WHERE OITL."ApplyEntry" = :v_keyAsInteger 
								AND Lines."LineStatus" = 'O'
								AND "StockEff" = 2
								AND PMX_PLPL."DocEntry" IS NULL
								GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code", OITM."U_PMX_ICAW", OITM."U_PMX_DQUM", Lines."U_PMX_SQOP"

								HAVING SUM("AllocQty") > 0;


						OPEN salesInvoiceBatchCursor;
						FETCH salesInvoiceBatchCursor 
						INTO v_batchQuantityToLock,v_batchToLock,v_expDateToLock,v_lineNumToLock,v_docEntryToLock,v_typeToLock,v_itemCodeToLock,v_pmxWarehouseToLock, v_catchWeightRatioToLock, v_shippingQualityOptionToLock;
						WHILE v_docEntryToLock IS NOT NULL DO

							--Check for each batch if we have a corresponding locking line
							SELECT COUNT(*) INTO v_cnt FROM PMX_INLD 
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = v_typeToLock
								AND PMX_INLD."BaseEntry" = v_docEntryToLock
								AND PMX_INLD."BaseLine" = v_lineNumToLock
								AND PMX_ITRI."BatchNumber" = v_batchToLock;
								--AND IFNULL( PMX_ITRI."BestBeforeDate", '') = IFNULL( v_expDateToLock, '');
							IF v_cnt < 1
							THEN

								--Entry does not exist, so we need to lock the stock
								--First try to get the itri key to lock
								v_remainingQuantityToLock := v_batchQuantityToLock;

								BEGIN --TRY
									DECLARE CURSOR itriSalesInvoiceCursor FOR SELECT PMX_ITRI."InternalKey", IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (
											SELECT "Code" FROM PMX_QYST 
											WHERE  (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
											) 
										GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (
											SELECT "Code" FROM PMX_QYST 
											WHERE  (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'CAN_USE_SUQ' AND ("CanBeShipped" = 'Y' OR "CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'MUST_USE_SUQ' AND ("CanBeShippedUnderQuarantine" = 'Y') )
											OR (IFNULL(:v_shippingQualityOptionToLock, 'RELEASED') = 'RELEASED' AND ("CanBeShipped" = 'Y') )
											) 
										GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = :v_batchToLock AND PMX_ITRI."ItemCode" = :v_itemCodeToLock
										AND Inventory."PmxWhsCode" = :v_pmxWarehouseToLock
										AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0;


									OPEN itriSalesInvoiceCursor;
									FETCH itriSalesInvoiceCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
									WHILE v_itriToLock IS NOT NULL AND v_remainingQuantityToLock > 0 DO

										IF v_remainingQuantityToLock < v_freeQuantityToLock THEN
											v_currentQuantityToLock := v_remainingQuantityToLock;
										ELSE
											v_currentQuantityToLock := v_freeQuantityToLock;
										END IF;

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ("InternalKey","BaseType","BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
										SELECT PMX_INLD_S.nextval, * FROM (
										SELECT v_typeToLock,v_docEntryToLock,v_lineNumToLock,MAX(OINV."CardCode"),MAX(OINV."CardName"),
											v_itemCodeToLock,v_currentQuantityToLock,v_itriToLock,NULL,
											MAX(OINV."UserSign"),v_updateDate,
											v_updateTime,NULL, v_qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, v_currentQuantityToLock * v_catchWeightRatioToLock
										FROM OINV 
										INNER JOIN INV1 ON INV1."DocEntry" = OINV."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = INV1."WhsCode"
										WHERE OINV."DocEntry" = v_docEntryToLock
										AND INV1."LineNum" = v_lineNumToLock
										);

										--Reduce remaining
										v_remainingQuantityToLock := v_remainingQuantityToLock - v_currentQuantityToLock;


										FETCH itriSalesInvoiceCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
									END WHILE;
									CLOSE itriSalesInvoiceCursor;

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(v_remainingQuantityToLock > 0)
									THEN
										p_error := 51050;
										p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock || '. Missing qty: ' || v_remainingQuantityToLock;
										SELECT p_error, p_error_message FROM dummy;
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN;
									END IF;

								END; /* TRY
								BEGIN CATCH
									p_error := 51090
									p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
									|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE itriSalesInvoiceCursor 
										DEALLOCATE itriSalesInvoiceCursor
										SELECT p_error, p_error_message
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE itriSalesInvoiceCursor
										SELECT p_error, p_error_message
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN
									END CATCH
								END CATCH*/

							ELSE 

								--The line exists, so now we need to check if the quantity is the same/more/less
								SELECT SUM(PMX_INLD."Quantity") -  v_batchQuantityToLock INTO v_remainingQuantityToLock
									FROM PMX_INLD 
									INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
									WHERE PMX_INLD."BaseType" = v_typeToLock
									AND PMX_INLD."BaseEntry" = v_docEntryToLock
									AND PMX_INLD."BaseLine" = v_lineNumToLock
									AND PMX_ITRI."BatchNumber" = v_batchToLock
									GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine";
									
												
								IF v_remainingQuantityToLock <> 0 THEN
								--Only if deviation, then we try to update/insert
								--Loop tru all locking lines
									BEGIN -- TRY
										DECLARE CURSOR inldCursor FOR SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey" FROM PMX_INLD 
										INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
										WHERE PMX_INLD."BaseType" = :v_typeToLock
										AND PMX_INLD."BaseEntry" = :v_docEntryToLock
										AND PMX_INLD."BaseLine" = :v_lineNumToLock
										AND PMX_ITRI."BatchNumber" = :v_batchToLock;

										OPEN inldCursor;
										FETCH inldCursor INTO v_currentLockInternalKey, v_lockedQuantity, v_itriToLock;
										WHILE v_currentLockInternalKey IS NOT NULL AND v_remainingQuantityToLock <> 0 DO

											IF v_remainingQuantityToLock > 0 THEN
												--Remaining is more than 0, so we can just remove the locking
												IF v_remainingQuantityToLock >= v_lockedQuantity THEN
													--Delete
													DELETE FROM PMX_INLD WHERE "InternalKey" = v_currentLockInternalKey;
													v_remainingQuantityToLock := v_remainingQuantityToLock - v_lockedQuantity;
												ELSE
													--update
													UPDATE PMX_INLD SET "Quantity" = "Quantity" - v_remainingQuantityToLock, "QuantityUom2" = "QuantityUom2" - (v_remainingQuantityToLock * v_catchWeightRatioToLock) WHERE "InternalKey" = v_currentLockInternalKey;
													v_remainingQuantityToLock := 0;
												END IF;
											ELSE 
												--Remaining is less than 0, so we need to add locking
												--If there is still free stock for current itri, then take that one
												SELECT IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) 
												INTO v_freeQuantityToLock
												FROM PMX_ITRI 
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
												INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
												LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
												WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
												AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
												WHERE PMX_ITRI."BatchNumber" = v_batchToLock AND PMX_ITRI."ItemCode" = v_itemCodeToLock
												AND Inventory."PmxWhsCode" = v_pmxWarehouseToLock
												AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0
												AND PMX_ITRI."InternalKey" = v_itriToLock;

												IF v_remainingQuantityToLock > v_freeQuantityToLock THEN
													v_currentQuantityToLock := v_freeQuantityToLock;
												ELSE
													v_currentQuantityToLock := v_remainingQuantityToLock;
												END IF;

												UPDATE PMX_INLD SET "Quantity" = "Quantity" + v_currentQuantityToLock, "QuantityUom2" = "QuantityUom2" + (v_currentQuantityToLock * v_catchWeightRatioToLock) WHERE PMX_INLD."InternalKey" = v_currentLockInternalKey;

												-- add qty, because v_remainingQuantityToLock is negative
												v_remainingQuantityToLock := v_remainingQuantityToLock + v_currentQuantityToLock;
												
											END IF;

											FETCH inldCursor INTO v_currentLockInternalKey, v_lockedQuantity, v_itriToLock;
										END WHILE;
										CLOSE inldCursor;

									END; /* TRY
									BEGIN CATCH
										p_error := 51091
										p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
										|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
										BEGIN TRY
											CLOSE inldCursor 
											DEALLOCATE inldCursor
											SELECT p_error, p_error_message
											--EXEC xp_logevent p_error, p_error_message, ERROR
											RETURN
										END TRY
										BEGIN CATCH
											DEALLOCATE inldCursor
											SELECT p_error, p_error_message
											--EXEC xp_logevent p_error, p_error_message, ERROR
											RETURN
										END CATCH
									END CATCH*/

									IF v_remainingQuantityToLock <> 0 THEN
										-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
										--Set v_remainingQuantityToLock as positive quanitty
										v_remainingQuantityToLock := ABS( v_remainingQuantityToLock );
										

										BEGIN --TRY
											DECLARE CURSOR itriSalesInvoiceCursor FOR
                                            SELECT PMX_ITRI."InternalKey", IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
											LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
											INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
											WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
											LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
											WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
											AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
											WHERE PMX_ITRI."BatchNumber" = :v_batchToLock AND PMX_ITRI."ItemCode" = :v_itemCodeToLock
											AND Inventory."PmxWhsCode" = :v_pmxWarehouseToLock
											AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0;


											OPEN itriSalesInvoiceCursor;
											FETCH itriSalesInvoiceCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
											WHILE v_itriToLock IS NOT NULL AND v_remainingQuantityToLock > 0 DO

												IF(v_remainingQuantityToLock < v_freeQuantityToLock) THEN
													v_currentQuantityToLock := v_remainingQuantityToLock;
												ELSE
													v_currentQuantityToLock := v_freeQuantityToLock;
												END IF;

												--Insert INTO the PMX_INLD table

												INSERT INTO PMX_INLD ("InternalKey","BaseType","BaseEntry","BaseLine","CardCode","CardName",
													"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
													"UserSign","CreateDate",
													"CreateTime","LogUnitIdentKey","QualityStatusCode", 
													"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version", "QuantityUom2") 
												SELECT PMX_INLD_S.nextval, * FROM (
												SELECT v_typeToLock,v_docEntryToLock,v_lineNumToLock,MAX(OINV."CardCode"),MAX(OINV."CardName"),
													v_itemCodeToLock,v_currentQuantityToLock,v_itriToLock,NULL,
													MAX(OINV."UserSign"),v_updateDate,
													v_updateTime,NULL, v_qualityStatusToLock,
													'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0, v_currentQuantityToLock * v_catchWeightRatioToLock
												FROM OINV 
												INNER JOIN INV1 ON INV1."DocEntry" = OINV."DocEntry"
												INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = INV1."WhsCode"
												WHERE OINV."DocEntry" = v_docEntryToLock
												AND INV1."LineNum" = v_lineNumToLock
												);

												--Reduce remaining
												v_remainingQuantityToLock := v_remainingQuantityToLock - v_currentQuantityToLock;

												FETCH itriSalesInvoiceCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock ;
											END WHILE;
											CLOSE itriSalesInvoiceCursor;

											--If there is remaining QTY, there is not enough free stock, so throw error
											IF(v_remainingQuantityToLock > 0) THEN
												p_error := 51052;
												p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock|| '. Missing qty: ' || v_remainingQuantityToLock;
												SELECT p_error, p_error_message FROM dummy;
												--EXEC xp_logevent p_error, p_error_message, ERROR
												RETURN;
											END IF;

										END; /* TRY
										BEGIN CATCH
											p_error := 51092
											p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
											|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
											BEGIN TRY
												CLOSE itriSalesInvoiceCursor 
												DEALLOCATE itriSalesInvoiceCursor
												SELECT p_error, p_error_message
												--EXEC xp_logevent p_error, p_error_message, ERROR
												RETURN
											END TRY
											BEGIN CATCH
												DEALLOCATE itriSalesInvoiceCursor
												SELECT p_error, p_error_message
												--EXEC xp_logevent p_error, p_error_message, ERROR
												RETURN
											END CATCH
										END CATCH*/

									END IF;

								END IF;

								IF v_remainingQuantityToLock <> 0 THEN
								    -- if quantity is still different from 0, then there was not enough stock to reserve!
									CLOSE salesInvoiceBatchCursor;
									p_error := 51053;
									p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock|| '. Missing qty: ' || v_remainingQuantityToLock;
									SELECT p_error, p_error_message FROM dummy;
									--EXEC xp_logevent p_error, p_error_message, ERROR
									RETURN;
								END IF;

							END IF;

							FETCH salesInvoiceBatchCursor
							INTO v_batchQuantityToLock,v_batchToLock,v_expDateToLock,v_lineNumToLock,v_docEntryToLock,v_typeToLock,v_itemCodeToLock, v_pmxWarehouseToLock, v_catchWeightRatioToLock, v_shippingQualityOptionToLock;
						END WHILE;
						CLOSE salesInvoiceBatchCursor;
						
					END; /* TRY
					BEGIN CATCH
						p_error := 50699
						p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
						|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
						BEGIN TRY
							CLOSE salesInvoiceBatchCursor 
							DEALLOCATE salesInvoiceBatchCursor
							SELECT p_error, p_error_message
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN
						END TRY
						BEGIN CATCH
							DEALLOCATE salesInvoiceBatchCursor
							SELECT p_error, p_error_message
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN
						END CATCH
					END CATCH*/
				END IF;
				--**********************************************************************************************************
				--END: Add/update locking based on batch numbers
				--**********************************************************************************************************
			END IF;





			

			--UPDATE LOCKING TABLE
			DELETE FROM PMX_INLD WHERE "InternalKey" IN ( SELECT "InternalKey" 
                FROM PMX_INLD LOCKING
                INNER JOIN OINV ON OINV."DocEntry" = LOCKING."BaseEntry"
				AND CAST(OINV."ObjType" AS nvarchar(20)) = LOCKING."BaseType"
				WHERE OINV."InvntSttus" = 'C' AND OINV."isIns" = 'Y'
				AND OINV."DocEntry" = v_keyAsInteger);


			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF v_isTransactionTypeAdd = 1 THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OINV."ObjType", OINV."DocEntry", OINV."DocNum", OINV."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OINV."CardCode", OINV."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OINV."UserSign", OINV."CreateDate", OINV."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  OINV 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OINV."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on purchase invoice and add lines to detail inventory
		--**********************************************************************************************************
		IF v_isPurchaseInvoice = 1 THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd=1) THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OPCH."ObjType", OPCH."DocEntry", OPCH."DocNum", OPCH."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OPCH."CardCode", OPCH."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OPCH."UserSign", OPCH."CreateDate", OPCH."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  OPCH 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OPCH."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on purchase credit note and add lines to detail inventory
		--**********************************************************************************************************
		IF v_isPurchaseCreditNote = 1 THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF v_isTransactionTypeAdd = 1 THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT ORPC."ObjType", ORPC."DocEntry", ORPC."DocNum", ORPC."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORPC."CardCode", ORPC."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORPC."UserSign", ORPC."CreateDate", ORPC."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  ORPC 
						  INNER JOIN TMP_TN_documentLine AS Lines ON ORPC."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;

		--**********************************************************************************************************
		--Do checks on sales credit note and add lines to detail inventory
		--**********************************************************************************************************
		IF v_isSalesCreditNote = 1 THEN

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

			--The base document cannot have picklist(proposals)
			--We have to do the check here because the close of the sales invoice by sales credit note is not triggered!!
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(PMX_PLPL."DocEntry" AS VARCHAR)
			FROM OINV 
			INNER JOIN PMX_PLPL ON OINV."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = OINV."ObjType"
			INNER JOIN INV1 ON OINV."DocEntry" = INV1."DocEntry" AND INV1."LineNum" = PMX_PLPL."BaseLine" 
			INNER JOIN RIN1 ON RIN1."BaseEntry" = INV1."DocEntry" AND RIN1."BaseType" = INV1."ObjType" AND RIN1."BaseLine" = INV1."LineNum"
			INNER JOIN ORIN ON ORIN."DocEntry" = RIN1."DocEntry"
			WHERE ORIN."DocEntry" = v_keyAsInteger AND (ORIN."DocStatus" = 'C' OR RIN1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0
			AND OINV."isIns" = 'Y';
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51100;
				p_error_message := 'The sales credit note is linked to a sales invoice with open pick list proposal(s): ' || v_errorList || '. The pick list proposal(s) must be closed first before adding the invoice.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE: Delete locks for closed reserve invoices
			DELETE FROM PMX_INLD WHERE "InternalKey" IN ( SELECT "InternalKey" 
                FROM PMX_INLD LOCKING
                INNER JOIN OINV ON OINV."DocEntry" = LOCKING."BaseEntry"
				AND OINV."ObjType" = LOCKING."BaseType"
				WHERE OINV."DocStatus" = 'C' AND OINV."isIns" = 'Y' );


			--**********************************************************************************************************
			--Insert inventory detail lines in memory table (these are only the inventory items lines)
			--**********************************************************************************************************
			IF v_isTransactionTypeAdd = 1 THEN
				INSERT INTO TMP_TN_InventoryDetail
					SELECT ORIN."ObjType", ORIN."DocEntry", ORIN."DocNum", ORIN."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  ORIN."CardCode", ORIN."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity" * v_stockDirection, Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", ORIN."UserSign", ORIN."CreateDate", ORIN."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2" * v_stockDirection, Lines."InventoryQuantity"
					FROM  ORIN 
						  INNER JOIN TMP_TN_documentLine AS Lines ON ORIN."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y';
			END IF;
		END IF;
		
		
		END;

		--**********************************************************************************************************
		--Do checks for move and add lines to detail inventory
		--If a move line is indicated that it is a logistic carrier then this logistic carrier is put on the
		--logistic unit.
		--**********************************************************************************************************
		IF v_isMoveDoc = 1 AND v_isTransactionTypeAdd = 1 THEN
			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************
			--"ItemCode" filled in and correct
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines LEFT JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" AND Items."InvntItem" = 'Y'
			WHERE Items."ItemCode" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50100;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The item code does not exist, is not filled in or is not an inventory item.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

	--		--"BatchNumber" filled in if needed
	--		v_errorList := ''
	--		SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' ' 
	--		FROM PMX_MVLI AS Lines INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
	--		LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
	--		WHERE PMX_ITRI."BatchNumber" IS NULL AND Items."ManBtchNum" = 'Y' AND Lines."DocEntry" = v_keyAsInteger
	--		IF LENGTH(v_errorList) <> 0 THEN
	--			p_error := 50101
	--			p_error_message := 'Error on line(s): ' || v_errorList || ' The batchnumber must be filled in for item.'
	--			SELECT p_error, p_error_message
	--			--EXEC xp_logevent p_error, p_error_message, ERROR
	--			RETURN
	--		END
	--
	--		-- Check if best before date is filled in for expirable goods
	--		v_errorList := ''
	--		SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' ' 
	--		FROM PMX_MVLI AS Lines INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" 
	--		LEFT JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = Lines."ItemTransactionalInfoKey"
	--		WHERE PMX_ITRI."BestBeforeDate" IS NULL AND Items.U_PMX_HBBD = 'Y' AND Lines."DocEntry" = v_keyAsInteger
	--		IF LENGTH(v_errorList) <> 0 THEN
	--			p_error := 50102
	--			p_error_message := 'Error on line(s): ' || v_errorList || ' The best before date must be filled in for expirable item.'
	--			SELECT p_error, p_error_message
	--			--EXEC xp_logevent p_error, p_error_message, ERROR
	--			RETURN
	--		END

			--SrcStorLocCode filled in and correct?
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSEL AS Locations ON Lines."SrcStorLocCode" = Locations."Code" AND Locations."IsStorageLocation" = 'Y'
			WHERE Locations."Code" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50103;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The source storage location does not exist or is not filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--"DestStorLocCode" filled in and correct?
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSEL AS Locations ON Lines."DestStorLocCode" = Locations."Code" AND Locations."IsStorageLocation" = 'Y'
			WHERE Locations."Code" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50104;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The destination storage location does not exist or is not filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--SrcLogUnitIdentKey correct
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_LUID AS LogicalUnitID ON Lines."SrcLogUnitIdentKey" = LogicalUnitID."InternalKey" 
			WHERE Lines."SrcLogUnitIdentKey" IS NOT NULL AND LogicalUnitID."InternalKey" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50105;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The source logistic unit identification does not exist.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--DestLogUnitIdentKey correct
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines
			LEFT JOIN PMX_LUID AS LogicalUnitID ON Lines."DestLogUnitIdentKey" = LogicalUnitID."InternalKey" 
			WHERE Lines."DestLogUnitIdentKey" IS NOT NULL AND LogicalUnitID."InternalKey" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50106;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The destination logistic unit identification does not exist.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--SrcQualityStatus filled in and correct
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_QYST AS QualityStatus ON Lines."SrcQualityStatusCode" = QualityStatus."Code" 
			WHERE QualityStatus."Code" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50107;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The entered source quality status does not exist or is not filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--DestQualityStatus filled in and correct
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines
			LEFT JOIN PMX_QYST AS QualityStatus ON Lines."DestQualityStatusCode" = QualityStatus."Code" 
			WHERE QualityStatus."Code" IS NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50108;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The entered destination quality status does not exist or is not filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- If a line is set as logisic carrier then the LUID must be filled in
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Lines."DestLogUnitIdentKey" IS NULL AND Lines."DocEntry" = v_keyAsInteger ;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50109;
				p_error_message := 'Error on line(s): ' || v_errorList || ' If the flag is set that this line is the logistic carrier of the logistic unit then the logistic unit ID must be filled in.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if logistic carrier is filled in on the logistic unit and that there are no lines
			-- indicated as logistic carrier.
			-- Also check that not 2 logistic carriers are defined on the same logistic unit
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."DestLogUnitIdentKey" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			 INNER JOIN PMX_LUID AS LogisticUnitID ON Lines."DestLogUnitIdentKey" = LogisticUnitID."InternalKey"
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Lines."DocEntry" = v_keyAsInteger
			GROUP BY lines."DocEntry",lines."DestLogUnitIdentKey",LogisticUnitID."LogisticCarrierCode"
			HAVING (count(lines."DocEntry") > 1) or (count(lines."DocEntry") = 1 AND  LogisticUnitID."LogisticCarrierCode" IS NOT NULL);
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50110;
				p_error_message := 'Error for logistic unit(s): ' || v_errorList || ' There is already a logistic carrier defined or there is more then 1 line is defined as logistic carrier for these logistic unit(s).';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- If a line is set as logisic carrier then the item must be a logistic carrier
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines
			INNER JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode"
			WHERE Lines."IsLogisticCarrier" = 'Y' AND Items.U_PMX_LOCA <> 'Y' AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50111;
				p_error_message := 'Error on line(s): ' || v_errorList || ' If the flag is set that this line is the logistic carrier of the logistic unit then the item must be a logistic carrier.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- If a line is set as logisic carrier then its quantity must be 1
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			WHERE Lines."DocEntry" = v_keyAsInteger AND Lines."IsLogisticCarrier" = 'Y' AND ((Lines."Quantity"*Lines."QuantityPerUom") <> 1);
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50112;
				p_error_message := 'Error on line(s): ' || v_errorList || ' If a line is a logistic carrier then its quantity must be 1.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Check if the warehouse of the source storage location is the same
			-- warehouse as the destination.
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
				 INNER JOIN PMX_OSEL AS OSE ON Lines."DestStorLocCode" = OSE."Code"
				 INNER JOIN PMX_OSEL AS OSE2 ON Lines."SrcStorLocCode" = OSE2."Code"
			WHERE Lines."DocEntry" = v_keyAsInteger AND OSE."PmxWhsCode" <> OSE2."PmxWhsCode";
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50113;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The source storage location code is of a different warehouse than the destination storage location.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--"SrcStorLocCode" locked?
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines
			LEFT JOIN PMX_OSSL AS Locations ON Lines."SrcStorLocCode" = Locations."Code"
			WHERE Locations."LockedBy" IS NOT NULL AND Lines."DocEntry" = v_keyAsInteger
			AND IFNULL( Lines."BaseType", '') <> 'PMX_CYCO';
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50114;
				p_error_message := 'Error on line(s): ' || v_errorList || ' The source storage location is locked.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--"DestStorLocCode" locked?
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
			LEFT JOIN PMX_OSSL AS Locations ON Lines."DestStorLocCode" = Locations."Code"
			WHERE Locations."LockedBy" IS NOT NULL AND Lines."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50115;
				p_error_message := 'Error on line(s) ' || v_errorList || ': The destination storage location is locked.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--"Quantity" uom2 filled in for catch weight?
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines LEFT JOIN OITM AS Items ON Lines."ItemCode" = Items."ItemCode" AND Items."InvntItem" = 'Y'
			WHERE Items.U_PMX_ICAW = 'Y' AND Lines."DocEntry" = v_keyAsInteger
			AND Lines."QuantityUom2" IS NULL;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50116;
				p_error_message := 'Error on line(s) ' || v_errorList || ': The quantity for "Uom2" needs to be filled in for catch weight items.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

            -- Check if best before date is filled in for expirable goods
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM PMX_MVLI AS Lines
			INNER JOIN WOR1 ON Lines."BaseEntry" = WOR1."DocEntry"
			AND Lines."BaseLine" = WOR1."LineNum" 
			AND Lines."BaseType" = N'202'
			INNER JOIN OWOR ON WOR1."DocEntry" = OWOR."DocEntry"
			INNER JOIN PMX_ITRI ON Lines."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			WHERE CASE WHEN "WOR1"."U_PMX_BBDO" = N'P' THEN CURRENT_TIMESTAMP WHEN "WOR1"."U_PMX_BBDO" = N'B' THEN IFNULL( "OWOR"."U_PMX_BBDT", "OWOR"."DueDate" ) ELSE "OWOR"."DueDate" END > PMX_ITRI."BestBeforeDate"
			AND WOR1."DocEntry" = v_keyAsInteger;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50118;
				p_error_message := 'Error on line(s): ' || v_errorList || ' Cannot move stock for production order. Invalid calculation for BestBeforeDate option.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			--Check if location matches quality status
			DELETE FROM TMP_TN_CharListTableNoIndex;
	        INSERT INTO TMP_TN_CharListTableNoIndex
	        SELECT CAST(Lines."DestStorLocCode" AS VARCHAR) || '-' || CAST(Lines."DestQualityStatusCode" AS VARCHAR)
			FROM PMX_MVLI AS Lines 
		    LEFT JOIN PMX_OSSL AS OSE ON Lines."DestStorLocCode" = OSE."Code"
			WHERE IFNULL(OSE."QualityStatusCode", Lines."DestQualityStatusCode") <> Lines."DestQualityStatusCode"
			AND Lines."DocEntry" = v_keyAsInteger ;
			SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 50119;
				p_error_message := 'Quality status does not match fixed quality status on location: ' || v_errorList;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;


			--**********************************************************************************************************
			--If a move line is indicated that it is a logistic carrier then this logistic carrier is put on the
			--logistic unit.
			--**********************************************************************************************************
			UPDATE PMX_LUID 
			SET "LogisticCarrierCode" = Lines."ItemCode" FROM PMX_MVLI AS Lines INNER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."DestLogUnitIdentKey"
			WHERE Lines."IsLogisticCarrier" = 'Y'AND Lines."DocEntry" = v_keyAsInteger;

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table
			--**********************************************************************************************************
			IF v_isTransactionTypeAdd = 1 THEN
				--One move line represents 2 lines in the detail inventory table:
				-- *The destination as positive lines
				-- *The source as negative lines

				--Insert source lines -> negative
				INSERT INTO TMP_TN_InventoryDetail
					SELECT PMX_MVHE."ObjType", PMX_MVHE."DocEntry", PMX_MVHE."DocEntry", PMX_MVHE."CreateDate", PMX_MVLI."LineNum", PMX_MVLI."BaseType", PMX_MVLI."BaseEntry", PMX_MVLI."BaseLine",
						  NULL, NULL, PMX_MVLI."ItemCode", 
						  SUBSTRING(PMX_MVLI."Dscription", 0, 100), - PMX_MVLI."Quantity", PMX_OSSL."Code", PMX_MVLI."ItemTransactionalInfoKey", 
						  NULL AS Serial, PMX_MVLI."SrcLogUnitIdentKey", PMX_MVHE."UserSign", PMX_MVHE."CreateDate", PMX_MVHE."CreateTime", 
						  PMX_LUID.SSCC, PMX_MVLI."SrcQualityStatusCode", /*Use inventory "Uom"!*/IFNULL( OUOM."UomCode", OITM."InvntryUom"), IFNULL( OUOM."UomCode", OITM."InvntryUom"), PMX_MVLI."QuantityPerUom", PMX_MVLI."Uom2", -PMX_MVLI."QuantityUom2", -(PMX_MVLI."Quantity"*PMX_MVLI."QuantityPerUom")
					FROM  PMX_MVHE INNER JOIN
						  PMX_MVLI ON PMX_MVHE."DocEntry" = PMX_MVLI."DocEntry" INNER JOIN
						  PMX_OSSL ON PMX_MVLI."SrcStorLocCode" = PMX_OSSL."Code" LEFT OUTER JOIN
						  PMX_LUID ON PMX_LUID."InternalKey" = PMX_MVLI."SrcLogUnitIdentKey"
						  INNER JOIN OITM ON PMX_MVLI."ItemCode" = OITM."ItemCode"
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE PMX_MVHE."DocEntry" = v_keyAsInteger;

				--Insert destination lines -> positive
				INSERT INTO TMP_TN_InventoryDetail
					SELECT PMX_MVHE."ObjType", PMX_MVHE."DocEntry", PMX_MVHE."DocEntry", PMX_MVHE."CreateDate", PMX_MVLI."LineNum", PMX_MVLI."BaseType", PMX_MVLI."BaseEntry", PMX_MVLI."BaseLine",
						  NULL, NULL, PMX_MVLI."ItemCode", 
						  SUBSTRING(PMX_MVLI."Dscription", 0, 100), PMX_MVLI."Quantity", PMX_OSSL."Code", PMX_MVLI."ItemTransactionalInfoKey", 
						  NULL AS Serial, PMX_MVLI."DestLogUnitIdentKey", PMX_MVHE."UserSign", PMX_MVHE."CreateDate", PMX_MVHE."CreateTime", 
						  PMX_LUID.SSCC, PMX_MVLI."DestQualityStatusCode", /*Use inventory "Uom"!*/IFNULL( OUOM."UomCode", OITM."InvntryUom"), IFNULL( OUOM."UomCode", OITM."InvntryUom"), PMX_MVLI."QuantityPerUom", PMX_MVLI."Uom2", PMX_MVLI."QuantityUom2", (PMX_MVLI."Quantity"*PMX_MVLI."QuantityPerUom")
					FROM  PMX_MVHE INNER JOIN
						  PMX_MVLI ON PMX_MVHE."DocEntry" = PMX_MVLI."DocEntry" INNER JOIN
						  PMX_OSSL ON PMX_MVLI."DestStorLocCode" = PMX_OSSL."Code" LEFT OUTER JOIN						  
						  PMX_LUID ON PMX_LUID."InternalKey" = PMX_MVLI."DestLogUnitIdentKey"
						  INNER JOIN OITM ON PMX_MVLI."ItemCode" = OITM."ItemCode"
						  LEFT JOIN OUOM ON OUOM."UomEntry" = OITM."IUoMEntry"
					WHERE PMX_MVHE."DocEntry" = v_keyAsInteger;
			END IF;
			
			IF (v_isTransactionTypeAdd=1) THEN
			
			    --Delete the previous links
			    --DELETE FROM PMX_LMLU WHERE "InternalKey" IN (
				    --SELECT Links."InternalKey"
				    --FROM PMX_LMLU Links
				    --INNER JOIN PMX_MVLI Lines ON Links.LUID = Lines."DestLogUnitIdentKey"
				    --WHERE Lines."DocEntry" = v_keyAsInteger
			    --);
			    
			    			
				--Delete the previous links
			    DELETE FROM PMX_LMLU WHERE "InternalKey" IN (
				    SELECT Links."InternalKey"
				    FROM PMX_LMLU Links
					INNER JOIN PMX_MVLI Lines ON Links."LUID" = Lines."SrcLogUnitIdentKey"
					WHERE Lines."DocEntry" = v_keyAsInteger
					AND Lines."MasterLUID" <> Links."MasterLUID"
			    );
			    
			    --Delete links when all stock from luid is moved
				DELETE FROM PMX_LMLU WHERE "LUID" IN 
				(
					SELECT Inventory."LogUnitIdentKey" FROM PMX_MVLI Lines
					INNER JOIN PMX_LMLU Links ON Links."LUID" = Lines."SrcLogUnitIdentKey"
					INNER JOIN PMX_INVT Inventory ON Inventory."LogUnitIdentKey" = Links."LUID" 
					WHERE Lines."DocEntry" = v_keyAsInteger AND Lines."MasterLUID" IS NULL 
					GROUP BY Inventory."LogUnitIdentKey"
					HAVING SUM(Inventory."Quantity") = SUM(Lines."Quantity" * Lines."QuantityPerUom")
				);

			    
    				
			    --Add new links
			    INSERT INTO PMX_LMLU ( "InternalKey", "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "MasterLUID", "LUID" )	
			    SELECT PMX_LMLU_S.nextval, * FROM (
			    SELECT DISTINCT 'N', Header."UserSign", v_tsNow, v_tsNow, 1, Lines."MasterLUID", Lines."DestLogUnitIdentKey"
			    FROM	    PMX_MVLI Lines
			    INNER JOIN  PMX_MVHE Header ON Lines."DocEntry" = Header."DocEntry"
			    LEFT JOIN   PMX_LMLU Links  ON Lines."DestLogUnitIdentKey" = Links.LUID AND Lines."MasterLUID" = Links."MasterLUID"
			    WHERE   Lines."DocEntry" = v_keyAsInteger 
				    AND Lines."MasterLUID" IS NOT NULL
				    AND Links."InternalKey" IS NULL
				);									
    				
			END IF;
			
			--Check if all subluids are on the same location as master luid
			IF (v_isTransactionTypeAdd=1) THEN

				DELETE FROM TMP_TN_CharListTableNoIndex;
		        INSERT INTO TMP_TN_CharListTableNoIndex
		        SELECT CAST(Lines."LineNum" AS VARCHAR)
				FROM PMX_MVLI AS Lines
				INNER JOIN (
					SELECT PMX_INVT."StorLocCode", PMX_LMLU.LUID, PMX_LMLU."MasterLUID"
					FROM PMX_LMLU
					INNER JOIN PMX_INVT ON PMX_LMLU.LUID = PMX_INVT."LogUnitIdentKey"
				) t ON t."MasterLUID" = Lines."MasterLUID"
				WHERE t."StorLocCode" <> Lines."DestStorLocCode"
				AND t."MasterLUID" <> Lines."MasterLUID"
				AND Lines."DocEntry" = v_keyAsInteger; 
                SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
				IF LENGTH(v_errorList) <> 0 THEN
				    p_error := 50119;
				    p_error_message := 'Error on line(s) ' || v_errorList || ': SubLuids all need to be on the same location.';
				    SELECT p_error, p_error_message FROM dummy;
				    --EXEC xp_logevent p_error, p_error_message, ERROR
				    RETURN;		
				 END IF;

			END IF;

		END IF;

		IF v_isWarehouseTransferRequestDocument = 1 THEN

			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );
			
			--Check if the warehouse inventory request is closed that there are no open pick list proposals of this warehouse inventory request
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(PMX_PLPL."DocEntry" AS VARCHAR)
			FROM OWTQ 
			INNER JOIN PMX_PLPL ON OWTQ."DocEntry" = PMX_PLPL."BaseEntry" AND PMX_PLPL."BaseType" = OWTQ."ObjType"
			INNER JOIN WTQ1 ON OWTQ."DocEntry" = WTQ1."DocEntry" AND WTQ1."LineNum" = PMX_PLPL."BaseLine" 
			WHERE OWTQ."DocEntry" = v_keyAsInteger AND (OWTQ."DocStatus" = 'C' OR WTQ1."LineStatus" = 'C' ) AND PMX_PLPL."LineStatus" <> 'C' AND PMX_PLPL."OpenQty" <> 0;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51200;
				p_error_message := 'The warehouse inventory request ''' || p_list_of_cols_val_tab_del || ''' has open pick list proposal(s): ' || v_errorList || '. The pick list proposal(s) must be closed first before closing the warehouse inventory request (line).';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if the full quantity is selected when entering batch numbers
--			v_errorList := ''
--			SELECT v_errorList = v_errorList || CAST(Lines."LineNum" AS VARCHAR) || ' '
--			FROM WTQ1 AS Lines
--			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
--			LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = WhsInvRequest."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
--			WHERE WhsInvRequest."DocEntry" = v_keyAsInteger AND Lines."LineStatus" = 'O'
--			GROUP BY Lines."LineNum", Lines."Quantity", Lines."NumPerMsr"
--			HAVING SUM(IBT1."Quantity") <> (Lines."Quantity" * Lines."NumPerMsr") AND SUM(IBT1."Quantity") > 0
--			IF LENGTH(v_errorList) <> 0 THEN
--				p_error := 51201
--				p_error_message := 'Error on line(s):  ' || v_errorList || ' You need to fill the full quantity when entering a batch number'
--				SELECT p_error, p_error_message
--				--EXEC xp_logevent p_error, p_error_message, ERROR
--				RETURN
--			END

			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM WTQ1 AS Lines
            INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
            INNER JOIN OITL ON OITL."AllocatEnt" = Lines."DocEntry" AND OITL."AllocatEnt" = Lines."LineNum" AND OITL."AllocateTp" = Lines."ObjType"
            INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
            INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
            AND ITL1."SysNumber" = OBTN."SysNumber"
            WHERE OITL."AllocatEnt" = v_keyAsInteger 
			AND Lines."LineStatus" = 'O'
            AND "StockEff" = 2
            GROUP BY Lines."LineNum", Lines."OpenQty", Lines."NumPerMsr"
            HAVING SUM("AllocQty") <> (Lines."OpenQty" * Lines."NumPerMsr") AND SUM("AllocQty") > 0;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51201;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' You need to fill the full quantity when entering a batch number';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;


			--Check if warehouse inventory request lines are deleted that have open pickListProposals or open pick list
			--or that the quantity on the warehouse inventory request line is lower than on the open pick list proposal line / open pick list line
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CASE WHEN        WTQ1."ItemCode"   <> PMX_PLPL."ItemCode" THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) || ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN WTQ1 ON PMX_PLPL."BaseEntry" = WTQ1."DocEntry" AND PMX_PLPL."BaseLine" = WTQ1."LineNum"  AND PMX_PLPL."BaseType" = WTQ1."ObjType"
			LEFT JOIN PMX_PLLI ON PMX_PLLI."BaseEntry" = PMX_PLPL."DocEntry" AND PMX_PLLI."BaseLine" = PMX_PLPL."LineNum" 
			WHERE PMX_PLPL."BaseType" = 1250000001 AND PMX_PLPL."BaseEntry" = v_keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C' OR IFNULL( PMX_PLLI."LineStatus", 'C') <> 'C');
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51203;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' The item code on a warehouse inventory request line cannot be changed if a picklist proposal is generated for it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CASE WHEN IFNULL( WTQ1."OpenQty",0) < SUM(PMX_PLPL."Quantity") THEN ( CAST(PMX_PLPL."BaseLine" AS VARCHAR) || ' ' ) ELSE '' END
			FROM PMX_PLPL
			LEFT JOIN WTQ1 ON PMX_PLPL."BaseEntry" = WTQ1."DocEntry" AND PMX_PLPL."BaseLine" = WTQ1."LineNum"  AND PMX_PLPL."BaseType" = WTQ1."ObjType"
			WHERE PMX_PLPL."BaseType" = 1250000001 AND PMX_PLPL."BaseEntry" = v_keyAsInteger
			AND (PMX_PLPL."LineStatus" <> 'C') 
			GROUP BY PMX_PLPL."BaseLine", WTQ1."OpenQty";
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51202;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' The quantity on a warehouse inventory request line cannot be lowered if a picklist proposal is generated for it.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Check if maximum 1 batchnumber is selected when multiple batches are not allowed
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM WTQ1 AS Lines
			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
			--LEFT OUTER JOIN IBT1 ON Lines."DocEntry" = IBT1."BaseEntry" AND IBT1."BaseType" = WhsInvRequest."ObjType" AND IBT1.BaseLinNum = Lines."LineNum"
			INNER JOIN 
			(
				SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH
			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"
						
			WHERE WhsInvRequest."DocEntry" = v_keyAsInteger AND Lines."LineStatus" = 'O' AND Lines."U_PMX_AMBA" = 'N'
			GROUP BY Lines."LineNum"
			HAVING (Count(*)) > 1;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
			IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51204;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' When no multiple batches are allowed, you cannot add more than 1 batch to the line.';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- change of batch only allowed when not linked in PMX_PLPL and PMX_DOLL
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(Lines."LineNum" AS VARCHAR)
			FROM WTQ1 AS Lines
			INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"	
			
			INNER JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType" 
			AND PMX_PLPL."BaseEntry" = Lines."DocEntry" 
			AND PMX_PLPL."BaseLine" = Lines."LineNum" 
			
			INNER JOIN PMX_DOLL ON PMX_DOLL."ObjType" = Lines."ObjType" 
			AND PMX_DOLL."DocEntry" = Lines."DocEntry" 
			AND PMX_DOLL."LineNum" = Lines."LineNum" 
			
			INNER JOIN PMX_ITRI ON PMX_PLPL."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
			
			LEFT JOIN 
			(	SELECT "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", SUM("AllocQty") AS "AllocQty"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"
				WHERE "StockEff" = 2
								GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType"
				HAVING SUM("AllocQty") > 0
			) AS BATCH

			ON BATCH."ApplyEntry" = Lines."DocEntry" AND BATCH."ApplyLine" = Lines."LineNum" AND BATCH."ApplyType" = Lines."ObjType"		

			WHERE Lines."DocEntry" = v_keyAsInteger
			AND Lines."LineStatus" = 'O'
			AND( IFNULL( BATCH."AllocQty", PMX_DOLL."LockedQuantity") < PMX_DOLL."LockedQuantity"
			OR BATCH."DistNumber" <> IFNULL( PMX_ITRI."BatchNumber", BATCH."DistNumber"));
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51205;
				p_error_message := 'Error on line(s):  ' || v_errorList || ' Proposal exists: Changing the batch number(s) not allowed';
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Multiple batches on serial numbers are only allowed when multiple batches on line level are allowed
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(WTQ1."LineNum" AS VARCHAR)
			FROM PMX_SELD	
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN WTQ1 on WTQ1."DocEntry" = PMX_SELD."BaseEntry"
			AND WTQ1."LineNum" = PMX_SELD."BaseLine"
			AND WTQ1."ObjType" = PMX_SELD."BaseType"
			WHERE WTQ1.U_PMX_AMBA = 'N'
			AND WTQ1."DocEntry" = v_keyAsInteger	
			GROUP BY WTQ1."LineNum", PMX_SENU."ItriKey"
			HAVING COUNT(PMX_SENU."ItriKey") > 1;
            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51206;
				p_error_message := 'Serial numbers linked with multiple batches, but multiple batches are not allowed. Error on line(s):  ' || v_errorList ;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			--Linked serial numbers cannot be more than open quantity
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT 'Line ' || CAST(WTQ1."LineNum" AS VARCHAR) || ', Link qty:' || CAST(COUNT(PMX_SELD."InternalKey") AS VARCHAR)
			FROM PMX_SELD
			INNER JOIN PMX_SENU
			ON PMX_SENU."InternalKey" = PMX_SELD."SerialKey"
			INNER JOIN WTQ1 on WTQ1."DocEntry" = PMX_SELD."BaseEntry"
			AND WTQ1."LineNum" = PMX_SELD."BaseLine"
			AND WTQ1."ObjType" = PMX_SELD."BaseType"
			WHERE WTQ1."DocEntry" = v_keyAsInteger
			AND PMX_SENU."InStock" = 'Y'
			AND WTQ1."LineStatus" = 'O'
			GROUP BY WTQ1."LineNum", WTQ1."OpenQty", WTQ1."NumPerMsr"
			HAVING COUNT(PMX_SELD."InternalKey") > WTQ1."OpenQty" * WTQ1."NumPerMsr";

            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, '; ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51207;
				p_error_message := 'Number of linked serial numbers cannot be more than open quantity. Error on line(s):  ' || v_errorList ;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;
			
			-- Due date minutes
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(OWTQ.U_PMX_DDTM AS VARCHAR) 
			FROM OWTQ
			WHERE OWTQ."DocEntry" = v_keyAsInteger
			AND ( IFNULL( OWTQ.U_PMX_DDTM, 0) > 59 OR
				IFNULL( OWTQ.U_PMX_DDTM, 0) < 0 );

            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51208;
				p_error_message := 'The due date minutes: ' || v_errorList || ' are not valid for document ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;

			-- Due date Hour
			DELETE FROM TMP_TN_CharListTableNoIndex;
            INSERT INTO TMP_TN_CharListTableNoIndex
            SELECT CAST(OWTQ.U_PMX_DDTH AS VARCHAR)
			FROM OWTQ
			WHERE OWTQ."DocEntry" = v_keyAsInteger
			AND (IFNULL( OWTQ.U_PMX_DDTH, 0) < 0
					OR (IFNULL( OWTQ.U_PMX_DDTH, 0) > 23 AND IFNULL( OWTQ.U_PMX_DDTA, '') NOT IN ('AM', 'PM'))  
					OR (IFNULL( OWTQ.U_PMX_DDTH, 0) > 11 AND IFNULL( OWTQ.U_PMX_DDTA, '') IN ('AM', 'PM'))  
				);

            SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
            IF LENGTH(v_errorList) <> 0 THEN
				p_error := 51209;
				p_error_message := 'The due date hours: ' || v_errorList || ' are not valid for document ' || p_list_of_cols_val_tab_del;
				SELECT p_error, p_error_message FROM dummy;
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN;
			END IF;


			--**********************************************************************************************************
			--Updates
			--**********************************************************************************************************

			--UPDATE LOCKING TABLE
			DELETE FROM PMX_INLD WHERE "InternalKey" IN ( SELECT "InternalKey" 
                FROM PMX_INLD LOCKING
                INNER JOIN OWTQ ON OWTQ."DocEntry" = LOCKING."BaseEntry"
    			AND OWTQ."ObjType" = LOCKING."BaseType"
    			WHERE OWTQ."DocStatus" = 'C' AND OWTQ."DocEntry" = v_keyAsInteger );


			--IF(v_isTransactionTypeAdd=0 AND
			--	EXISTS(SELECT * FROM WTQ1 WHERE WTQ1."LineStatus" = 'C' AND WTQ1."DocEntry" = v_keyAsInteger)) 
			--THEN

			--	--Close container (lines) if needed
			--	UPDATE PMX_COLI SET "LineStatus" = 'C', "Version" = "Version" + 1 
			--	FROM PMX_COLI
			--	INNER JOIN WTQ1
			--	ON PMX_COLI."BaseEntry" = WTQ1."DocEntry"
			--	AND PMX_COLI."BaseLine" = WTQ1."LineNum"
			--	AND PMX_COLI."BaseType" = WTQ1."ObjType"
			--	WHERE PMX_COLI."LineStatus" <> 'C'
			--	AND WTQ1."DocEntry" = v_keyAsInteger
			--	AND WTQ1."LineStatus" = 'C'

			--	--Close header if no open lines are found
			--	UPDATE PMX_COHE SET PMX_COHE."DocStatus" = 'C', PMX_COHE."ContainerStatus" = 'C',
			--	PMX_COHE."Version" = PMX_COHE."Version" + 1, PMX_COHE."UpdateDateTime" = v_tsNow, 
			--	PMX_COHE."UserSign" = (SELECT MAX("UserSign") FROM OWTQ  WHERE OWTQ."DocEntry" = v_keyAsInteger)
			--	WHERE PMX_COHE."DocStatus" = 'O'
			--	AND PMX_COHE."DocEntry" NOT IN (SELECT "DocEntry" FROM PMX_COLI WHERE PMX_COLI."LineStatus" = 'O')

			--END


			--**********************************************************************************************************
			--START: Add/update locking based on batch numbers
			--**********************************************************************************************************



			--Start with deleting the lockings that are not correct
			DELETE FROM PMX_INLD
			WHERE PMX_INLD."InternalKey" NOT IN 
			(--Query to get the correct lockings
				SELECT 	PMX_INLD."InternalKey"
				FROM WTQ1 AS Lines
				INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
				INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( "DistNumber", '')
				--AND IFNULL( PMX_ITRI."BestBeforeDate", 1) = IFNULL( "ExpDate", 1)

				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"
				AND PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"

				WHERE OITL."ApplyEntry" = v_keyAsInteger 
				AND Lines."LineStatus" = 'O'
				AND "StockEff" = 2

				AND PMX_INLD."InternalKey" IS NOT NULL


				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				HAVING SUM("AllocQty") > 0

				-- Exclude locked stock created on goods receipt.
				UNION SELECT PMX_INLD."InternalKey"
				FROM WTQ1 AS Lines 
				INNER JOIN POR1 ON 
				POR1."BaseType" = Lines."ObjType"
				AND POR1."BaseEntry" = Lines."DocEntry"
				AND POR1."BaseLine" = Lines."LineNum"

				INNER JOIN PDN1 ON 
				PDN1."BaseType" = POR1."ObjType"
				AND PDN1."BaseEntry" = POR1."DocEntry"
				AND PDN1."BaseLine" = POR1."LineNum"

				INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
				LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
				LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
				AND ITL1."SysNumber" = OBTN."SysNumber"

				LEFT JOIN PMX_ITRI ON
				 IFNULL( PMX_ITRI."BatchNumber", '') = IFNULL( "DistNumber", '')


				LEFT JOIN PMX_INLD ON PMX_INLD."BaseType" = Lines."ObjType"
				AND PMX_INLD."BaseEntry" = Lines."DocEntry"
				AND PMX_INLD."BaseLine" = Lines."LineNum"

				WHERE Lines."DocEntry" = v_keyAsInteger				

				GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
				Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_ITRI."BatchNumber", PMX_INLD."InternalKey"

				--Only allow detailed locking from purchase order when no batch has been linked
				HAVING SUM( IFNULL( "AllocQty", 0) ) = 0 

				--Exclude manual lockings on LUID level when no batch has been selected on warehouse inventory request
				UNION SELECT PMX_INLD."InternalKey"
				FROM PMX_INLD 
				LEFT JOIN
				(
					SELECT Lines."LineNum", Lines."DocEntry", Lines."ObjType"
					FROM WTQ1 AS Lines
									INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
									LEFT JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
									LEFT JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
									LEFT JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
									AND ITL1."SysNumber" = OBTN."SysNumber"

					WHERE OITL."ApplyEntry"  = v_keyAsInteger 
									AND Lines."LineStatus" = 'O'
									AND IFNULL( "StockEff", 2) = 2

						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", 
									Lines."NumPerMsr", "DistNumber", "ExpDate"

					HAVING SUM( IFNULL( "AllocQty", 0) ) > 0
				) AS OWTQ

				ON OWTQ."LineNum" = PMX_INLD."BaseLine"
				AND OWTQ."DocEntry" = PMX_INLD."BaseEntry"
				AND OWTQ."ObjType" = PMX_INLD."BaseType"


				WHERE PMX_INLD."InvLockLevel" = 'L'
				AND PMX_INLD."BaseEntry" = v_keyAsInteger
				AND OWTQ."DocEntry" IS NULL

				

			) 

			AND PMX_INLD."BaseType" = '1250000001'
			AND PMX_INLD."BaseEntry" = v_keyAsInteger;



			BEGIN --TRY

				DECLARE CURSOR WhsInvRequestBatchCursor FOR 
						SELECT 	SUM("AllocQty") AS "AllocQty", "DistNumber", "ExpDate", Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."ItemCode", PMX_OSWH."Code"
						FROM WTQ1 AS Lines
						INNER JOIN OWTQ AS WhsInvRequest ON Lines."DocEntry" = WhsInvRequest."DocEntry"
						INNER JOIN OITL ON OITL."ApplyEntry" = Lines."DocEntry" AND OITL."ApplyLine" = Lines."LineNum" AND OITL."ApplyType" = Lines."ObjType"
						INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
						INNER JOIN OBTN ON ITL1."MdAbsEntry" = OBTN."AbsEntry"
						AND ITL1."SysNumber" = OBTN."SysNumber"

						INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = WhsInvRequest."Filler"

						LEFT JOIN PMX_PLPL ON PMX_PLPL."BaseType" = Lines."ObjType"
						AND PMX_PLPL."BaseEntry" = Lines."DocEntry"
						AND PMX_PLPL."BaseLine" = Lines."LineNum"
						AND PMX_PLPL."LineStatus" = 'O'

						WHERE OITL."ApplyEntry" = :v_keyAsInteger 
						AND Lines."LineStatus" = 'O'
						AND "StockEff" = 2
						--Only when no openpick list proposal is found
						AND PMX_PLPL."DocEntry" IS NULL
						GROUP BY Lines."LineNum", Lines."DocEntry", Lines."ObjType", Lines."Quantity", Lines."ItemCode", Lines."NumPerMsr", "DistNumber", "ExpDate", PMX_OSWH."Code"

						HAVING SUM("AllocQty") > 0;


				OPEN WhsInvRequestBatchCursor;
				FETCH WhsInvRequestBatchCursor 
				INTO v_batchQuantityToLock,v_batchToLock,v_expDateToLock,v_lineNumToLock,v_docEntryToLock,v_typeToLock,v_itemCodeToLock,v_pmxWarehouseToLock;
				WHILE v_docEntryToLock IS NOT NULL DO

					--Check for each batch if we have a corresponding locking line
					SELECT COUNT(*) INTO v_cnt FROM PMX_INLD 
					INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
					WHERE PMX_INLD."BaseType" = v_typeToLock
					AND PMX_INLD."BaseEntry" = v_docEntryToLock
					AND PMX_INLD."BaseLine" = v_lineNumToLock
					AND PMX_ITRI."BatchNumber" = v_batchToLock;
					--AND IFNULL( PMX_ITRI."BestBeforeDate", '') = IFNULL( v_expDateToLock, '');
					IF v_cnt < 1 THEN

						--Entry does not exist, so we need to lock the stock
						--First try to get the itri key to lock
						v_remainingQuantityToLock := v_batchQuantityToLock;

						BEGIN --TRY
							DECLARE CURSOR WhsInvRequestinldCursor FOR SELECT PMX_ITRI."InternalKey", IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
								LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
								INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
								WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
								LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
								WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
								AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
								WHERE PMX_ITRI."BatchNumber" = :v_batchToLock AND PMX_ITRI."ItemCode" = :v_itemCodeToLock
								AND Inventory."PmxWhsCode" = :v_pmxWarehouseToLock
								AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0;


							OPEN WhsInvRequestinldCursor;
							FETCH WhsInvRequestinldCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
							WHILE v_itriToLock IS NOT NULL AND v_remainingQuantityToLock > 0 DO

								IF(v_remainingQuantityToLock < v_freeQuantityToLock) THEN
									v_currentQuantityToLock := v_remainingQuantityToLock;
								ELSE
									v_currentQuantityToLock := v_freeQuantityToLock;
								END IF;

								--Insert INTO the PMX_INLD table
								INSERT INTO PMX_INLD ( "InternalKey", "BaseType","BaseEntry","BaseLine","CardCode","CardName",
									"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
									"UserSign","CreateDate",
									"CreateTime","LogUnitIdentKey","QualityStatusCode", 
									"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
								SELECT PMX_INLD_S.nextval, * FROM (
								SELECT v_typeToLock,v_docEntryToLock,v_lineNumToLock,MAX(OWTQ."CardCode"),MAX(OWTQ."CardName"),
									v_itemCodeToLock,v_currentQuantityToLock,v_itriToLock,NULL,
									MAX(OWTQ."UserSign"),v_updateDate,
									v_updateTime,NULL, v_qualityStatusToLock,
									'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0
								FROM OWTQ 
								INNER JOIN WTQ1 ON WTQ1."DocEntry" = OWTQ."DocEntry"
								INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OWTQ."Filler"
								WHERE OWTQ."DocEntry" = v_docEntryToLock
								AND WTQ1."LineNum" = v_lineNumToLock
								);

								--Reduce remaining
								v_remainingQuantityToLock := v_remainingQuantityToLock - v_currentQuantityToLock;

								FETCH WhsInvRequestinldCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock ;
							END WHILE;
							CLOSE WhsInvRequestinldCursor ;

							--If there is remaining QTY, there is not enough free stock, so throw error
							IF(v_remainingQuantityToLock > 0)
							THEN
								p_error := 51250;
								p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock || '. Missing qty: ' || v_remainingQuantityToLock;
								SELECT p_error, p_error_message FROM dummy;
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN;
							END IF;
							
						END; /*TRY
						BEGIN CATCH
							p_error := 51290
							p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
							|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
							BEGIN TRY
								CLOSE WhsInvRequestinldCursor 
								DEALLOCATE WhsInvRequestinldCursor
								SELECT p_error, p_error_message
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN
							END TRY
							BEGIN CATCH
								DEALLOCATE WhsInvRequestinldCursor
								SELECT p_error, p_error_message
								--EXEC xp_logevent p_error, p_error_message, ERROR
								RETURN
							END CATCH
						END CATCH*/

					ELSE 

						--The line exists, so now we need to check if the quantity is the same/more/less
						SELECT SUM(PMX_INLD."Quantity") -  v_batchQuantityToLock
                        INTO v_remainingQuantityToLock
						FROM PMX_INLD 
						INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
						WHERE PMX_INLD."BaseType" = v_typeToLock
						AND PMX_INLD."BaseEntry" = v_docEntryToLock
						AND PMX_INLD."BaseLine" = v_lineNumToLock
						AND PMX_ITRI."BatchNumber" = v_batchToLock
						GROUP BY PMX_INLD."BaseType", PMX_INLD."BaseEntry", PMX_INLD."BaseLine";

						IF v_remainingQuantityToLock <> 0 THEN
						--Only if deviation, then we try to update/insert
						--Loop through all locking lines
							BEGIN --TRY
								DECLARE CURSOR WhsInvRequestinldCursor FOR SELECT PMX_INLD."InternalKey", PMX_INLD."Quantity", PMX_INLD."ItemTransactionalInfoKey" FROM PMX_INLD 
								INNER JOIN PMX_ITRI ON PMX_ITRI."InternalKey" = PMX_INLD."ItemTransactionalInfoKey"
								WHERE PMX_INLD."BaseType" = :v_typeToLock
								AND PMX_INLD."BaseEntry" = :v_docEntryToLock
								AND PMX_INLD."BaseLine" = :v_lineNumToLock
								AND PMX_ITRI."BatchNumber" = :v_batchToLock;

								OPEN WhsInvRequestinldCursor;
								FETCH WhsInvRequestinldCursor INTO v_currentLockInternalKey, v_lockedQuantity, v_itriToLock;
								WHILE v_currentLockInternalKey IS NOT NULL AND v_remainingQuantityToLock <> 0 DO

									IF v_remainingQuantityToLock > 0 THEN

										--Remaining is more than 0, so we can just remove the locking
										IF v_remainingQuantityToLock >= v_lockedQuantity THEN
											--Delete
											DELETE FROM PMX_INLD WHERE "InternalKey" = v_currentLockInternalKey;
											v_remainingQuantityToLock := v_remainingQuantityToLock - v_lockedQuantity;
										ELSE 
											--update
											UPDATE PMX_INLD SET "Quantity" = "Quantity" - v_remainingQuantityToLock WHERE "InternalKey" = v_currentLockInternalKey;
											v_remainingQuantityToLock := 0;
										END IF;

									ELSE

										--Remaining is less than 0, so we need to add locking
										--If there is still free stock for current itri, then take that one
										SELECT IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) 
										INTO v_freeQuantityToLock
										FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = v_batchToLock AND PMX_ITRI."ItemCode" = v_itemCodeToLock
										AND Inventory."PmxWhsCode" = v_pmxWarehouseToLock
										AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0
										AND PMX_ITRI."InternalKey" = v_itriToLock;

										IF v_remainingQuantityToLock > v_freeQuantityToLock THEN
											v_currentQuantityToLock := v_freeQuantityToLock;
										ELSE
											v_currentQuantityToLock := v_remainingQuantityToLock;
										END IF;

										UPDATE PMX_INLD SET "Quantity" = "Quantity" + v_currentQuantityToLock WHERE PMX_INLD."InternalKey" = v_currentLockInternalKey;

										-- add qty, because v_remainingQuantityToLock is negative
										v_remainingQuantityToLock := v_remainingQuantityToLock + v_currentQuantityToLock;
										
									END IF;

									FETCH WhsInvRequestinldCursor INTO v_currentLockInternalKey, v_lockedQuantity, v_itriToLock;
								END WHILE;
								CLOSE WhsInvRequestinldCursor;
								
							END; /*TRY
							BEGIN CATCH
								p_error := 51291
								p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
								|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
								BEGIN TRY
									CLOSE WhsInvRequestinldCursor 
									DEALLOCATE WhsInvRequestinldCursor
									SELECT p_error, p_error_message
									--EXEC xp_logevent p_error, p_error_message, ERROR
									RETURN
								END TRY
								BEGIN CATCH
									DEALLOCATE WhsInvRequestinldCursor
									SELECT p_error, p_error_message
									--EXEC xp_logevent p_error, p_error_message, ERROR
									RETURN
								END CATCH
							END CATCH*/

							IF v_remainingQuantityToLock <> 0 THEN
								-- IF there is still remaining quantity, it could not be added to the existing locking lines, so new locking lines need to be added
								--Set v_remainingQuantityToLock as positive quanitty
								v_remainingQuantityToLock := ABS( v_remainingQuantityToLock );
								

								BEGIN --TRY
									DECLARE CURSOR WhsInvRequestinldCursor FOR SELECT PMX_ITRI."InternalKey", IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0), Inventory."QualityStatusCode" FROM PMX_ITRI 
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode" FROM PMX_INVT 
										INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = PMX_INVT."StorLocCode" 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", PMX_OSEL."PmxWhsCode") AS Inventory ON Inventory."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey"
										LEFT JOIN (SELECT SUM("Quantity") AS "Quantity", "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode" FROM PMX_INLD 
										WHERE "QualityStatusCode" IN (SELECT "Code" FROM PMX_QYST WHERE "CanBeShipped" = 'Y') GROUP BY "ItemTransactionalInfoKey", "QualityStatusCode", "PmxWhsCode") AS Locking ON Locking."ItemTransactionalInfoKey" = PMX_ITRI."InternalKey" AND Locking."QualityStatusCode" = Inventory."QualityStatusCode"
										AND Locking."PmxWhsCode" = Inventory."PmxWhsCode"
										WHERE PMX_ITRI."BatchNumber" = :v_batchToLock AND PMX_ITRI."ItemCode" = :v_itemCodeToLock
										AND Inventory."PmxWhsCode" = :v_pmxWarehouseToLock
										AND IFNULL( Inventory."Quantity", 0) - IFNULL( Locking."Quantity", 0) > 0;

									OPEN WhsInvRequestinldCursor;
									FETCH WhsInvRequestinldCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock;
									WHILE v_itriToLock IS NOT NULL AND v_remainingQuantityToLock > 0 DO

										IF v_remainingQuantityToLock < v_freeQuantityToLock THEN
											v_currentQuantityToLock := v_remainingQuantityToLock;
										ELSE
											v_currentQuantityToLock := v_freeQuantityToLock;
										END IF;

										--Insert INTO the PMX_INLD table
										INSERT INTO PMX_INLD ( "InternalKey", "BaseType","BaseEntry","BaseLine","CardCode","CardName",
											"ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
											"UserSign","CreateDate",
											"CreateTime","LogUnitIdentKey","QualityStatusCode", 
											"InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
										SELECT PMX_INLD_S.nextval, * FROM (
										SELECT v_typeToLock,v_docEntryToLock,v_lineNumToLock,MAX(OWTQ."CardCode"),MAX(OWTQ."CardName"),
											v_itemCodeToLock,v_currentQuantityToLock,v_itriToLock,NULL,
											MAX(OWTQ."UserSign"),v_updateDate,
											v_updateTime,NULL, v_qualityStatusToLock,
											'RESERVED', 'B' , MAX(PMX_OSWH."Code"), 0
										FROM OWTQ 
										INNER JOIN WTQ1 ON WTQ1."DocEntry" = OWTQ."DocEntry"
										INNER JOIN PMX_OSWH ON PMX_OSWH."SboWhsCode" = OWTQ."Filler"
										WHERE OWTQ."DocEntry" = v_docEntryToLock
										AND WTQ1."LineNum" = v_lineNumToLock
										);

										--Reduce remaining
										v_remainingQuantityToLock := v_remainingQuantityToLock - v_currentQuantityToLock;

										FETCH WhsInvRequestinldCursor INTO v_itriToLock, v_freeQuantityToLock, v_qualityStatusToLock ;
									END WHILE;
									CLOSE WhsInvRequestinldCursor ;

									--If there is remaining QTY, there is not enough free stock, so throw error
									IF(v_remainingQuantityToLock > 0) THEN
										p_error := 51252;
										p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock|| '. Missing qty: ' || v_remainingQuantityToLock;
										SELECT p_error, p_error_message FROM dummy;
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN;
									END IF;

								END; /*TRY
								BEGIN CATCH
									p_error := 51291
									p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
									|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
									BEGIN TRY
										CLOSE WhsInvRequestinldCursor 
										DEALLOCATE WhsInvRequestinldCursor
										SELECT p_error, p_error_message
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN
									END TRY
									BEGIN CATCH
										DEALLOCATE WhsInvRequestinldCursor
										SELECT p_error, p_error_message
										--EXEC xp_logevent p_error, p_error_message, ERROR
										RETURN
									END CATCH
								END CATCH*/

							END IF;

						END IF;

						IF v_remainingQuantityToLock <> 0 THEN
						-- if quantity is still different from 0, then there was not enough stock to reserve!
							CLOSE WhsInvRequestBatchCursor;
							p_error := 51253;
							p_error_message := 'Not enough free stock to allocate batch ' || v_batchToLock || ' of item ' || v_itemCodeToLock|| '. Missing qty: ' || v_remainingQuantityToLock;
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;

					END IF;

					FETCH WhsInvRequestBatchCursor
					INTO v_batchQuantityToLock,v_batchToLock,v_expDateToLock,v_lineNumToLock,v_docEntryToLock,v_typeToLock,v_itemCodeToLock, v_pmxWarehouseToLock;
				END WHILE;
				CLOSE WhsInvRequestBatchCursor;

			END; /*TRY
			BEGIN CATCH
				p_error := 51299
				p_error_message := 'Error On batch allocation: (' || CAST(ERROR_NUMBER() AS VARCHAR) || ') ' || ERROR_MESSAGE()
				|| IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
				BEGIN TRY
					CLOSE WhsInvRequestBatchCursor 
					DEALLOCATE WhsInvRequestBatchCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE WhsInvRequestBatchCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END CATCH
			END CATCH*/

			--**********************************************************************************************************
			--END: Add/update locking based on batch numbers
			--**********************************************************************************************************

			--**********************************************************************************************************
			--START: SAP Serial numbers
			--**********************************************************************************************************

			--Delete system serial links
			DELETE FROM PMX_SELD WHERE "LinkStatus" = 'S'
			AND PMX_SELD."BaseEntry" = v_keyAsInteger
			AND PMX_SELD."BaseType" = '1250000001';

			--Add to PMX serial numbers table

			INSERT INTO PMX_SELD ( "InternalKey", "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "SerialKey", "BaseType", "BaseEntry", "BaseLine", "LinkStatus") 
			SELECT PMX_SELD_S.nextval, 'N', SERIAL."UserSign", v_tsNow, v_tsNow, 1, PMX_SENU."InternalKey", "ApplyType", "ApplyEntry", "ApplyLine", 'S'	
			FROM 
				(SELECT OSRN."ItemCode", "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", CAST(OITL."ApplyType" AS NVARCHAR) AS "ApplyType", SUM("AllocQty") AS "AllocQty", "StockEff", OWTQ."UserSign"
				FROM OITL 
				INNER JOIN ITL1 ON ITL1."LogEntry" = OITL."LogEntry"
				INNER JOIN OSRN ON ITL1."MdAbsEntry" = OSRN."AbsEntry"
				AND ITL1."SysNumber" = OSRN."SysNumber"
				INNER JOIN OWTQ ON OWTQ."DocEntry" = OITL."ApplyEntry"
				INNER JOIN WTQ1 ON WTQ1."DocEntry" = OITL."ApplyEntry"
				AND WTQ1."LineNum" = OITL."ApplyLine"
				AND WTQ1."ObjType" = OITL."ApplyType"

				WHERE "StockEff" = 2
				AND OITL."ApplyEntry" = v_keyAsInteger
				GROUP BY "DistNumber", OITL."ApplyEntry", OITL."ApplyLine", OITL."ApplyType", OSRN."ItemCode", "StockEff", OWTQ."UserSign"
				HAVING SUM("AllocQty") > 0 AND "DistNumber" IS NOT NULL
				) AS SERIAL

			INNER JOIN PMX_SENU ON PMX_SENU."ItemCode" = SERIAL."ItemCode"
			AND PMX_SENU."SerialNumber" = SERIAL."DistNumber"
			LEFT JOIN PMX_SELD ON PMX_SELD."SerialKey" = PMX_SENU."InternalKey"
			AND PMX_SELD."BaseEntry" = SERIAL."ApplyEntry"
			AND PMX_SELD."BaseLine" = SERIAL."ApplyLine"
			AND PMX_SELD."BaseType" = SERIAL."ApplyType"
			WHERE PMX_SELD."InternalKey" IS NULL;



			--**********************************************************************************************************
			--END: SAP Serial numbers
			--**********************************************************************************************************


			RETURN;

		END IF;

		--**********************************************************************************************************
		--Do checks for stock transfer and add lines to detail inventory
		--**********************************************************************************************************
		IF (v_isStockTransfer=1) AND (v_isTransactionTypeAdd=1 OR v_isTransactionTypeCancel=1) THEN
			--Convert the list of key columns to an integer key
			v_keyAsInteger := CAST( p_list_of_cols_val_tab_del AS INT );

			--**********************************************************************************************************
			--Checks
			--**********************************************************************************************************

			--**********************************************************************************************************
			--Insert inventory detail lines in memory table
			--**********************************************************************************************************
			IF (v_isTransactionTypeAdd=1 OR v_isTransactionTypeCancel=1) THEN
				--One stock transfer line represents 2 lines in the detail inventory table:
				-- *The destination as positive lines
				-- *The source as negative lines

				--Insert source lines -> negative
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OWTR."ObjType", OWTR."DocEntry", OWTR."DocNum", OWTR."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OWTR."CardCode", OWTR."CardName", Lines."ItemCode", 
						  Lines."Dscription", -Lines."Quantity", Lines."StorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."LogUnitIdentKey", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", PMX_LUID.SSCC, 
						  Lines."QualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", -Lines."QuantityUom2", -Lines."InventoryQuantity"
					FROM  OWTR 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OWTR."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
		 				  INNER JOIN OWHS ON OWHS."WhsCode" = Lines."WhsCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."LogUnitIdentKey"
					WHERE OITM."InvntItem" = 'Y' AND OWHS.U_PMX_IMBP = 'Y';


				--Insert destination lines -> positive
				INSERT INTO TMP_TN_InventoryDetail
					SELECT OWTR."ObjType", OWTR."DocEntry", OWTR."DocNum", OWTR."DocDate", Lines."LineNum", Lines."BaseType", Lines."BaseEntry", Lines."BaseLine",
						  OWTR."CardCode", OWTR."CardName", Lines."ItemCode", 
						  Lines."Dscription", Lines."Quantity", Lines."DestinationStorLocCode", Lines."ItemTransactionalInfoKey", 
						  NULL AS Serial, Lines."DestinationLUID", OWTR."UserSign", OWTR."CreateDate", OWTR."DocTime", PMX_LUID.SSCC, 
						  Lines."DestinationQualityStatusCode", Lines."Uom" , Lines."InventoryUom", Lines."QuantityPerUom", Lines."Uom2", Lines."QuantityUom2", Lines."InventoryQuantity"
					FROM  OWTR 
						  INNER JOIN TMP_TN_documentLine AS Lines ON OWTR."DocEntry" = Lines."DocEntry" 
		 				  INNER JOIN OITM ON Lines."ItemCode" = OITM."ItemCode"
		 				  INNER JOIN OWHS ON OWHS."WhsCode" = Lines."DestinationWhsCode"
						  LEFT OUTER JOIN PMX_LUID ON PMX_LUID."InternalKey" = Lines."DestinationLUID"
					WHERE OITM."InvntItem" = 'Y' AND OWHS.U_PMX_IMBP = 'Y';
					
					
								
			--			--Delete the previous links
				--DELETE FROM Links
    --            FROM PMX_LMLU Links
				--INNER JOIN PMX_MVLI Lines ON Links.LUID = Lines."DestLogUnitIdentKey"
				--WHERE Lines."DocEntry" = p_keyAsInteger;
				
			--Delete links when all stock from luid is moved
				DELETE FROM PMX_LMLU
				WHERE LUID IN 
				(
					SELECT Inventory."LogUnitIdentKey" FROM TMP_TN_DocumentLine Lines
					INNER JOIN PMX_LMLU Links ON Links.LUID = Lines."LogUnitIdentKey"
					INNER JOIN PMX_INVT Inventory ON Inventory."LogUnitIdentKey" = Links.LUID 
					WHERE Lines."DocEntry" = v_keyAsInteger AND Lines."MasterLUID" IS NULL 
					GROUP BY Inventory."LogUnitIdentKey"
					HAVING SUM(Inventory."Quantity") = SUM(Lines."Quantity" * Lines."QuantityPerUom")
				);
				
				IF (v_isTransactionTypeAdd=1 ) THEN	
				
    			    ----Add new links
			        INSERT INTO PMX_LMLU ( "InternalKey", "Canceled", "UserSign", "CreateDateTime", "UpdateDateTime", "Version", "MasterLUID", "LUID" )	
			        SELECT PMX_LMLU_S.nextval, * FROM (
				    SELECT DISTINCT 'N', Header."UserSign", v_tsNow ,v_tsNow, 0, Lines."MasterLUID", Lines."LogUnitIdentKey"
				    FROM	TMP_TN_DocumentLine Lines INNER JOIN
						    OWTR Header ON Lines."DocEntry" = Header."DocEntry" LEFT JOIN
						    PMX_LMLU Links ON Lines."LogUnitIdentKey" = Links.LUID AND Lines."LogUnitIdentKey" = Links."MasterLUID"
				    WHERE	Lines."DocEntry" = v_keyAsInteger 
					    AND Lines."MasterLUID" IS NOT NULL
					    AND Links."InternalKey" IS NULL
                    );				
                    				
				END IF;
					
					
			END IF;

		END IF;

        -- *** Check if destination location can accept more LU's ***			
		DELETE FROM TMP_TN_CharListTableNoIndex;
        INSERT INTO TMP_TN_CharListTableNoIndex
        SELECT t."StorLocCode"
        FROM (
           SELECT "StorLocCode", "LogUnitIdentKey"
           FROM TMP_TN_InventoryDetail
           WHERE "Quantity" > 0
           UNION  -- 'UNION' will only get the distinct rows as needed
           SELECT PMX_INVT."StorLocCode", PMX_INVT."LogUnitIdentKey"
           FROM TMP_TN_InventoryDetail idt
           INNER JOIN PMX_INVT ON PMX_INVT."StorLocCode" = idt."StorLocCode"
           WHERE idt."Quantity" > 0
        ) t
        INNER JOIN PMX_OSSL ON PMX_OSSL."Code" = "StorLocCode"
        GROUP BY t."StorLocCode", PMX_OSSL."MaxLogUnits"
        HAVING COUNT(*) > PMX_OSSL."MaxLogUnits";
		SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ', ' );
		IF LENGTH(v_errorList) > 0 THEN
			p_error := 50116;
			p_error_message := 'Error: The destination location ' || v_errorList || 'has insufficient free space.';
			SELECT p_error, p_error_message FROM dummy;
			--EXEC xp_logevent p_error, p_error_message, ERROR
			RETURN;
		END IF;


		--Check if an item with SAP serial number and track locations has an LUID (On a moveable location, no LUID is allowed
		DELETE FROM TMP_TN_CharListTableNoIndex;
        INSERT INTO TMP_TN_CharListTableNoIndex
        SELECT Lines."ItemCode"
		FROM TMP_TN_InventoryDetail AS Lines
		INNER JOIN OITM ON OITM."ItemCode" = Lines."ItemCode"
		LEFT JOIN PMX_SENU ON PMX_SENU.LUID = Lines."LogUnitIdentKey"
		--LEFT JOIN PMX_OSML ON PMX_OSML."Code" = Lines."StorLocCode"
		WHERE OITM.U_PMX_TLSN = 'Y' AND Lines."LogUnitIdentKey" IS NULL
		AND Lines."StorLocCode" NOT IN ( SELECT "StorLocLostAndFound" FROM PMX_OSWH WHERE "StorLocLostAndFound" IS NOT NULL )
		--AND PMX_OSML."Code" IS NULL
		AND PMX_SENU."InternalKey" IS NOT NULL;
		SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
		IF LENGTH(v_errorList) <> 0 THEN
			p_error := 50117;
			p_error_message := 'Serial nrs. locations are tracked, but no LUID is set for items(s): ' || v_errorList ;
			SELECT p_error, p_error_message FROM dummy;
			--EXEC xp_logevent p_error, p_error_message, ERROR
			RETURN;
		END IF;
		
		IF (v_isTransactionTypeAdd = 1 AND ( v_isPurchaseDeliveryDocument = 1 OR v_isStockTransfer = 1 OR v_isInventoryEntry = 1)) THEN
		
BEGIN
			DECLARE v_cardCodeAINL  NVARCHAR(15);
			DECLARE v_cardNameAINL  NVARCHAR(100);
			DECLARE v_internalKeyAINL  INT;
			DECLARE v_itemCodeAINL  NVARCHAR(20);
			DECLARE v_quantityAINL  NUMERIC(19,6);
			DECLARE v_sboWhsCodeAINL  NVARCHAR(8);

			--*****************************************************************
			--* Lock stock if stock is available in Advance inventory locking *
			--*****************************************************************

			--Declare memory table for storing the details of the inventory (Temp table)
			DELETE FROM TMP_TN_TempDetail;

			--insert into temp table, because we will perform data actions on it
			INSERT INTO TMP_TN_TempDetail( "ItemCode", "ItemName", "Quantity", "StorLocCode", "ItemTransactionalInfoKey", 
				"SerialNum", "LogUnitIdentKey", "CreatedBy", "CreateDate", "CreateTime", "SSCC", 
				"QualityStatusCode", "Uom", "InventoryUom", "QuantityPerUom", "Uom2", "QuantityUom2", "BaseType", 
				"BaseEntry", "BaseLine", "SboWhsCode", "InternalKey" ) 
			SELECT "ItemCode", "ItemName", "Quantity", "StorLocCode", "ItemTransactionalInfoKey", 
				"SerialNum", "LogUnitIdentKey", "CreatedBy", idt."CreateDate", idt."CreateTime", "SSCC", 
				"QualityStatusCode", "Uom","InventoryUom", "QuantityPerUom", "Uom2", "QuantityUom2", "BaseType", 
				"BaseEntry", "BaseLine", PMX_OSWH."SboWhsCode", ROW_NUMBER() OVER ( ORDER BY "ItemCode", "BaseType", "BaseEntry", "BaseLine" )
			FROM TMP_TN_InventoryDetail idt
			LEFT JOIN ( -- inner join because "StorLocCode" is mandatory ?
			    PMX_OSEL
			    INNER JOIN PMX_OSWH ON PMX_OSWH."Code" = PMX_OSEL."PmxWhsCode"
			) ON PMX_OSEL."Code" = idt."StorLocCode"
			INNER JOIN PMX_QYST ON PMX_QYST."Code" = idt."QualityStatusCode"
			WHERE idt."Quantity" > 0
			AND (PMX_QYST."CanBeShipped" = 'Y' OR PMX_QYST."CanBeShippedUnderQuarantine" = 'Y');

BEGIN

			DECLARE CURSOR advanceLockingCursor FOR
			SELECT PMX_AINL."ItemCode", PMX_AINL."Quantity", PMX_AINL."InternalKey", PMX_AINL."CardCode", PMX_AINL."CardName", PMX_AINL."SboWhsCode"
			FROM PMX_AINL
			WHERE "ItemCode" IN ( SELECT "ItemCode" FROM TMP_TN_TempDetail )
			ORDER BY PMX_AINL."ItemCode", PMX_AINL."Sequence";

			OPEN advanceLockingCursor;
			FETCH advanceLockingCursor INTO v_itemCodeAINL,v_quantityAINL,v_internalKeyAINL,v_cardCodeAINL, v_cardNameAINL, v_sboWhsCodeAINL;
			WHILE v_internalKeyAINL IS NOT NULL DO

				DECLARE v_tempQuantityAINL  NUMERIC(19,6);
				DECLARE v_currentInternalKey  INT;
				DECLARE v_currentQuantityAINL  NUMERIC(19,6);

				v_tempQuantityAINL := v_quantityAINL;
				v_currentInternalKey := -1;
				v_currentQuantityAINL := 0;

				SELECT TOP 1 "Quantity" * "QuantityPerUom", "InternalKey" 
				INTO v_currentQuantityAINL, v_currentInternalKey
				FROM TMP_TN_TempDetail AS INV 
				WHERE INV."ItemCode" = v_itemCodeAINL AND "Quantity" > 0 AND ( INV."SboWhsCode" = v_sboWhsCodeAINL OR v_sboWhsCodeAINL IS NULL OR v_sboWhsCodeAINL = '' );

				WHILE v_currentQuantityAINL > 0 AND v_tempQuantityAINL > 0 DO

					DECLARE v_quantityToReduce  NUMERIC(19,6);
					v_quantityToReduce := v_tempQuantityAINL;

					--check if we find a linked locking
					--IF this is the case, we do not need another locking 				
					SELECT COUNT(*) INTO v_cnt FROM TMP_TN_TempDetail AS InvTable
						INNER JOIN POR1 ON 
						    POR1."DocEntry" = InvTable."BaseEntry" AND
						    POR1."LineNum" = InvTable."BaseLine" AND
						    POR1."ObjType" = InvTable."BaseType" 
						INNER JOIN PMX_INLD ON 
						    POR1."BaseEntry" = PMX_INLD."BaseEntry" AND
						    POR1."BaseLine" = PMX_INLD."BaseLine" AND
						    POR1."BaseType" = PMX_INLD."BaseType" 
						WHERE InvTable."InternalKey" = v_currentInternalKey;
					IF v_cnt < 1 THEN

						IF(v_quantityToReduce> v_currentQuantityAINL) THEN
							v_quantityToReduce := v_currentQuantityAINL;
						END IF;

						SELECT COUNT(*) INTO v_cnt FROM PMX_INLD 
						INNER JOIN TMP_TN_TempDetail AS InvTable ON
						    InvTable."ItemCode" = PMX_INLD."ItemCode"
						AND InvTable."ItemTransactionalInfoKey" = PMX_INLD."ItemTransactionalInfoKey"
						AND InvTable."SerialNum" = PMX_INLD."SerialNum"
						AND InvTable."LogUnitIdentKey" = PMX_INLD."LogUnitIdentKey"
						AND InvTable."QualityStatusCode" = PMX_INLD."QualityStatusCode"
						WHERE "CardCode" = v_cardCodeAINL AND PMX_INLD."BaseType" IS NULL AND PMX_INLD."BaseEntry" IS NULL 
						AND PMX_INLD."BaseLine" IS NULL AND PMX_INLD."ItemCode" = v_itemCodeAINL 
						AND InvTable."InternalKey" = v_currentInternalKey;
						IF v_cnt > 0 THEN

							--The entry already exists, so update the data
							UPDATE PMX_INLD SET "Version" = "Version" + 1, PMX_INLD."Quantity" = PMX_INLD."Quantity" + v_quantityToReduce 
							FROM PMX_INLD 
							INNER JOIN TMP_TN_TempDetail AS InvTable ON
							    	InvTable."ItemCode" = PMX_INLD."ItemCode"
								AND InvTable."ItemCode" = PMX_INLD."ItemCode"
								AND InvTable."ItemTransactionalInfoKey" = PMX_INLD."ItemTransactionalInfoKey"
								AND InvTable."SerialNum" = PMX_INLD."SerialNum"
								AND InvTable."LogUnitIdentKey" = PMX_INLD."LogUnitIdentKey"
								AND InvTable."QualityStatusCode" = PMX_INLD."QualityStatusCode"
							WHERE "CardCode" = v_cardCodeAINL AND PMX_INLD."BaseType" IS NULL AND PMX_INLD."BaseEntry" IS NULL 
								AND PMX_INLD."BaseLine" IS NULL AND PMX_INLD."ItemCode" = v_itemCodeAINL 
								AND InvTable."InternalKey" = v_currentInternalKey;
						ELSE
							--Insert into locking
							INSERT INTO PMX_INLD ("InternalKey","BaseType","BaseEntry","BaseLine","CardCode","CardName",
							    "ItemCode","Quantity","ItemTransactionalInfoKey","SerialNum",
							    "UserSign","CreateDate",
							    "CreateTime","LogUnitIdentKey","QualityStatusCode", 
							    "InvLockStatusCode", "InvLockLevel", "PmxWhsCode", "Version") 
							SELECT PMX_INLD_S.nextval, NULL, NULL,NULL,v_cardCodeAINL,v_cardNameAINL,
							    InvTable."ItemCode",v_quantityToReduce,InvTable."ItemTransactionalInfoKey",InvTable."SerialNum",
							    InvTable."CreatedBy",InvTable."CreateDate",
							    InvTable."CreateTime",InvTable."LogUnitIdentKey", InvTable."QualityStatusCode",
							     'RESERVED', 'L' , PMX_OSEL."PmxWhsCode", 0
							FROM TMP_TN_TempDetail AS InvTable
							INNER JOIN PMX_OSEL ON PMX_OSEL."Code" = InvTable."StorLocCode"
							WHERE InvTable."ItemCode" = v_itemCodeAINL
							AND InvTable."InternalKey" = v_currentInternalKey;
						END IF;

						--Remove it from source temp table
						UPDATE TMP_TN_TempDetail SET "Quantity" = "Quantity" - v_quantityToReduce WHERE "InternalKey" = v_currentInternalKey;
						--Update the advance locking table
						UPDATE PMX_AINL SET "Quantity" = "Quantity" - v_quantityToReduce WHERE "InternalKey" = v_internalKeyAINL;

					ELSE
					
						--Delete the current line from the temp table, so next time this is not taken in account
						DELETE FROM TMP_TN_TempDetail WHERE "InternalKey" = v_currentInternalKey;
						
					END IF;

					v_tempQuantityAINL := v_tempQuantityAINL - v_quantityToReduce;

					IF (v_tempQuantityAINL <= 0) THEN
						BREAK;
					END IF;

					v_currentInternalKey := -1;
					v_currentQuantityAINL := 0;

					SELECT TOP 1 "Quantity", "InternalKey" 
					INTO v_currentQuantityAINL, v_currentInternalKey
					FROM TMP_TN_TempDetail AS INV 
					WHERE INV."ItemCode" = v_itemCodeAINL AND "Quantity" > 0 AND ( INV."SboWhsCode" = v_sboWhsCodeAINL OR v_sboWhsCodeAINL IS NULL OR v_sboWhsCodeAINL = '' );
				
				END WHILE;

				FETCH advanceLockingCursor INTO v_itemCodeAINL,v_quantityAINL,v_internalKeyAINL,v_cardCodeAINL, v_cardNameAINL, v_sboWhsCodeAINL;
			END WHILE;
			CLOSE advanceLockingCursor;

END;
END;

			END IF;

			--Remove all not needed lockings
			DELETE FROM PMX_AINL WHERE "Quantity" <= 0;
		
		
		--**********************************************************************************************************
		--Update Produmex inventory tables
		--**********************************************************************************************************

		--Insert inventory detail lines in real table
		INSERT INTO PMX_INVD ( "InternalKey", "TransType","DocEntry","DocNum","DocDate","DocLineNum","BaseType","BaseEntry","BaseLine","CardCode","CardName","ItemCode","ItemName","Quantity","StorLocCode","ItemTransactionalInfoKey","SerialNum","LogUnitIdentKey","CreatedBy","CreateDate","CreateTime",SSCC,"QualityStatusCode","Uom","QuantityPerUom","Uom2","QuantityUom2") 
		SELECT PMX_INVD_S.nextval, "TransType","DocEntry","DocNum","DocDate","DocLineNum","BaseType","BaseEntry","BaseLine","CardCode","CardName","ItemCode","ItemName","Quantity","StorLocCode","ItemTransactionalInfoKey","SerialNum","LogUnitIdentKey","CreatedBy","CreateDate","CreateTime",SSCC,"QualityStatusCode","Uom","QuantityPerUom","Uom2","QuantityUom2"
        FROM TMP_TN_InventoryDetail;

		--Update total inventory table
		BEGIN --TRY
			DECLARE v_itemCode  nvarchar(20);
			DECLARE v_storLocCode  nvarchar(50);
			DECLARE v_serialNum  nvarchar(17);
			DECLARE v_logUnitIdentKey  INT;
			DECLARE v_itemtransactionalInfoKey  INT;
			DECLARE v_quantity  numeric(19, 6);
			DECLARE v_quantityStr NVARCHAR(20);
			DECLARE v_quantityInventoryTotal  numeric(19, 6);
			DECLARE v_quantityUom2InventoryTotal  numeric(19, 6);
			DECLARE v_inventoryTotalId  INT;
			DECLARE v_sscc  nvarchar(18);
			DECLARE v_qualityStatusCode  NVARCHAR(8);
			DECLARE v_uom2  NVARCHAR(20);
			DECLARE v_quantityUom2  numeric(19, 6);
			DECLARE v_uom  NVARCHAR(20);
	
			DECLARE CURSOR invDetailCursor FOR
            SELECT InvTable."ItemCode","StorLocCode","ItemTransactionalInfoKey",InvTable."SerialNum","LogUnitIdentKey", ROUND( "Quantity"*"QuantityPerUom", IFNULL( OITM.U_PMX_UOMD, 6) ) AS "Quantity", SSCC, "QualityStatusCode", "Uom2", "QuantityUom2", "InventoryUom" 
			FROM TMP_TN_InventoryDetail InvTable
			INNER JOIN OITM on OITM."ItemCode" = InvTable."ItemCode";

			OPEN invDetailCursor;
			FETCH invDetailCursor INTO v_itemCode,v_storLocCode,v_itemTransactionalInfoKey,v_serialNum,v_logUnitIdentKey,v_quantity,v_sscc,v_qualityStatusCode, v_uom2, v_quantityUom2, v_uom;
			WHILE v_itemCode IS NOT NULL DO
			
				--Because it is not possible to do column = v_param when v_param is NULL we have to do something special. We can use SET ANSI_NULLS OFF but this will not be supported in the future by Microsoft.
				--for every parameter that can be NULL we have to create a new variable that tells us if the parameter is NULL
				DECLARE v_serialNumIsNull  SMALLINT;
				DECLARE v_logUnitIdentKeyIsNull  SMALLINT;
				DECLARE v_ssccIsNull  SMALLINT;
				DECLARE v_itemTransactionalInfoKeyIsNull  SMALLINT;

				v_ssccIsNull := 0;
				v_itemTransactionalInfoKeyIsNull := 0;

				IF v_serialNum IS NULL THEN v_serialNumIsNull := 1; ELSE v_serialNumIsNull := 0; END IF;
				IF v_logUnitIdentKey IS NULL THEN v_logUnitIdentKeyIsNull := 1; ELSE v_logUnitIdentKeyIsNull := 0; END IF;
				IF v_sscc IS NULL THEN v_ssccIsNull := 1; ELSE v_ssccIsNull := 0; END IF;
				IF v_itemTransactionalInfoKey IS NULL THEN v_itemTransactionalInfoKeyIsNull := 1; ELSE v_itemTransactionalInfoKeyIsNull := 0; END IF;

				--Get the summary line 
				v_inventoryTotalId := NULL;
				v_quantityInventoryTotal := 0;
				v_quantityUom2InventoryTotal := 0;
				
				BEGIN
                    DECLARE CURSOR cur FOR				
				    SELECT "InternalKey", "Quantity", "QuantityUom2"
				    FROM PMX_INVT
				    WHERE "ItemCode" = :v_itemCode AND 
					    "StorLocCode" = :v_storLocCode AND 
					    ((:v_serialNumIsNull=1 AND "SerialNum" IS NULL) OR (:v_serialNumIsNull=0 AND "SerialNum"=:v_serialNum)) AND 
					    ((:v_logUnitIdentKeyIsNull=1 AND "LogUnitIdentKey" IS NULL) OR (:v_logUnitIdentKeyIsNull=0 AND "LogUnitIdentKey"=:v_logUnitIdentKey)) AND
					    ((:v_itemTransactionalInfoKeyIsNull=1 AND "ItemTransactionalInfoKey" IS NULL) OR (:v_itemTransactionalInfoKeyIsNull=0 AND "ItemTransactionalInfoKey"=:v_itemTransactionalInfoKey)) AND
					    "QualityStatusCode"=:v_qualityStatusCode
				    FOR UPDATE;
				    OPEN cur;
    				FETCH cur INTO v_inventoryTotalId, v_quantityInventoryTotal, v_quantityUom2InventoryTotal;
    				CLOSE cur;
				END;

				--Check if found an inventory total line		
				IF v_inventoryTotalId IS NULL THEN

					SELECT COUNT(*) INTO v_cnt FROM PMX_OSWH WHERE "StorLocLostAndFound" = v_storLocCode;
					IF v_cnt < 1 THEN
						--Only check for quantity lower than 0 if it is not the lost and found location
						IF (v_quantity < 0) THEN
							CLOSE invDetailCursor ;

							IF v_quantity IS NULL THEN	v_quantityStr := '(NULL)';
							ELSE						v_quantityStr := '' || v_quantity;
							END IF;

							p_error := 60000;
							p_error_message := SUBSTRING('The quantity ' || v_quantityStr || ' goes below zero for item=' || IFNULL( v_itemCode,'') || ' location=' || IFNULL( v_storLocCode,'') || ' batchID=' || IFNULL( CAST(v_itemTransactionalInfoKey AS VARCHAR),'') || ' "SerialNum"=' || IFNULL( v_serialNum,'') || ' LUID=' || IFNULL( CAST(v_logUnitIdentKey AS VARCHAR),'') || ' quality=' || IFNULL( v_qualityStatusCode,'') || ' . This is not allowed.',1,200);
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;
					END IF;

					--Logging
	--				SET v_MSG =  SUBSTRING('No inventory total table found for item=' || IFNULL( v_itemCode,'') || ' storageLocation='  || IFNULL( v_storLocCode,'') || ' bestBeforeDate=' || IFNULL( CAST(v_bestBeforeDate AS VARCHAR),'') || ' batchNumber=' || IFNULL( v_batchNumber,'') || ' "SerialNum"=' || IFNULL( v_serialNum,'') || ' logisticUnitIdentificationKey=' || IFNULL( CAST(v_logUnitIdentKey AS VARCHAR),'') || ' "QualityStatusCode"=' || IFNULL( v_qualityStatusCode,''),1,255)
	--				--EXEC xp_logevent 999999, v_MSG, INFORMATIONAL
					INSERT INTO PMX_INVT ( "InternalKey", "ItemCode","StorLocCode","ItemTransactionalInfoKey","SerialNum","LogUnitIdentKey","Quantity", SSCC, "QualityStatusCode", "Uom2", "QuantityUom2", "Uom")
					VALUES (PMX_INVT_S.nextval, v_itemCode,v_storLocCode,v_itemTransactionalInfoKey,v_serialNum,v_logUnitIdentKey,v_quantity, v_sscc, v_qualityStatusCode, v_uom2, v_quantityUom2, v_uom);
				
				ELSE
				
					SELECT COUNT(*) INTO v_cnt FROM PMX_OSWH WHERE "StorLocLostAndFound" = v_storLocCode;
					IF v_cnt < 1 THEN
						IF (v_quantityInventoryTotal + v_quantity) < 0 THEN
							CLOSE invDetailCursor;
							
							IF v_quantity IS NULL THEN	v_quantityStr := '(NULL)';
							ELSE						v_quantityStr := '' || v_quantity;
							END IF;

							p_error := 60001;
							p_error_message := SUBSTRING('The quantity ' || v_quantityStr || ' goes below zero for item=' || IFNULL( v_itemCode,'') || ' storageLocation=' || IFNULL( v_storLocCode,'') || ' "ItemTransactionalInfoKey"=' || IFNULL( CAST(v_itemTransactionalInfoKey AS VARCHAR),'') || ' "SerialNum"=' || IFNULL( v_serialNum,'') || ' logisticUnitIdentificationKey=' || IFNULL( CAST(v_logUnitIdentKey AS VARCHAR),'') || ' "QualityStatusCode"=' || IFNULL( v_qualityStatusCode,'') || ' . This is not possible.',1,200);
							SELECT p_error, p_error_message FROM dummy;
							--EXEC xp_logevent p_error, p_error_message, ERROR
							RETURN;
						END IF;
					END IF;
					--Logging
	--				SET v_MSG = SUBSTRING('Inventory total table with key=' || IFNULL( CAST(v_inventoryTotalID AS VARCHAR),'') || ' found for item=' || IFNULL( v_itemCode,'') || ' storageLocation='  || IFNULL( v_storLocCode,'') || ' bestBeforeDate=' || IFNULL( CAST(v_bestBeforeDate AS VARCHAR),'') || ' batchNumber=' || IFNULL( v_batchNumber,'') || ' "SerialNum"=' || IFNULL( v_serialNum,'') || ' logisticUnitIdentificationKey=' || IFNULL( CAST(v_logUnitIdentKey AS VARCHAR),'') || ' "QualityStatusCode"=' || IFNULL( v_qualityStatusCode,''),1,255)
	--				--EXEC xp_logevent 999999, v_MSG, INFORMATIONAL

					UPDATE PMX_INVT SET "Quantity" = "Quantity" + v_quantity, "QuantityUom2" = "QuantityUom2" + v_quantityUom2
                    WHERE "InternalKey" = v_inventoryTotalId;
				END IF;
				
				FETCH invDetailCursor INTO v_itemCode, v_storLocCode,v_itemTransactionalInfoKey,v_serialNum,v_logUnitIdentKey,v_quantity,v_sscc,v_qualityStatusCode, v_uom2, v_quantityUom2, v_uom;
			END WHILE;
			CLOSE invDetailCursor;
			
			
			--Update catch weight variances
			BEGIN -- TRY

				DECLARE v_catchWeightBatchVarianceId int;
				DECLARE v_quantityCatchWeightBatchVariance numeric(19, 6);

				DECLARE CURSOR catchWeightVarianceCursor FOR 
					SELECT PMX_INVT."ItemCode",PMX_INVT."ItemTransactionalInfoKey",PMX_INVT."QuantityUom2", PMX_INVT."Uom2"
					FROM PMX_INVT
					LEFT JOIN PMX_CWBV ON PMX_CWBV."ItemCode" = PMX_INVT."ItemCode"
						AND IFNULL( PMX_CWBV."ItriKey", 0) = IFNULL( PMX_INVT."ItemTransactionalInfoKey", 0)
					--INNER JOIN TMP_TN_InventoryDetail InvTable ON InvTable.ItemCode = PMX_INVT.ItemCode
					--AND IFNULL( PMX_INVT."ItemTransactionalInfoKey", 0) = IFNULL( InvTable."ItemTransactionalInfoKey", 0)
					WHERE PMX_INVT."Quantity" = 0 AND PMX_INVT."QuantityUom2" IS NOT NULL;
					--AND PMX_CWBV."ItemCode" IS NOT NULL;

				OPEN catchWeightVarianceCursor;
				FETCH catchWeightVarianceCursor INTO v_itemCode,v_itemTransactionalInfoKey,v_quantity, v_uom;
				WHILE v_itemCode IS NOT NULL DO

					SELECT COUNT(1) 
					INTO v_cnt
					FROM PMX_CWBV
					WHERE  "ItemCode" = v_itemCode AND 
							IFNULL( "ItriKey", 0) = IFNULL( v_itemTransactionalInfoKey, 0);
							
					
					
					--Check if found an inventory total line		
					IF v_cnt = 0 THEN
						INSERT INTO PMX_CWBV ("InternalKey","ItemCode","ItriKey","Quantity","Uom") 
						VALUES (PMX_CWBV_S.nextval,v_itemCode, v_itemTransactionalInfoKey, v_quantity, v_uom);
					ELSE
							
						--Get the summary line 
						v_catchWeightBatchVarianceId := NULL;
						v_quantityCatchWeightBatchVariance := 0;
						SELECT "InternalKey", "Quantity" 
						INTO v_catchWeightBatchVarianceId, v_quantityCatchWeightBatchVariance
						FROM PMX_CWBV
						WHERE  "ItemCode" = v_itemCode AND 
								IFNULL( "ItriKey", 0) = IFNULL( v_itemTransactionalInfoKey, 0)
								FOR UPDATE;
					
						UPDATE PMX_CWBV SET "Quantity" = "Quantity" + v_quantity 
                        WHERE "InternalKey" = v_catchWeightBatchVarianceId;
					END IF;
					
					FETCH catchWeightVarianceCursor INTO v_itemCode, v_itemTransactionalInfoKey, v_quantity, v_uom;
				END WHILE;
				CLOSE catchWeightVarianceCursor;

			END; /* TRY
			BEGIN CATCH
				p_error := 60003
				p_error_message := 'Unhandled error update Produmex catch weight variance: Number:' || CAST(ERROR_NUMBER() AS NCLOB) || ' Message: ' || ERROR_MESSAGE()
				BEGIN TRY
					CLOSE catchWeightVarianceCursor 
					DEALLOCATE catchWeightVarianceCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END TRY
				BEGIN CATCH
					DEALLOCATE catchWeightVarianceCursor
					SELECT p_error, p_error_message
					--EXEC xp_logevent p_error, p_error_message, ERROR
					RETURN
				END CATCH
			END CATCH;*/

			DELETE FROM PMX_INVT WHERE "Quantity" = 0;

		END; /* TRY
		BEGIN CATCH
			p_error := 60002
			p_error_message := 'Unhandled error update Produmex inventory tables: Number:' || CAST(ERROR_NUMBER() AS NCLOB) || ' Message: ' || ERROR_MESSAGE()
			BEGIN TRY
				CLOSE invDetailCursor 
				DEALLOCATE invDetailCursor
				SELECT p_error, p_error_message
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN
			END TRY
			BEGIN CATCH
				DEALLOCATE invDetailCursor
				SELECT p_error, p_error_message
				--EXEC xp_logevent p_error, p_error_message, ERROR
				RETURN
			END CATCH
		END CATCH; */

		--**********************************************************************************************************
		--Do general checks
		--**********************************************************************************************************
		
		--Check if all Logistic Units are stored on the same storage location
		--It is not possible that a logistic unit is stored on more then 1 location.
		DELETE FROM TMP_TN_CharListTableNoIndex;
		INSERT INTO TMP_TN_CharListTableNoIndex
		SELECT CAST("LogUnitIdentKey" AS VARCHAR)
		FROM PMX_INVENTORY_LUID_STORLOC
		GROUP BY "LogUnitIdentKey"
		HAVING (count("LogUnitIdentKey") > 1);
		SELECT str INTO v_errorList FROM "WUK_4004_PMX"."PMX_FN_TableToCharListNoIndex"( "WUK_4004".TMP_TN_CharListTableNoIndex, ' ' );
		IF LENGTH(v_errorList) <> 0 THEN
			p_error := 70000;
			p_error_message := 'The logistic unit(s) ' || v_errorList || ' is/are stored on several storage locations. This is not possible. A logistic unit can only be stored on one storage location.';
			SELECT p_error, p_error_message FROM dummy;
			--EXEC xp_logevent p_error, p_error_message, ERROR
			RETURN;
		END IF;


	END; /* TRY
	BEGIN CATCH
		p_error := 60003
		p_error_message := 'PMX_SP: sql error ' || CAST(ERROR_NUMBER() AS NVARCHAR) || ' : ' || ERROR_MESSAGE()
		   || IFNULL( ' line ' || CAST(ERROR_LINE() AS NVARCHAR), '' ) || IFNULL( ' in ' || ERROR_PROCEDURE(), '' )
		SELECT p_error, p_error_message --> done at the end of SBO_SP_...
		--EXEC xp_logevent p_error, p_error_message, ERROR
	END CATCH */

END

----Following code must be added to the stored procedure SBO_SP_TransactionNotification
----to execute the PMX stored procedure:

----**********************************************************************************************************
----Start executing Produmex Logex Addon code
----**********************************************************************************************************
--IF error = 0 THEN
--	BEGIN
--		DECLARE EXIT HANDLER FOR SQLEXCEPTION
--		BEGIN
--			error := ::SQL_ERROR_CODE;
--			error_message := SUBSTRING( 'PMX_SP(' || object_type
--							|| ',' || transaction_type 
--							|| ',' || list_of_key_cols_tab_del
--							|| ',' || list_of_cols_val_tab_del || ')'
--							|| CHAR(13) || CHAR(10)
--							|| ::SQL_ERROR_MESSAGE, 1, 200 );
--		END;
--		CALL "PMX_SP_TransactionNotification" (
--		  object_type,
--		  transaction_type,
--		  num_of_cols_in_key,
--		  list_of_key_cols_tab_del,
--		  list_of_cols_val_tab_del,
--		  error,
--		  error_message );
--	END;
--END IF;
------**********************************************************************************************************
------End executing Produmex Logex Addon code
------**********************************************************************************************************